/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/security/NdsServiceIdentity.java $
  $Revision: 7 $
  $Modtime: 10/18/99 11:28a $

  Copyright (c) 1997-1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL PROPRIETARY
  AND TRADE SECRET INFORMATION OF NOVELL, INC. ACCESS  TO  THIS  WORK IS
  RESTRICTED TO (I) NOVELL, INC.  EMPLOYEES WHO HAVE A NEED TO  KNOW HOW
  TO  PERFORM  TASKS WITHIN  THE SCOPE  OF  THEIR   ASSIGNMENTS AND (II)
  ENTITIES OTHER  THAN  NOVELL, INC.  WHO  HAVE ENTERED INTO APPROPRIATE
  LICENSE   AGREEMENTS.  NO  PART  OF  THIS WORK MAY BE USED, PRACTICED,
  PERFORMED COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED, ABRIDGED,
  CONDENSED, EXPANDED, COLLECTED, COMPILED, LINKED,  RECAST, TRANSFORMED
  OR ADAPTED  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC.  ANY USE
  OR EXPLOITATION  OF  THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE
  PERPETRATOR  TO CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.security;

import com.novell.java.security.Identity;
import com.novell.java.security.IdentityScope;
import com.novell.java.security.KeyManagementException;
import java.util.StringTokenizer;

/**@internal
 * Represents an NDS identity for a Service (Process executing without necessarily 
 * requiring user intervention) that is integrated with NDS for storing configuration 
 * information etc. This is a subclass of NdsIdentity, which would include 
 * attributes specific to a Service.
 *
 */
 
public  
class NdsServiceIdentity extends NdsIdentity {

    private static String DOT = ".";
    
    /**
     * Constructs an NdsServiceIdentity with the passed-in name and scope.
     *
     * @param name  The name of the identity.  
     * @param scope The scope of the identity.
     *
     * @exception KeyManagementException Thrown if there is already
     *            an identity with the same name in the scope.
     */
    public NdsServiceIdentity(String name, NdsIdentityScope scope) throws
    KeyManagementException {
	   super(name, scope);
    }

    /**
     * Constructs an NdsServiceIdentity with the passed-in name but no scope.
     *
     * @param name The name of the identity.
     */
    public NdsServiceIdentity(String name) {
   	super(name);
    }


   /**
	 * Construct an NdsServiceIdentity with the specified name and scope
	 * passed-in as strings.
	 *
	 * @param name The name of the identity.
	 * @param scope The scope of the identity.
	 *
	 * @exception com.novell.java.security.KeyManagementException
	 * Thrown when an IdentityScope cannot be constructed due to
	 * key management issues.
	 */
   public NdsServiceIdentity(String name, String scope)
      throws KeyManagementException
   {
      super(name, new NdsIdentityScope(scope));
   }


   /**
	 * Construct an NdsServiceIdentity from an identity.
	 *
	 * @param identity The identity to copy
	 *
	 * @exception com.novell.java.security.KeyManagementException
	 * Thrown when an IdentityScope cannot be constructed due to
	 * key management issues.
	 */
	public NdsServiceIdentity(Identity identity)
		throws KeyManagementException
	{
		super(identity.getName(), new NdsIdentityScope(identity.getScope()));
	}


   /*
	 * Parses an identity to create a user domain string.
	 *
	 * @param identity            The identity whose domain is to be parsed.
	 * @return                    The user domain String.
	 * 
	 * NOTE: getUserDomain() provides same functionality 
   public String getNDSTreeName()
   {
      // Traverse the NdsIdentityScope list
      IdentityScope scope = this.getScope();
      IdentityScope nextScope = (scope == null) ? null : scope.getScope();
      
      while ((nextScope != null) && 
             (!nextScope.getName().equalsIgnoreCase(
                     NdsIdentityScope.ADMINISTRATIVE_DOMAIN_NAME))) {
         scope = nextScope;
         nextScope = scope.getScope();
      }

      if (scope.getName().equalsIgnoreCase(
               NdsIdentityScope.ADMINISTRATIVE_DOMAIN_NAME))
         return null;
      else 
         return scope.getName();
      
   } // getNDSTreeName ()
	 */


   /**
    * Parses a string consisting of a full dn of a service, written in 
    * root-right form, using DOT as the RDN separator. The Tree name must be 
    * the right-most entry. This method is provided as a convenient way 
    * of constructing an NdsServiceIdentity from a string.
    * 
    * <p>for example: To construct a service identity with a distinguished
    * name : agent007.managewise.services.novell in the tree named 
    * novell_inc, there are two alternative ways of instantiating an
    * NdsServiceIdentity.
    * <p> The first one is to manually build all the scope objects, as follows:
    * 
    * <pre>
    * IdentityScope administrativeDomain  = new NdsIdentityScope();
    * IdentityScope tree                  = new NdsIdentityScope("novell_inc",administrativeDomain);
    * IdentityScope context               = new NdsIdentityScope("services",tree);
    * IdentityScope subcontext            = new NdsIdentityScope("managewise",context);
    * Identity      service               = new NdsServiceIdentity("agent007",subcontext);
    * </pre>
    *
    * <p>The alternative is:
    * <pre>
    * Identity service  = NdsServiceIdentity,parse("agent007.managewise.services.novell.novell_inc");
    * </pre>
    * 
    * @param dn      The distinguished name of the NDS Service Object
    * @return        An NdsServiceIdentity object.
    */
   public static NdsServiceIdentity parse(String dn)
		throws KeyManagementException
   {
      StringTokenizer st = new StringTokenizer(dn, DOT);

      // This is a non-recursive implementation
      // Get all the tokens, so that we can traverse them backwards
      int tokenCount = st.countTokens();
      if (tokenCount == 0) 
         return null;
         
      String tokens[] = new String[tokenCount];
      for (int i=0; i<tokenCount; i++) {
         tokens[i] = st.nextToken();
      }

      // Traverse the tokens backwards, and build the objects
      NdsIdentityScope scope = new NdsIdentityScope();   // The administrative domain
      for (int i=tokenCount-1; i>0; i--)     {
         scope = new NdsIdentityScope(tokens[i], scope);
      }
      NdsServiceIdentity id = new NdsServiceIdentity(tokens[0], scope);
         
      return id;      
   }
      
} // class NdsServiceIdentity

