/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/session/Session.java $
  $Revision: 5 $
  $Modtime: 1/10/02 3:20p $

  Copyright (c) 1997 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/
package com.novell.service.session;

import java.rmi.*;
import java.rmi.server.UID;
import java.util.*;

import com.novell.java.security.*;

/**
 * Provides a communication channel between the owning/initiating
 * object that creates a session and the target domain (tree, server,
 * or any other object). It provides those methods that are common to
 * all Sessions. All objects returned by the SessionManager are of
 * type Session.
 *
 * @see SessionManager
 */
public interface Session extends Authenticatable
{
   /**
    * Attribute ID describing the provider of a Session.
    *
    * <p>(PROVIDER_NAME_ATTR_ID = "SESSION_PROVIDER")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String PROVIDER_NAME_ATTR_ID = "SESSION_PROVIDER";

   /**
    * Attribute ID describing the state of a Session.
    *
    * <p>(SESSION_STATE_ATTR_ID = "SESSION_STATE")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String SESSION_STATE_ATTR_ID = "SESSION_STATE";

   /**
    * Attribute ID describing the domain name of a Session.
    *
    * <p>(DOMAIN_NAME_ATTR_ID = "DOMAIN_NAME")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String DOMAIN_NAME_ATTR_ID = "DOMAIN_NAME";

   /**
    * Attribute ID describing the type of a Session. It
    * uniquely identifies sessions at a certain level (INITIAL_SESSION
    * for all initial sessions, TREE and SERVER types under the NDS
    * provider, and SERVER under the Bindery provider.
    *
    * <p>(SESSION_TYPE_ATTR_ID = "SESSION_TYPE")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    * @see  #MANAGER_SESSION_TYPE
    * @see  #INITIAL_SESSION_TYPE
    */
   public static final String SESSION_TYPE_ATTR_ID = "SESSION_TYPE";

   /**
    * Attribute ID describing the "realness" of a Session.
    * The associated value is a Boolean. Real sessions have real
    * endpoints (servers, trees, and so forth), and are those
    * sessions whose domain name is non-deterministic.
    *
    * <p>(IS_REAL_ATTR_ID =  "IS_REAL")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String IS_REAL_ATTR_ID = "IS_REAL";

   /**
    * Attribute ID describing the unique ID of a Session.
    *
    * <p>(UID_ATTR_ID = "UID")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String UID_ATTR_ID = "UID";

   /**@deprecated Use MANAGER_SESSION_TYPE
    * <p>Attribute value returned for SESSION_TYPE_ATTR_ID for SessionManager
    * Sessions.
    *
    * <p>(SESSION_MANAGER_TYPE = "SESSION_MANAGER")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    */
   public static final String SESSION_MANAGER_TYPE = "SESSION_MANAGER";

   /**
    * Attribute value returned for SESSION_TYPE_ATTR_ID in SessionManager
    * Sessions.
    *
    * <p>(MANAGER_SESSION_TYPE = "SESSION_MANAGER")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    * @see  #SESSION_TYPE_ATTR_ID
    */
   public static final String MANAGER_SESSION_TYPE = "SESSION_MANAGER";

   /**
    * Attribute value returned for SESSION_TYPE_ATTR_ID in initial Sessions.
    *
    * <p>(INITIAL_SESSION_TYPE = "INITIAL_SESSION")
    *
    * @see  SessionAttr
    * @see  #getAttributes
    * @see  #SESSION_TYPE_ATTR_ID
    */
   public static final String INITIAL_SESSION_TYPE = "INITIAL_SESSION";

   /**
    * Closes and invalidates this session and its children, removing all
    * stored credentials. With the exception of SessionManager and
    * InitialSessions, the session state will change to InvalidSessionState.
    * This is a terminal state that cannot be changed back into another
    * state. This effectively invalidates the session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public void close()
   throws SessionException, RemoteException;

   /**
    * Adds a child session to the parent based on the session from
    * which this operation is called. The parent session can be a
    * Session or SessionManager.
    *
    * @param domainName The domain name of the session to add. The
    *                   domain name depends on the provider.
    *
    * @return Session The child session added.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getSession(java.lang.String SessionEnv)
    * @see #getSessionTop
    * @see com.novell.service.session.xplat.Xplat#LOOKUP_DOMAIN
    * @see com.novell.service.session.xplat.Xplat#DOMAIN_ADDRESS
    */
   public Session getSession
   (
      String domainName
   )
   throws SessionException, RemoteException;

   /**
    * Adds a child session to the parent based on the session from
    * which this operation is called. It uses environment to control
    * behavior when applicable.
    *
    * @param domainName  The domain name of the session to add. The
    *                    domain name depends on the provider.
    * @param environment The environment passed-in to control the
    *                    the creation behavior of this session.
    *
    * @return Session The child session added.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getSession(java.lang.String)
    * @see #getSessionTop
    * @see SessionEnv
    */
   public Session getSession
   (
      String domainName,
      SessionEnv environment
   )
   throws SessionException, RemoteException;

   /**
    * Creates a session with no consideration of the session from
    * which this operation was called.
    *
    * @param domainName The domain name of the session to add.
    *
    * @return Session The session created.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getSession
    */
   public Session getSessionTop
   (
      String domainName
   )
   throws SessionException, RemoteException;

   /**
    * Creates a session with no consideration of the session from
    * which this operation was called. It uses environment to
    * control behavior when applicable.
    *
    * @param domainName  The domain name of the session to add.
    * @param environment The environment passed to control the creation
    *                    behavior of this session.
    *
    * @returns Session The Session created.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getSession
    * @see SessionEnv
    */
   public Session getSessionTop
   (
      String domainName,
      SessionEnv environment
   )
   throws SessionException, RemoteException;

   /**
    * Returns a selected default subset of all attributes for this
    * session. Operational attributes will not be returned with
    * this method unless the FORCE_OPERATIONAL_ATTRIBUTES variable
    * value is set to TRUE. You should be aware that session attribute
    * IDs are provider specific.
    *
    * @return SessionAttrs The attributes of this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getAttributes(java.lang.String[])
    * @see com.novell.service.session.xplat.Xplat#FORCE_OPERATIONAL_ATTRIBUTES
    */
   public SessionAttrs getAttributes
   (
   )
   throws SessionException, RemoteException;

   /**
    * Returns the attributes for this session listed in the attrIds[]
    * parameter. All attributes, including operational attributes,
    * specified in the parameter will be returned. You should be aware
    * that session attribute IDs are provider specific.
    *
    * @param attrIds An array of session attribute IDs.
    *
    * @return SessionAttrs The attributes of this Session that match
    *                      the passed in IDs.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getAttributes()
    * @see com.novell.service.session.Authenticatable
    * @see com.novell.service.session.bindery.Bindery
    * @see com.novell.service.session.nds.NDS
    * @see com.novell.service.session.xplat.Xplat
    */
   public SessionAttrs getAttributes
   (
      String attrIds[]
   )
   throws SessionException, RemoteException;

   /**
    * Returns the environment for this session.
    *
    * @return SessionEnv The environment for this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @return SessionEnv The environment for this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getEffectiveEnvironment
    * @see #setEnvironment
    * @see SessionEnv
    */
   public SessionEnv getEnvironment
   (
   )
   throws SessionException, RemoteException;

   /**
    * Returns the environment affecting this session.
    *
    * <p>The effective environment includes the environment of all parent
    * sessions above this session, in addition to this session's environment.
    * SessionEnv entries in this session override duplicates in upper
    * sessions.
    *
    * @return SessionEnv The environment affecting this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getEnvironment
    * @see SessionEnv
    */
   public SessionEnv getEffectiveEnvironment
   (
   )
   throws SessionException, RemoteException;

   /**
    * Sets/replaces the environment for this session and returns
    * the old environment. The environment may be empty or NULL.
    *
    * @param environment The environment passed-in to control the
    *                    behavior ofthis Session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    *
    * @see #getEnvironment
    * @see SessionEnv
    */
   public SessionEnv setEnvironment
   (
      SessionEnv environment
   )
   throws SessionException, RemoteException;

   /**
    * Searches for sessions with matching attributes. All string
    * attribute values specified in a SessionAttrs instance when
    * calling search() should be case-exact.
    *
    * <p>Interesting sessions are typically those sessions other than
    * initial sessions and the session manager (i.e. those sessions
    * that can be added or closed and those that have IS_REAL_ATTR_ID
    * = TRUE).
    *
    * @param matchingSessionAttrSet Attributes to match. An empty
    *                               SessionAttrs will return all Sessions.
    *
    * @return SessionEnumerator The Session with matching attributes.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public SessionEnumerator search
   (
      SessionAttrs matchingSessionAttrSet
   )
   throws SessionException, RemoteException;

   /**
    * Returns the domain name of this session.
    *
    * @return String Containing the domain name of this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects.
    */
   public String getDomainName
   (
   )
   throws SessionException, RemoteException;

   /**
    * Returns an enumeration of child sessions of this session.
    *
    * @return SessionEnumerator An enumeration of child sessions.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects.
    */
   public SessionEnumerator getChildren
   (
   )
   throws SessionException, RemoteException;

   /**
    * Returns this session's parent session, or NULL if its parent
    * Session doesn't exist.
    */
   public Session getParent() throws RemoteException;

   /**
    * Ivalidate this session and all children.
    * <p>Note: Because a user of the session may still have a reference
    * to it, they may still attempt to call a method on that Session. Upon
    * execution of invalidate(), the session's state is changed to invalid
    * and most subsequent method calls on that session will throw
    * InvalidSessionException.
    *
    * @param rootException The reason (if non null) to invalidate
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public void invalidate
   (
      Throwable rootException
   )
   throws SessionException, RemoteException;

   /**
    * Validate the children of this session.
    *
    * <p>A session implementation may sit on top of an existing client that
    * has other entry points besides through a single JVM. This operation
    * allows a user of the session to make sure data in the session is
    * up-to-date.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public void validateLinks()
   throws SessionException, RemoteException;

   /**
    * Returns a service extension for this session.
    *
    * Typically, sessions are only used for gaining access to a set of
    * services. Those services are made available via the SessionService
    * returned by this method.
    *
    * @param serviceKey  The key defined by the provider for accessing a
    *                    particular service.
    *
    * @return SessionService The service extensions for this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public SessionService getService
   (
      String serviceKey
   )
   throws SessionException, RemoteException;

   /**
    * Return the unique ID for this session.
    *
    * @return The unique ID for this session.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public UID getUID() throws SessionException, RemoteException;

   /**
    * Determines if this session has a parent.
    *
    * @return A boolean set to TRUE if this session has a parent,
    *         otherwise set to FALSE.
    */
   public boolean hasParent() throws RemoteException;

   /**
    * Determines if this session has children.
    *
    * @return A boolean set to TRUE if this session has children,
    *         otherwise set to FALSE.
    */
   public boolean hasChildren() throws RemoteException;

   /**
    * Returns the first session found with a  domain name matching the
    * passed-in domaine parameter value.
    *
    * @return The Session with a matching domain name.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public Session findSession
   (
      String domainName
   )
   throws SessionException, RemoteException;

   /**
    * Returns the first session found with a matching domain name, starting
    * at the session manager.
    *
    * @return The first session found with a matching domain name.
    *
    * @exception SessionException Top level exception thrown by Session
    *            objects or a subclass thereof.
    */
   public Session findSessionTop
   (
      String domainName
   )
   throws SessionException, RemoteException;

   /**
    * Determines if this session is in a valid state.
    *
    * @return A boolean set to TRUE if this session is valid,
    *         otherwise set to FALSE.
    */
   public boolean isValid() throws RemoteException;
}

