/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/session/spi/SessionManagerState.java $
  $Revision: 28 $
  $Modtime: 11/14/00 10:23a $

  Copyright (c) 1997 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/
package com.novell.service.session.spi;

import java.rmi.RemoteException;
import java.util.StringTokenizer;
import java.util.Vector;
import com.novell.service.session.*;
import com.novell.service.session.util.Debug;

/** @internal
 * Top-level Session.
 *
 * Allows SessionManager to be treated like a Session.
 */
public class SessionManagerState extends SessionState
{
   // Always access debug via final static...hopefully a final static of
   // false will be optimized out by the compiler
   final static private boolean DEBUG = false;
   final static private boolean I_DEBUG = false; // Ignore exception
   final static private boolean S_DEBUG = false; // May have side effects
   final static private boolean FAIL_DEBUG = false; // when eval fails

   public SessionManagerState
   (
      SessionManager attachedSession
   )
   throws SessionException, RemoteException
   {
      super((SessionImpl)attachedSession);

      // We're ready to be used!
      this.attachedSession.setState(this);
   }

   /**
    * Inherited from Session
    *
    * @see Session#getSession
    */
   public Session getSession
   (
      String domainName,
      SessionEnv environment
   )
   throws SessionException, RemoteException
   {
      Session is, session = null;

      session = findSession(domainName);
      if (null == session)
      {
         if (FAIL_DEBUG)
         {
            Debug.printlnOnly("Failed as existing session: '" +
               domainName + "'");
         }
         Vector exceptions = new Vector();
          // Children are initial sessions...
          // pass operation on to each until success.
         SessionEnumerator initialSessionEnumeration = getChildren();

         while (null == session && initialSessionEnumeration.hasMoreElements())
         {
            is = initialSessionEnumeration.next();
            try
            {
               session = is.getSession(
                  domainName,
                  environment);
            }
            catch (Exception e)
            {
               exceptions.addElement(e);
                // Do nothing...try next initial session
                // NOTE: May need to fail on first exception...?
               if (I_DEBUG || FAIL_DEBUG)
               {
                  Debug.printlnOnly("Failed under provider (" +
                     session.getDomainName()
                     + "):  '" +
                     domainName + "'");
               }
               if (I_DEBUG)
               {
                  Debug.ignoreException(e);
               }
            }
         }
         if (null == session)
         {
            InvalidDomainNameException t =
               new InvalidDomainNameException(domainName);
            t.setRootCauses(exceptions);
            if (DEBUG)
            {
               Debug.dumpException(t);
            }
            throw t;
         }
      }
      if (DEBUG)
      {
         Debug.println("SMS.getSession() = " + session);
      }
      return session;
   }

   /**
    * Inherited from Session
    *
    * @see Session#getSessionTop
    */
   public Session getSessionTop
   (
      String domainName,
      SessionEnv environment
   )
   throws SessionException, RemoteException
   {
       // For SessionManagerState, this operation is the same as getSession().
       // Since SessionManager doesn't have children it can add, it must defer
       // the operation to each InitialSession.
      return this.getSession(domainName, environment);
   }

   /**
    * Inherited from Session
    *
    * @see Session#getAttributes()
    */
   public SessionAttrs getAttributes
   (
   )
   throws SessionException, RemoteException
   {
      // Get inherited attributes
      SessionAttrs attributes = super.getAttributes();

      // Override with this session's
      String attrIds[] =
      {
         Session.SESSION_TYPE_ATTR_ID,
         Session.IS_REAL_ATTR_ID,
         SessionManager.SCOPE_ATTR_ID
      };
      attributes.modify(
         getStateAttrs(attrIds));

      return attributes;
   }

   /**
    * Inherited from Session
    *
    * @see Session#getAttributes(String[])
    */
   public SessionAttrs getAttributes
   (
      String attrIds[]
   )
   throws SessionException, RemoteException
   {
/*
      // Get inherited attributes
      SessionAttrs attributes = super.getAttributes(attrIds);

      // Override with this session's
      attributes.modify(getStateAttrs(attrIds));
*/
      // Get this state's attributes
      SessionAttrs attributes = getStateAttrs(attrIds);

      if (attributes.count() != attrIds.length)
      {
         // Get inherited attributes
         attributes.merge(
            super.getAttributes(attrIds));
      }

      return attributes;
   }

   private SessionAttrs getStateAttrs
   (
      String attrIds[]
   )
   throws SessionException
   {
      // Get inherited attributes
      SessionAttrs attributes = new SessionAttrs();

      // Override with this session's
      for (int i = 0; i < attrIds.length; i++)
      {
         if (Session.SESSION_TYPE_ATTR_ID.equals(attrIds[i]))
         {
            attributes.modify(
               Session.SESSION_TYPE_ATTR_ID,
               Session.MANAGER_SESSION_TYPE);
         }
         if (Session.IS_REAL_ATTR_ID.equals(attrIds[i]))
            attributes.add(
               Session.IS_REAL_ATTR_ID,
               new Boolean(false));
         if (SessionManager.SCOPE_ATTR_ID.equals(attrIds[i]))
            attributes.add(
               SessionManager.SCOPE_ATTR_ID,
               new Integer(((SessionManagerImpl)attachedSession).getScope()));
      }
      return attributes;
   }

   protected void validateChildren
   (
   )
   throws SessionException, RemoteException
   {
      if (!getChildren().hasMoreElements())
      {
         String isfList = (String)getEffectiveEnvironment().
            get(SessionEnv.INITIAL_SESSION_FACTORY);

         if (null == isfList)
            isfList =
            "com.novell.service.session.nds.NDSInitialSessionFactory:" +
            "com.novell.service.session.bindery.BinderyInitialSessionFactory";


         StringTokenizer isfParser = new StringTokenizer(isfList, ":");
         while (isfParser.hasMoreTokens())
         {
            InitialSessionFactory isf = null;
            try
            {
               String name = isfParser.nextToken();
               if (DEBUG)
               {
                  Debug.println("SMS.validateChildren(): Instantiating: " + name);
               }
               Class isfClass = Class.forName(name);
               isf = (InitialSessionFactory)isfClass.newInstance();
               Session initialSession = isf.getInitialSession(
                  (SessionManager)attachedSession);
               attachedSession.addChild((SessionImpl)initialSession);
               if (DEBUG)
               {
                  Debug.println(
                     "SMS.validateChildren(): Added: " +
                     initialSession.getClass().
                        getName());
               }
            }
            catch (Exception e)
            {
               throw new SessionException(e);
            }
         }
      }
   }

   /**
    * Inherited from Session
    */
   public Session findSessionTop(String domainName)
   throws SessionException, RemoteException
   {
       // For SessionManagerState, this operation is the same as findSession().
      return this.findSession(domainName);
   }

   synchronized protected void lock()
   {
      attachedSession.lock();
   }

   synchronized protected void unlock()
   {
      attachedSession.unlock();
   }
}


