/*
 * @(#)ClassFile.java        1.13 98/09/21
 *
 * Copyright 1995-1998 by Sun Microsystems, Inc.,
 * 901 San Antonio Road, Palo Alto, California, 94303, U.S.A.
 * All rights reserved.
 * 
 * This software is the confidential and proprietary information
 * of Sun Microsystems, Inc. ("Confidential Information").  You
 * shall not disclose such Confidential Information and shall use
 * it only in accordance with the terms of the license agreement
 * you entered into with Sun.
 */

package com.novell.utility.java;

import java.io.File;
import java.io.InputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.zip.*;

/**
 * This class is used to represent a file loaded from the class path, and
 * can either be a regular file or a zip file entry.
 */
public
class ClassFile {
    /*
     * Non-null if this represents a regular file
     */
    private File file;

    /*
     * Non-null if this represents a zip file entry
     */
    private ZipFile zipFile;
    private ZipEntry zipEntry;

    /**
     * Constructor for instance representing a regular file
     */
    public ClassFile(File file) {
        this.file = file;
    }

    /**
     * Contructor for instance representing a zip file entry
     */
    public ClassFile(ZipFile zf, ZipEntry ze) {
        this.zipFile = zf;
        this.zipEntry = ze;
    }

    /**
     * Returns true if this is zip file entry
     */
    public boolean isZipped() {
        return zipFile != null;
    }

    /**
     * Returns input stream to either regular file or zip file entry
     */
    public InputStream getInputStream() throws IOException {
        if (file != null) {
            return new FileInputStream(file);
        } else {
            try {
                return zipFile.getInputStream(zipEntry);
            } catch (ZipException e) {
                throw new IOException(e.getMessage());
            }
        }
    }

    /**
     * Returns true if file exists.
     */
    public boolean exists() {
        return file != null ? file.exists() : true;
    }

    /**
     * Returns true if this is a directory.
     */
    public boolean isDirectory() {
        return file != null ? file.isDirectory() :
                              zipEntry.getName().endsWith("/");
    }

    /**
     * Return last modification time
     */
    public long lastModified() {
        return file != null ? file.lastModified() : zipEntry.getTime();
    }

    /**
     * Get file path. The path for a zip file entry will also include
     * the zip file name.
     */
    public String getPath() {
        if (file != null) {
            return file.getPath();
        } else {
            return zipFile.getName() + "(" + zipEntry.getName() + ")";
        }
    }

    /**
     * Get name of file entry excluding directory name
     */
    public String getName() {
        return file != null ? file.getName() : zipEntry.getName();
    }

//JCOV
    /**
     * Get absolute name of file entry
     */
    public String getAbsoluteName() {
        String absoluteName;
        if (file != null) {
            try {
                absoluteName = file.getCanonicalPath();
            } catch (IOException e) {
                absoluteName = file.getAbsolutePath();
            }
        } else {
            absoluteName = zipFile.getName() + "(" + zipEntry.getName() + ")";
        }
        return absoluteName;
    }
// end JCOV

    /**
     * Get length of file
     */
    public long length() {
        return file != null ? file.length() : zipEntry.getSize();
    }

    public String toString() {
        return (file != null) ? file.toString() : zipEntry.toString();
    }
}
