/********************************************************************

   Copyright (c) 1996 Novell, Inc.  All Rights Reserved.

   With respect to this file, Novell hereby grants to Developer a
   royalty-free, non-exclusive license to include this sample code
   and derivative binaries in its product. Novell grants to Developer
   worldwide distribution rights to market, distribute or sell this
   sample code file and derivative binaries as a component of
   Developer's product(s).  Novell shall have no obligations to
   Developer or Developer's customers with respect to this code.

   DISCLAIMER:

   Novell disclaims and excludes any and all express, implied, and
   statutory warranties, including, without limitation, warranties
   of good title, warranties against infringement, and the implied
   warranties of merchantability and fitness for a particular purpose.
   Novell does not warrant that the software will satisfy customer's
   requirements or that the licensed works are without defect or error
   or that the operation of the software will be uninterrupted.
   Novell makes no warranties respecting any technical services or
   support tools provided under the agreement, and disclaims all other
   warranties, including the implied warranties of merchantability and
   fitness for a particular purpose. */

/*********************************************************************
   LISTPART.C
**********************************************************************

   Description:

   Use NWDSListPartitions to return a list of partitions on a
   specified server. NWDSGetServerName and NWDSGetPartitionInfo
   are called to retrieve results from the output buffer.  This
   example requires a valid server name.

   syntax: LISTPART <server name>

*********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <nwnet.h>
#include <nwcalls.h>
#include <nwlocale.h>

/* cleanup functions */
void FreeBuf(pBuf_T pBuf, NWDSContextHandle context);
void FreeContext(NWDSContextHandle context);
void FreeUnicodeTables();

void main(int argc, char *argv[])
{

   NWDSContextHandle  context;
   nint32             lIterHandle;
   nuint32            luReplicaType;
   pBuf_T             pOutBuf;
   nuint32            luTotalPart;
   NWDSCCODE          ccode;
   nstr8              strPartition[MAX_DN_CHARS+1];
   nstr8              strServer[MAX_DN_CHARS+1];
   nuint              i;
   LCONV              lConvInfo;

   if(argc < 2)
   {
      printf("\nUsage: LISTPART <server name>\n");
      exit(1);
   }

   /*  Initialize NWCalls and unicode tables */
   ccode = NWCallsInit(NULL,NULL);
   if(ccode)
   {
      printf("\nNWCallsInit returned %X", ccode);
      exit(1);
   }

   NWLlocaleconv(&lConvInfo);

   ccode = NWInitUnicodeTables(lConvInfo.country_id,
                               lConvInfo.code_page);
   if(ccode)
   {
      printf("\nNWInitUnicodeTables error %X \n", ccode);
      exit(1);
   }

   /* Initialize context and allocate output buffer.  */
   ccode = NWDSCreateContextHandle(&context);
   if(ccode)
   {
      printf("\nNWDSCreateContextHandle failed");
      FreeUnicodeTables();
   }

   ccode = NWDSAllocBuf(DEFAULT_MESSAGE_LEN,&pOutBuf);
   if(ccode)
   {
      printf("\nNWDSAllocBuf returned %X", ccode);
      FreeContext(context);
   }

   /* Set lIterHandle for loop control  */
   lIterHandle = NO_MORE_ITERATIONS;

   do
   {
      /* Call NWDSListPartitions to get the list of partitions for
         the server entered on the command line and test its
         completion. */
      ccode = NWDSListPartitions(context, &lIterHandle,
                                 argv[1], pOutBuf);
      if(ccode)
      {
         printf("\nNWDSListParitions returned %X", ccode);
         FreeBuf(pOutBuf, context);
      }
      else
      {
         /* Call NWDSGetServerName to get the server name and the
            number of partitions found from the output buffer. */
         ccode = NWDSGetServerName(context, pOutBuf, strServer,
                                   &luTotalPart);
         if(ccode)
         {
            printf("\nNWDSGetServerName returned %X", ccode);
            FreeBuf(pOutBuf, context);
         }

         printf("\nPartitions on server %s:\n", strServer);

         /* Loop for each partition found */
         for(i = 0; i < luTotalPart; i++)
         {
            /* Use NWDSGetPartitionInfo to retrieve the
               partition name and its replica type from the
               output buffer */
            ccode = NWDSGetPartitionInfo(context, pOutBuf,
                             strPartition, &luReplicaType);
            if(ccode)
            {
               printf("\nNWDSGetPartition returned %X", ccode);
               FreeBuf(pOutBuf, context);
            }
            printf("\nPartition %s (Replica Type %d)", strPartition,
                   luReplicaType);
         }
      }
   } while(lIterHandle != NO_MORE_ITERATIONS);

   /* Clean up, normal termination */
   NWDSFreeBuf(pOutBuf);
   NWDSFreeContext(context);
   NWFreeUnicodeTables();
}

/* Clean up and exit, called on error condition only */
void FreeBuf(pBuf_T pBuf, NWDSContextHandle context)
{
   NWDSFreeBuf(pBuf);
   FreeContext(context);
}

void FreeContext(NWDSContextHandle context)
{
   NWDSFreeContext(context);
   FreeUnicodeTables();
}

void FreeUnicodeTables()
{
   NWFreeUnicodeTables();
   exit(1);
}

