/*
   $RCSfile: RegistrationItem.java,v $
   $Revision: 1.1.1.1 $ $Name:  $
   $Date: 2001/05/09 12:43:25 $

   $ProjectName: V:/PROJECT/houston/snapin.pj $
   $ProjectRevision: 1.172 $

   For revision information, see the history log at the end of this file.

   Copyright (c) 1997-1999 Novell, Inc.  All Rights Reserved.

   THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL,
   PROPRIETARY AND TRADE SECRET INFORMATION OF NOVELL, INC. ACCESS TO
   THIS WORK IS RESTRICTED TO (I) NOVELL, INC. EMPLOYEES WHO HAVE A
   NEED TO KNOW HOW TO PERFORM TASKS WITHIN THE SCOPE OF THEIR
   ASSIGNMENTS AND (II) ENTITIES OTHER THAN NOVELL, INC. WHO HAVE
   ENTERED INTO APPROPRIATE LICENSE AGREEMENTS. NO PART OF THIS WORK
   MAY BE USED, PRACTICED, PERFORMED, COPIED, DISTRIBUTED, REVISED,
   MODIFIED, TRANSLATED, ABRIDGED, CONDENSED, EXPANDED, COLLECTED,
   COMPILED, LINKED, RECAST, TRANSFORMED OR ADAPTED WITHOUT THE PRIOR
   WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION OF THIS WORK
   WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO CRIMINAL AND
   CIVIL LIABILITY.
*/

package com.novell.application.console.snapin;

import java.util.*;
import java.io.*;
import java.beans.*;

import com.novell.application.console.snapin.scope.*;
import com.novell.application.console.snapin.context.*;

import com.novell.application.console.shell.D;
import com.novell.application.console.shell.Resources;
import com.novell.application.console.shell.ShellNamespace;
/**
 * Specifies to ConsoleOne the conditions under which snap-ins are loaded.
 *
 * <p>The specified conditions are based on the snap-in's Scope. The Scope
 * narrows down the circumstances under which the snap-in will appear. The
 * class name is the name of the snap-in. The instance data is optional
 * and may be used to instantiate multiple instances of the same snap-in,
 * making each instance of the snap-in appear to be different (see the
 * Template interface for further details).
 *
 * @see Template
 * @see Registration
 */
public final class RegistrationItem implements java.io.Serializable
{
	// Serialized data
   private Scope		theScope = null;
   private String		className = null;
   private Object		instanceData = null;		// must implement Serializable
   private String		uniqueID = null;
   private String    registrarName = null;
   private String    namespaceUniqueID = null;
   private boolean   export = false;

	// Nonserialized data
   private transient String   snapinName = null;
   private transient String   snapinDescription = null;

   private transient boolean  everBeenLoaded = false;

   private transient ClassLoader	classLoader = null;
   private transient Snapin		cacheSnapin = null;             // used to cache this snapin
   private transient boolean		cacheNeedsShutdown = false;

   private transient Vector trackedList = new Vector();    // used for snapin Tracking
                                                   // both dynamic and static

	// this table is used for all registration items
	private static Hashtable dynamicCache = new Hashtable();
   
   private boolean isEnabled = false;
   private int priority = 0;
   
	// custom readObject to st up variables after default deserialization
	private void readObject( ObjectInputStream in )
		throws IOException, ClassNotFoundException
	{
		// first get the non static, non transient variables
		in.defaultReadObject();

		// now, set up the remaining variables
		snapinName = null;
		snapinDescription = null;

		everBeenLoaded = false;

		classLoader = null;
		cacheSnapin = null;             // used to cache this snapin
		cacheNeedsShutdown = false;

		trackedList = new Vector();    // used for snapin Tracking
	}

  /**
   * Builds a RegistrationItem based on the scope and the snap-in full
   * class name.
   *
   * <p>The resulting RegistrationItem array contains information identifying
   * what kind of snap-in each object is, where to find it, and when
   * ConsoleOne will load the snap-in.
   *
   * @param theScope  Identifies the conditions under which the snap-in will
   *                  appear in ConsoleOne.
   * @param className A string containing the full name of the snap-in class
   *                  that is to be registered to ConsoleOne.
   *
   * @see Scope
   */
   public RegistrationItem( Scope theScope, String className )
   {
      this( theScope, className, null );
   }

   /**
    * @exclude
    */
   public RegistrationItem( Scope theScope, String className, boolean export )
   {
      this( theScope, className, null );
      setExport(export);
   }
   
  /**
   * Builds a RegistrationItem based on the scope, the snap-in full
   * class name, and instance data.
   *
   * <p>The resulting RegistrationItem contains information identifying
   * what kind of snap-in each object is, where to find it, and when
   * ConsoleOne will load the snap-in.
   * </p>
   * <p>This constructor has an additional parameter, instanceData. The
   * instanceData parameter may contain any Java object that will be passed
   * to the snap-in through the Snapin.setInstanceData(Object instanceData)
   * method, which is defined in the Template interface.
   * </p>
   * <p>Note: Any time you use the instanceData parameter in RegistrationItem,
   * you must implement the Template interface in the snap-in you create in
   * order to retrieve the instanceData object.
   *
   * @param theScope     Identifies the conditions under which the snap-in will
   *                     appear in ConsoleOne.
   * @param className    A string containing the full name of the snap-in class
   *                     that is to be registered to ConsoleOne.
   * @param instanceData May contain any Java object.
   *
   * @see Scope
   * @see Template#setInstanceData
   */
   public RegistrationItem( Scope theScope, String className, Object instanceData )
   {
      this.theScope = theScope;
      this.className = className;
      this.instanceData = instanceData;
      this.uniqueID = className;
      if(instanceData != null)
      {
         this.uniqueID += instanceData;
      }
   }

  /**
   * @internal
   */
   public RegistrationItem( Scope theScope, String className, Object instanceData, String namespaceUniqueID )
   {
      this.theScope = theScope;
      this.className = className;
      this.instanceData = instanceData;
      this.uniqueID = new String( className + instanceData );
      this.namespaceUniqueID = namespaceUniqueID;
   }
   
   /**
    * @internal
    */
   public void setExport(boolean export)
   {
      this.export = export;
   }
   
   /**
    * @internal
    */
   public boolean getExport()
   {
      return export;
   }
   
   /**
    * @internal
    */
	public boolean isEnabled()
	{
		return isEnabled;
	}
	
   /**
    * @internal
    */
	public void setEnabled( boolean flag )
	{
		isEnabled = flag;
	}
   
  /**
   * Returns the scope of the participating snap-in.
   *
   * @return The scope of where the snap-in applies.
   */
   public Scope getScope()
   {
      return theScope;
   }

  /**
   * Returns the name of the participating snap-in class.
   *
   * @return The name of the snap-in class as a String.
   */
   public String getClassName()
   {
      return className;
   }

   /**
   * Sets the name of the registrar for this registration item.
   *
   * @param The name of the registrar for this registration item.
   */
   public void setRegistrarName(String registrarName)
   {
      this.registrarName = registrarName;
   }

   /**
   * Gets the name of the registrar for this registration item.
   *
   * @return The name of the registrar for this registration item.
   */
   public String getRegistrarName()
   {
      return registrarName;
   }

   /**
    * @internal
    */
   // Added for the file system, they wanted the ability to register two namespaces with
   // the same unique id and for us to just use one of them, so we pick the one to based on this priority.
   public void setNamespacePriority(int priority)
   {
      this.priority = priority;
   }
   
   /**
    * @internal
    */
   public int getNamespacePriority()
   {
      return priority;
   }
   
  /**
   * @internal
   * Returns the dynamicCache.
   *
   * @return The dynamicCache, for internal use only.
   */
   static public Hashtable getDynamicCache()
   {
      return dynamicCache;
   }

  /**
   * Returns the instance data as an Object.
   *
   * <p>The getInstanceData() method returns the instance data object as
   * defined and set by the setInstanceData() method in the Template interface.
   *
   * @return The instance data as an Object, or null if the constructor does
   *         not build a RegistrationItem array that includes the instanceData
   *         parameter.
   *
   * @see Template#setInstanceData
   */
   public Object getInstanceData()
   {
      return instanceData;
   }

  /**
   * @internal
   * Returns the unique identifier (name) of the object.
   *
   * @return The unique identifier as a String.
   */
   public String getUniqueID()
   {
      return uniqueID;
   }

  /**
   * @internal
   * Clears the snap-in cache.
   */
   public void clearCache()
   {
      if( cacheSnapin != null )
      {
         if( cacheNeedsShutdown )
         {
            try
            {
               cacheSnapin.shutdownSnapin();
            }
            catch(Exception e)
            {
               D.reportSnapinError(e);
            }
         }

         cacheNeedsShutdown = false;
         cacheSnapin = null;
      }
   }

  /**
   * @internal
   * Shuts down all snap-ins that are being tracked.
   */
   public void shutdownTrackedSnapins()
   {
		if( trackedList == null )
		{
			return;
		}

      Snapin aSnapin = null;

      // First shutdown all tracked snapins in the vector.
      Enumeration aEnum = trackedList.elements();
      while( aEnum.hasMoreElements())
      {
         aSnapin = (Snapin) aEnum.nextElement();
         try
         {
            aSnapin.shutdownSnapin();  // call shutDownSnapin on all tracked snapins
         }
         catch(Exception e)
         {
            D.reportSnapinError(e);
         }
      }

      // clear out the tracked vector
      trackedList = new Vector();       // lose references
   }

  /**
   * @internal
   * Sets the class loader.
   *
   * @param classLoader The class loader to be set.
   */
   public void setClassLoader( ClassLoader classLoader )
   {
      this.classLoader = classLoader;     // use this classLoader
   }
   
   /**
    * @internal
    */
   public String getNamespaceUniqueID()
   {
      return namespaceUniqueID;
   }

   /**
   * @internal
   */
   private Snapin loadSnapin()
   {
      Snapin   aSnapin = null;

      // first check in the dynamicCache for this puppy
      if(( aSnapin = (Snapin) dynamicCache.get( uniqueID )) == null )
      {
         // not there, so lets instantiate it
         if( classLoader == null )
         {
            classLoader = getClass().getClassLoader();   // use my classLoader
         }

         //D.startTime("Time to create snapin:" + className);
         // load the snapin
         try
         {
            aSnapin = (Snapin) (classLoader.loadClass( className ).newInstance());
         }
         catch (ExceptionInInitializerError e)
         {
            D.reportException("ExceptionInInitializerError while loading snapin " + className + " - " + e, e);
         }
         catch (NoSuchMethodError e) // class does not have a no-arg constructor
         {
            D.reportException("NoSuchMethodError while loading snapin " + className + " - " + e, e);
         }
			catch (ClassNotFoundException e)
         {
            try
            {
               aSnapin = (Snapin) (Beans.instantiate( classLoader, className ));
            }
            catch (Exception ex)
            {
               D.out("Exception while loading snapin " + className + " - " + ex);
            }
            catch (NoSuchMethodError ex)
            {
               D.out("NoSuchMethodError while loading snapin " + className + " - " + ex);
            }
         }
         catch (InstantiationException e)
         {
            D.reportException("InstantiationException while loading snapin " + className + " - " + e, e);
         }
         catch (IllegalAccessException e)
         {
            D.reportException("IllegalAccessException while loading snapin " + className + " - " + e, e);
         }
         catch (Exception e)
         {
            D.reportException("Exception while loading snapin " + className + " - " + e, e);
         }
         catch (Error e)
         {
            D.reportException("Error while loading snapin " + className + " - " + e, e);
         }
         //D.stopTime("Time to create snapin:" + className);
      }

      // This is only called each time the snapin is loaded...
      // For static or dynamic snapins...but not for snapins of a dynamic
      if( aSnapin instanceof Template )
      {
         ((Template)aSnapin).setInstanceData( instanceData );
      }

      // Only do this loop once per ConsoleOne Session, even if not tracked...
      if( !everBeenLoaded )
      {
         everBeenLoaded = true;

         try
         {
            snapinName = aSnapin.getSnapinName();
         }
         catch(Exception e)
         {
            D.reportSnapinError(e);
         }

         if(snapinName == null )
         {
            snapinName = uniqueID;     // if the snapin writer didn't return a name, use uniqueID
         }

         try
         {
            snapinDescription = aSnapin.getSnapinDescription();
         }
         catch(Exception e)
         {
            D.reportSnapinError(e);
         }

         if(snapinDescription == null )
         {
            // get this in the resource file
            snapinDescription = new String( Resources.getString( "NoDescriptionString" ));
         }
      }

      return aSnapin;
   }

   /**
   * @internal
   */
   private Snapin loadAndCache( InitSnapinInfo theInfo )
   {
      Snapin   aSnapin=null;              // get cached version, if there

      // Check if this snapin is cached.
      
      if( cacheSnapin == null )
      {
         // This sections handles loading and initializing namespace snapins, which
         // should be treated differently since we only want them to be initialized once
         // and only have one instance of each snapin.
         if(theScope.getSnapinType().equals(Shell.SNAPIN_NAMESPACE))
         {
            aSnapin = (Snapin) ShellNamespace.getNamespaceCache().get(uniqueID);
            if(aSnapin == null)
            {
               aSnapin = loadSnapin();

               //REMIND: How to handle null here, report?
               if(aSnapin != null)
               {
                  try
                  {
                     if( aSnapin.initSnapin( theInfo ) )
                     {
                        //cacheSnapin = aSnapin;    // cache it
                        //cacheNeedsShutdown = true;
                     }
                     else  // initSnapin returned false
                     {
                        // Return empty array if cached snapin does not init.
                        return null;
                     }
                  }
                  catch( Exception e )
                  {
                     D.reportSnapinError(e);
                     return null;
                  }
               }
               
               ShellNamespace.getNamespaceCache().put(uniqueID, aSnapin);
            }
         }
         else
         {
            aSnapin = loadSnapin();    // not cached, so load it!
         }
      }
      else       // get the cached snapin
      {
         aSnapin = cacheSnapin;
      }

      // This snapin was loaded successfully, now check for caching needs
      if( aSnapin != null )
      {
         Object   snapinType = theScope.getSnapinType();

         // If dynamic, cache and init it
         if( aSnapin instanceof DynamicSnapin && !dynamicCache.containsKey( uniqueID ) )
         {
            // build a dummy initSnapinInfo for this dynamic snapin's initSnapin()
            // with the context portion set to null
            if(aSnapin.initSnapin( new InitSnapinInfo( MainShell.getInstance(), Shell.SNAPIN_DYNAMIC ) ))
            {
               dynamicCache.put( uniqueID, aSnapin );
            }
            else
            {
               // Return empty array if dynamic snapin does not init.
               return null;
            }
         }

         // Only cache these snapins
         if( (snapinType.equals(Shell.SNAPIN_VIEW) ||
              snapinType.equals(Shell.SNAPIN_MAP_OBJECTENTRY) ||
              snapinType.equals(Shell.SNAPIN_DISPLAYICON) ||
              snapinType.equals(Shell.SNAPIN_DISPLAYNAME) ||
              snapinType.equals(Shell.SNAPIN_TOOLBARITEM) ||
              snapinType.equals(Shell.SNAPIN_STATUSBARITEM) ||
              snapinType.equals(Shell.SNAPIN_MENU) ||
              snapinType.equals(Shell.SNAPIN_MAP_OBJECTENTRY))
             && cacheSnapin == null
             && !(aSnapin instanceof DynamicSnapin))
         {
            if( theInfo != null )
            {
               if(theInfo.getTracking())  // do they want tracking?
               {
                  // Call initSnapin on these snapins
                  try
                  {
                     if( aSnapin.initSnapin( theInfo ) )
                     {
                        cacheSnapin = aSnapin;    // cache it
                        cacheNeedsShutdown = true;
                     }
                     else  // initSnapin returned false
                     {
                        // Return empty array if cached snapin does not init.
                        return null;
                     }
                  }
                  catch( Exception e )
                  {
                     D.reportSnapinError(e);
                     return null;
                  }
               }
               else  // just cache it, no init
               {
                  cacheSnapin = aSnapin;    // cache it
               }
            }
            else  // just cache it, no init
            {
               cacheSnapin = aSnapin;    // cache it
            }
         }

      }

      return aSnapin;
   }


   /**
   * @internal
   */
   public boolean checkNonDynamicSnapins( InitSnapinInfo theInfo )
   {
      if(theInfo != null && theInfo.exportMapping && !export)
      {
         return false;
      }
      
      Snapin   aSnapin=null;              // get cached version, if there

      // takes care of loading and caching needs
      if( (aSnapin = loadAndCache( theInfo )) == null )
      {
         /*
         if(theScope.getSnapinType().equals(Shell.SNAPIN_ACTION))
         {
            D.out("init failed for action snapin &&&&&&&&");
         }
         */
         return false;  // return false bcause init failed
      }

      if( aSnapin instanceof DynamicSnapin )
      {
         return false;
      }

      return true;
   }

   /**
   * @internal
   */
   public boolean checkDynamicSnapins( InitSnapinInfo theInfo )
   {
      if(theInfo != null && theInfo.exportMapping && !export)
      {
         return false;
      }
      
      Snapin   aSnapin=null;              // get cached version, if there

      // takes care of loading and caching needs
      if( (aSnapin = loadAndCache( theInfo )) == null )
      {
         return false;  // return false bcause init failed
      }

      if( aSnapin instanceof DynamicSnapin )
      {
         SnapinContext aContext=null;

         if( theInfo != null )
         {
            aContext = theInfo.getSnapinContext();
         }

         return ((DynamicSnapin)aSnapin).doSnapinsExist( theScope, aContext);
      }
      else
      {
         return false;
      }
   }
   
  /**
   * @internal
   * Returns an array of snap-ins based on the scope and snap-in context specified.
   *
   * <p>If there is no array of snap-ins to return, null can be returned instead.
   * Also, the reference the to the snap-in context can be null.
   *
   * @param theInfo The InitSnapinInfo containing references to the shell, the snap-in
   *                type, and the snap-in context.
   *
   * @return The array of snap-ins based on the scope and snap-in context.
   *
   * @see InitSnapinInfo
   * @see DynamicSnapin#getSnapins(Scope, SnapinContext)
   */
   // This is the only place where snapins are actually loaded, cached, and tracked.
   //   public Snapin[] getSnapins( Object theObject, SnapinContext theContext, boolean tracking )
   public Snapin[] getSnapins( InitSnapinInfo theInfo )
   {
      if(theInfo != null && theInfo.exportMapping && !export)
      {
         return new Snapin[0];
      }
      
      Snapin   aSnapin=null;              // get cached version, if there
      Snapin[] sa = new Snapin[0];        // used to hold results from dynamic snapins getSnapins() call
      Object   snapinType = theScope.getSnapinType();

      // takes care of loading and caching needs
      if(( aSnapin = loadAndCache( theInfo )) == null )
      {
         return sa;  // return empty array bcause init failed
      }

      // Convert DynamicSnapins into a snapinArray or
      // Convert aSnapin into the array

      boolean dynamic = (aSnapin instanceof DynamicSnapin);

      if( dynamic )
      {
         // Load the snapins from the DynamicSnapin
         try
         {
            SnapinContext aContext=null;

            if( theInfo != null )
            {
               aContext = theInfo.getSnapinContext();
            }

            try
            {
               sa = ((DynamicSnapin)aSnapin).getSnapins( theScope, aContext);
            }
            catch(Exception e)
            {
               D.reportException("Exception while calling getSnapins() for snapin " + aSnapin.getClass().getName() + " - " + e, e);
            }
            catch(Error e)
            {
               D.reportException("Error while calling getSnapins() for snapin " + aSnapin.getClass().getName() + " - " + e, e);
            }

            if(sa == null)
            {
               sa = new Snapin[0];
            }
         }
         catch(Exception e)
         {
            D.reportSnapinError(e);
         }
      }
      else  // static snapin, so load the 1st element
      {
         sa = new Snapin[1];
         sa[0] = aSnapin;
      }

      // Only track if theInfo is valid and getTracking() is true.
      if( theInfo != null && theInfo.getTracking())
      {
         // track the array
         shutdownTrackedSnapins();

         // Don't init cached static snapins.
         if( cacheSnapin == null || dynamic )
         {
            // Now track this snapinArray
            for( int i=0; i<sa.length; i++ )
            {
               D.assert(sa[i] != null);
               if(sa[i] != null)
               {
                  try
                  {
                     if( sa[i].initSnapin( theInfo ) )
                     {
//								if( trackedList == null )
//								{
//									trackedList = new Array();
//								}

                        trackedList.addElement( sa[i] );   // save reference
                     }
                     else
                     {
                        sa[i] = null;  // this snapin didn't start
                     }
                  }
                  catch(Exception e)
                  {
                     D.reportSnapinError(e);
                     sa[i] = null;
                  }
               }
            }     // for loop
         }
      }        // if theInfo is valid

      return sa;
   }

  /**
   * @internal
   * Returns the localized name of the participating snap-in.
   *
   * @return The localized name of the snap-in as a String.
   */
   public String getSnapinName()
   {
      if( !everBeenLoaded )
      {
         loadSnapin();     // load snapin, call setInstanceData if Dynamic, getSnapinName
      }

      return snapinName;   // default to uniqueID if snapin writer returns null
   }

  /**
   * @internal
   * Returns the localized description of the participating snap-in.
   *
   * @return The localized description of the snap-in as a String.
   */
   public String getSnapinDescription()
   {
      if( !everBeenLoaded )
      {
         loadSnapin();     // load snapin, call setInstanceData if Dynamic, getSnapinName
      }

      return snapinDescription;
   }

  /**
   * @internal
   * Sets the localized name of the participating snap-in.
   *
   * @param snapinName The localized name of the snap-in as a String.
   */
   public void setSnapinName( String snapinName )
   {
      everBeenLoaded = true;
      this.snapinName = snapinName;
   }

  /**
   * @internal
   * Sets the localized description of the participating snap-in.
   *
   * @param snapinDescription The localized description of the snap-in as a String.
   */
   public void setSnapinDescription( String snapinDescription )
   {
      everBeenLoaded = true;
      this.snapinDescription = snapinDescription;
   }
}

/*
$Log: RegistrationItem.java,v $
Revision 1.1.1.1  2001/05/09 12:43:25  rommel
Sandbox initial version uploaded from novell cd 1/3

Revision 1.66  2000/07/26 16:54:45  DChamberlain
Discovered @@exclude was the right tag, changed back.
Revision 1.65  2000/06/29 19:26:37Z  DChamberlain
Changed @@exclude javadoc tag to proper @@excluded tag.
Revision 1.64  2000/05/03 00:11:00Z  DChamberlain
Catch and report errors and exceptions made from calling getSnapins().
Revision 1.63  2000/04/26 18:01:45Z  DChamberlain
Catch error on instantiation.  Partial fix for Bug# 227627.
Revision 1.62  2000/02/22 22:52:03Z  DChamberlain
Added constructor for export flag.
Revision 1.61  2000/01/12 15:48:59Z  DChamberlain
Took out new constructor until SP1.
Revision 1.59  1999/12/20 23:39:08  dwilson
Added ability to instantiate a snapin bean -- coming soon to a theater near you.
Revision 1.58  1999/10/20 00:14:38  DChamberlain
Added export stuff.
Revision 1.57  1999/10/05 19:56:43  DChamberlain
Added namespace priority methods, removed beans stuff.
Revision 1.56  1999/09/08 17:43:10  DChamberlain
Bug fix from last checkin
Revision 1.55  1999/09/08 16:25:04  DChamberlain
Added setPriority and set and get Enabled stuff.
Revision 1.54  1999/07/28 22:04:30  DChamberlain
Moved namespace unique from scopes to registration item.
Revision 1.53  1999/07/19 20:07:21  DChamberlain
Fixes so that namespaces are single instances.
Revision 1.52  1999/06/28 14:57:03  DChamberlain
Changed the way checkSnapins works to make faster.
Revision 1.51  1999/06/15 16:16:38  DChamberlain
Cache namespaces for placeholder stuff.
Revision 1.50  1999/05/24 15:54:52  DChamberlain
Adding node namespace.
Revision 1.49  1999/05/03 16:02:21  DChamberlain
Report errors differently.
Revision 1.48  1999/04/01 21:51:58  DChamberlain
Removed dependencies on j g l.
Revision 1.47  1999/02/05 00:56:13  mharris

*/