/*
   $RCSfile: NamespaceTypeScope.java,v $
   $Revision: 1.1.1.1 $ $Name:  $
   $Date: 2001/05/09 12:43:27 $

   $ProjectName: V:/PROJECT/houston/scopes.pj $
   $ProjectRevision: 1.164 $

   For revision information, see the history log at the end of this file.

   Copyright (c) 1997-1999 Novell, Inc.  All Rights Reserved.

   THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL,
   PROPRIETARY AND TRADE SECRET INFORMATION OF NOVELL, INC. ACCESS TO
   THIS WORK IS RESTRICTED TO (I) NOVELL, INC. EMPLOYEES WHO HAVE A
   NEED TO KNOW HOW TO PERFORM TASKS WITHIN THE SCOPE OF THEIR
   ASSIGNMENTS AND (II) ENTITIES OTHER THAN NOVELL, INC. WHO HAVE
   ENTERED INTO APPROPRIATE LICENSE AGREEMENTS. NO PART OF THIS WORK
   MAY BE USED, PRACTICED, PERFORMED, COPIED, DISTRIBUTED, REVISED,
   MODIFIED, TRANSLATED, ABRIDGED, CONDENSED, EXPANDED, COLLECTED,
   COMPILED, LINKED, RECAST, TRANSFORMED OR ADAPTED WITHOUT THE PRIOR
   WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION OF THIS WORK
   WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO CRIMINAL AND
   CIVIL LIABILITY.
*/
package com.novell.application.console.snapin.scope;

//REMIND: Remember to comment out these lines when building non-checked version in all scopes.
import com.novell.application.console.shell.*;
import com.novell.application.console.snapin.*;

/**
 * Defines a scope that is restricted to the namespace and object type
 * specified at registration.
 *
 * <p>Page, View, Map Object, Menu, Toolbar, Status Bar, Display Name, Display Icon, Popup Menu and
 * Dynamic snap-ins may be registered with the NamespaceTypeScope scope. When
 * registered, the functionality provided by these snap-ins becomes active when
 * an object of the specified type and in the namespace specified at registration
 * is selected (highlighed) by the user.
 * </p>
 * <p>The following code snippet is an example of registering a view snap-in
 * to the specified namespace with the NamespaceTypeScope scope.
 * <pre><code>
 *      public RegistrationItem[] getRegistration()
 *      {
 *          NamespaceTypeScope scope;
 *          scope = new NamespaceTypeScope(Shell.SNAPIN_VIEW,
 *                                         "Namespace UniqueID",Type");
 *          return new RegistrationItem[]
 *          {
 *              new RegistrationItem(scope, getClass().getName())
 *          };
 *      }
 * </code></pre>
 *
 * @see Scope
 */
public class NamespaceTypeScope implements Scope
{
   private static final String SCOPE_KEY = Constants.NamespaceTypeScopeKey;
   private Object    snapinType;
   private String    namespace;
   private String    type;
   private String    key;

  /**
   * Builds a NamespaceTypeScope for the specified snapinType, but restricted
   * to the namespace and object type ('type') specified at registration.
   *
   * <p>The snap-in types that can be used for the NamespaceTypeScope are:
   * <ul>
   * <li>Shell.SNAPIN_DISPLAYICON
   * <li>Shell.SNAPIN_DISPLAYNAME
   * <li>Shell.SNAPIN_DYNAMIC
   * <li>Shell.SNAPIN_MAP_OBJECTENTRY
   * <li>Shell.SNAPIN_MENU
   * <li>Shell.SNAPIN_PAGE
   * <li>Shell.SNAPIN_POPUP_MENU
   * <li>Shell.SNAPIN_RESULTMODIFIER
   * <li>Shell.SNAPIN_STATUSBARITEM
   * <li>Shell.SNAPIN_TOOLBARITEM
   * <li>Shell.SNAPIN_VIEW
   * </ul>
   *
   * @param snapinType The type of the snap-in as defined above.
   * @param namespace  The uniqueID string for which the namespace applies.
   * @param type       The object type specified at registration.
   */
   public NamespaceTypeScope( Object snapinType, String namespace, String type )
   {
      Debug.n_assert(
               snapinType != Shell.SNAPIN_SERVICE,
               "Cannot use given snapin type with this scope.");
      this.snapinType = snapinType;
      this.namespace = namespace;
      this.type = type;
      // Found during optimizing that it is better to compute key once and save.
      key = namespace + SCOPE_KEY + type;
   }

   /**
   * @internal
   * Returns the scopeKey string for this registration point.
   *
   * <p>The getScopeKey() method is called by the RegistrationItem class.
   *
   * @return The scopeKey string.
   */
   public String getScopeKey()
   {
      return key;
   }

   /**
   * @internal
   * Returns the scope description string for this registration point.
   *
   * <p>The getScopeDescription() method is called by the ConsoleOne Configurator.
   *
   * @return The scope description string.
   */
   public String getScopeDescription()
   {
      return new String( namespace + " " + type );
   }

   /**
   * Returns the snap-in type object that was passed in the constructor.
   *
   * @return The snap-in type object.
   *
   * @see Scope
   */
   public Object getSnapinType()
   {
      return snapinType;
   }

   /**
   * Returns the namespace unique ID for this scope instance.
   *
   * @return The unique ID of the specified namespace.
   */
   public String getNamespaceUniqueID()
   {
      return namespace;
   }

   /**
   * Returns the type name for this scope instance.
   *
   * @return The type name.
   */
   public String getTypeName()
   {
      return type;
   }
}

/*
$Log: NamespaceTypeScope.java,v $
Revision 1.1.1.1  2001/05/09 12:43:27  rommel
Sandbox initial version uploaded from novell cd 1/3

Revision 1.32  1999/07/28 16:44:36  DChamberlain
Moved namespace unique id to ri.
Revision 1.30  1999/06/02 17:01:18  DChamberlain
Allow namespaces to be registered to this scope.
Revision 1.29  1999/05/05 14:30:50  arlandes
Remove snapin persistence scope
Revision 1.28  1999/04/01 21:53:38  DChamberlain
Use Debug instead of D.
Revision 1.27  1999/03/17 00:01:53  DChamberlain
Cleaned up javadoc.
Revision 1.26  1999/01/07 19:26:51  DChamberlain
Updated copyright to include 1999.
Revision 1.25  1998/12/15 21:40:42  mharris
Revised javadoc comments to refect Snapin types allowed with scope.
Revision 1.24  1998/11/05 18:56:00  DChamberlain
Removed assertions to allow tb, sb and main menu to switch on type.
Revision 1.23  1998/10/26 17:02:58  DChamberlain
Fixed asserts to exclude 3rd party snapin types.
Revision 1.22  1998/10/23 15:36:27  DChamberlain
Changed asserts to not disallow new snapin types, but still do most common
mistake checking.
Revision 1.21  1998/10/14 22:42:33  dwilson
Removed assert that check for valid snapin types
Revision 1.20  1998/10/08 19:27:30  ksondere
Removed references to ExtendChildrenSnapin.
Revision 1.19  1998/09/25 15:12:01  DChamberlain
Cache key for optimization, found during optimizing that it took too
long to compute everytime getScopeKey() is called.
Revision 1.18  1998/06/05 19:59:40  DChamberlain
Added SNAPIN_DYNAMIC to all asserts, removed assert from GlobalScope.
Revision 1.17  1998/06/04 22:20:20  SHerrin
Corrected and enhanced Javadoc comments.
Revision 1.16  1998/05/27 19:52:27  SHerrin
Added  Shell.SNAPIN_MAP_OBJECTENTRY to Javadoc comments.
Revision 1.15  1998/05/21 17:18:32  DChamberlain
Added map object entry snapin to scope.
Revision 1.14  1998/05/12 16:34:42  SHerrin
Corrected code snippet in Javadoc comments.
Revision 1.13  1998/05/12 15:44:44  SHerrin
Corrected code snippet in Javadoc comments.
Revision 1.12  1998/05/11 20:04:25  SHerrin
Added code snippet to Javadoc comments.
Revision 1.11  1998/04/27 16:40:40  DChamberlain
Moved non-translated strings out of resource bundle.  (SPD# 188198)
Revision 1.10  1998/04/08 16:54:49  DChamberlain
Added accessor methods.
Revision 1.9  1998/04/07 16:12:29  DChamberlain
Marked internal methods for javadoc.
Revision 1.8  1998/03/26 16:19:32  SHerrin
Corrections to Javadoc comments.
Revision 1.7  1998/03/20 16:06:13  DChamberlain
Use shorter scope keys defined in resources.
Revision 1.6  1998/03/11 21:10:52  DChamberlain
Use icons instead of images.
Revision 1.5  1998/03/07 02:56:41  DChamberlain
Removed name map snapin.
Revision 1.4  1998/03/05 23:49:39  dwilson
Added getScopeDescription()
Revision 1.3  1998/03/05 03:49:10  SHerrin
Added Javadoc comments.
Revision 1.2  1998/02/24 01:36:53  DChamberlain
import snapin.
Revision 1.1  1998/02/24 01:04:43  DChamberlain
Initial revision
Revision 1.3  1998/02/23 17:42:06  DChamberlain
New registration.
Revision 1.2  1998/02/17 17:05:43  dwilson
Revision 1.1  1998/02/16 22:24:50  DChamberlain
Initial revision
*/
