/*
   $RCSfile: Resources.java,v $
   $Revision: 1.1.1.1 $ $Name:  $
   $Date: 2001/05/09 12:43:14 $

   $ProjectName: V:/PROJECT/houston/shell.pj $
   $ProjectRevision: 1.171 $

   For revision information, see the history log at the end of this file.

   Copyright (c) 1997-1999 Novell, Inc.  All Rights Reserved.

   THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL, 
   PROPRIETARY AND TRADE SECRET INFORMATION OF NOVELL, INC. ACCESS TO 
   THIS WORK IS RESTRICTED TO (I) NOVELL, INC. EMPLOYEES WHO HAVE A 
   NEED TO KNOW HOW TO PERFORM TASKS WITHIN THE SCOPE OF THEIR 
   ASSIGNMENTS AND (II) ENTITIES OTHER THAN NOVELL, INC. WHO HAVE 
   ENTERED INTO APPROPRIATE LICENSE AGREEMENTS. NO PART OF THIS WORK 
   MAY BE USED, PRACTICED, PERFORMED, COPIED, DISTRIBUTED, REVISED, 
   MODIFIED, TRANSLATED, ABRIDGED, CONDENSED, EXPANDED, COLLECTED, 
   COMPILED, LINKED, RECAST, TRANSFORMED OR ADAPTED WITHOUT THE PRIOR 
   WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION OF THIS WORK 
   WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO CRIMINAL AND 
   CIVIL LIABILITY. 
*/
package com.novell.application.console.shell;

import java.awt.*;
import java.net.*;
import java.util.*;

/**
* Class that is used to funnel resources through.  It wraps the resource bundle.
* All shell classes should use it to get resources.  The only thing it does not
* retrieve from the resource file is images, which it retrieves directly from
* the classpath.
*/
public class Resources
{
   private static ResourceBundle res;

   public static Font              DEFAULT_FONT = null; 

   public static Font getDefaultFont()
   {
      if(DEFAULT_FONT == null)
      {
         DEFAULT_FONT = new Font("Dialog", Font.PLAIN, 11);         
      }
      
      return DEFAULT_FONT;
   }
   
   /**
   * Sets the resource bundle to use.  This has to happen after the Console.snapinBundleLoader is
   * is set, which has to happen after the snapin class loader's path is set up.
   */
   static void setBundle()
   {
      res = ConsoleShell.snapinBundleLoader.getBundle("com.novell.application.console.shell.resources.ConsoleOneResourceBundle");
   }

   /**
   * Returns an object from the resource file associated with a given key.
   * @param keyName Key for object in resource.
   * @return Localized object associated with key.  If object is not found, a MissingResource exception is throw.
   */
   public static Object getObject(String keyName)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      Object ret = null;
      try
      {
         ret = res.getObject(keyName);
      }
      catch(MissingResourceException e)
      {
         //REMIND: Do we really need to report?
         D.out("Missing resource, throwing exception to caller:" + keyName);
         throw(e);
      }
      return ret;
   }

   /**
   * Returns an object from the resource file associated with a given key.
   * @param keyName Key for object in resource.
   * @param defaultObject Default Object to return if resource is not found (MissingResourceException is caught).
   * @return Localized object associated with key.  If object is not found, the default object is returned.
   */
   public static Object getObject(String keyName, Object defaultObject)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      Object ret = null;
      try
      {
         ret = res.getObject(keyName);
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, returning default:" + keyName);
         ret = defaultObject;
      }
      return ret;
   }

   /**
   * Returns a string from the resource file associated with a given key.
   * @param keyName Key for string in resource.
   * @return Localized string associated with key.  If string is not found, a MissingResourceException is thrown.
   */
   public static String getString(String keyName)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      String ret = null;
      try
      {
         D.assert(res.getObject(keyName) instanceof String);
         ret = res.getString(keyName);
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, throwing exception to caller:" + keyName);
         throw(e);
      }
      return ret;
   }

   /**
   * Returns a string from the resource file associated with a given key.
   * This getString will not throw an exception if the resource is not found.
   * @param keyName Key for string in resource.
   * @param defaultString Default string to return if resource is not found (MissingResourceException is caught).
   * @return Localized string associated with key.  If the string is not found, defaultString is returned.
   */
   public static String getString(String keyName, String defaultString)
   {
      D.assert(keyName != null && !keyName.equals(""));
      String ret = null;
      try
      {
         D.assert(res.getObject(keyName) instanceof String);
         ret = res.getString(keyName);
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, returning default:" + keyName);
         ret = defaultString;
      }
      return ret;
   }

   /**
   * Returns an image from the shell's image directory.  Note that a class
   * must be passed in.  This is because to get a url off of the classpath,
   * the Class class getResource method must be used (probably for the 
   * classloader to use), and since this is a static method, we must use
   * another class.  Just pass in a class that is using the same classloader
   * as Houston, for most cases, just use "this".
   * REMIND: Is there a cleaner way to do than passing in an object?
   * @param imageName Name of image file in image dir to get.
   * @return Image.  If image is not found or cannot be loaded, returns null.
   */
   public static Image getImage(String imageName)
   {
      //REMIND: No longer pass in object.
      //REMIND: should image name be a key?
      //REMIND: Can we return a default if image if not in resource, one that is hardcoded
      //        as bytes in this class?
      D.assert(imageName != null && !imageName.equals(""));
      Image i = null;
      String path = Constants.ConsoleOneImageDirectoryKey;
      path += imageName;
      // Use the resource bundle as the class to use to get the resource since it was loaded 
      // with the same class loader where the gifs are.
      URL url = res.getClass().getResource(path);
      if(url != null)
      {
         i = Toolkit.getDefaultToolkit().getImage(url);
      }
      D.assert(i != null, "Cannot load image file:" + imageName);
      return i;
   }

   /**
   * Returns a font from the resource file.
   * @param keyName Key for Font.
   * @return Font associated with key.  If font is not found, a MissingResourceException is thrown.
   */
   public static Font getFont(String keyName)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      Font font = null;
      try
      {
         D.assert(res.getObject(keyName) instanceof Font);
         font = (Font) res.getObject(keyName);
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource:" + keyName);
         font = getDefaultFont();
      }
      return font;
   }

   /**
   * Returns a font from the resource file.
   * @param keyName Key for Font.
   * @param defaultFont Default font to pass back if resource is not found.  (A good one to pass is Resources.DEFUALT_FONT)
   * @return Font associated with key.  If font is not found, defaultFont is returned.
   */
   public static Font getFont(String keyName, Font defaultFont)
   {
      D.assert(keyName != null && !keyName.equals(""));
      Font font = null;
      try
      {
         D.assert(res.getObject(keyName) instanceof Font);
         font = (Font) res.getObject(keyName);
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource:" + keyName);
         font = defaultFont;
      }
      return font;
   }

   /**
   * Returns an int from the resource file.  This getInt will not throw
   * an exception if the resource is not found.
   * @param keyName Key for int.
   * @param defaultValue Value returned if cannot find or load resource.
   * @return int associated with key.  If int is not found, defaultValue is returned.  
   */
   public static int getInt(String keyName, int defaultValue)
   {
      D.assert(keyName != null && !keyName.equals(""));
      int ret = 0;
      try
      {
         D.assert(res.getObject(keyName) instanceof Integer);
         Integer i = (Integer) res.getObject(keyName);
         ret = i.intValue();
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, returning default passed in:" + keyName);
         ret = defaultValue;
      }
      return ret;
   }

   /**
   * Returns an int from the resource file.
   * @param keyName Key for int.
   * @return int associated with key.  If int is not found, a MissingResourceException is thrown.
   */
   public static int getInt(String keyName)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      int ret = 0;
      try
      {
         D.assert(res.getObject(keyName) instanceof Integer);
         Integer i = (Integer) res.getObject(keyName);
         ret = i.intValue();
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, throwing exception to caller:" + keyName);
         throw(e);
      }
      return ret;
   }

   /**
   * Returns an float from the resource file.  This getFloat will
   * not throw an exception if resource is not found.
   * @param keyName Key for float.
   * @param defaultValue Value returned if cannot find or load resource.
   * @return float associated with key.  If float is not found, defaultValue is returned.
   */
   public static float getFloat(String keyName, float defaultValue)
   {
      D.assert(keyName != null && !keyName.equals(""));
      float ret = 0;
      try
      {
         D.assert(res.getObject(keyName) instanceof Float);
         Float i = (Float) res.getObject(keyName);
         ret = i.floatValue();
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, returning default passed in:" + keyName);
         ret = defaultValue;
      }
      return ret;
   }

   /**
   * Returns an float from the resource file.
   * @param keyName Key for float.
   * @return float associated with key.  If float is not found, a MissingResourceException is thrown.
   */
   public static float getFloat(String keyName)
      throws MissingResourceException
   {
      D.assert(keyName != null && !keyName.equals(""));
      float ret = 0;
      try
      {
         D.assert(res.getObject(keyName) instanceof Float);
         Float i = (Float) res.getObject(keyName);
         ret = i.floatValue();
      }
      catch(MissingResourceException e)
      {
         D.out("Missing resource, throwing exception to caller:" + keyName);
         throw(e);
      }
      return ret;
   }
}

/*
$Log: Resources.java,v $
Revision 1.1.1.1  2001/05/09 12:43:14  rommel
Sandbox initial version uploaded from novell cd 1/3

Revision 1.16  1999/02/25 23:40:31  DChamberlain
Moved fonts to constants.  This is to allow NonGuiShell to work.  SPD# 211543.
Revision 1.15  1999/01/22 15:37:57  DChamberlain
Renamed Console to ConsoleShell.
Revision 1.14  1999/01/07 19:27:35  DChamberlain
Updated copyright to include 1999.
Revision 1.13  1998/09/29 15:44:38  DChamberlain
Moved items that will not be translated from resources to constants.
Revision 1.12  1998/09/17 21:40:37  DChamberlain
Resources.getImage() no longer takes objectreference.
Revision 1.11  1998/09/14 17:52:51  DChamberlain
Setup resource bundle in the setBundle() call.
Revision 1.10  1998/09/14 17:44:28  DChamberlain
Updated to get resources using snapin class loader.
Revision 1.9  1998/04/03 22:00:48  Paul_MacKay
Changed name of resource bundle to be ConsoleOneResourceBundle
Revision 1.8  1998/04/03 00:51:30  DChamberlain
Renamed globalsnapins to consolesnapins.
Revision 1.7  1998/04/01 03:27:25  DChamberlain
Cleaned up error reporting.
Revision 1.6  1998/03/20 04:06:24  DChamberlain
Renamed resource bundle to ConsoleResourceBundle.
Revision 1.5  1998/02/13 17:06:51  DChamberlain
New copyright.  Make sure file had log.  Wrap calls to snapins in try blocks.  
Revision 1.4  1998/01/16 16:39:26  DChamberlain
Removed NCC in resources keys.
Revision 1.3  1998/01/12 18:32:32  DChamberlain
Moved from ncc package to console and from ncc.console package to console.shell. 
Revision 1.2  1998/01/05 23:32:58  DChamberlain
Added getObject that takes a default.  Brought up to coding standards.
Revision 1.1  1997/12/11 22:57:34  DChamberlain
Initial revision
Revision 1.16  1997/12/11 18:06:54  DChamberlain
Made two sets of calls, one that takes default and another that throws MissingResouce exception.
Revision 1.15  1997/12/04 23:58:22  DChamberlain
Added getObject and comments.
Revision 1.14  1997/11/13 23:52:42  DChamberlain
Added float and passing defaults as parameters.
Revision 1.13  1997/11/13 18:13:04  DChamberlain
Made image directory key a static final variable.
Revision 1.12  1997/11/13 16:43:04  DChamberlain
Changed Debug to D.
Revision 1.11  1997/11/12 21:27:08  DChamberlain
Added reminder comment.
Revision 1.10  1997/11/12 00:01:20  DChamberlain
Adding messages to asserts.
Revision 1.9  1997/11/11 18:55:32  DChamberlain
Small cleanup, rearranging logic and adding asserts.
Revision 1.8  1997/11/07 23:59:16  DChamberlain
Catch exception on ints.
Revision 1.7  1997/11/07 16:32:46  DChamberlain
Massive package name changes.
Revision 1.6  1997/11/07 15:19:34  DChamberlain
Cleaned up comments and docs.
Revision 1.5  1997/11/07 00:41:50  DChamberlain
Javadoced and added getInt().
Revision 1.4  1997/10/10 22:40:58  DChamberlain
Added getFont() method for getting fonts from resources.
Revision 1.3  1997/09/18 23:11:34  DChamberlain
Uppercase COM to lowercase com for novell packages.
Revision 1.2  1997/09/03 21:01:52  DChamberlain
Removed comments above imports, they break coverage build.
Revision 1.1  1997/08/29 00:32:34  DChamberlain
Initial revision
Revision 1.1  1997/08/28 23:45:24  DChamberlain
Initial revision
Revision 2.2  1997/08/25 17:19:56  DChamberlain
Added body to getImage().
Revision 2.1  1997/08/18 19:38:14  DChamberlain
Cleaned up source file.
Revision 2.0  1997/08/08 23:10:08  DChamberlain
Initial revision
*/
