#ifndef   _NWCCS_H_
#define   _NWCCS_H_
/**************************************************************************
 * Novell Software Development Kit
 * (C) Copyright 1998-2005 by Novell, Inc. All Rights Reserved.
 *
 *
 * This work is subject to U.S. and international copyright laws and treaties.
 * Use and redistribution of this work is subject  to  the  license  agreement
 * accompanying  the  software  development kit (SDK) that contains this work.
 * However, no part of this work may be revised and/or  modified  without  the
 * prior  written consent of Novell, Inc. Any use or exploitation of this work
 * without authorization could subject the perpetrator to criminal  and  civil
 * liability.
 *
 *  NAME OF FILE: nwccs.h
 **************************************************************************
*/

/*
 **************************************************************************
 *
 *     NICI SDK header file for the Controlled Cryptographic Services (CCS)
 *     cryptographic library (XLIB).
 *
 *   TABS set to width=3
 *
 **************************************************************************
*/

#if defined(__cplusplus)
extern "C" {
#endif

#ifndef KERNEL
/* WIN32 must be user defined if compiling under Windows */
/* ntypes.h will then add N_PLAT_MSW, and N_ARCH_32 */
#ifdef WIN32
#include <windows.h>
#endif
#endif

/* CCS types are based on Novell standard types */
#include <ntypes.h>

/* NICI Error codes occupy the range from -1400 to -1499 */


#define NICI_E_OK                               (0)
#define NICI_E_INTERNAL_ERROR                   (-1400) /* 0xFA88 */

#define NICI_E_ALGORITHM_INVALID                (-1401) /* 0xFA87 */
#define NICI_E_ALGORITHM_NOT_SUPPORTED          (-1402) /* 0xFA86 */
#define NICI_E_ASN1_DECODE                      (-1403) /* 0xFA85 */
#define NICI_E_ASN1_ENCODE                      (-1404) /* 0xFA84 */
#define NICI_E_ATTRIBUTE_READ_ONLY              (-1405) /* 0xFA83 */
#define NICI_E_ATTRIBUTE_SENSITIVE              (-1406) /* 0xFA82 */
#define NICI_E_ATTRIBUTE_TYPE_INVALID           (-1407) /* 0xFA81 */
#define NICI_E_ATTRIBUTE_VALUE_INVALID          (-1408) /* 0xFA80 */
#define NICI_E_BUFFER_OVERFLOW                  (-1409) /* 0xFA7F */
#define NICI_E_CC_COUNT                         (-1410) /* 0xFA7E */
#define NICI_E_CC_HANDLE_INVALID                (-1411) /* 0xFA7D */
#define NICI_E_CERTIFICATE_CHAIN                (-1412) /* 0xFA7C */
#define NICI_E_CERTIFICATE_DATA_INVALID         (-1413) /* 0xFA7B */
#define NICI_E_CERTIFICATE_DATA_LEN_RANGE       (-1414) /* 0xFA7A */
#define NICI_E_CRITICAL_EXTENSION               (-1415) /* 0xFA79 */
#define NICI_E_DATA_INVALID                     (-1416) /* 0xFA78 */
#define NICI_E_DATA_LEN_RANGE                   (-1417) /* 0xFA77 */
#define NICI_E_ENCRYPTED_DATA_INVALID           (-1418) /* 0xFA76 */
#define NICI_E_ENCRYPTED_DATA_LEN_RANGE         (-1419) /* 0xFA75 */
#define NICI_E_FLAGS_INVALID                    (-1420) /* 0xFA74 */
#define NICI_E_FUNCTION_NOT_SUPPORTED           (-1421) /* 0xFA73 */
#define NICI_E_INVALID_PARAMETER                (-1422) /* 0xFA72 */
#define NICI_E_INVALID_USAGE                    (-1423) /* 0xFA71 */
#define NICI_E_KEY_HANDLE_INVALID               (-1424) /* 0xFA70 */
#define NICI_E_KEY_PAIR                         (-1425) /* 0xFA6F */
#define NICI_E_KEY_POLICY_VIOLATION             (-1426) /* 0xFA6E */
#define NICI_E_KEY_SENSITIVE                    (-1427) /* 0xFA6D */
#define NICI_E_KEY_SIZE_RANGE                   (-1428) /* 0xFA6C */
#define NICI_E_KEY_TYPE_INCONSISTENT            (-1429) /* 0xFA6B */
#define NICI_E_KM_STRENGTH_INVALID              (-1430) /* 0xFA6A */
#define NICI_E_MEMORY                           (-1431) /* 0xFA69 */
#define NICI_E_NULL_INVALID                     (-1432) /* 0xFA68 */
#define NICI_E_OBJECT_CLASS_INCONSISTENT        (-1433) /* 0xFA67 */
#define NICI_E_OBJECT_CLASS_INVALID             (-1434) /* 0xFA66 */
#define NICI_E_OBJECT_CLASS_NOT_SUPPORTED       (-1435) /* 0xFA65 */
#define NICI_E_OBJECT_COUNT                     (-1436) /* 0xFA64 */
#define NICI_E_OBJECT_HANDLE_INVALID            (-1437) /* 0xFA63 */
#define NICI_E_OBJECT_PERMANENT                 (-1438) /* 0xFA62 */
#define NICI_E_OPERATION_ACTIVE                 (-1439) /* 0xFA61 */
#define NICI_E_OPERATION_INCONSISTENT           (-1440) /* 0xFA60 */
#define NICI_E_OPERATION_NOT_INITIALIZED        (-1441) /* 0xFA5F */
#define NICI_E_RESOURCE                         (-1442) /* 0xFA5E */
#define NICI_E_SIGNATURE_INVALID                (-1443) /* 0xFA5D */
#define NICI_E_SIGNATURE_LEN_RANGE              (-1444) /* 0xFA5C */
#define NICI_E_TEMPLATE_INCOMPLETE              (-1445) /* 0xFA5B */
#define NICI_E_TEMPLATE_INCONSISTENT            (-1446) /* 0xFA5A */
#define NICI_E_UNAVAILABLE                      (-1447) /* 0xFA59 */
#define NICI_E_UNWRAPPING_KEY_HANDLE_INVALID    (-1448) /* 0xFA58 */
#define NICI_E_UNWRAPPING_KEY_SIZE_RANGE        (-1449) /* 0xFA57 */
#define NICI_E_UNWRAPPING_KEY_TYPE_INCONSISTENT (-1450) /* 0xFA56 */
#define NICI_E_VALIDITY_EXPIRED                 (-1451) /* 0xFA55 */
#define NICI_E_WRAPPED_KEY_INVALID              (-1452) /* 0xFA54 */
#define NICI_E_WRAPPED_KEY_LEN_RANGE            (-1453) /* 0xFA53 */
#define NICI_E_WRAPPING_KEY_HANDLE_INVALID      (-1454) /* 0xFA52 */
#define NICI_E_WRAPPING_KEY_SIZE_RANGE          (-1455) /* 0xFA51 */
#define NICI_E_WRAPPING_KEY_TYPE_INCONSISTENT   (-1456) /* 0xFA50 */
#define NICI_E_WRONG_ORDER                      (-1457) /* 0xFA4F */
#define NICI_E_XS_HANDLE_INVALID                (-1458) /* 0xFA4E */
#define NICI_E_XS_RESOURCE                      (-1459) /* 0xFA4D */

#define NICI_E_NOT_FOUND                        (-1460) /* 0xFA4C */
#define NICI_E_VERSION_MISMATCH                 (-1461) /* 0xFA4B */

/* note: -1462 to -1469 are reserved. */
#define NICI_E_1462                             (-1462)  /* 0xFA4A */
#define NICI_E_1463                             (-1463)  /* 0xFA49 */
#define NICI_E_1464                             (-1464)  /* 0xFA48 */
#define NICI_E_1465                             (-1465)  /* 0xFA47 */
#define NICI_E_1466                             (-1466)  /* 0xFA46 */
#define NICI_E_1467                             (-1467)  /* 0xFA45 */
#define NICI_E_1468                             (-1468)  /* 0xFA44 */
#define NICI_E_1469                             (-1469)  /* 0xFA43 */

#define NICI_E_FIPS140CNRG_ERR                  (-1470)  /* 0xFA42 */
#define NICI_E_SELF_VERIFICATION                (-1471)  /* 0xFA41 */
#define NICI_E_CRYPTO_DOWNGRADE                 (-1472)  /* 0xFA40 */
#define NICI_E_1473                             (-1473)  /* 0xFA3F */
#define NICI_E_1474                             (-1474)  /* 0xFA3E */
#define NICI_E_POINTER_MISMATCH                 (-1475)  /* 0xFA3D */

/* CCS error codes */

   /*
      Developers, please note that some of these CCS error codes may be
      returned if you do not supply the correct data to CCS_Init or if
      you have linked with the wrong library for your target environment.
      For example, CCS_E_EXPIRED may be returned on NetWare if you do
      not pass the correct NLM handle to CCS_Init.  The parameter to CCS_Init
      is platform specific.
   */

#define CCS_E_NOT_INITIALIZED                   (-1494) /* 0xFA2A */
#define CCS_E_NO_SERVER_KEY                     (-1495) /* 0xFA29 */
#define CCS_E_NOT_AUTHENTICATED                 (-1496) /* 0xFA28 */
#define CCS_E_AUTHENTICATION_FAILURE            (-1497) /* 0xFA27 */
#define CCS_E_ALREADY_AUTHENTICATED             (-1498) /* 0xFA26 */
#define CCS_E_EXPIRED                           (-1499) /* 0xFA25 */


/* Attribute Type Identifiers */
#define NICI_A_VARIABLELEN       0x80000000L /* Top bit set in varaible length attributes */
#define NICI_A_SPECIAL           0x40000000L /* Special attributes are only set through "special" APIs.*/
#define NICI_A_UNUSED            0           /* The attribute is not intialized */

/* Fv represents a fixed value and Vv represents a variable value*/

#define Fv(x)  (x)
#define Vv(x)  (x + NICI_A_VARIABLELEN)

#define NICI_A_ARCHIVE           Fv(0x01)
#define NICI_A_ARCHIVE_NAME      Vv(0x02)
#define NICI_A_ARCHIVE_SERIAL    Vv(0x03)
#define NICI_A_CA_ISSUER         Vv(0x04)
#define NICI_A_CA_KEYID          Vv(0x05)
#define NICI_A_CA_SERIAL         Vv(0x06)
#define NICI_A_CLASS             Fv(0x07)
#define NICI_A_DERIV_ID          Vv(0x08)
#define NICI_A_EXTENSIONS        Vv(0x09)
#define NICI_A_FEATURE           Fv(0x0a)
#define NICI_A_FOREIGN           Fv(0x0b)
#define NICI_A_GLOBAL            Fv(0x0c)
#define NICI_A_ISSUER            Vv(0x0d)
#define NICI_A_ISSUER_ALT        Vv(0x0e)
#define NICI_A_ISSUER_UID        Vv(0x0f)
#define NICI_A_KEY_FORMAT        Vv(0x10)
#define NICI_A_KEY_ID            Vv(0x11)
#define NICI_A_KEY_NUMBER        Fv(0x12)
#define NICI_A_KEY_QUALITY       Fv(0x13)
#define NICI_A_KEY_SIZE          Fv(0x14)
#define NICI_A_KEY_TYPE          Fv(0x15)
#define NICI_A_KEY_USAGE         Fv(0x16)
#define NICI_A_KEY_VALUE         Vv(0x17)
#define NICI_A_MAX_KEY_SIZE      Fv(0x18)
#define NICI_A_MIN_KEY_SIZE      Fv(0x19)
#define NICI_A_OP_USAGE          Fv(0x1a)
#define NICI_A_ORIGINAL_DATA     Vv(0x1b)
#define NICI_A_PARTITION         Vv(0x1c)
#define NICI_A_RELOAD            Fv(0x1d)
#define NICI_A_SERIAL            Vv(0x1e)
#define NICI_A_SENSITIVE         Fv(0x1f)
#define NICI_A_SIGN_ALGORITHM    Vv(0x20)
#define NICI_A_SIGN_PARAMETER    Vv(0x21)
#define NICI_A_SIGN_QUALITY      Fv(0x22)
#define NICI_A_SUBJECT           Vv(0x23)
#define NICI_A_SUBJECT_UID       Vv(0x24)
#define NICI_A_VALIDITY          Vv(0x25)
#define NICI_A_VERSION           Vv(0x26)
#define NICI_A_XENG_HANDLE       Fv(0x27)
#define NICI_A_SUBJECT_ALT       Vv(0x28)
#define NICI_A_KEY_ID_ALT        Vv(0x29)

/* Special Attributes   */

#define NICI_A_SESSIONKEY        (NICI_A_SPECIAL + 0x0000434f /* 'CO' */)  /* Value is connection ID */
#define NICI_A_DOMAINKEY         (NICI_A_SPECIAL + 0x00004E41 /* 'NA'*/ )  /* Value is a name */
#define NICI_A_INTERNAL          (NICI_A_SPECIAL + 0x00004944 /* 'ID' */)  /* Value is identifier */

/*
   Attribute values (AV) are well know handles to internal objects, objects created inside NICI but
   available outside NICI - the "well known" nici objects.
   They are used in the value of the NICI_A_FEATURE attribute with FindObjects.
*/

#define NICI_AV_CA                     (0x01)
#define NICI_AV_DISTRIBUTION           (0x02)
#define NICI_AV_FOREIGN                (0x03)
#define NICI_AV_PARTITION              (0x04)
#define NICI_AV_SESSION                (0x05)
#define NICI_AV_STORAGE                (0x06)

/* Object Class Identifiers */
#define NICI_O_POLICY                  1
#define NICI_O_CERTIFICATE             2
#define NICI_O_SECRET_KEY              3
#define NICI_O_PRIVATE_KEY             4
#define NICI_O_PUBLIC_KEY              5

/* Key & Algorithm Usage Flags */

#define NICI_F_NO_USAGE                0L

#define NICI_F_ENCRYPT                 0x00000001L
#define NICI_F_DECRYPT                 0x00000002L
#define NICI_F_SIGN                    0x00000004L
#define NICI_F_SIGN_RECOVER            0x00000008L
#define NICI_F_VERIFY                  0x00000010L
#define NICI_F_VERIFY_RECOVER          0x00000020L
#define NICI_F_WRAP                    0x00000040L
#define NICI_F_UNWRAP                  0x00000080L
#define NICI_F_DERIVE                  0x00000100L
#define NICI_F_GENERATE_KEY            0x00000200L
#define NICI_F_GENERATE_KEYPAIR        0x00000400L
#define NICI_F_DIGEST                  0x00000800L
#define NICI_F_RANDOM                  0x00001000L
#define NICI_F_OBFUSCATE               0x00002000L

/* Key type flags use upper 4 bits of the flags.*/
#define NICI_F_KEY_TYPE_MASK           0xE0000000L
#define NICI_F_KEY_TYPE_NONE           0x00000000L
#define NICI_F_KEY_TYPE_SECRET         0x20000000L
#define NICI_F_KEY_TYPE_PUBLIC         0x40000000L
#define NICI_F_KEY_TYPE_PRIVATE        0x80000000L

/* Additional Key Usage Flags */
#define NICI_F_DATA_ENCRYPT            0x00010000L
#define NICI_F_AUTH_ENCRYPT            0x00020000L
#define NICI_F_KM_ENCRYPT              0x00040000L
#define NICI_F_LIC_ENCRYPT             0x00080000L
#define NICI_F_DATA_DECRYPT            0x00100000L
#define NICI_F_AUTH_DECRYPT            0x00200000L
#define NICI_F_KM_DECRYPT              0x00400000L
#define NICI_F_LIC_DECRYPT             0x00800000L
#define NICI_F_EXTRACT                 0x01000000L
#define NICI_F_DISALLOW_GENERATE       0x02000000L

/* Operational Key Usage Flags */

#define NICI_F_NO_OP_USAGE             0L

#define NICI_F_OP_ROOT                 0x00000001L
#define NICI_F_OP_XENG_SIGN            0x00000002L
#define NICI_F_OP_XENG_VERIFY          0x00000004L
#define NICI_F_OP_XLIB_SIGN            0x00000008L
#define NICI_F_OP_XLIB_VERIFY          0x00000010L
#define NICI_F_OP_XMGR_SIGN            0x00000020L
#define NICI_F_OP_XMGR_VERIFY          0x00000040L
#define NICI_F_OP_CLI_XMGR_SIGN        0x00000080L
#define NICI_F_OP_CLI_XMGR_VERIFY      0x00000100L
#define NICI_F_OP_XMGR_ASST_SIGN       0x00000200L
#define NICI_F_OP_XMGR_ASST_VERIFY     0x00000400L
#define NICI_F_OP_XSUP_SIGN            0x00000800L
#define NICI_F_OP_XSUP_VERIFY          0x00001000L
#define NICI_F_OP_ARCHIVE              0x00002000L
#define NICI_F_OP_ARCHIVE_RECOVER      0x00004000L
#define NICI_F_OP_CA_SIGN              0x00008000L
#define NICI_F_OP_CA_VERIFY            0x00010000L
#define NICI_F_OP_POLICY_SIGN          0x00020000L
#define NICI_F_OP_POLICY_VERIFY        0x00040000L



/************************************************************************/

/*
	Flags for use with CCS_CreateContext

	Warning to developers:

	Use of NICI_F_CTX_FORCE_CCS_INIT is not portable across platforms
	and carries with it some serious side effects.

	1. It is not available on NLM platforms due to complications in
		manipulating the moduleID.  On NLM platforms using this flag
		will result in error NICI_E_FUNCTION_NOT_SUPPORTED.

	2. On other platforms it supplies its own modlueID parameter,
		which may conflict with that provided by other calls to CCS_Init.
		Because of this calls to CCS_Init and CCS_CreateContext using the
		flag may cause serious conflicts.

	3. It may not be possible to determine how many times CCS_Init has been
	   called inorder to supply matching calls to CCS_Shutdown.  In particular,
		the internal counter used in CCS_Init is much more likely to wrap.  The
		only acceptable alternative is probably to call CCS_ShutdownAll.

	4. Since the main reason for using CLIENT_NICI_FORCE_INITIALIZATION is
		because you cannot determine when to call CCS_Init yourself you may
		have equal difficulty calling CCS_Shutdown or CCS_ShutdownAll.

	5. If the call to CCS_Init returns an error code then that error code
		will be returned from CCS_CreateContext immediately and the context
		will not be created.
*/

#define NICI_F_CTX_FORCE_CCS_INIT 0x00000001L

/************************************************************************/
/*
   NICI_KEY_TYPE values used in NICI_GetPolicyInfo and elsewhere.
*/
#define NICI_K_DES            0x00000001L
#define NICI_K_DES3X          0x00000002L
#define NICI_K_DH             0x00000003L
#define NICI_K_DSA            0x00000004L
#define NICI_K_RC2            0x00000005L
#define NICI_K_RC4            0x00000006L
#define NICI_K_RSA            0x00000007L
#define NICI_K_SECRET         0x00000008L
#define NICI_K_CAST           0x00000009L
#define NICI_K_DES2X          0x0000000AL
#define NICI_K_RC5            0x0000000BL
#define NICI_K_AES            0x0000000CL

/*
   The null key types are used for information for the NULL XENG.
*/

#define NICI_K_NULL_SECRET    0x00008000L
#define NICI_K_NULL_PUBLIC    0x00004000L

/************************************************************************/

/* Reserved Object Handle Value */
/*
   These values must always be "-1" in order to be compatible
   with other constants used in checking for valid handles.
*/
#define NICI_H_INVALID        0xFFFFFFFFL
#define NICI_H_NULL           0xFFFFFFFFL

/* Reserved Key Type Value */
#define NICI_K_INVALID        NULL

/* Quality values */
#define NICI_Q_UNKNOWN        0


/* The following data types are unique to NICI */
typedef struct NICI_VERSION
{
   nuint8      major;        /* Integer portion of version*/
   nuint8      minor;        /* hundredths portion of version*/
}NICI_VERSION;

typedef NICI_VERSION N_FAR *NICI_VERSION_PTR;

typedef struct  NICI_VALIDITY
{
   nuint32      begin;
   nuint32      end;
}NICI_VALIDITY;

typedef NICI_VALIDITY N_FAR    *NICI_VALIDITY_PTR;

typedef struct NICI_MODULE_INFO
{
   pnuint8             modName;
   pnuint8             fileName;
   NICI_VERSION        version;
   NICI_VALIDITY       validity;
}NICI_MODULE_INFO;

typedef NICI_MODULE_INFO      N_FAR  *NICI_MODULE_INFO_PTR;

typedef nuint32               NICI_CC_HANDLE;
typedef NICI_CC_HANDLE        N_FAR  *NICI_CC_HANDLE_PTR;

typedef nuint32               NICI_OBJECT_HANDLE;
typedef NICI_OBJECT_HANDLE    N_FAR  *NICI_OBJECT_HANDLE_PTR;

typedef nuint32               NICI_OBJECT_CLASS;
typedef NICI_OBJECT_CLASS     N_FAR  *NICI_OBJECT_CLASS_PTR;

typedef nuint32               NICI_ATTRIBUTE_TYPE;
typedef NICI_ATTRIBUTE_TYPE   N_FAR *NICI_ATTRIBUTE_TYPE_PTR;

typedef nuint32               NICI_KEY_TYPE;
typedef NICI_KEY_TYPE         N_FAR  *NICI_KEY_TYPE_PTR;

typedef pnuint8               NICI_ALGORITHM_ID;
typedef NICI_ALGORITHM_ID     N_FAR *NICI_ALGORITHM_ID_PTR;

typedef nuint32               NICI_KM_STRENGTH;
typedef NICI_KM_STRENGTH      N_FAR  *NICI_KM_STRENGTH_PTR;

/* These are the values NICI_KM_STRENGTH can take on.*/
#define NICI_KM_UNSPECIFIED   0
#define NICI_KM_NULL          1
#define NICI_KM_EXPORTABLE    2
#define NICI_KM_DOMESTIC      3

/*
   This structure is used when an attribute is an "interger" value.
   Two values are allowed, just for flexibility.
   This structure is 32 bit sensitve because the nuint32    must
   occupy the same space as a pointer in the NICI_ATTRIBUTE structure.
*/

typedef  struct   NICI_FIXED_LEN_ATTRIBUTE
{
   nuint32     hasValue;      /* Zero if no value has been assigned */
   nuint32     value;         /* Value, if present. */
   nuint32     valueInfo;     /* Optional field specific to a type*/
}NICI_FIXED_LEN_ATTRIBUTE;

typedef struct    NICI_VAR_LEN_ATTRIBUTE
{
   nptr        valuePtr;
   nuint32     valueLen;
   nuint32     valueInfo;   /* Optional field specific to a type*/
}NICI_VAR_LEN_ATTRIBUTE;

typedef struct NICI_ATTRIBUTE
{
   NICI_ATTRIBUTE_TYPE type;   /* High bit set if fixed length types*/
   union
   {
     NICI_FIXED_LEN_ATTRIBUTE f; /* Fixed length attribute */
     NICI_VAR_LEN_ATTRIBUTE   v; /* Variable length attribute */
   }u;
}NICI_ATTRIBUTE;


typedef NICI_ATTRIBUTE           N_FAR *NICI_ATTRIBUTE_PTR;

typedef  nuint32                 NICI_ALGORITHM_HANDLE;
typedef  NICI_ALGORITHM_HANDLE   N_FAR *NICI_ALGORITHM_HANDLE_PTR;

typedef enum NICI_PACKET_SIGN_CONST
{
   PS_RESET=0,
   PS_CURRENT_UPDATE,
   PS_CURRENT_NOUPDATE,
   PS_PREVIOUS_UPDATE,
   PS_PREVIOUS_NOUPDATE
}NICI_PACKET_STATE;

/************************************************************************/

/*
   Parameter information is exchanged between the XMGR and XENG to convey
   information about any special parameters needed by a particular algorithm.

   The defined parameter types are algorithm specific.

   Allocate a NICI_PARAMETER_INFO structure with enough space to hold
   as many NICI_PARAMETER_DATA structures as are needed to satisfy the
   count.

   Those parameters with the NICI_P_VARIABLE flag set require the pointer
   and length (the struct Block part of the union) when used as input
   parameters.  The others use only the value, since they are integers.

   When an XENG signals that it supports an optional parameter the caller may
   choose to supply or omit it.
*/

#define NICI_P_VARIABLE             0x80000000L
#define NICI_P_OPTIONAL             0x40000000L /* Parameter need not be supplied */

#define NICI_P_BASE                 (NICI_P_VARIABLE + 1)
#define NICI_P_DLDH                 (NICI_P_VARIABLE + 2)
#define NICI_P_EFF_BITS             3
#define NICI_P_IV                   (NICI_P_VARIABLE + 4)
#define NICI_P_PRIME                (NICI_P_VARIABLE + 5)
#define NICI_P_SUBPRIME             (NICI_P_VARIABLE + 6)
#define NICI_P_SALT                 (NICI_P_VARIABLE + 7)
#define NICI_P_COUNT                8
#define NICI_P_PRIME_LENGTH         9
#define NICI_P_PUBLIC               (NICI_P_VARIABLE + 10)
#define NICI_P_ROUNDS               11
#define NICI_P_WORDSIZE             12
#define NICI_P_DSAPARAMS            (NICI_P_VARIABLE + 13)
#define NICI_P_FIPS_MODE            (NICI_P_OPTIONAL + 14)  /* Choose FIPS mode of operation.*/
#define NICI_P_HASH_FUNC            (NICI_P_VARIABLE + 15)  /* RSA OAEP */
#define NICI_P_MASK_GEN_FUNC        (NICI_P_VARIABLE + 16)  /* RSA OAEP */
#define NICI_P_MASK_GEN_FUNC_ALG    (NICI_P_VARIABLE + 17)  /* RSA OAEP */
#define NICI_P_PSOURCE_FUNC         (NICI_P_VARIABLE + 18)  /* RSA OAEP */


typedef struct NICI_PARAMETER_DATA
{
   nuint32        parmType;
   union
   {
     nuint32      value;

     struct Range
     {
       nuint32    min;
       nuint32    max;
     }r;

      struct Block
     {
       nuint32    len;
       pnuint8    ptr;
     }b;
   }u;
}NICI_PARAMETER_DATA;

typedef NICI_PARAMETER_DATA *NICI_PARAMETER_DATA_PTR;

typedef struct NICI_PARAMETER_INFO
{
   nuint32              count;  /* Count of elements in the parameter array */
   NICI_PARAMETER_DATA  parms[1];   /* An array of NICI_PARAMETER_DATA starts here */
}NICI_PARAMETER_INFO;

typedef NICI_PARAMETER_INFO   N_FAR *NICI_PARAMETER_INFO_PTR;

/************************************************************************/

/* flags for use with CCS_InitializeClient() */

#define CLIENT_NICI_FORCE_INITIALIZATION    0x00000001

/************************************************************************/

typedef struct NICI_ALGORITHM
{
    NICI_ALGORITHM_ID   algorithm;
    NICI_PARAMETER_INFO N_FAR *parameter;
    nuint32                   parameterLen;
} NICI_ALGORITHM;

typedef NICI_ALGORITHM     N_FAR *NICI_ALGORITHM_PTR;

typedef struct NICI_ALGORITHM_INFO
{
   NICI_OBJECT_HANDLE   algHandle;
   NICI_KEY_TYPE        keyType;
   nflag32              usages;
   nuint32              mayRepeat;  /* May call update after final*/
   nuint32              mayRestart; /* May call init after final      */
   nuint32              blockLen;
   nuint32              algIDLen;
   nuint32              keyFmtLen;
   nuint32              parmCount;
   nuint32              minKeySize; /* Minimum valid size.  Zero if no key is used */
   nuint32              maxKeySize; /* Maximum valid size.  Zero if no key is used */
} NICI_ALGORITHM_INFO;

typedef NICI_ALGORITHM_INFO N_FAR *NICI_ALGORITHM_INFO_PTR;

typedef nuint32                  NICI_QUALITY;
typedef NICI_QUALITY N_FAR       *NICI_QUALITY_PTR;


/* The following definitions are extracted from asn1def.h (T.A.)
 *
 * OID Prefix definitions.
 * The macro parameter is the additional number of bytes to be appended
 *
 * USPrefix {iso(1) member-body(2) us(840)}
 * OIWPrefix {iso(1) identified-organization(3) oiw(14) secsig(3)}
 * Novell {joint-iso-ccitt(2) country(16)  us(840) organization (1) novell(113719)}
 * RSADSI {iso(1) member-body(2) us(840) rsadsi(113549)}
 * PKCS {iso(1) member-body(2) us(840) rsadsi(113549) pkcs(1)}
 * RSADigest {iso(1) member-body(2) us(840) rsadsi(113549) digest(2)}
 * PKCS-1 {iso(1) member-body(2) us(840) rsadsi(113549) pkcs(1) pkcs-1(1)}
 * OIWAlgorithm {iso(1) identified-organization(3) oiw(14) secsig(3) algorithm(2)}
 * dsa-with-sha1 {iso(1) member-body(2) us(840) x9-57(10040)}
 * X9_57 {iso(1) member-body(1) us(840) x9-57(10040)}
 * ANSI_X942 {iso(1) member-body(2) us(840) ansi-x942(10046)}
 */
#define IDV_ISOMemberBodyPrefix(l)              0x06, ((l)+1), 40*1+2
#define IDV_OIWPrefix(l)                        0x06, ((l)+2), 40*1+3, 14
#define IDV_ISOCCITTCountryPrefix(l)            0x06, ((l)+1), 40*2+16

#define IDV_ISOCCITTUSOrganizationPrefix(l)     IDV_ISOCCITTCountryPrefix((l)+3), 0x86, 0x48, 1
#define IDV_USPrefix(l)                         IDV_ISOMemberBodyPrefix((l)+2), 0x86, 0x48
#define IDV_OIWSecSigPrefix(l)                  IDV_OIWPrefix((l)+1), 3
#define IDV_RSADSIPrefix(l)                     IDV_USPrefix((l)+3), 0x86, 0xF7, 0x0D
#define IDV_NovellPrefix(l)                     IDV_ISOCCITTUSOrganizationPrefix((l)+3), 0x86, 0xF8, 0x37
#define IDV_NovellApplicationsPrefix(l)         IDV_NovellPrefix((l)+1), 1
#define IDV_NovellSecurityPrefix(l)             IDV_NovellApplicationsPrefix((l)+1), 2
#define IDV_NortelPrefix(l)                     IDV_USPrefix((l)+3), 0x86, 0xF6, 0x7D
#define IDV_NortelAlgorithmPrefix(l)            IDV_NortelPrefix((l)+2), 7, 66
#define IDV_PKCSPrefix(l)                       IDV_RSADSIPrefix((l)+1), 1
#define IDV_RSADSIDigestAlgorithmPrefix(l)      IDV_RSADSIPrefix((l)+1), 2
#define IDV_RSADSIEncryptionAlgorithmPrefix(l)  IDV_RSADSIPrefix((l)+1), 3
#define IDV_PKCS1Prefix(l)                      IDV_PKCSPrefix((l)+1), 1
#define IDV_PKCS3Prefix(l)                      IDV_PKCSPrefix((l)+1), 3
#define IDV_PKCS5Prefix(l)                      IDV_PKCSPrefix((l)+1), 5
#define IDV_PKCS12Prefix(l)                     IDV_PKCSPrefix((l)+1), 12
#define IDV_PKCS12pbePrefix(l)                  IDV_PKCS12Prefix((l)+1), 1
#define IDV_OIWAlgorithmPrefix(l)               IDV_OIWSecSigPrefix((l)+1), 2
#define IDV_X9_57Prefix(l)                      IDV_USPrefix((l)+2), 0xCE, 0x38
#define IDV_ANSI_X942Prefix(l)                  IDV_USPrefix((l)+2), 0xCE, 0x3E
#define IDV_USGovPrefix(l)                      IDV_ISOCCITTUSOrganizationPrefix((l)+1), 0x65
#define IDV_CSORPrefix(l)                       IDV_USGovPrefix((l)+1), 3
#define IDV_NISTAlgorithmsPrefix(l)             IDV_CSORPrefix((l)+1), 4
#define IDV_AESPrefix(l)                        IDV_NISTAlgorithmsPrefix((l)+1), 1


/*
 * Encryption
 */
#define IDV_PKCS1_rsaEncryption                 IDV_PKCS1Prefix(1), 1
#define IDV_DES_CBC                             IDV_OIWAlgorithmPrefix(1), 7
#define IDV_RC2_CBC                             IDV_RSADSIEncryptionAlgorithmPrefix(1), 2
#define IDV_RC4                                 IDV_RSADSIEncryptionAlgorithmPrefix(1), 4
#define IDV_DES_EDE3_CBC                        IDV_RSADSIEncryptionAlgorithmPrefix(1), 7
#define IDV_RC5_CBCPad                          IDV_RSADSIEncryptionAlgorithmPrefix(1), 9
#define IDV_cast5CBC                            IDV_NortelAlgorithmPrefix(1), 10

/* Also see Novell specific AES CBCPad modes defined later. */
#define IDV_AES128CBC                           IDV_AESPrefix(1), 2
#define IDV_AES192CBC                           IDV_AESPrefix(1), 22
#define IDV_AES256CBC                           IDV_AESPrefix(1), 42

/*
 * Signature
 */
#define IDV_PKCS1_md2withRSAEncryption          IDV_PKCS1Prefix(1), 2
#define IDV_PKCS1_md4withRSAEncryption          IDV_PKCS1Prefix(1), 3
#define IDV_PKCS1_md5withRSAEncryption          IDV_PKCS1Prefix(1), 4
#define IDV_PKCS1_sha1withRSAEncryption         IDV_PKCS1Prefix(1), 5
#define IDV_PKCS1_ripemd160withRSAEncryption    IDV_PKCS1Prefix(1), 6
#define IDV_PKCS1_rsaES_OAEP                    IDV_PKCS1Prefix(1), 7
#define IDV_PKCS1_MFG1                          IDV_PKCS1Prefix(1), 8
#define IDV_PKCS1_pSpecified                    IDV_PKCS1Prefix(1), 9
#define IDV_DSA                                 IDV_X9_57Prefix(2), 4, 1
#define IDV_DSAWithSHA1                         IDV_X9_57Prefix(2), 4, 3

#define IDV_RSADSI_hmacWithMD5                  IDV_RSADSIDigestAlgorithmPrefix(1), 6
#define IDV_RSADSI_hmacWithSHA1                 IDV_RSADSIDigestAlgorithmPrefix(1), 7

/*
 * Password Based Encryption
 */
#define IDV_pbeWithMD2AndDES_CBC                IDV_PKCS5Prefix(1), 1
#define IDV_pbeWithMD5AndDES_CBC                IDV_PKCS5Prefix(1), 3
#define IDV_pbeWithMD2AndRC2_CBC                IDV_PKCS5Prefix(1), 4
#define IDV_pbeWithMD5AndRC2_CBC                IDV_PKCS5Prefix(1), 6
#define IDV_pbeWithSHA1AndDES_CBC               IDV_PKCS5Prefix(1), 10
#define IDV_pbeWithSHA1AndRC2_CBC               IDV_PKCS5Prefix(1), 11
#define IDV_pbkdf2                              IDV_PKCS5Prefix(1), 12
#define IDV_pbes2                               IDV_PKCS5Prefix(1), 13
#define IDV_pbmac1                              IDV_PKCS5Prefix(1), 14

#define IDV_pbeWithSHA1And128BitRC4             IDV_PKCS12pbePrefix(1), 1
#define IDV_pbeWithSHA1And40BitRC4              IDV_PKCS12pbePrefix(1), 2
#define IDV_pbeWithSHA1And3Key3xDES_CBC         IDV_PKCS12pbePrefix(1), 3
#define IDV_pbeWithSHA1And2Key3xDES_CBC         IDV_PKCS12pbePrefix(1), 4
#define IDV_pbeWithSHA1And128BitRC2_CBC         IDV_PKCS12pbePrefix(1), 5
#define IDV_pbeWithSHA1And40BitRC2_CBC          IDV_PKCS12pbePrefix(1), 6

/*
 * Key Exchange
 */
#define IDV_DHKeyAgreement                      IDV_PKCS3Prefix(1), 1
#define IDV_DHPublicNumber                      IDV_ANSI_X942Prefix(2), 2, 1


/*
   NOVELL Object Identifier is registered with ANSI.

   NOVELL OID := {joint-iso-ccitt(2) country(16) US(840) organization (1) novell(113719)}
   concatenated with
   Applications OID := {novell applications(1)}
   security OID := {applications security(2)}

   In what follows:
   NICI_AttributePrefix is for application security ATTRIBUTES.
   NICI_ObjectPrefix is for application security OBJECTS.
   NICI_AlgorithmPrefix is for application security ALGORITHMS.

   NOVELL OID :=  0x06, 0x0B, 0x60, 0x86, 0x48, 0x01, 0x86, 0xF8, 0x37

   The integer values in some identifiers are coded base 128 (in standard
   ASN.1 Integer formats) with the high bit set in each byte except the last.
   (See The PKCS Standards documents from RSA:  A Layman's Guide to a Subset
   of ASN.1, BER and DER.

   TB is a macro which produces a two byte intger value in the format just
   described.
*/

#define TB(N)  (0x80 | (N / 128)), (N % 128)

#define NICI_AttributePrefix(l)     IDV_NovellSecurityPrefix((l)+1), 4
#define NICI_ObjectPrefix(l)        IDV_NovellSecurityPrefix((l)+1), 6
#define NICI_AlgorithmPrefix(l)     IDV_NovellSecurityPrefix((l)+1), 8

/************************************************************************/
/*
** There is no guarantee that any of these values are actually supported
** by a particular NICI implementation.
**
** The external byte arrays identified as NICI_OID are also reserved for
** future use.  They are intended to suggest standard names for referring
** to the IDV values.
*/
/************************************************************************/

/*
 * Legacy PKCS#1 definitions
*/

#define PKCS1_rsaEncryption            IDV_PKCS1_rsaEncryption
#define PKCS1_md2withRSAEncryption     IDV_PKCS1_md2withRSAEncryption
#define PKCS1_md5withRSAEncryption     IDV_PKCS1_md5withRSAEncryption
#define PKCS1_sha1withRSAEncryption    IDV_PKCS1_sha1withRSAEncryption
#define RSADSI_RC2CBC                  IDV_RC2_CBC
#define RSADSI_RC4                     IDV_RC4


/* These algorithm ID defintions are useful for initializations.*/

#define IDV_NULL                          NICI_AlgorithmPrefix(1), 1 /* 0x01*/
#define IDV_NULL_SECRET                   NICI_AlgorithmPrefix(1), 2 /* 0x02*/
#define IDV_NULL_PUBLIC                   NICI_AlgorithmPrefix(1), 3 /* 0x03*/

/************************************************************************/

/*
   Some missing numbers are due to removal of some of the algorithms
   defined by other standards, such as PKCS#1, PKCS#3, and PKCS#5 above.
*/
#define IDV_DES_CBC_IV8                   NICI_AlgorithmPrefix(1), 22 /* 0x16 */
#define IDV_DES_CBCPadIV8                 NICI_AlgorithmPrefix(1), 23 /* 0x17 */
#define IDV_DES_EDE2_CBC_IV8              NICI_AlgorithmPrefix(1), 24 /* 0x18 */
#define IDV_DES_EDE2_CBCPadIV8            NICI_AlgorithmPrefix(1), 25 /* 0x19 */
#define IDV_DES_EDE3_CBC_IV8              NICI_AlgorithmPrefix(1), 26 /* 0x1a */
#define IDV_DES_EDE3_CBCPadIV8            NICI_AlgorithmPrefix(1), 27 /* 0x1b */
#define IDV_RC5_CBC                       NICI_AlgorithmPrefix(1), 28 /* 0x1c */
#define IDV_md2withRSAEncryptionBSafe1    NICI_AlgorithmPrefix(1), 29 /* 0x1d */
#define IDV_md5withRSAEncryptionBSafe1    NICI_AlgorithmPrefix(1), 30 /* 0x1e */
#define IDV_sha1withRSAEncryptionBSafe1   NICI_AlgorithmPrefix(1), 31 /* 0x1f */
#define IDV_LMDigest                      NICI_AlgorithmPrefix(1), 32 /* 0x20 */
#define IDV_UNIXCrypt                     NICI_AlgorithmPrefix(1), 33 /* 0x21 */
#define IDV_MD2                           NICI_AlgorithmPrefix(1), 40 /* 0x28 */
#define IDV_MD5                           NICI_AlgorithmPrefix(1), 50 /* 0x32 */
#define IDV_IKEhmacWithSHA1_RSA           NICI_AlgorithmPrefix(1), 51 /* 0x33 */
#define IDV_IKEhmacWithMD5_RSA            NICI_AlgorithmPrefix(1), 52 /* 0x34 */
#define IDV_RC2_CBCPad                    NICI_AlgorithmPrefix(1), 69 /* 0x45 */
#define IDV_SHA1                          NICI_AlgorithmPrefix(1), 82 /* 0x52 */
#define IDV_RC2_CBC_BSAFE1                NICI_AlgorithmPrefix(1), 92 /* 0x5C */
#define IDV_MD4                           NICI_AlgorithmPrefix(1), 95 /* 0x5F */
#define IDV_NOV_MD4_NOPAD                 NICI_AlgorithmPrefix(1), 96 /* 0x60 */

/*
   AES Padding modes needed for key wrapping and other uses.
   These OIDs are unique to Novell, unlike some other AES OIDs
   defined by NIST.
*/
#define IDV_NOV_AES128CBCPad              NICI_AlgorithmPrefix(1), 97 /* 0x61 */
#define IDV_NOV_AES192CBCPad              NICI_AlgorithmPrefix(1), 98 /* 0x62 */
#define IDV_NOV_AES256CBCPad              NICI_AlgorithmPrefix(1), 99 /* 0x63 */

/*
   Higher bit SHA algorithms:  See  FIPS 180-2

   Reserve an ID for SHA 224 if it is ever implemented
   #define IDV_SHA224                     NICI_AlgorithmPrefix(1), 100  0x64
*/
#define IDV_SHA256                     NICI_AlgorithmPrefix(1), 101 /* 0x65 */
#define IDV_SHA384                     NICI_AlgorithmPrefix(1), 102 /* 0x66 */
#define IDV_SHA512                     NICI_AlgorithmPrefix(1), 103 /* 0x67 */


/*--------------------------------------------------------------------------*/
/*  NICI_AlgorithmPrefix(1) OIDs with values above 127 require two bytes    */
/*--------------------------------------------------------------------------*/

/*
   Generic IDs used by NICI to represent groups of BSAFE info types.
   These IDs have a two byte value after the basic Novell algorithm ID.
   The high bit in the first byte is set and the rest of the two bytes
   are a base 128 representation of the ID value.
*/
#define IDV_NOV_MD4_PACKET                NICI_AlgorithmPrefix(2), TB(130) /* 0x81, 0x02 MD4 variant for packet signature*/
#define IDV_NOV_RSA_BSAFE1                NICI_AlgorithmPrefix(2), TB(131) /* 0x81, 0x03 old NDS BSAFE1 (B_BSAFE1_PAD_CHECKSUM)*/
#define IDV_NOV_NW_PASSWORD               NICI_AlgorithmPrefix(2), TB(132) /* 0x81, 0x04 NW password hash */
#define IDV_NOV_OBFUSCATE_1               NICI_AlgorithmPrefix(2), TB(133) /* 0x81, 0x05 Novell obfuscation*/
#define IDV_NOV_MABLE_SIGN_1              NICI_AlgorithmPrefix(2), TB(134) /* 0x81, 0x06 Mable SHA1 RSA Raw Encrypt Signature */
#define IDV_NOV_RSA_BSAFE1Pad             NICI_AlgorithmPrefix(2), TB(135) /* 0x81, 0x07 RSA BSAFE1 (B_BSAFE1_PAD)*/
#define IDV_NOV_RSA_BSAFE1Raw             NICI_AlgorithmPrefix(2), TB(136) /* 0x81, 0x08 RSA BSAFE1 (B_BSAFE1_RAW)*/
#define IDV_NOV_BSAFE1_MD                 NICI_AlgorithmPrefix(2), TB(137) /* 0x81, 0x09 RSA BSAFE1 MD */
#define IDV_TLS_KEYEXCHANGE_RSASIGN       NICI_AlgorithmPrefix(2), TB(140)



#if !defined(_NWCCS_H_SUPPRESS_EXTERNS)

/*
   DEVELOPERS:  If the presence of the following external declarations
   causes problems you may suppress them though the above definition.
*/


/************************************************************************
* These external declarations are for use by the XENG, XMGR, and XLIBS.
* They are declared here to encourage the use of standard names within
* all the modules.
*
* They correspond to the IDVs defined above and should be initialized
* to that IDV when used.
*
* For example: NICI_BYTE NICI_OID_NULL[] = {IDV_NULL};
*
************************************************************************/

N_EXTERN_VAR nuint8  NICI_OID_NULL[];
N_EXTERN_VAR nuint8  NICI_OID_NULL_SECRET[];
N_EXTERN_VAR nuint8  NICI_OID_NULL_PUBLIC[];
N_EXTERN_VAR nuint8  NICI_OID_DES_CBC_IV8[];
N_EXTERN_VAR nuint8  NICI_OID_DES_EDE3_CBC_IV8[];
N_EXTERN_VAR nuint8  NICI_OID_MD5[];
N_EXTERN_VAR nuint8  NICI_OID_RC2_CBCPad[];
N_EXTERN_VAR nuint8  NICI_OID_RC2_CBC_BSAFE1[];
N_EXTERN_VAR nuint8  NICI_OID_RC5_CBC[];
N_EXTERN_VAR nuint8  NICI_OID_RC5_CBCPad[];
N_EXTERN_VAR nuint8  NICI_OID_MD4[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_MD4_NOPAD[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_MD4_PACKET[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_RSA_BSAFE1[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_RSA_BSAFE1Pad[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_NW_PASSWORD[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_OBFUSCATE[];
N_EXTERN_VAR nuint8  NICI_OID_NOV_MABLE_SIGN_1[];
N_EXTERN_VAR nuint8  NICI_OID_AES128CBC[];
N_EXTERN_VAR nuint8  NICI_OID_AES192CBC[];
N_EXTERN_VAR nuint8  NICI_OID_AES256CBC[];

N_EXTERN_VAR nuint8  OID_PKCS1_rsaEncryption[];
N_EXTERN_VAR nuint8  OID_PKCS1_md2withRSAEncryption[];
N_EXTERN_VAR nuint8  OID_PKCS1_md5withRSAEncryption[];
N_EXTERN_VAR nuint8  OID_PKCS1_sha1withRSAEncryption[];
N_EXTERN_VAR nuint8  NICI_OID_TLS_KeyExchangeRSASign[];
N_EXTERN_VAR nuint8  OID_RSADSI_hmacWithMD5[];
N_EXTERN_VAR nuint8  OID_RSADSI_hmacWithSHA1[];
N_EXTERN_VAR nuint8  OID_RSADSI_DHKeyAgreement[];
N_EXTERN_VAR nuint8  OID_RSADSI_RC4[];     /* initialize with RSADSI_RC4 - BSAFE AI_RC4 non-BER key*/
N_EXTERN_VAR nuint8  OID_RSADSI_RC2CBC[];  /* initialize with RSADSI_RC2CBC*/

#endif /* !defined(_NWCCS_H_SUPPRESS_EXTERNS) */

/************************************************************************/


/* CCS function prototypes - alphabetical by function name */

N_EXTERN_LIBRARY(int)
CCS_CreateContext(
   nflag32                    flags,
   NICI_CC_HANDLE_PTR         cryptoContext);

N_EXTERN_LIBRARY(int)
CCS_DataDecryptInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_DataEncryptInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_Decrypt(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    encryptedData,
   nuint32                    encryptedDataLen,
   pnuint8                    data,
   pnuint32                   dataLen);

N_EXTERN_LIBRARY(int)
CCS_DecryptFinal(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    lastDecryptedPart,
   pnuint32                   lastDecryptedPartLen);

N_EXTERN_LIBRARY(int)
CCS_DecryptRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_DecryptUpdate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    part,
   nuint32                    partLen,
   pnuint8                    decryptedPart,
   pnuint32                   decryptedPartLen);

N_EXTERN_LIBRARY(int)
CCS_DeObfuscate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    clearData,
   pnuint32                   clearDataLen);

N_EXTERN_LIBRARY(int)
CCS_DestroyContext(
   NICI_CC_HANDLE             cryptoContext);

N_EXTERN_LIBRARY(int)
CCS_DestroyObject(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         object);

N_EXTERN_LIBRARY(int)
CCS_Digest(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    digest,
   pnuint32                   digestLen);

N_EXTERN_LIBRARY(int)
CCS_DigestFinal(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    digest,
   pnuint32                   digestLen);

N_EXTERN_LIBRARY(int)
CCS_DigestInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm);

N_EXTERN_LIBRARY(int)
CCS_DigestRestart(
   NICI_CC_HANDLE             cryptoContext);

N_EXTERN_LIBRARY(int)
CCS_DigestUpdate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    part,
   nuint32                    partLen);

N_EXTERN_LIBRARY(int)
CCS_Encrypt(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    encryptedData,
   pnuint32                   encryptedDataLen);

N_EXTERN_LIBRARY(int)
CCS_EncryptFinal(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    lastEncryptedPart,
   pnuint32                   lastEncryptedPartLen);

N_EXTERN_LIBRARY(int)
CCS_EncryptRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_EncryptUpdate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    part,
   nuint32                    partLen,
   pnuint8                    encryptedPart,
   pnuint32                   encryptedPartLen);

N_EXTERN_LIBRARY(int)
CCS_FindObjects(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE_PTR     object,
   pnuint32                   objectCount);

N_EXTERN_LIBRARY(int)
CCS_FindObjectsInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ATTRIBUTE_PTR         findTemplate,
   nuint32                    attributeCount);

N_EXTERN_LIBRARY(int)
CCS_GenerateCertificate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         signatureAlgorithm,
   NICI_OBJECT_HANDLE         signatureKey,
   NICI_OBJECT_HANDLE         issuerCertificate,
   NICI_OBJECT_HANDLE         key,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   pnuint8                    certificateData,
   pnuint32                   certificateDataLen);

N_EXTERN_LIBRARY(int)
CCS_GenerateCertificateFromRequest(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         signKeyH,
   NICI_OBJECT_HANDLE         issuerCertificate,
   pnuint8                    CSR,
   nuint32                    CSRLen,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   pnuint8                    certData,
   pnuint32                   certDataLen);

N_EXTERN_LIBRARY(int)
CCS_GenerateCertificateFromNSRequest(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         signKeyH,
   NICI_OBJECT_HANDLE         issuerCertificate,
   pnuint8                    CSR,
   nuint32                    CSRLen,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   pnuint8                    certData,
   pnuint32                   certDataLen);

N_EXTERN_LIBRARY(int)
CCS_GenerateKey(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   pnbool8                    sizeChanged,
   NICI_OBJECT_HANDLE_PTR     key,
   nuint32                    archive);

N_EXTERN_LIBRARY(int)
CCS_GenerateKeyPair(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   pnbool8                    sizeChanged,
   NICI_OBJECT_HANDLE_PTR     publicKey,
   NICI_OBJECT_HANDLE_PTR     privateKey,
   nuint32                    archive);

N_EXTERN_LIBRARY(int)
CCS_GetAlgorithmInfo(
   NICI_ALGORITHM_ID          algorithm,
   NICI_ALGORITHM_INFO_PTR    algorithmInfo);

N_EXTERN_LIBRARY(int)
CCS_GetAlgorithmList(
   NICI_ALGORITHM_INFO_PTR    algInfoList,
   pnuint32                   count);

N_EXTERN_LIBRARY(int)
CCS_GetAttributeValue(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         object,
   NICI_ATTRIBUTE_PTR         attrTemplate,
   nuint32                    count);

N_EXTERN_LIBRARY(int)
CCS_GetCertificate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key,
   pnuint8                    certificateData,
   pnuint32                   certificateDataLen,
   NICI_OBJECT_HANDLE_PTR     issuerKey);

N_EXTERN_LIBRARY(int)
CCS_GetInfo(
   NICI_VERSION_PTR           version,
   pnuint8                    serial,
   pnuint32                   serialLen);

N_EXTERN_LIBRARY(int)
CCS_GetKMStrength(
   NICI_KM_STRENGTH_PTR       level);

N_EXTERN_LIBRARY(int)
CCS_GetLocalCertificate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key,
   pnuint8                    certificateData,
   pnuint32                   certificateDataLen,
   NICI_OBJECT_HANDLE_PTR     issuerKey);

N_EXTERN_LIBRARY(int)
CCS_GetModuleInfo(
   NICI_MODULE_INFO_PTR       info,
   pnuint32                   count);

N_EXTERN_LIBRARY(int)
CCS_GetMoreAlgorithmInfo(
   NICI_OBJECT_HANDLE         algHandle,
   NICI_ALGORITHM_ID_PTR      algID,
   NICI_ALGORITHM_ID_PTR      keyFormat,
   NICI_PARAMETER_INFO_PTR    parameters);

N_EXTERN_LIBRARY(int)
CCS_GetPartitionKey(
   NICI_CC_HANDLE             cryptoContext,
   nuint32                    entryID,
   pnuint8                    keyID,
   pnuint32                   keyIDLen,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_GetPolicyInfo(
   NICI_KEY_TYPE              keyType,
   nflag32                    keyUsage,
   pnuint32                   maxKeySize,
   pnuint32                   minKeySize,
   pnuint32                   archiveType,
   pnuint8                    archiveRqmts,
   pnuint32                   archiveRqmtsLen);

N_EXTERN_LIBRARY(int)
CCS_GetRandom(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    buffer,
   nuint32                    length);

N_EXTERN_LIBRARY(int)
CCS_Init(
   nptr                       moduleID);

N_EXTERN_LIBRARY(int)
CCS_InitializeClient(
   unsigned long              connHandle,
   unsigned long              flags,
   void                       *reserved);

N_EXTERN_LIBRARY(int)
CCS_InitializeClientGeneric(
   npproc                     genericSendReceive,
   unsigned long              connHandle,
   unsigned long              flags,
   void                       *reserved);

N_EXTERN_LIBRARY(int)
CCS_LoadCertificate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         verificationKey,
   pnuint8                    certificateData,
   nuint32                    certificateDataLen,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_LoadSelfSignedCertificate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    certificateData,
   nuint32                    certificateDataLen,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_LoadUnverifiedCertificate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    certificateData,
   nuint32                    certificateDataLen,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   NICI_OBJECT_HANDLE_PTR     certificate,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_Obfuscate(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    obfuscatedData,
   pnuint32                   obfuscatedDataLen);

N_EXTERN_LIBRARY(int)
CCS_PacketReplySign(
   NICI_CC_HANDLE             cryptoContext,
   nuint32                    encodedLen,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_PacketReSign(
   NICI_CC_HANDLE             cryptoContext,
   nuint32                    encodedLen,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_PacketSign(
   NICI_CC_HANDLE             cryptoContext,
   nuint32                    encodedLen,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_PacketSignInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_pbeEncrypt(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    encrypted,
   pnuint32                   encryptedLen);

N_EXTERN_LIBRARY(int)
CCS_pbeDecrypt(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   pnuint8                    encrypted,
   nuint32                    encryptedLen,
   pnuint8                    data,
   pnuint32                   dataLen);

N_EXTERN_LIBRARY(int)
CCS_pbeShroudPrivateKey(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   NICI_OBJECT_HANDLE         key,
   pnuint8                    shrouded,
   pnuint32                   shroudedLen);

N_EXTERN_LIBRARY(int)
CCS_pbeUnshroudPrivateKey(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attributeCount,
   pnuint8                    shrouded,
   nuint32                    shroudedLen,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_pbeSign(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_pbeVerify(
   NICI_CC_HANDLE             hContext,
   NICI_ALGORITHM_PTR         algorithm,
   nwchar                     *passwd,      /* Null-terminated unicode password */
   pnuint8                    data,         /* Data to sign */
   nuint32                    dataLen,
   pnuint8                    signature,
   nuint32                    signatureLen);

N_EXTERN_LIBRARY(int)
CCS_SetAttributeValue(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         object,
   NICI_ATTRIBUTE_PTR         attrTemplate,
   nuint32                    count);

N_EXTERN_LIBRARY(void)
CCS_Shutdown(void);

N_EXTERN_LIBRARY(void)
CCS_ShutdownAll(void);

N_EXTERN_LIBRARY(int)
CCS_Sign(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_SignFinal(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_SignInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_SignRecover(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   pnuint32                   signatureLen);

N_EXTERN_LIBRARY(int)
CCS_SignRecoverInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_SignRecoverRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_SignRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_SignUpdate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    part,
   nuint32                    partLen);

N_EXTERN_LIBRARY(int)
CCS_UnwrapKey(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         unwrappingKey,
   pnuint8                    wrappedKey,
   nuint32                    wrappedKeyLen,
   NICI_OBJECT_HANDLE_PTR     key);

N_EXTERN_LIBRARY(int)
CCS_UnwrapSessionKey(
   pnuint8                    wrappedKey,
   nuint32                    wrappedKeyLen,
   nint32                     wrappedKeyVersion,
   NICI_OBJECT_HANDLE_PTR     hKey);

N_EXTERN_LIBRARY(int)
CCS_Verify(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    data,
   nuint32                    dataLen,
   pnuint8                    signature,
   nuint32                    signatureLen);

N_EXTERN_LIBRARY(int)
CCS_VerifyFinal(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    signature,
   nuint32                    signatureLen);

N_EXTERN_LIBRARY(int)
CCS_VerifyInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_VerifyRecover(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    signature,
   nuint32                    signatureLen,
   pnuint8                    data,
   pnuint32                   dataLen);

N_EXTERN_LIBRARY(int)
CCS_VerifyRecoverInit(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_VerifyRecoverRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_VerifyRestart(
   NICI_CC_HANDLE             cryptoContext,
   NICI_OBJECT_HANDLE         key);

N_EXTERN_LIBRARY(int)
CCS_VerifyUpdate(
   NICI_CC_HANDLE             cryptoContext,
   pnuint8                    part,
   nuint32                    partLen);

N_EXTERN_LIBRARY(int)
CCS_WrapKey(
   NICI_CC_HANDLE             cryptoContext,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_KM_STRENGTH           kmLevel,
   NICI_OBJECT_HANDLE         sealingKey,
   NICI_OBJECT_HANDLE         wrappingKey,
   NICI_OBJECT_HANDLE         key,
   pnuint8                    wrappedKey,
   pnuint32                   wrappedKeyLen);

N_EXTERN_LIBRARY(int)
CCS_GenerateKeyExchangeParameters (
   NICI_CC_HANDLE             ctx,
   NICI_ALGORITHM             *algorithm,
   NICI_OBJECT_HANDLE         key,
   NICI_ATTRIBUTE             *attributeTemplate,
   nuint32                    attributeCount,
   nuint8                     *out,
   nuint32                    *outLength,
   NICI_OBJECT_HANDLE         *parameterObject,
   nuint32                    archive);

N_EXTERN_LIBRARY(int)
CCS_KeyExchangePhase1 (
   NICI_CC_HANDLE             ctx,
   NICI_ALGORITHM             *algorithm,
   NICI_OBJECT_HANDLE         key,
   nuint8                     *out,
   nuint32                    *outLength,
   nuint32                    archive);

N_EXTERN_LIBRARY(int)
CCS_KeyExchangePhase2 (
   NICI_CC_HANDLE             ctx,
   NICI_ALGORITHM_PTR         algorithm,
   NICI_OBJECT_HANDLE         key,
   pnuint8                    in,
   nuint32                    inLength,
   NICI_ATTRIBUTE_PTR         attributeTemplate,
   nuint32                    attrCount,
   nbool8                     *sizeChanged,
   NICI_OBJECT_HANDLE_PTR     keyObject,
   nuint32                    archive);

N_EXTERN_LIBRARY(int)
CCS_LoadPFXPrivateKeyWithPassword (NICI_CC_HANDLE           ctx,
                                   nwchar                   *passwd,       /* Null-terminated unicode password */
                                   pnuint8                  data,
                                   nuint32                  dataLen,
                                   NICI_OBJECT_HANDLE_PTR   privateKey,
                                   pnuint32                 privateKeyCount);

N_EXTERN_LIBRARY(int)
CCS_LoadPFXCertificateWithPassword (NICI_CC_HANDLE          ctx,
                                    nwchar                  *passwd,       /* Null-terminated unicode password */
                                    pnuint8                 data,
                                    nuint32                 dataLen,
                                    NICI_OBJECT_HANDLE_PTR  certificate,
                                    pnuint32                certificateCount,
                                    NICI_OBJECT_HANDLE_PTR  publicKey,
                                    pnuint32                publicKeyCount);

N_EXTERN_LIBRARY(int)
IKE_Verify (NICI_CC_HANDLE       hContext,
            NICI_ALGORITHM       *algo,
            NICI_OBJECT_HANDLE   verificationKey,
            nuint8               *prdKeyData,
            nuint32              prfKeyDataLength,
            nuint8               *data,
            nuint32              dataLength,
            nuint8               *signature,
            nuint32              signatureLength);

N_EXTERN_LIBRARY(int)
IKE_Sign (NICI_CC_HANDLE      hContext,
          NICI_ALGORITHM      *algo,
          NICI_OBJECT_HANDLE  signatureKey,
          nuint8              *prfKeyData,
          nuint32             prfKeyDataLength,
          nuint8              *data,
          nuint32             dataLength,
          nuint8              *signature,
          nuint32             *signatureLength);

N_EXTERN_LIBRARY(int)
CCS_SetNewIV(
   NICI_CC_HANDLE          hContext,
   pnuint8                 IV,
   nuint32                 IVLen);

N_EXTERN_LIBRARY(int)
CCS_InjectKey (
   NICI_CC_HANDLE          hContext,
   NICI_ATTRIBUTE_PTR      attributeTemplate,
   nuint32                 attributecount,
   NICI_OBJECT_HANDLE_PTR  key);

N_EXTERN_LIBRARY(int)
CCS_ExtractKey (
   NICI_CC_HANDLE       hContext,
   NICI_OBJECT_HANDLE   key,
   NICI_ATTRIBUTE_PTR   attrTemplate,
   nuint32              attributeCount);

#if defined(__cplusplus)
}
#endif

#endif    /*   _NWCCS_H_ */

