param (
    [String] $Secret = $null,
    [String] $WebState = $null,
    [Switch] $help = $false,
    [Switch] $ActiveMQ = $false,
    [Switch] $All = $false,
    [Switch] $Base = $false,
    [Switch] $Everything = $false,
    [Switch] $Gov = $false,
    [Switch] $Governance = $false,
    [Switch] $IDGov = $false,
    [Switch] $Java = $false,
    [Switch] $JDK = $false,
    [Switch] $JRE = $false,
    [Switch] $Mail = $false,
    [Switch] $PG = $false,
    [Switch] $Postgres = $false,
    [Switch] $PostgreSQL = $false,
    [Switch] $Tomcat = $false,
    [Switch] $Web = $false
)

Add-Type -AssemblyName System.IO.Compression.FileSystem

#Versions
$VSN_ACTIVEMQ           = "5.15.12"
$VSN_JDK                = "8.44.0.11-ca-jdk8.0.242"
$VSN_POSTGRESQL         = "11.10-1"
$VSN_POSTGRESQL_SERVICE = "11"
$VSN_TOMCAT             = "9.0.33"

#Names
$NAME_ACTIVEMQ          = "activemq"
$NAME_POSTGRES          = "postgres"
$NAME_TOMCAT            = "tomcat"
$NAME_TOMCAT_LAUNCH     = "start-tomcat.bat"
$NAME_TOMCAT_QUIT       = "stop-tomcat.bat"
$NAME_TOMCAT_SERVICE    = "TomcatService.exe"

#Directories
$DIR_ACTIVEMQ           = "${NAME_ACTIVEMQ}"
$DIR_BACKUP             = "backup"
$DIR_BASE               = "c:\netiq\idm\apps"
$DIR_COMPRESSED         = "${PSScriptRoot}\compressed"
$DIR_JDK                = "jdk"
$DIR_LOG_BASE           = "${PSScriptRoot}"
$DIR_LOG_PARENT         = "logs"
$DIR_LOG                = "${DIR_LOG_BASE}\${DIR_LOG_PARENT}"
$DIR_POSTGRES           = "${NAME_POSTGRES}"
$DIR_POSTGRES_DATA      = "data"
$DIR_PRISTINE           = "${PSScriptRoot}\untouched"
$DIR_TOMCAT             = "${NAME_TOMCAT}"

#Files
$FILE_ACTIVEMQ          = "${DIR_COMPRESSED}\apache-activemq-${VSN_ACTIVEMQ}-bin.zip"
$FILE_JDK               = "${DIR_COMPRESSED}\zulu${VSN_JDK}-win_x64.zip"
$FILE_POSTGRESQL        = "${DIR_COMPRESSED}\postgresql-${VSN_POSTGRESQL}-windows-x64.exe"
$FILE_LOG               = "${DIR_LOG}\IDGov-Core-Helper.log"
$FILE_TOMCAT            = "${DIR_COMPRESSED}\apache-tomcat-${VSN_TOMCAT}-windows-x64.zip"

#Services (final state can be `start`, `stop`, and `restart`)
$SERV_TOMCAT_SERVICE    = "IDM Apps Tomcat Service"
$SERV_TOMCAT_STATE      = ""

#Parameters
$PARAM_OSP_LDAP_TYPE    = "edir"
$PARAM_OSP_LOGGING      = "WARN"
$PARAM_POSTGRES_PORT    = "5432"
$PARAM_TOMCAT_SERVER    = "localhost"
$PARAM_TOMCAT_OPTS_COMMON_ALL = "-Dnetiqtomcatcustomsrvidentify=netiqtomcatcustomsrvidentify;-Djava.awt.headless=true;-Dcom.netiq.ism.config=${DIR_BASE}\${DIR_TOMCAT}\conf\ism-configuration.properties;"
$PARAM_TOMCAT_OPTS_OSP  = "-Dcom.netiq.osp.ext-context-file=${DIR_BASE}\osp\lib\osp-conf-${PARAM_OSP_LDAP_TYPE}.jar;-Dcom.netiq.idm.osp.logging.level=${PARAM_OSP_LOGGING};-Dcom.netiq.idm.osp.client.host=${PARAM_TOMCAT_SERVER};-Dcom.netiq.idm.osp.logging.file.dir=${DIR_BASE}\${DIR_TOMCAT}\logs;-Dcom.netiq.idm.osp.audit.enabled=false;"
$PARAM_TOMCAT_OPTS_COMMON_IDENT = "-Dlog4j.configuration=file:${DIR_BASE}\${DIR_TOMCAT}\conf\log4j.properties;-Dinternal.atlaslite.jcce.xml.w3c.XMLUtil.suppressSecurityWarning=true;"
$PARAM_TOMCAT_OPTS_GOV  = "-Dcom.netiq.ism.config.is.jndi.env=true;-Dnovell.logging.config.dir=${DIR_BASE}\${DIR_TOMCAT}\conf;"
$PARAM_TOMCAT_OPTS_RPT  = "-Dcom.netiq.rpt.config.file=${DIR_BASE}\${DIR_TOMCAT}\conf\ism-configuration.properties;-Dextend.local.config.dir=${DIR_BASE}\${DIR_TOMCAT}\conf;-Didmuserapp.logging.config.dir=${DIR_BASE}\${DIR_TOMCAT}\conf;"
$PARAM_TOMCAT_JVM_OPTS  = "${PARAM_TOMCAT_OPTS_COMMON_ALL}${PARAM_TOMCAT_OPTS_OSP}${PARAM_TOMCAT_OPTS_COMMON_IDENT}${PARAM_TOMCAT_OPTS_GOV}${PARAM_TOMCAT_OPTS_RPT}"
$PARAM_TOMCAT_JVM_MEM_INIT  = "1024"
$PARAM_TOMCAT_JVM_MEM_FINAL = "1024"

#Parse and consume input arguments provided
$USAGE_THIS = "Usage: " + $MyInvocation.MyCommand.Name + " -all|-gov|-base|-tomcat|-activeMQ|-postgreSQL|-java [<PG password>]"

$PARAM_INSTALL_ACTIVE_MQ = $false
$PARAM_INSTALL_JAVA = $false
$PARAM_INSTALL_POSTGRESQL = $false
$PARAM_INSTALL_WEB = $false

If ($help) {
    Write-Host "$USAGE_THIS"
    Exit
}
If ($All -or $Everything) {
    $PARAM_INSTALL_ACTIVE_MQ = $true
    $PARAM_INSTALL_JAVA = $true
    $PARAM_INSTALL_POSTGRESQL = $true
    $PARAM_INSTALL_WEB = $true
}
If ($Governance -or $Gov -or $IDGov) {
    $PARAM_INSTALL_ACTIVE_MQ = $true
    $PARAM_INSTALL_JAVA = $true
    $PARAM_INSTALL_WEB = $true
}
If ($Base -or $Tomcat -or $Web) {
    $PARAM_INSTALL_JAVA = $true
    $PARAM_INSTALL_WEB = $true
}
If ($ActiveMQ -or $Mail) {
    $PARAM_INSTALL_ACTIVE_MQ = $true
    $PARAM_INSTALL_JAVA = $true
}
If ($Java -or $JDK -or $JRE) {
    $PARAM_INSTALL_JAVA = $true
}
If ($Postgres -or $PostgreSQL -or $PG) {
    $PARAM_INSTALL_POSTGRESQL = $true
}

If (-not $WebState.equals("")) {
    $SERV_TOMCAT_STATE = $WebState
}

If (-not $Secret.equals("")) {
    $PARAM_POSTGRES_SECRET = $Secret
}

If (-not $PARAM_INSTALL_ACTIVE_MQ -and -not $PARAM_INSTALL_JAVA -and -not $PARAM_INSTALL_POSTGRESQL -and -not $PARAM_INSTALL_WEB) {
    Write-Host "Nothing specified for installation!"
    Write-Host "$USAGE_THIS"
    Exit
}
If ($PARAM_INSTALL_POSTGRESQL -and $PARAM_POSTGRES_SECRET -eq $null) {
    Write-Host "Required PostgreSQL password missing!"
    Write-Host "$USAGE_THIS"
    Exit
}

Function Unzip([string]${zip}, [string]${destination})
{
    ${private:parent} = Split-Path -Path ${destination} -Parent
    ${private:leaf} = Split-Path -Path ${destination} -Leaf
    If (-not (Test-Path ${destination})) {
        ${null} = (New-Item -Path "${parent}" -Name "${leaf}" -ItemType "directory")
    }
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Extracting `"${zip}`" into `"${destination}`":"
    Write-Host "######################################################################################"
    [System.IO.Compression.ZipFile]::ExtractToDirectory(${zip}, ${destination})
    ${private:children} = @(Get-ChildItem -Path "${destination}")
    If (($(${children}.count) -eq 1) -and (${children}[0].Attributes -match "Directory")) {
        # When the only child-item is a directory we should strip off a layer, which should be safe
        #   Otherwise the (extracted) files in this folder are those that should stay
        # On some systems we cannot rename the extracted folder immediately
        ${private:rename_waiting} = ${TRUE}
        ${private:rename_count} = 0
        While (${private:rename_waiting}) {
            try {
                Rename-Item -Path "$(${children}[0].FullName)" -newName "${leaf}" -ErrorAction Stop
                ${private:rename_waiting} = ${FALSE}
            } catch [System.IO.IOException] {
                If (${private:rename_count}++ -ge 15) {
                    # We have waited long enough; report the error
                    ${private:rename_waiting} = ${FALSE}
                    throw $_
                } Else {
                    Start-Sleep -Seconds 2
                }
            }
        }
        Get-ChildItem -Path "${destination}\${leaf}" -Recurse | Move-Item -Destination "${destination}"
        Remove-Item -Path "${destination}\${leaf}"
    }
}

Function Disable-TomcatAJPConnector {
  Write-Host ""
  Write-Host "######################################################################################"
  Write-Host " Disabling Tomcat AJP connector within `"${DIR_BASE}\${DIR_TOMCAT}\conf\server.xml`""
  Write-Host "######################################################################################"

  Set-Content `
    -Path "${DIR_BASE}\${DIR_TOMCAT}\conf\server.xml" `
    -Value "$((Get-Content `"${DIR_BASE}\${DIR_TOMCAT}\conf\server.xml`" -Raw) -replace -join `
      (
        "(?ms)",
        "(\s+<!-- Define an AJP 1\.3 Connector on port [0-9]+ -->)(\s*)", # $1 $2
        "(<Connector port=``"[0-9]+``" protocol=``"AJP/1\.3``" redirectPort=``"[0-9]+``"\s*/>)(\s*)" # $3 $4
      ), -join `
      (
        "`$1`$2",
        "<!--`$2",
        "`$3`$2",
        "-->",
        "`$2`$2"
      )
    )"
}

Function Enable-CatalinaOutput {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Enable Output to CATALINA_OUT:"
    Write-Host "######################################################################################"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\catalina.bat" -ReadCount 0) -join "`r`n"
    $text -replace 'set _EXECJAVA=start "%TITLE%" %_RUNJAVA%', 'set _EXECJAVA=cmd /c "%_RUNJAVA%' | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\catalina.bat"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\catalina.bat" -ReadCount 0) -join "`r`n"
    $text -replace '%ACTION%', '%ACTION%" >> %CATALINA_OUT% 2>&1' | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\catalina.bat"
}

Function Install-ActiveMQ {
    Unzip "${FILE_ACTIVEMQ}" "${DIR_BASE}\${DIR_ACTIVEMQ}"

    ((Get-Content -path "${DIR_BASE}\${DIR_ACTIVEMQ}\bin\activemq.bat" -Raw) -replace
        "if `"%JAVA_HOME%`" == `"`" goto noJavaHome",
        "if `"%JAVA_HOME%`" == `"`" set JAVA_HOME=${DIR_BASE}\jre") |
        Set-Content -Path "${DIR_BASE}\${DIR_ACTIVEMQ}\bin\activemq.bat"
}

Function Install-JDK {
    Unzip "${FILE_JDK}" "${DIR_BASE}\${DIR_JDK}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Creating symbolic link at ${DIR_BASE}\jre for ${DIR_JDK}\jre"
    Write-Host "######################################################################################"
    $null = (New-Item -ItemType SymbolicLink -Path "${DIR_BASE}\jre" -Target "${DIR_BASE}\${DIR_JDK}\jre")
}

Function Install-PostgreSQL {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Installing PostgreSQL to ${DIR_BASE}\${DIR_POSTGRES}:"
    Write-Host "######################################################################################"
    Start-Process ${FILE_POSTGRESQL} -Wait -NoNewWindow -ArgumentList `
        "--unattendedmodeui none", `
        "--mode unattended", `
        "--superaccount ${NAME_POSTGRES}", `
        "--superpassword ${PARAM_POSTGRES_SECRET}", `
        "--serverport ${PARAM_POSTGRES_PORT}", `
        "--prefix ${DIR_BASE}\${DIR_POSTGRES}", `
        "--datadir ${DIR_BASE}\${DIR_POSTGRES}\${DIR_POSTGRES_DATA}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Updating PostgreSQL to allow remote access"
    Write-Host "######################################################################################"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_POSTGRES}\${DIR_POSTGRES_DATA}\pg_hba.conf" -ReadCount 0) -join "`n"
    $text -replace '/32', '/0 ' | Set-Content -Path "${DIR_BASE}\${DIR_POSTGRES}\${DIR_POSTGRES_DATA}\pg_hba.conf"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Restarting PostgreSQL"
    Write-Host "######################################################################################"
    Restart-Service -Name postgresql-x64-${VSN_POSTGRESQL_SERVICE}
}

Function Install-Tomcat {
    Unzip "$FILE_TOMCAT" "${DIR_BASE}\${DIR_TOMCAT}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Moving the default Tomcat wars to ${DIR_BASE}\${DIR_BACKUP}:"
    Write-Host "######################################################################################"
    $null = (New-Item -Path "${DIR_BASE}" -Name "${DIR_BACKUP}" -ItemType "directory")
    Get-ChildItem -Path "${DIR_BASE}\${DIR_TOMCAT}\webapps" -Recurse | Move-Item -Destination "${DIR_BASE}\${DIR_BACKUP}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Creating new default Tomcat index file at ${DIR_BASE}\${DIR_TOMCAT}\webapps\ROOT\index.html:"
    Write-Host "######################################################################################"
    $null = (New-Item -Path "${DIR_BASE}\${DIR_TOMCAT}\webapps" -Name "ROOT" -ItemType "directory")
    Copy-Item -Path "${DIR_PRISTINE}\index.html" -Destination "${DIR_BASE}\${DIR_TOMCAT}\webapps\ROOT\index.html"

    Disable-TomcatAJPConnector
}

Function Join-ActiveMQTomcat {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Copying the ActiveMQ jar to ${DIR_BASE}\${DIR_TOMCAT}\lib:"
    Write-Host "######################################################################################"
    Copy-Item "${DIR_BASE}\${DIR_ACTIVEMQ}\${DIR_ACTIVEMQ}-all*.jar" -Destination "${DIR_BASE}\${DIR_TOMCAT}\lib"
}

Function Setup-TomcatService {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Prepare ${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}:"
    Write-Host "######################################################################################"
    Copy-Item -Path "${DIR_PRISTINE}\${NAME_TOMCAT_LAUNCH}" -Destination "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}" -ReadCount 0) -join "`r`n"
    $text -replace "REPLACE_INSTALL_BASE", "${DIR_BASE}" | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}" -ReadCount 0) -join "`r`n"
    $text -replace "REPLACE_INSTALL_JAVA", "${DIR_BASE}\${DIR_JDK}" | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_LAUNCH}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Prepare ${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}:"
    Write-Host "######################################################################################"
    Copy-Item -Path "${DIR_PRISTINE}\${NAME_TOMCAT_QUIT}" -Destination "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}" -ReadCount 0) -join "`r`n"
    $text -replace "REPLACE_INSTALL_BASE", "${DIR_BASE}" | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}"
    $text = (Get-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}" -ReadCount 0) -join "`r`n"
    $text -replace "REPLACE_INSTALL_JAVA", "${DIR_BASE}\${DIR_JDK}" | Set-Content -Path "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_QUIT}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Create the Tomcat service ${SERV_TOMCAT_SERVICE}:"
    Write-Host "######################################################################################"
    Copy-Item -Path "${DIR_PRISTINE}\${NAME_TOMCAT_SERVICE}" -Destination "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_SERVICE}"
    New-Service -Name "${SERV_TOMCAT_SERVICE}" -BinaryPathName "${DIR_BASE}\${DIR_TOMCAT}\bin\${NAME_TOMCAT_SERVICE} ${DIR_BASE}" -StartupType Manual
}

Function Restart-TomcatService {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Restart the Tomcat service, ${SERV_TOMCAT_SERVICE}:"
    Write-Host "######################################################################################"
    Restart-Service -Name "${SERV_TOMCAT_SERVICE}"
}

Function Start-TomcatService {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Start the Tomcat service, ${SERV_TOMCAT_SERVICE}:"
    Write-Host "######################################################################################"
    Start-Service -Name "${SERV_TOMCAT_SERVICE}"
}

Function Stop-TomcatService {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Stop the Tomcat service, ${SERV_TOMCAT_SERVICE}:"
    Write-Host "######################################################################################"
    Stop-Service -Name "${SERV_TOMCAT_SERVICE}"
}

Function Start-Recording {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Stopping existing transcripts"
    Write-Host "######################################################################################"
    $ErrorActionPreference="SilentlyContinue"
    Stop-Transcript | out-null

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Creating log directory at: ${DIR_LOG}"
    Write-Host "######################################################################################"
    $null = (New-Item -Path "${DIR_LOG_BASE}" -Name "${DIR_LOG_PARENT}" -ItemType "directory")

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Removing previous log if exists: ${FILE_LOG}"
    Write-Host "######################################################################################"
    Remove-Item -Path "${FILE_LOG}"

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Starting transcript: ${FILE_LOG}"
    Write-Host "######################################################################################"
    $ErrorActionPreference = "Continue"
    Start-Transcript -path "${FILE_LOG}" -append
}

Function Stop-Recording {
    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Complete the transcript"
    Write-Host "######################################################################################"
    Stop-Transcript

    Write-Host ""
    Write-Host "######################################################################################"
    Write-Host " Completed"
    Write-Host " Review the log file: ${FILE_LOG}"
    Write-Host "######################################################################################"
}

Start-Recording
If ($PARAM_INSTALL_JAVA) { Install-JDK }
If ($PARAM_INSTALL_ACTIVE_MQ) { Install-ActiveMQ }
If ($PARAM_INSTALL_WEB) {
    Install-Tomcat
    Enable-CatalinaOutput
    If ($PARAM_INSTALL_ACTIVE_MQ) { Join-ActiveMQTomcat }
    Setup-TomcatService
    If ($SERV_TOMCAT_STATE.equals("start")) { Start-TomcatService }
    ElseIf ($SERV_TOMCAT_STATE.equals("restart")) { Restart-TomcatService }
    ElseIf ($SERV_TOMCAT_STATE.equals("stop")) { Stop-TomcatService }
}
if ($PARAM_INSTALL_POSTGRESQL) { Install-PostgreSQL }
Stop-Recording

# SIG # Begin signature block
# MIIdcwYJKoZIhvcNAQcCoIIdZDCCHWACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCwGfZDyk7E93hf
# D4XJYjEzUQpOyG9PeAShcaRwUBCwoaCCGFwwggO3MIICn6ADAgECAhAM5+DlF9hG
# /o/lYPwb8DA5MA0GCSqGSIb3DQEBBQUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNV
# BAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0wNjExMTAwMDAwMDBa
# Fw0zMTExMTAwMDAwMDBaMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNVBAMTG0RpZ2lD
# ZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC
# AQoCggEBAK0OFc7kQ4BcsYfzt2D5cRKlrtwmlIiq9M71IDkoWGAM+IDaqRWVMmE8
# tbEohIqK3J8KDIMXeo+QrIrneVNcMYQq9g+YMjZ2zN7dPKii72r7IfJSYd+fINcf
# 4rHZ/hhk0hJbX/lYGDW8R82hNvlrf9SwOD7BG8OMM9nYLxj+KA+zp4PWw25EwGE1
# lhb+WZyLdm3X8aJLDSv/C3LanmDQjpA1xnhVhyChz+VtCshJfDGYM2wi6YfQMlqi
# uhOCEe05F52ZOnKh5vqk2dUXMXWuhX0irj8BRob2KHnIsdrkVxfEfhwOsLSSplaz
# vbKX7aqn8LfFqD+VFtD/oZbrCF8Yd08CAwEAAaNjMGEwDgYDVR0PAQH/BAQDAgGG
# MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFEXroq/0ksuCMS1Ri6enIZ3zbcgP
# MB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA0GCSqGSIb3DQEBBQUA
# A4IBAQCiDrzf4u3w43JzemSUv/dyZtgy5EJ1Yq6H6/LV2d5Ws5/MzhQouQ2XYFwS
# TFjk0z2DSUVYlzVpGqhH6lbGeasS2GeBhN9/CTyU5rgmLCC9PbMoifdf/yLil4Qf
# 6WXvh+DfwWdJs13rsgkq6ybteL59PyvztyY1bV+JAbZJW58BBZurPSXBzLZ/wvFv
# hsb6ZGjrgS2U60K3+owe3WLxvlBnt2y98/Efaww2BxZ/N3ypW2168RJGYIPXJwS+
# S86XvsNnKmgR34DnDDNmvxMNFG7zfx9jEB76jRslbWyPpbdhAbHSoyahEHGdreLD
# +cOZUbcrBwjOLuZQsqf6CkUvovDyMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHO
# FADw3TANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGln
# aUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhE
# aWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEw
# MTAwMDAwMFoXDTMxMDEwNjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
# DkRpZ2lDZXJ0LCBJbmMuMSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAy
# MTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpj
# a1HXOhFCvQp1dU2UtAxQtSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9
# NmqB+in43Stwhd4CGPN4bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaP
# xqPhLxs6t2HWc+xObTOKfF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsO
# jizVI9r0TXhG4wODMSlKXAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7Ed
# MMKbaYK02/xWVLwfoYervnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmE
# vQECAwEAAaOCAbgwggG0MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkw
# JwYIKwYBBQUHAgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSME
# GDAWgBT0tuEgHf4prtLkYaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgt
# HUQ23eNqerwwcQYDVR0fBGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQu
# Y29tL3NoYTItYXNzdXJlZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2lj
# ZXJ0LmNvbS9zaGEyLWFzc3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZD
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJ
# RFRpbWVzdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7zn
# GucgDo5nRv1CclF0CiNHo6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJm
# Te1ppA/2uHDPYuj1UUp4eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3
# Y18jUmmDgvoaU+2QzI2hF3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74
# wOFcz8QRcucbZEnYIpp1FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPW
# H1ub9y4bTxMd90oNcX6Xt/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHL
# KNpbh6aKLzCCBTAwggQYoAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcN
# AQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
# A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJl
# ZCBJRCBSb290IENBMB4XDTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjEL
# MAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3
# LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElE
# IENvZGUgU2lnbmluZyBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB
# APjTsxx/DhGvZ3cH0wsxSRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5
# idQ3Gde2qvCchqXYJawOeSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQ
# dLEoJrskacLCUvIUZ4qJRdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+
# rHCiq85/6XzLkqHlOzEcz+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE
# 86xnTrXE94zRICUj6whkPlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pz
# INAPZHM8np+mM6n9Gd8lk9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8C
# AQAwDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUF
# BwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMG
# CCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3Js
# NC5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2
# hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3JsME8GA1UdIARIMEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxo
# dHRwczovL3d3dy5kaWdpY2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQW
# BBRaxLl7KgqjpepxA8Bg+S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYun
# pyGd823IDzANBgkqhkiG9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwt
# OhrE7zBh134LYP3DPQ/Er4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GO
# Yw4TS63XX0R58zYUBor3nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEo
# AjwukaPAJRHinBRHoXpoaK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhC
# MrI2iiQC/i9yfhzXSUWW6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOB
# m0/GkxAG/AeB+ova+YJJ92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEw
# ggQZoAMCAQICEAqhJdbWMht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkG
# A1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRp
# Z2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENB
# MB4XDTE2MDEwNzEyMDAwMFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGlu
# ZyBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6
# gpnFOVQoV7YjSsQOB0UzURB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjW
# ahQAOPcuHjvuzKb2Mln+X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oi
# PhisEeTwmQNtO4V8CdPuXciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTG
# TSQjMF287DxgaqwvB8z98OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgP
# hH+fMRTWrdXyZMt7HgXQhBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2s
# rOlW/5MCAwEAAaOCAc4wggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1
# bjAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAG
# AQH/AgEAMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5Bggr
# BgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNv
# bTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0QXNzdXJlZElEUm9vdENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDov
# L2NybDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6
# oDigNoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElE
# Um9vdENBLmNybDBQBgNVHSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcC
# ARYcaHR0cHM6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJ
# KoZIhvcNAQELBQADggEBAHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9
# xafDDiBCLK938ysfDCFaKrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIl
# HsS6HHssIeLWWywUNUMEaLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8p
# ieV4H9YLFKWA1xJHcLN11ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4Ui
# jGHKeZR+WfyMD+NvtQEmtmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8
# QkIoxhhWz0E0tmZdtnR79VYzIi8iNrJLokqV2PWmjlIwggUyMIIEGqADAgECAhAB
# Pe4iVgqMX6vbiX1xPVYPMA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUw
# EwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20x
# MTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcg
# Q0EwHhcNMTgwNjExMDAwMDAwWhcNMjEwODIxMTIwMDAwWjBvMQswCQYDVQQGEwJH
# QjEQMA4GA1UEBxMHTmV3YnVyeTEmMCQGA1UEChMdTWljcm8gRm9jdXMgSW50ZXJu
# YXRpb25hbCBwbGMxJjAkBgNVBAMTHU1pY3JvIEZvY3VzIEludGVybmF0aW9uYWwg
# cGxjMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA2Vb7QjccsV83UO3b
# YCw/nTB6Bl3tGX0eSCCPsBtPTspWKz2GXWK7/1KeDUNPLnNBT1InLwNcaDxiX1TG
# SC5+gBfJiaTWB03ll7FpDbayyQ532smNvlQop2v5rgDkXQLu4Jg1JE3OLrppxBfP
# tBsjHw76kCqePA8cvTMWxUc6b7loX26dQe8v7+SeOI0qZ0tXCNwtgTu2513frUxj
# d/fY2F/toc9OjateJZQ5v5+WV7dOG/89bkvMAEHXegLcPZw7PT4aZP/ZVOYtoflS
# g6oT0m0N1LdsjyTEDbqmRm6/dt5sI9akmdln7hpOp1y9EMCgkGiNAwf9u4Lw2iUj
# NvoT5QIDAQABo4IBxTCCAcEwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt9mV1
# DlgwHQYDVR0OBBYEFLyS6yL4fTRV+hftOctsNinkVorLMA4GA1UdDwEB/wQEAwIH
# gDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRwOi8v
# Y3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOgMYYv
# aHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5jcmww
# TAYDVR0gBEUwQzA3BglghkgBhv1sAwEwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93
# d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsGAQUFBwEBBHgwdjAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tME4GCCsGAQUFBzAC
# hkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRTSEEyQXNzdXJl
# ZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsF
# AAOCAQEAZllSiMymdGRhqGhTP8p8Xdjmu1IZu6wg09n7oKLaH7qg3TfjqX+jjaQ2
# CTTyr2cwblYPa1Dp2sOqjGUNyrUvg22cfaSJAtbTNkpb8VtU8nEMjc3IaTdAw0hM
# +qHgYUaX1he6HrP8x51xJW67Z+Q6junwCPjo4fhEonJ4D9PBqQpRzpHtHy3D9rI6
# sgTSX3uWHqUet4Ehq49dSb2GZB8qAweUeM8ntCsEe2Trrei+HbrWn2F6DjpFCRA8
# VjhNvIsqOyltdsGT1HOGHz3hIi2G1Xr/T43sBmeqndopqavoj9GZsUMVp6RPRohn
# yGDTxINU2+Cs981b0UUjO5BQkkQfrzGCBG0wggRpAgEBMIGGMHIxCzAJBgNVBAYT
# AlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2Vy
# dC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNp
# Z25pbmcgQ0ECEAE97iJWCoxfq9uJfXE9Vg8wDQYJYIZIAWUDBAIBBQCggYQwGAYK
# KwYBBAGCNwIBDDEKMAigAoAAoQKAADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIB
# BDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg
# M+E5gOanJhfBRbT8C+TJ/re0vPpfQyGt8g+qndSRJi4wDQYJKoZIhvcNAQEBBQAE
# ggEAD1WXXeeGe0VArQE63PZ9vK2vs5nz3dLQUAJ9JtD5GSO8BfG1IuLvTUouGw/w
# Nadxp6MGKdWnXHJ8ZuuQyxhDO6YIekN1RtDShBCuo4T0QQ9tvtOWhp/FW1qf/6KS
# cmHhTBWxij7qAECbvsR699got0/EJhSh6XcYrn2gH53DxpkT6KsnQBsfO9cD/Hy6
# ZajaFetxBig+TpiFHGmtZgxcF3xRJ/MRPIfT36Pdx0pt8Or32W+J6ay5YXVgdJ9W
# EwHFTD67iz5+f2OLM8xIkxcw0gJxHxukqrF/Uev8xbq0Qe5bz+/BfP7mTFwtpQdD
# 2QdWBFaJoEM46wCBdqE0MD28SKGCAjAwggIsBgkqhkiG9w0BCQYxggIdMIICGQIB
# ATCBhjByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBAhANQkrgvjqI/2BAIc4UAPDdMA0GCWCG
# SAFlAwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0B
# CQUxDxcNMjEwMzAyMTgxMjUwWjAvBgkqhkiG9w0BCQQxIgQggwF91QYNuCm9XTWi
# PAAotNpkeuv+YvLieu6+/QjvP0MwDQYJKoZIhvcNAQEBBQAEggEAB+XELSG1Q8h7
# twp6D590k444hp4ep2vGe7TGPU/R95quVrVH03Kw6FXTWMc49E9U3+8tMZF7neil
# iag7fPBucdUiOr2sKIcGYYyCgbIqYj2tWp2SStDn/gsrss/TaHRFz4FxIg6I3l5Y
# bQXwFqMDXAIBX+Tye13BJOHUjXu0AtnbIeYauytbxGtocMCMXLC5L5f1pc5x4RsF
# B++PdmbbrdtupNBDwkNjHa+n51oEQxnntizGoOOaeLyQr3NWk9MS1AjNOIefM44b
# a8W/VVyOmrfF4X6dwKeNgsUN9MY+LBjfCJxI/QJ3FAC8EYr9C/B9lUoyBwUK9J+Y
# cKIc4rYXYQ==
# SIG # End signature block
