. ${PSScriptRoot}\def_copies.ps1

# Define variables
${ACTIVE_MQ_DEFAULT} = "C:\netiq\idm\apps\activemq"
${JAVA_DEFAULT} = "C:\netiq\idm\apps\jre"
${POSTGRESQL_ADMIN_DEFAULT} = "postgres"
${POSTGRESQL_PATH_DEFAULT} = "C:\netiq\idm\apps\postgres"
${POSTGRESQL_PORT_DEFAULT} = "5432"
${POSTGRESQL_SERVER_DEFAULT} = "localhost"
${TOMCAT_DEFAULT} = "C:\netiq\idm\apps\tomcat"

${DIR_COMPRESSED} = "${PSScriptRoot}\compressed"

${FILE_ACTIVE_MQ} = "${DIR_COMPRESSED}\apache-activemq-${ACTIVE_MQ_VERSION_TARGET}-bin.zip"
${FILE_JAVA} = "${DIR_COMPRESSED}\zulu${JAVA_DOWNLOAD_CODE}-win_x64.zip"
${FILE_POSTGRESQL} = "${DIR_COMPRESSED}\postgresql-${POSTGRESQL_VERSION_TARGET}-windows-x64.exe"
${FILE_TOMCAT} = "${DIR_COMPRESSED}\apache-tomcat-${TOMCAT_VERSION_TARGET}-windows-x64.zip"

${global:ALREADY_OBTAINED_JAVA_LOC} = ${FALSE}

${TIMESTAMP} = (Get-Date -Format "yyyyMMdd-HHmmss")

${COMPARE_CODE_EQ} = 0
${COMPARE_CODE_LT} = 1
${COMPARE_CODE_GT} = 2

${global:SUMMARY_REGISTRATION}=@()
${SUMMARY_CODE_COMPLETED} = 0
${SUMMARY_CODE_SKIPPED} = 1
${SUMMARY_CODE_MISSED} = 2
${SUMMARY_CODE_FAILED} = 3

Add-Type -AssemblyName System.IO.Compression.FileSystem

# Define functions
Function Get-PathValidity ([string]${base}, [string[]]${array}) {
  ${valid} = ${TRUE}
  ForEach (${path} in ${array}) {
    If (-not (Test-Path ${base}\${path})) {
      Write-Information "    Expected to find '${base}\${path}'" -InformationAction Continue
      ${valid}=${FALSE}
    }
  }
  return ${valid}
}

Function Get-ValidJavaPath () {
  If (-not ${global:ALREADY_OBTAINED_JAVA_LOC}) {
    ${global:ALREADY_OBTAINED_JAVA_LOC} = ${TRUE}
    ${input_invalid} = ${TRUE}
    while (${input_invalid}) {
      ${java_loc} = Read-Host "  Where is your Java (JRE) located? [${java_loc_final}]"
      If ("${java_loc}" -ne "") { ${java_loc_final} = ${java_loc} }
      ${result} = (Get-PathValidity ${java_loc_final} ${JAVA_VALIDITY})
      If ($result -eq ${TRUE}) {
        ${input_invalid} = ${FALSE}
      }
    }
  }
}

Function Parse-Version ([string]${version}) {
  return (${version} -replace '\.', ' ' -replace '-', ' ' -replace '_', ' ').split(" ")
}

Function Compare-Versions ([string]${A}, [string]${B}) {
  ${version_A} = Parse-Version ${A}
  ${version_B} = Parse-Version ${B}

  ${max} = ${version_B}.count
  For ((${i} = 0); ${i} -lt ${max}; ${i}++) {
    try {
      ${actual} = [int]::Parse(${version_A}[${i}])
      ${expected} = [int]::Parse(${version_B}[${i}])
    } catch [System.Management.Automation.MethodInvocationException] {
      # Compare as strings in case of an error
      ${actual} = ${version_A}[${i}]
      ${expected} = ${version_B}[${i}]
    }
    If ( ${actual} -lt ${expected} ) {
      return ${COMPARE_CODE_LT}
    } ElseIf ( ${actual} -gt ${expected} ) {
      return ${COMPARE_CODE_GT}
    }
  }
  return ${COMPARE_CODE_EQ}
}

Function Compare-VersionsWithResult ([string]${name}, [string]${actual}, [string]${expected}) {
  ${comparison} = Compare-Versions ${actual} ${expected}
  If ( ${comparison} -eq ${COMPARE_CODE_LT} ) {
    Write-Host "  Expected ${name} version ${expected}, but instead found the older ${actual}"
  } ElseIf ( ${comparison} -eq ${COMPARE_CODE_GT} ) {
    Write-Host "  Your ${name} version (${actual}) surpasses the target (${expected})`n    We cannot guarantee your version will work, but it might."
  } Else { # If ( ${comparison} -eq ${COMPARE_CODE_EQ} )
    Write-Host "  Your ${name} version (${expected}) appears to be valid."
  }
}

Function Wait-StopProcesses([string]${component}, [string]${search}) {
  ${private:process_all} = (Get-WmiObject -ClassName Win32_Process -Filter "CommandLine LIKE '%${search}%'")
  While (${process_all} -ne ${null}) {
    ${private:do_please} = Read-Host "    It appears that ${component} is still running.`n      May we forcefully stop it [y/N]?"
    If ("${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
      ForEach (${process} in ${process_all}) {
        Write-Host "    Stopping `"$(${process}.Name)`":"
        ${null} = ${process}.Terminate()
        Start-Sleep 3
      }
    } Else {
      Read-Host "    Go ahead and stop your ${component} instances.`n      When the ${component} instances have completely stopped, press ENTER."
    }
    ${private:process_all} = (Get-WmiObject -ClassName Win32_Process -Filter "CommandLine LIKE '%${search}%'")
  }
  Write-Host "    It appears no instances of ${component} are running. Proceeding ..."
}

Function Wait-StopServices([string]${component}, [string]${search}) {
  ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.Status -eq "Running"})
  While (${service_all} -ne ${null}) {
    ${private:do_please} = Read-Host "    It appears that ${component} is still running.`n      May we forcefully stop it [y/N]?"
    If ("${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
      ForEach (${service} in ${service_all}) {
        Write-Host "    Stopping `"$(${service}.Name)`":"
        Stop-Service -Force -Name ${service}.Name
        Start-Sleep 3
      }
    } Else {
      Read-Host "    Go ahead and stop your ${component} instances.`n      When the ${component} instances have completely stopped, press ENTER."
    }
    ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.Status -eq "Running"})
  }
  Write-Host "    It appears no instances of ${component} are running. Proceeding ..."
}

Function Wait-DisableServices([string]${component}, [string]${search}) {
  ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.StartType -ne "Disabled"})
  While (${service_all} -ne ${null}) {
    ForEach (${service} in ${service_all}) {
      Write-Host "    Disabling `"$(${service}.Name)`""
      Set-Service -Name ${service}.Name -StartupType Disabled
      Start-Sleep 3
    }
    ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.StartType -ne "Disabled"})
  }
  Write-Host "    It appears no instances of ${component} are enabled. Proceeding ..."
}

Function Copy-OldFiles([string]${name}, [string]${source}, [string]${destination}, [string[]]$files) {
  ForEach (${file} in ${files}) {
    If ((${file} -ne ${null}) -and (${file} -ne "")) {
      If (Test-Path -Path ${source}\${file}) {
        If (Test-Path -Path ${destination}\${file}) {
          ${private:item} = (Get-Item ${destination}\${file})
          ${private:backup_name} = "$(${item}.BaseName)-backup$(${item}.Extension)"
          ${private:backup_full} = "$(${item}.Directory)\${backup_name}"
          Write-Host "    Backing up existing ${name} '${file}' to '${backup_name}'"
          Move-Item -Path ${item} -Destination ${backup_full} -Force
        }
        Write-Host "    Copying ${name} '${file}'"
        Copy-Item -Path "${source}\${file}" -Destination "$(Split-Path -Path "${destination}\${file}" -Parent)"
      }
    }
  }
}

Function Discard-SpecificFiles([string]${name}, [string]${destination}, [string[]]$files) {
  ForEach (${file} in ${files}) {
    If ((${file} -ne ${null}) -and (${file} -ne "")) {
      If (Test-Path ${destination}\${file}) {
        Write-Host "    Removing existing ${name} '${file}'"
        Remove-Item -Path "${destination}\${file}" -Recurse
      }
    }
  }
}

Function Disable-AJPConnector([string]${tomcat}) {
  Write-Host "    Disabling Tomcat AJP connector within `"${tomcat}\conf\server.xml`""

  Set-Content `
    -Path "${tomcat}\conf\server.xml" `
    -Value "$((Get-Content `"${tomcat}\conf\server.xml`" -Raw) -replace -join `
      (
        "(?ms)",
        "(\s+<!-- Define an AJP 1\.3 Connector on port [0-9]+ -->)(\s*)", # $1 $2
        "(<Connector port=``"[0-9]+``" protocol=``"AJP/1\.3``" redirectPort=``"[0-9]+``"\s*/>)(\s*)" # $3 $4
      ), -join `
      (
        "`$1`$2",
        "<!--`$2",
        "`$3`$2",
        "-->",
        "`$2`$2"
      )
    )"
}


Function Unzip([string]${zip}, [string]${destination})
{
  ${private:parent} = Split-Path -Path ${destination} -Parent
  ${private:leaf} = Split-Path -Path ${destination} -Leaf
  If (-not (Test-Path ${destination})) {
    ${null} = (New-Item -Path "${parent}" -Name "${leaf}" -ItemType "directory")
  }
  Write-Host "  Unpacking `"${zip}`" into `"${destination}`""
  [System.IO.Compression.ZipFile]::ExtractToDirectory(${zip}, ${destination})
  ${private:children} = @(Get-ChildItem -Path "${destination}")
  If (($(${children}.count) -eq 1) -and (${children}[0].Attributes -match "Directory")) {
    # When the only child-item is a directory we should strip off a layer, which should be safe
    #   Otherwise the (extracted) files in this folder are those that should stay
    # On some systems we cannot rename the extracted folder immediately
    ${private:rename_waiting} = ${TRUE}
    ${private:rename_count} = 0
    While (${private:rename_waiting}) {
      try {
        Rename-Item -Path "$(${children}[0].FullName)" -newName "${leaf}" -ErrorAction Stop
        ${private:rename_waiting} = ${FALSE}
      } catch [System.IO.IOException] {
        If (${private:rename_count}++ -ge 15) {
          # We have waited long enough; report the error
          ${private:rename_waiting} = ${FALSE}
          throw $_
        } Else {
          Start-Sleep -Seconds 2
        }
      }
    }
    Get-ChildItem -Path "${destination}\${leaf}" -Recurse | Move-Item -Destination "${destination}"
    Remove-Item -Path "${destination}\${leaf}"
  }
}

Function Is-Cloud([string]${server}) {
  return ((${server} -like "*.postgres.database.azure.com") -or (${server} -like "*.rds.amazonaws.com"))
}

Function Backup-Databases(
    [string]${admin},
    [string]${server},
    [string]${port},
    [string]${postgresql_path},
    [string]${destination_files},
    [string[]]${databases}
) {
  If (-not (Test-Path ${destination_files})) {
    ${null} = (New-Item -Path "${destination_files}" -ItemType "directory")
  }

  ForEach (${db} in ${databases}) {
    ${private:file} = "${destination_files}\${db}.dump"
    Write-Host "    Backing up `"${db}`" to `"${file}`""
    Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\pg_dump.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-Fc`",",
        "`"-f ${file}`",",
        "`"${db}`""
    )
    ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
    ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
    ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
    ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_dump.exe"
    ${PROCESS_START_INFO}.Arguments = `
      "-h ${server}", `
      "-p ${port}", `
      "-U ${admin}", `
      "-Fc", `
      "-f ${file}", `
      "${db}"
    ${PROCESS_RUN} = New-Object System.Diagnostics.Process
    ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
    ${PROCESS_RUN}.Start() | Out-Null
    ${PROCESS_RUN}.WaitForExit()
    Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
      -replace -join ( `"(?ms)`",
        "pg_dump: \[archiver \(db\)\] connection to database ``"${db}``" failed: FATAL:  database ``"${db}``" does not exist"
      ), "    pg_dump: [archiver (db)] connection to database ``"${db}``" failed: ``"${db}.dump``" will safely remain empty"
    )"
  }

  ${private:roles} = "${destination_files}\roles.sql"
  Write-Host "    Backing up roles to `"${roles}`""
  ${private:cloud} = $(Is-Cloud ${server})
  If (${cloud}) {
    ${private:extra_params} = "--no-role-passwords"
  } Else {
    ${private:extra_params} = " "
  }
  Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\pg_dumpall.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-f ${roles}`",",
        "`"--roles-only`",",
        "`"${extra_params}`""
  )
  ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
  ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
  ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
  ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_dumpall.exe"
  ${PROCESS_START_INFO}.Arguments = `
    "-h ${server}", `
    "-p ${port}", `
    "-U ${admin}", `
    "-f ${roles}", `
    "--roles-only", `
    "${extra_params}"
  ${PROCESS_RUN} = New-Object System.Diagnostics.Process
  ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
  ${PROCESS_RUN}.Start() | Out-Null
  ${PROCESS_RUN}.WaitForExit()
  Write-Host "$(${PROCESS_RUN}.StandardError.ReadToEnd() `
  )"
}

Function Restore-Databases(
    [string]${admin},
    [string]${server},
    [string]${port},
    [string]${postgresql_path},
    [string]${destination_files},
    [string[]]${databases}
) {
  If (-not (Test-Path ${destination_files})) { return }

  ${private:roles} = "${destination_files}\roles.sql"
  Write-Host "    Restoring roles from `"${roles}`""
  Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\psql.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-d postgres`",",
        "`"-f ${roles}`""
  )
  ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
  ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
  ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
  ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\psql.exe"
  ${PROCESS_START_INFO}.Arguments = `
    "-h ${server}", `
    "-p ${port}", `
    "-U ${admin}", `
    "-d postgres", `
    "-f ${roles}"
  ${PROCESS_RUN} = New-Object System.Diagnostics.Process
  ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
  ${PROCESS_RUN}.Start() | Out-Null
  ${PROCESS_RUN}.WaitForExit()
  Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
    -replace -join ( `"(?ms)`",
      "ERROR:  role ``"(.*?)``" already exists"
    ), "Redundant query:  role ``"`$1``" already exists" `
    -replace -join ( `"(?ms)`",
      "NOTICE:  role ``"(.*?)``" is already a member of role ``"(.*?)``""
    ), "Redundant assignment:  role ``"`$1``" is already a member of role ``"`$2``""
  )"

  ForEach (${db} in ${databases}) {
    ${private:file} = "${destination_files}\${db}.dump"
    If ((Get-Content "${private:file}") -ne ${NULL}) {
      Write-Host "    Restoring `"${db}`" from `"${file}`""
      # Delete the DB if it exists, which it should not
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\dropdb.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"${db}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\dropdb.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}", `
        "-p ${port}", `
        "-U ${admin}", `
        "${db}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          "dropdb: database removal failed: ERROR:  database ``"${db}``" does not exist"
        ), "    dropdb: database removal redundant:  database ``"${db}``" does not exist"
      )"

      # Create the DB
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\createdb.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"-T template0`",",
            "`"${db}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\createdb.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}", `
        "-p ${port}", `
        "-U ${admin}", `
        "-T template0", `
        "${db}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          "createdb: database creation failed: ERROR:  database ``"${db}``" already exists"
        ), "    createdb: database creation redundant:  database ``"${db}``" already exists"
      )"

      # Populate the database with the backed-up data
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\pg_restore.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"-d ${db}`",",
            "`"-Fc ${file}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_restore.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}",
        "-p ${port}",
        "-U ${admin}",
        "-d ${db}",
        "-Fc ${file}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$(${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          `"(pg_restore: \[archiver \(db\)\] )(Error)( while PROCESSING TOC:`r`n)`",
          `"(pg_restore: \[archiver \(db\)\] )(Error)( from TOC entry 3; 2615 2200 SCHEMA public postgres`r`n)`",
          `"(pg_restore: \[archiver \(db\)\] )(could not execute query: ERROR)(:  schema ``"public``" already exists)`"
        ), `"    `$1Redundancy`$3    `$4Redundancy`$6    `$7Skipping query execution`$9`" `
        -replace -join ( `"(?ms)`",
          "WARNING: errors ignored on restore: [0-9]+"
        ), `"`" `
        -replace -join ( `"(?ms)`",
          "(pg_restore: \[custom archiver\] )(could not read from)( input file: end of file)"
        ), `"    `$1ignoring empty input file ``"${db}.dump``"`"
      )"
    } Else {
      Write-Host "    Ignoring empty file `"${db}.dump`"`n"
    }
  }
}

Function Show-Differences(
    [string]${reference},
    [string]${difference}
) {
  ${do_please} = Read-Host "  Do you want to diff/compare `"${reference}`" with `"${difference}`"?`n    It can potentially take several minutes. [Y/n]"
  If ("${do_please}" -eq "" -Or "${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
    ${comparisons} = @{
      ReferenceObject = (Get-ChildItem -Recurse -Path ${reference})
      DifferenceObject = (Get-ChildItem -Recurse -Path ${difference})
    }

    Compare-Object @comparisons -PassThru
  }
}

Function Get-ResolvedPath(
    [string]${path}
) {
  ${item} = (Get-Item -Path ${path})
  If (${item}.Attributes -match "ReparsePoint") {
    ${local:path_target} = ${item}.Target
    If ([System.IO.Path]::IsPathRooted(${local:path_target})) {
      ${response} = "${local:path_target}"
    } Else {
      ${response} = "$(Split-Path -Path "$(${item}.FullName)" -Parent)\${local:path_target}"
    }
  } Else {
    ${response} = "$(${item}.FullName)"
  }
  return ${response}
}

Function Summary-Print() {
  Write-Host "`nSummary"
  ForEach (${local:register} in ${global:SUMMARY_REGISTRATION}) {
    If ( "${register}" -eq "activemq" ) { Summary-PrintOne "ActiveMQ" ${active_mq_summary_code} }
    If ( "${register}" -eq "java" ) { Summary-PrintOne "Java" ${java_summary_code} }
    If ( "${register}" -eq "postgresql" ) { Summary-PrintOne "PostgreSQL" ${postgresql_summary_code} }
    If ( "${register}" -eq "tomcat" ) { Summary-PrintOne "Tomcat" ${tomcat_summary_code} }
  }
}

Function Summary-PrintOne(
  [string]${name},
  [int]${code}
) {
  Switch(${code}) {
    ${SUMMARY_CODE_COMPLETED} { Write-Host "  ${name} updated successfully" }
    ${SUMMARY_CODE_SKIPPED} { Write-Host "  ${name} update was possible but skipped" }
    ${SUMMARY_CODE_MISSED} { Write-Host "  ${name} update unavailable due to missing installation file" }
    ${SUMMARY_CODE_FAILED} { Write-Host "  ${name} failed to update" }
  }
}

Function Summary-Register(
  [string[]]${components}
) {
  ForEach (${local:component} in ${components}) {
    ${private:can_add} = ${TRUE}
    ForEach (${local:register} in ${global:SUMMARY_REGISTRATION}) {
      If ("${register}" -eq "${component}") {
        ${private:can_add} = ${FALSE}
        continue
      }
    }
    If (${can_add}) {
      ${global:SUMMARY_REGISTRATION} = @(
        ${global:SUMMARY_REGISTRATION}
        ${component}
      )
    }
  }
}


# SIG # Begin signature block
# MIIdcwYJKoZIhvcNAQcCoIIdZDCCHWACAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBkJfvfjJ+HUukK
# rRA0BdR5YWvRNEClaO9ZBi7ng+Si8aCCGFwwggO3MIICn6ADAgECAhAM5+DlF9hG
# /o/lYPwb8DA5MA0GCSqGSIb3DQEBBQUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNV
# BAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0wNjExMTAwMDAwMDBa
# Fw0zMTExMTAwMDAwMDBaMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNVBAMTG0RpZ2lD
# ZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCC
# AQoCggEBAK0OFc7kQ4BcsYfzt2D5cRKlrtwmlIiq9M71IDkoWGAM+IDaqRWVMmE8
# tbEohIqK3J8KDIMXeo+QrIrneVNcMYQq9g+YMjZ2zN7dPKii72r7IfJSYd+fINcf
# 4rHZ/hhk0hJbX/lYGDW8R82hNvlrf9SwOD7BG8OMM9nYLxj+KA+zp4PWw25EwGE1
# lhb+WZyLdm3X8aJLDSv/C3LanmDQjpA1xnhVhyChz+VtCshJfDGYM2wi6YfQMlqi
# uhOCEe05F52ZOnKh5vqk2dUXMXWuhX0irj8BRob2KHnIsdrkVxfEfhwOsLSSplaz
# vbKX7aqn8LfFqD+VFtD/oZbrCF8Yd08CAwEAAaNjMGEwDgYDVR0PAQH/BAQDAgGG
# MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFEXroq/0ksuCMS1Ri6enIZ3zbcgP
# MB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA0GCSqGSIb3DQEBBQUA
# A4IBAQCiDrzf4u3w43JzemSUv/dyZtgy5EJ1Yq6H6/LV2d5Ws5/MzhQouQ2XYFwS
# TFjk0z2DSUVYlzVpGqhH6lbGeasS2GeBhN9/CTyU5rgmLCC9PbMoifdf/yLil4Qf
# 6WXvh+DfwWdJs13rsgkq6ybteL59PyvztyY1bV+JAbZJW58BBZurPSXBzLZ/wvFv
# hsb6ZGjrgS2U60K3+owe3WLxvlBnt2y98/Efaww2BxZ/N3ypW2168RJGYIPXJwS+
# S86XvsNnKmgR34DnDDNmvxMNFG7zfx9jEB76jRslbWyPpbdhAbHSoyahEHGdreLD
# +cOZUbcrBwjOLuZQsqf6CkUvovDyMIIE/jCCA+agAwIBAgIQDUJK4L46iP9gQCHO
# FADw3TANBgkqhkiG9w0BAQsFADByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGln
# aUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhE
# aWdpQ2VydCBTSEEyIEFzc3VyZWQgSUQgVGltZXN0YW1waW5nIENBMB4XDTIxMDEw
# MTAwMDAwMFoXDTMxMDEwNjAwMDAwMFowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoT
# DkRpZ2lDZXJ0LCBJbmMuMSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAy
# MTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAMLmYYRnxYr1DQikRcpj
# a1HXOhFCvQp1dU2UtAxQtSYQ/h3Ib5FrDJbnGlxI70Tlv5thzRWRYlq4/2cLnGP9
# NmqB+in43Stwhd4CGPN4bbx9+cdtCT2+anaH6Yq9+IRdHnbJ5MZ2djpT0dHTWjaP
# xqPhLxs6t2HWc+xObTOKfF1FLUuxUOZBOjdWhtyTI433UCXoZObd048vV7WHIOsO
# jizVI9r0TXhG4wODMSlKXAwxikqMiMX3MFr5FK8VX2xDSQn9JiNT9o1j6BqrW7Ed
# MMKbaYK02/xWVLwfoYervnpbCiAvSwnJlaeNsvrWY4tOpXIc7p96AXP4Gdb+DUmE
# vQECAwEAAaOCAbgwggG0MA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAAMBYG
# A1UdJQEB/wQMMAoGCCsGAQUFBwMIMEEGA1UdIAQ6MDgwNgYJYIZIAYb9bAcBMCkw
# JwYIKwYBBQUHAgEWG2h0dHA6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzAfBgNVHSME
# GDAWgBT0tuEgHf4prtLkYaWyoiWyyBc1bjAdBgNVHQ4EFgQUNkSGjqS6sGa+vCgt
# HUQ23eNqerwwcQYDVR0fBGowaDAyoDCgLoYsaHR0cDovL2NybDMuZGlnaWNlcnQu
# Y29tL3NoYTItYXNzdXJlZC10cy5jcmwwMqAwoC6GLGh0dHA6Ly9jcmw0LmRpZ2lj
# ZXJ0LmNvbS9zaGEyLWFzc3VyZWQtdHMuY3JsMIGFBggrBgEFBQcBAQR5MHcwJAYI
# KwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBPBggrBgEFBQcwAoZD
# aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0U0hBMkFzc3VyZWRJ
# RFRpbWVzdGFtcGluZ0NBLmNydDANBgkqhkiG9w0BAQsFAAOCAQEASBzctemaI7zn
# GucgDo5nRv1CclF0CiNHo6uS0iXEcFm+FKDlJ4GlTRQVGQd58NEEw4bZO73+RAJm
# Te1ppA/2uHDPYuj1UUp4eTZ6J7fz51Kfk6ftQ55757TdQSKJ+4eiRgNO/PT+t2R3
# Y18jUmmDgvoaU+2QzI2hF3MN9PNlOXBL85zWenvaDLw9MtAby/Vh/HUIAHa8gQ74
# wOFcz8QRcucbZEnYIpp1FUL1LTI4gdr0YKK6tFL7XOBhJCVPst/JKahzQ1HavWPW
# H1ub9y4bTxMd90oNcX6Xt/Q/hOvB46NJofrOp79Wz7pZdmGJX36ntI5nePk2mOHL
# KNpbh6aKLzCCBTAwggQYoAMCAQICEAQJGBtf1btmdVNDtW+VUAgwDQYJKoZIhvcN
# AQELBQAwZTELMAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
# A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJl
# ZCBJRCBSb290IENBMB4XDTEzMTAyMjEyMDAwMFoXDTI4MTAyMjEyMDAwMFowcjEL
# MAkGA1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3
# LmRpZ2ljZXJ0LmNvbTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElE
# IENvZGUgU2lnbmluZyBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEB
# APjTsxx/DhGvZ3cH0wsxSRnP0PtFmbE620T1f+Wondsy13Hqdp0FLreP+pJDwKX5
# idQ3Gde2qvCchqXYJawOeSg6funRZ9PG+yknx9N7I5TkkSOWkHeC+aGEI2YSVDNQ
# dLEoJrskacLCUvIUZ4qJRdQtoaPpiCwgla4cSocI3wz14k1gGL6qxLKucDFmM3E+
# rHCiq85/6XzLkqHlOzEcz+ryCuRXu0q16XTmK/5sy350OTYNkO/ktU6kqepqCquE
# 86xnTrXE94zRICUj6whkPlKWwfIPEvTFjg/BougsUfdzvL2FsWKDc0GCB+Q4i2pz
# INAPZHM8np+mM6n9Gd8lk9ECAwEAAaOCAc0wggHJMBIGA1UdEwEB/wQIMAYBAf8C
# AQAwDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHkGCCsGAQUF
# BwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEMG
# CCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3J0MIGBBgNVHR8EejB4MDqgOKA2hjRodHRwOi8vY3Js
# NC5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3JsMDqgOKA2
# hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290
# Q0EuY3JsME8GA1UdIARIMEYwOAYKYIZIAYb9bAACBDAqMCgGCCsGAQUFBwIBFhxo
# dHRwczovL3d3dy5kaWdpY2VydC5jb20vQ1BTMAoGCGCGSAGG/WwDMB0GA1UdDgQW
# BBRaxLl7KgqjpepxA8Bg+S32ZXUOWDAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYun
# pyGd823IDzANBgkqhkiG9w0BAQsFAAOCAQEAPuwNWiSz8yLRFcgsfCUpdqgdXRwt
# OhrE7zBh134LYP3DPQ/Er4v97yrfIFU3sOH20ZJ1D1G0bqWOWuJeJIFOEKTuP3GO
# Yw4TS63XX0R58zYUBor3nEZOXP+QsRsHDpEV+7qvtVHCjSSuJMbHJyqhKSgaOnEo
# AjwukaPAJRHinBRHoXpoaK+bp1wgXNlxsQyPu6j4xRJon89Ay0BEpRPw5mQMJQhC
# MrI2iiQC/i9yfhzXSUWW6Fkd6fp0ZGuy62ZD2rOwjNXpDd32ASDOmTFjPQgaGLOB
# m0/GkxAG/AeB+ova+YJJ92JuoVP6EpQYhS6SkepobEQysmah5xikmmRR7zCCBTEw
# ggQZoAMCAQICEAqhJdbWMht+QeQF2jaXwhUwDQYJKoZIhvcNAQELBQAwZTELMAkG
# A1UEBhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRp
# Z2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENB
# MB4XDTE2MDEwNzEyMDAwMFoXDTMxMDEwNzEyMDAwMFowcjELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTExMC8GA1UEAxMoRGlnaUNlcnQgU0hBMiBBc3N1cmVkIElEIFRpbWVzdGFtcGlu
# ZyBDQTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL3QMu5LzY9/3am6
# gpnFOVQoV7YjSsQOB0UzURB90Pl9TWh+57ag9I2ziOSXv2MhkJi/E7xX08PhfgjW
# ahQAOPcuHjvuzKb2Mln+X2U/4Jvr40ZHBhpVfgsnfsCi9aDg3iI/Dv9+lfvzo7oi
# PhisEeTwmQNtO4V8CdPuXciaC1TjqAlxa+DPIhAPdc9xck4Krd9AOly3UeGheRTG
# TSQjMF287DxgaqwvB8z98OpH2YhQXv1mblZhJymJhFHmgudGUP2UKiyn5HU+upgP
# hH+fMRTWrdXyZMt7HgXQhBlyF/EXBu89zdZN7wZC/aJTKk+FHcQdPK/P2qwQ9d2s
# rOlW/5MCAwEAAaOCAc4wggHKMB0GA1UdDgQWBBT0tuEgHf4prtLkYaWyoiWyyBc1
# bjAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzASBgNVHRMBAf8ECDAG
# AQH/AgEAMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB5Bggr
# BgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNv
# bTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0QXNzdXJlZElEUm9vdENBLmNydDCBgQYDVR0fBHoweDA6oDigNoY0aHR0cDov
# L2NybDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNybDA6
# oDigNoY0aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElE
# Um9vdENBLmNybDBQBgNVHSAESTBHMDgGCmCGSAGG/WwAAgQwKjAoBggrBgEFBQcC
# ARYcaHR0cHM6Ly93d3cuZGlnaWNlcnQuY29tL0NQUzALBglghkgBhv1sBwEwDQYJ
# KoZIhvcNAQELBQADggEBAHGVEulRh1Zpze/d2nyqY3qzeM8GN0CE70uEv8rPAwL9
# xafDDiBCLK938ysfDCFaKrcFNB1qrpn4J6JmvwmqYN92pDqTD/iy0dh8GWLoXoIl
# HsS6HHssIeLWWywUNUMEaLLbdQLgcseY1jxk5R9IEBhfiThhTWJGJIdjjJFSLK8p
# ieV4H9YLFKWA1xJHcLN11ZOFk362kmf7U2GJqPVrlsD0WGkNfMgBsbkodbeZY4Ui
# jGHKeZR+WfyMD+NvtQEmtmyl7odRIeRYYJu6DC0rbaLEfrvEJStHAgh8Sa4TtuF8
# QkIoxhhWz0E0tmZdtnR79VYzIi8iNrJLokqV2PWmjlIwggUyMIIEGqADAgECAhAB
# Pe4iVgqMX6vbiX1xPVYPMA0GCSqGSIb3DQEBCwUAMHIxCzAJBgNVBAYTAlVTMRUw
# EwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20x
# MTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNpZ25pbmcg
# Q0EwHhcNMTgwNjExMDAwMDAwWhcNMjEwODIxMTIwMDAwWjBvMQswCQYDVQQGEwJH
# QjEQMA4GA1UEBxMHTmV3YnVyeTEmMCQGA1UEChMdTWljcm8gRm9jdXMgSW50ZXJu
# YXRpb25hbCBwbGMxJjAkBgNVBAMTHU1pY3JvIEZvY3VzIEludGVybmF0aW9uYWwg
# cGxjMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA2Vb7QjccsV83UO3b
# YCw/nTB6Bl3tGX0eSCCPsBtPTspWKz2GXWK7/1KeDUNPLnNBT1InLwNcaDxiX1TG
# SC5+gBfJiaTWB03ll7FpDbayyQ532smNvlQop2v5rgDkXQLu4Jg1JE3OLrppxBfP
# tBsjHw76kCqePA8cvTMWxUc6b7loX26dQe8v7+SeOI0qZ0tXCNwtgTu2513frUxj
# d/fY2F/toc9OjateJZQ5v5+WV7dOG/89bkvMAEHXegLcPZw7PT4aZP/ZVOYtoflS
# g6oT0m0N1LdsjyTEDbqmRm6/dt5sI9akmdln7hpOp1y9EMCgkGiNAwf9u4Lw2iUj
# NvoT5QIDAQABo4IBxTCCAcEwHwYDVR0jBBgwFoAUWsS5eyoKo6XqcQPAYPkt9mV1
# DlgwHQYDVR0OBBYEFLyS6yL4fTRV+hftOctsNinkVorLMA4GA1UdDwEB/wQEAwIH
# gDATBgNVHSUEDDAKBggrBgEFBQcDAzB3BgNVHR8EcDBuMDWgM6Axhi9odHRwOi8v
# Y3JsMy5kaWdpY2VydC5jb20vc2hhMi1hc3N1cmVkLWNzLWcxLmNybDA1oDOgMYYv
# aHR0cDovL2NybDQuZGlnaWNlcnQuY29tL3NoYTItYXNzdXJlZC1jcy1nMS5jcmww
# TAYDVR0gBEUwQzA3BglghkgBhv1sAwEwKjAoBggrBgEFBQcCARYcaHR0cHM6Ly93
# d3cuZGlnaWNlcnQuY29tL0NQUzAIBgZngQwBBAEwgYQGCCsGAQUFBwEBBHgwdjAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tME4GCCsGAQUFBzAC
# hkJodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRTSEEyQXNzdXJl
# ZElEQ29kZVNpZ25pbmdDQS5jcnQwDAYDVR0TAQH/BAIwADANBgkqhkiG9w0BAQsF
# AAOCAQEAZllSiMymdGRhqGhTP8p8Xdjmu1IZu6wg09n7oKLaH7qg3TfjqX+jjaQ2
# CTTyr2cwblYPa1Dp2sOqjGUNyrUvg22cfaSJAtbTNkpb8VtU8nEMjc3IaTdAw0hM
# +qHgYUaX1he6HrP8x51xJW67Z+Q6junwCPjo4fhEonJ4D9PBqQpRzpHtHy3D9rI6
# sgTSX3uWHqUet4Ehq49dSb2GZB8qAweUeM8ntCsEe2Trrei+HbrWn2F6DjpFCRA8
# VjhNvIsqOyltdsGT1HOGHz3hIi2G1Xr/T43sBmeqndopqavoj9GZsUMVp6RPRohn
# yGDTxINU2+Cs981b0UUjO5BQkkQfrzGCBG0wggRpAgEBMIGGMHIxCzAJBgNVBAYT
# AlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2Vy
# dC5jb20xMTAvBgNVBAMTKERpZ2lDZXJ0IFNIQTIgQXNzdXJlZCBJRCBDb2RlIFNp
# Z25pbmcgQ0ECEAE97iJWCoxfq9uJfXE9Vg8wDQYJYIZIAWUDBAIBBQCggYQwGAYK
# KwYBBAGCNwIBDDEKMAigAoAAoQKAADAZBgkqhkiG9w0BCQMxDAYKKwYBBAGCNwIB
# BDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTAvBgkqhkiG9w0BCQQxIgQg
# GS/6+b9xk2repOFnwN6eHXMmRiVOUYfPUyFZQTnik9cwDQYJKoZIhvcNAQEBBQAE
# ggEAdFiHfuz0gYEP0vX5QQEjoaUnfcunZcdOncu9+W5949jgUI8y16YSbvKpy71l
# HHlkUElt8yf0qjmxxx2BB+sSt5eHW1ULhedBqDwMbQnBvd2wNrzKxH/qtsWH55GL
# 21UJC2Bqt9qwvcUtZUOJU1M4H5pF56ronFSzrt2Cs98ZPTs0qMQFdmjI+jJEx/10
# 1B6SfTsFdTiCouR1veHhvxU21O+0Ne21WndrwyyeHBCbZ5Hjy9ZfQft+3owfoxfl
# 2WrPI45eXcrtgpJ3Tk6uM1Xj3NyajwzEU/fAeeMQTgTg02JW7n+KVIIWKO1O0Z+u
# kxZUbOq9bN+4QDh6TIx5WTu4xqGCAjAwggIsBgkqhkiG9w0BCQYxggIdMIICGQIB
# ATCBhjByMQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYD
# VQQLExB3d3cuZGlnaWNlcnQuY29tMTEwLwYDVQQDEyhEaWdpQ2VydCBTSEEyIEFz
# c3VyZWQgSUQgVGltZXN0YW1waW5nIENBAhANQkrgvjqI/2BAIc4UAPDdMA0GCWCG
# SAFlAwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG9w0B
# CQUxDxcNMjEwMzAyMTgxMzAzWjAvBgkqhkiG9w0BCQQxIgQg8jequp6KdKVdGzAP
# /V+7EOiffbEWLCahVY9gLURD718wDQYJKoZIhvcNAQEBBQAEggEAQLPaPe+23OdE
# wytuI87ACNyxUvvtoz+xLnmYtq1gki7lhIQoIKfUjR+u2TCmzmt7ygi44oUmb/gf
# /mXyHSxs/9ZuEuLvuKvW4umx7g3ZujvZbI1vnMdce+QUkr5EEXCvuQrQpd5Lt2IK
# lcTCLMCgPxZOAmY433cBONt0TQnxdPMA5YdjmWkPnu+VByzqTO7UomNpWL38auUv
# EveK3Mnux7/JftwEpVDZTSg5pS1MMiMoN4uTnbyITTq1WLWKB7ZnFCCkxoJ9sJw4
# 0q7uRUGxOc0AsPtnbfNx2S8g2rfc7YUN/xdxmVbTQGQL+E6JYDq5UYbnKV1NQ3xh
# oz3xf8V9TA==
# SIG # End signature block
