. ${PSScriptRoot}\def_copies.ps1

# Define variables
${ACTIVE_MQ_DEFAULT} = "C:\netiq\idm\apps\activemq"
${JAVA_DEFAULT} = "C:\netiq\idm\apps\jre"
${POSTGRESQL_ADMIN_DEFAULT} = "postgres"
${POSTGRESQL_PATH_DEFAULT} = "C:\netiq\idm\apps\postgres"
${POSTGRESQL_PORT_DEFAULT} = "5432"
${POSTGRESQL_SERVER_DEFAULT} = "localhost"
${TOMCAT_DEFAULT} = "C:\netiq\idm\apps\tomcat"
${ACTIVE_MQ_REQUIED_1}="activemq-all-5.17*.jar"
${ACTIVE_MQ_REQUIED_2}="log4j-api*.jar"

${DIR_COMPRESSED} = "${PSScriptRoot}\compressed"

${FILE_ACTIVE_MQ} = "${DIR_COMPRESSED}\apache-activemq-${ACTIVE_MQ_VERSION_TARGET}-bin.zip"
${FILE_JAVA} = "${DIR_COMPRESSED}\zulu${JAVA_DOWNLOAD_CODE}-win_x64.zip"
${FILE_POSTGRESQL} = "${DIR_COMPRESSED}\postgresql-${POSTGRESQL_VERSION_TARGET}-windows-x64.exe"
${FILE_TOMCAT} = "${DIR_COMPRESSED}\apache-tomcat-${TOMCAT_VERSION_TARGET}-windows-x64.zip"

${global:ALREADY_OBTAINED_JAVA_LOC} = ${FALSE}

${TIMESTAMP} = (Get-Date -Format "yyyyMMdd-HHmmss")

${COMPARE_CODE_EQ} = 0
${COMPARE_CODE_LT} = 1
${COMPARE_CODE_GT} = 2

${global:SUMMARY_REGISTRATION}=@()
${SUMMARY_CODE_COMPLETED} = 0
${SUMMARY_CODE_SKIPPED} = 1
${SUMMARY_CODE_MISSED} = 2
${SUMMARY_CODE_FAILED} = 3

Add-Type -AssemblyName System.IO.Compression.FileSystem

# Define functions
Function Get-PathValidity ([string]${base}, [string[]]${array}) {
  ${valid} = ${TRUE}
  ForEach (${path} in ${array}) {
    If (-not (Test-Path ${base}\${path})) {
      Write-Information "    Expected to find '${base}\${path}'" -InformationAction Continue
      ${valid}=${FALSE}
    }
  }
  return ${valid}
}

Function Get-ValidJavaPath () {
  If (-not ${global:ALREADY_OBTAINED_JAVA_LOC}) {
    ${global:ALREADY_OBTAINED_JAVA_LOC} = ${TRUE}
    ${input_invalid} = ${TRUE}
    while (${input_invalid}) {
      ${java_loc} = Read-Host "  Where is your Java (JRE) located? [${java_loc_final}]"
      If ("${java_loc}" -ne "") { ${java_loc_final} = ${java_loc} }
      ${result} = (Get-PathValidity ${java_loc_final} ${JAVA_VALIDITY})
      If ($result -eq ${TRUE}) {
        ${input_invalid} = ${FALSE}
      }
    }
  }
}

Function Parse-Version ([string]${version}) {
  return (${version} -replace '\.', ' ' -replace '-', ' ' -replace '_', ' ').split(" ")
}

Function Compare-Versions ([string]${A}, [string]${B}) {
  ${version_A} = Parse-Version ${A}
  ${version_B} = Parse-Version ${B}

  ${max} = ${version_B}.count
  For ((${i} = 0); ${i} -lt ${max}; ${i}++) {
    try {
      ${actual} = [int]::Parse(${version_A}[${i}])
      ${expected} = [int]::Parse(${version_B}[${i}])
    } catch [System.Management.Automation.MethodInvocationException] {
      # Compare as strings in case of an error
      ${actual} = ${version_A}[${i}]
      ${expected} = ${version_B}[${i}]
    }
    If ( ${actual} -lt ${expected} ) {
      return ${COMPARE_CODE_LT}
    } ElseIf ( ${actual} -gt ${expected} ) {
      return ${COMPARE_CODE_GT}
    }
  }
  return ${COMPARE_CODE_EQ}
}

Function Compare-VersionsWithResult ([string]${name}, [string]${actual}, [string]${expected}) {
  ${comparison} = Compare-Versions ${actual} ${expected}
  If ( ${comparison} -eq ${COMPARE_CODE_LT} ) {
    Write-Host "  Expected ${name} version ${expected}, but instead found the older ${actual}"
  } ElseIf ( ${comparison} -eq ${COMPARE_CODE_GT} ) {
    Write-Host "  Your ${name} version (${actual}) surpasses the target (${expected})`n    We cannot guarantee your version will work, but it might."
  } Else { # If ( ${comparison} -eq ${COMPARE_CODE_EQ} )
    Write-Host "  Your ${name} version (${expected}) appears to be valid."
  }
}

Function Wait-StopProcesses([string]${component}, [string]${search}) {
  ${private:process_all} = (Get-WmiObject -ClassName Win32_Process -Filter "CommandLine LIKE '%${search}%'")
  While (${process_all} -ne ${null}) {
    ${private:do_please} = Read-Host "    It appears that ${component} is still running.`n      May we forcefully stop it [y/N]?"
    If ("${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
      ForEach (${process} in ${process_all}) {
        Write-Host "    Stopping `"$(${process}.Name)`":"
        ${null} = ${process}.Terminate()
        Start-Sleep 3
      }
    } Else {
      Read-Host "    Go ahead and stop your ${component} instances.`n      When the ${component} instances have completely stopped, press ENTER."
    }
    ${private:process_all} = (Get-WmiObject -ClassName Win32_Process -Filter "CommandLine LIKE '%${search}%'")
  }
  Write-Host "    It appears no instances of ${component} are running. Proceeding ..."
}

Function Wait-StopServices([string]${component}, [string]${search}) {
  ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.Status -eq "Running"})
  While (${service_all} -ne ${null}) {
    ${private:do_please} = Read-Host "    It appears that ${component} is still running.`n      May we forcefully stop it [y/N]?"
    If ("${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
      ForEach (${service} in ${service_all}) {
        Write-Host "    Stopping `"$(${service}.Name)`":"
        Stop-Service -Force -Name ${service}.Name
        Start-Sleep 3
      }
    } Else {
      Read-Host "    Go ahead and stop your ${component} instances.`n      When the ${component} instances have completely stopped, press ENTER."
    }
    ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.Status -eq "Running"})
  }
  Write-Host "    It appears no instances of ${component} are running. Proceeding ..."
}

Function Wait-DisableServices([string]${component}, [string]${search}) {
  ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.StartType -ne "Disabled"})
  While (${service_all} -ne ${null}) {
    ForEach (${service} in ${service_all}) {
      Write-Host "    Disabling `"$(${service}.Name)`""
      Set-Service -Name ${service}.Name -StartupType Disabled
      Start-Sleep 3
    }
    ${private:service_all} = (Get-Service -DisplayName "*${search}*" | Where-Object {$_.StartType -ne "Disabled"})
  }
  Write-Host "    It appears no instances of ${component} are enabled. Proceeding ..."
}

Function Copy-OldFiles([string]${name}, [string]${source}, [string]${destination}, [string[]]$files) {
  ForEach (${file} in ${files}) {
    If ((${file} -ne ${null}) -and (${file} -ne "")) {
      If (Test-Path -Path ${source}\${file}) {
        If (Test-Path -Path ${destination}\${file}) {
          ${private:item} = (Get-Item ${destination}\${file})
          ${private:backup_name} = "$(${item}.BaseName)-backup$(${item}.Extension)"
          ${private:backup_full} = "$(${item}.Directory)\${backup_name}"
          Write-Host "    Backing up existing ${name} '${file}' to '${backup_name}'"
          Move-Item -Path ${item} -Destination ${backup_full} -Force
        }
        Write-Host "    Copying ${name} '${file}'"
        Copy-Item -Path "${source}\${file}" -Destination "$(Split-Path -Path "${destination}\${file}" -Parent)"
      }
    }
  }
}

Function Discard-SpecificFiles([string]${name}, [string]${destination}, [string[]]$files) {
  ForEach (${file} in ${files}) {
    If ((${file} -ne ${null}) -and (${file} -ne "")) {
      If (Test-Path ${destination}\${file}) {
        Write-Host "    Removing existing ${name} '${file}'"
        Remove-Item -Path "${destination}\${file}" -Recurse
      }
    }
  }
}

Function addRequiredMQJar{
 
 If (Test-Path -Path ${ACTIVE_MQ_DEFAULT}\${ACTIVE_MQ_REQUIED_1}) {   
   If (-Not (Test-Path -Path ${TOMCAT_DEFAULT}\lib\${ACTIVE_MQ_REQUIED_1})) {
    write-output "  Adding required jar #1 for ActiveMQ 5.17"
	  copy-item -Path ${ACTIVE_MQ_DEFAULT}\${ACTIVE_MQ_REQUIED_1} -Destination ${TOMCAT_DEFAULT}\lib\
   }
 } 
  If (Test-Path -Path ${ACTIVE_MQ_DEFAULT}\lib\optional\${ACTIVE_MQ_REQUIED_2}) {   
   If (-Not (Test-Path -Path ${TOMCAT_DEFAULT}\lib\${ACTIVE_MQ_REQUIED_2})) {
    write-output "  Adding required jar #2 for ActiveMQ 5.17"
	  copy-item -Path ${ACTIVE_MQ_DEFAULT}\lib\optional\${ACTIVE_MQ_REQUIED_2} -Destination ${TOMCAT_DEFAULT}\lib\
   }
 }
  If (Test-Path -Path ${TOMCAT_DEFAULT}\lib\activemq-all-5.16*.jar ) {
   write-output "  Deleting old ActiveMQ 5.16 jar"
   remove-item -Path ${TOMCAT_DEFAULT}\lib\activemq-all-5.16*.jar -Force
 }
}

Function Disable-AJPConnector([string]${tomcat}) {
  Write-Host "    Disabling Tomcat AJP connector within `"${tomcat}\conf\server.xml`""

  Set-Content `
    -Path "${tomcat}\conf\server.xml" `
    -Value "$((Get-Content `"${tomcat}\conf\server.xml`" -Raw) -replace -join `
      (
        "(?ms)",
        "(\s+<!-- Define an AJP 1\.3 Connector on port [0-9]+ -->)(\s*)", # $1 $2
        "(<Connector port=``"[0-9]+``" protocol=``"AJP/1\.3``" redirectPort=``"[0-9]+``"\s*/>)(\s*)" # $3 $4
      ), -join `
      (
        "`$1`$2",
        "<!--`$2",
        "`$3`$2",
        "-->",
        "`$2`$2"
      )
    )"
}


Function Unzip([string]${zip}, [string]${destination})
{
  ${private:parent} = Split-Path -Path ${destination} -Parent
  ${private:leaf} = Split-Path -Path ${destination} -Leaf
  If (-not (Test-Path ${destination})) {
    ${null} = (New-Item -Path "${parent}" -Name "${leaf}" -ItemType "directory")
  }
  Write-Host "  Unpacking `"${zip}`" into `"${destination}`""
  [System.IO.Compression.ZipFile]::ExtractToDirectory(${zip}, ${destination})
  ${private:children} = @(Get-ChildItem -Path "${destination}")
  If (($(${children}.count) -eq 1) -and (${children}[0].Attributes -match "Directory")) {
    # When the only child-item is a directory we should strip off a layer, which should be safe
    #   Otherwise the (extracted) files in this folder are those that should stay
    # On some systems we cannot rename the extracted folder immediately
    ${private:rename_waiting} = ${TRUE}
    ${private:rename_count} = 0
    While (${private:rename_waiting}) {
      try {
        Rename-Item -Path "$(${children}[0].FullName)" -newName "${leaf}" -ErrorAction Stop
        ${private:rename_waiting} = ${FALSE}
      } catch [System.IO.IOException] {
        If (${private:rename_count}++ -ge 15) {
          # We have waited long enough; report the error
          ${private:rename_waiting} = ${FALSE}
          throw $_
        } Else {
          Start-Sleep -Seconds 2
        }
      }
    }
    Get-ChildItem -Path "${destination}\${leaf}" -Recurse | Move-Item -Destination "${destination}"
    Remove-Item -Path "${destination}\${leaf}"
  }
}

Function Is-Cloud([string]${server}) {
  return ((${server} -like "*.postgres.database.azure.com") -or (${server} -like "*.rds.amazonaws.com"))
}

Function Backup-Databases(
    [string]${admin},
    [string]${server},
    [string]${port},
    [string]${postgresql_path},
    [string]${destination_files},
    [string[]]${databases}
) {
  If (-not (Test-Path ${destination_files})) {
    ${null} = (New-Item -Path "${destination_files}" -ItemType "directory")
  }

  ForEach (${db} in ${databases}) {
    ${private:file} = "${destination_files}\${db}.dump"
    Write-Host "    Backing up `"${db}`" to `"${file}`""
    Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\pg_dump.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-Fc`",",
        "`"-f ${file}`",",
        "`"${db}`""
    )
    ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
    ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
    ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
    ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_dump.exe"
    ${PROCESS_START_INFO}.Arguments = `
      "-h ${server}", `
      "-p ${port}", `
      "-U ${admin}", `
      "-Fc", `
      "-f ${file}", `
      "${db}"
    ${PROCESS_RUN} = New-Object System.Diagnostics.Process
    ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
    ${PROCESS_RUN}.Start() | Out-Null
    ${PROCESS_RUN}.WaitForExit()
    Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
      -replace -join ( `"(?ms)`",
        "pg_dump: \[archiver \(db\)\] connection to database ``"${db}``" failed: FATAL:  database ``"${db}``" does not exist"
      ), "    pg_dump: [archiver (db)] connection to database ``"${db}``" failed: ``"${db}.dump``" will safely remain empty"
    )"
  }

  ${private:roles} = "${destination_files}\roles.sql"
  Write-Host "    Backing up roles to `"${roles}`""
  ${private:cloud} = $(Is-Cloud ${server})
  If (${cloud}) {
    ${private:extra_params} = "--no-role-passwords"
  } Else {
    ${private:extra_params} = " "
  }
  Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\pg_dumpall.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-f ${roles}`",",
        "`"--roles-only`",",
        "`"${extra_params}`""
  )
  ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
  ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
  ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
  ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_dumpall.exe"
  ${PROCESS_START_INFO}.Arguments = `
    "-h ${server}", `
    "-p ${port}", `
    "-U ${admin}", `
    "-f ${roles}", `
    "--roles-only", `
    "${extra_params}"
  ${PROCESS_RUN} = New-Object System.Diagnostics.Process
  ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
  ${PROCESS_RUN}.Start() | Out-Null
  ${PROCESS_RUN}.WaitForExit()
  Write-Host "$(${PROCESS_RUN}.StandardError.ReadToEnd() `
  )"
}

Function Restore-Databases(
    [string]${admin},
    [string]${server},
    [string]${port},
    [string]${postgresql_path},
    [string]${destination_files},
    [string[]]${databases}
) {
  If (-not (Test-Path ${destination_files})) { return }

  ${private:roles} = "${destination_files}\roles.sql"
  Write-Host "    Restoring roles from `"${roles}`""
  Write-Host (
    "    ",
    "Start-Process",
      "-FilePath ${postgresql_path}\bin\psql.exe",
      "-Wait",
      "-NoNewWindow",
      "-ArgumentList",
        "`"-h ${server}`",",
        "`"-p ${port}`",",
        "`"-U ${admin}`",",
        "`"-d postgres`",",
        "`"-f ${roles}`""
  )
  ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
  ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
  ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
  ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\psql.exe"
  ${PROCESS_START_INFO}.Arguments = `
    "-h ${server}", `
    "-p ${port}", `
    "-U ${admin}", `
    "-d postgres", `
    "-f ${roles}"
  ${PROCESS_RUN} = New-Object System.Diagnostics.Process
  ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
  ${PROCESS_RUN}.Start() | Out-Null
  ${PROCESS_RUN}.WaitForExit()
  Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
    -replace -join ( `"(?ms)`",
      "ERROR:  role ``"(.*?)``" already exists"
    ), "Redundant query:  role ``"`$1``" already exists" `
    -replace -join ( `"(?ms)`",
      "NOTICE:  role ``"(.*?)``" is already a member of role ``"(.*?)``""
    ), "Redundant assignment:  role ``"`$1``" is already a member of role ``"`$2``""
  )"

  ForEach (${db} in ${databases}) {
    ${private:file} = "${destination_files}\${db}.dump"
    If ((Get-Content "${private:file}") -ne ${NULL}) {
      Write-Host "    Restoring `"${db}`" from `"${file}`""
      # Delete the DB if it exists, which it should not
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\dropdb.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"${db}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\dropdb.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}", `
        "-p ${port}", `
        "-U ${admin}", `
        "${db}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          "dropdb: database removal failed: ERROR:  database ``"${db}``" does not exist"
        ), "    dropdb: database removal redundant:  database ``"${db}``" does not exist"
      )"

      # Create the DB
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\createdb.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"-T template0`",",
            "`"${db}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\createdb.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}", `
        "-p ${port}", `
        "-U ${admin}", `
        "-T template0", `
        "${db}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$( ${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          "createdb: database creation failed: ERROR:  database ``"${db}``" already exists"
        ), "    createdb: database creation redundant:  database ``"${db}``" already exists"
      )"

      # Populate the database with the backed-up data
      Write-Host (
        "    ",
        "Start-Process",
          "-FilePath ${postgresql_path}\bin\pg_restore.exe",
          "-Wait",
          "-NoNewWindow",
          "-ArgumentList",
            "`"-h ${server}`",",
            "`"-p ${port}`",",
            "`"-U ${admin}`",",
            "`"-d ${db}`",",
            "`"-Fc ${file}`""
      )
      ${PROCESS_START_INFO} = New-Object System.Diagnostics.ProcessStartInfo
      ${PROCESS_START_INFO}.RedirectStandardError = ${TRUE}
      ${PROCESS_START_INFO}.UseShellExecute = ${FALSE}
      ${PROCESS_START_INFO}.FileName = "${postgresql_path}\bin\pg_restore.exe"
      ${PROCESS_START_INFO}.Arguments = `
        "-h ${server}",
        "-p ${port}",
        "-U ${admin}",
        "-d ${db}",
        "-Fc ${file}"
      ${PROCESS_RUN} = New-Object System.Diagnostics.Process
      ${PROCESS_RUN}.StartInfo = ${PROCESS_START_INFO}
      ${PROCESS_RUN}.Start() | Out-Null
      ${PROCESS_RUN}.WaitForExit()
      Write-Host "$(${PROCESS_RUN}.StandardError.ReadToEnd() `
        -replace -join ( `"(?ms)`",
          `"(pg_restore: \[archiver \(db\)\] )(Error)( while PROCESSING TOC:`r`n)`",
          `"(pg_restore: \[archiver \(db\)\] )(Error)( from TOC entry 3; 2615 2200 SCHEMA public postgres`r`n)`",
          `"(pg_restore: \[archiver \(db\)\] )(could not execute query: ERROR)(:  schema ``"public``" already exists)`"
        ), `"    `$1Redundancy`$3    `$4Redundancy`$6    `$7Skipping query execution`$9`" `
        -replace -join ( `"(?ms)`",
          "WARNING: errors ignored on restore: [0-9]+"
        ), `"`" `
        -replace -join ( `"(?ms)`",
          "(pg_restore: \[custom archiver\] )(could not read from)( input file: end of file)"
        ), `"    `$1ignoring empty input file ``"${db}.dump``"`"
      )"
    } Else {
      Write-Host "    Ignoring empty file `"${db}.dump`"`n"
    }
  }
}

Function Show-Differences(
    [string]${reference},
    [string]${difference}
) {
  ${do_please} = Read-Host "  Do you want to diff/compare `"${reference}`" with `"${difference}`"?`n    It can potentially take several minutes. [Y/n]"
  If ("${do_please}" -eq "" -Or "${do_please}" -eq "y" -Or "${do_please}" -eq "Y") {
    ${comparisons} = @{
      ReferenceObject = (Get-ChildItem -Recurse -Path ${reference})
      DifferenceObject = (Get-ChildItem -Recurse -Path ${difference})
    }

    Compare-Object @comparisons -PassThru
  }
}

Function Get-ResolvedPath(
    [string]${path}
) {
  ${item} = (Get-Item -Path ${path})
  If (${item}.Attributes -match "ReparsePoint") {
    ${local:path_target} = ${item}.Target
    If ([System.IO.Path]::IsPathRooted(${local:path_target})) {
      ${response} = "${local:path_target}"
    } Else {
      ${response} = "$(Split-Path -Path "$(${item}.FullName)" -Parent)\${local:path_target}"
    }
  } Else {
    ${response} = "$(${item}.FullName)"
  }
  return ${response}
}

Function Summary-Print() {
  Write-Host "`nSummary"
  ForEach (${local:register} in ${global:SUMMARY_REGISTRATION}) {
    If ( "${register}" -eq "activemq" ) { Summary-PrintOne "ActiveMQ" ${active_mq_summary_code} }
    If ( "${register}" -eq "java" ) { Summary-PrintOne "Java" ${java_summary_code} }
    If ( "${register}" -eq "postgresql" ) { Summary-PrintOne "PostgreSQL" ${postgresql_summary_code} }
    If ( "${register}" -eq "tomcat" ) { Summary-PrintOne "Tomcat" ${tomcat_summary_code} }
  }
}

Function Summary-PrintOne(
  [string]${name},
  [int]${code}
) {
  Switch(${code}) {
    ${SUMMARY_CODE_COMPLETED} { Write-Host "  ${name} updated successfully" }
    ${SUMMARY_CODE_SKIPPED} { Write-Host "  ${name} update was possible but skipped" }
    ${SUMMARY_CODE_MISSED} { Write-Host "  ${name} update unavailable due to missing installation file" }
    ${SUMMARY_CODE_FAILED} { Write-Host "  ${name} failed to update" }
  }
}

Function Summary-Register(
  [string[]]${components}
) {
  ForEach (${local:component} in ${components}) {
    ${private:can_add} = ${TRUE}
    ForEach (${local:register} in ${global:SUMMARY_REGISTRATION}) {
      If ("${register}" -eq "${component}") {
        ${private:can_add} = ${FALSE}
        continue
      }
    }
    If (${can_add}) {
      ${global:SUMMARY_REGISTRATION} = @(
        ${global:SUMMARY_REGISTRATION}
        ${component}
      )
    }
  }
}


# SIG # Begin signature block
# MIIsvAYJKoZIhvcNAQcCoIIsrTCCLKkCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDKTbE+qUD4+l5m
# Wmf+8AzXVd5SNapFJ8xvktR94iBP4qCCJj8wggWNMIIEdaADAgECAhAOmxiO+dAt
# 5+/bUOIIQBhaMA0GCSqGSIb3DQEBDAUAMGUxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xJDAiBgNV
# BAMTG0RpZ2lDZXJ0IEFzc3VyZWQgSUQgUm9vdCBDQTAeFw0yMjA4MDEwMDAwMDBa
# Fw0zMTExMDkyMzU5NTlaMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
# dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lD
# ZXJ0IFRydXN0ZWQgUm9vdCBHNDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoC
# ggIBAL/mkHNo3rvkXUo8MCIwaTPswqclLskhPfKK2FnC4SmnPVirdprNrnsbhA3E
# MB/zG6Q4FutWxpdtHauyefLKEdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iTcMKy
# unWZanMylNEQRBAu34LzB4TmdDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGbNOsF
# xl7sWxq868nPzaw0QF+xembud8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclPXuU1
# 5zHL2pNe3I6PgNq2kZhAkHnDeMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCrVYJB
# MtfbBHMqbpEBfCFM1LyuGwN1XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFPObUR
# WBf3JFxGj2T3wWmIdph2PVldQnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTvkpI6
# nj3cAORFJYm2mkQZK37AlLTSYW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWMcCxB
# YKqxYxhElRp2Yn72gLD76GSmM9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls5Q5S
# UUd0viastkF13nqsX40/ybzTQRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBRa2+x
# q4aLT8LWRV+dIPyhHsXAj6KxfgommfXkaS+YHS312amyHeUbAgMBAAGjggE6MIIB
# NjAPBgNVHRMBAf8EBTADAQH/MB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qYrhwP
# TzAfBgNVHSMEGDAWgBRF66Kv9JLLgjEtUYunpyGd823IDzAOBgNVHQ8BAf8EBAMC
# AYYweQYIKwYBBQUHAQEEbTBrMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdp
# Y2VydC5jb20wQwYIKwYBBQUHMAKGN2h0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNv
# bS9EaWdpQ2VydEFzc3VyZWRJRFJvb3RDQS5jcnQwRQYDVR0fBD4wPDA6oDigNoY0
# aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENB
# LmNybDARBgNVHSAECjAIMAYGBFUdIAAwDQYJKoZIhvcNAQEMBQADggEBAHCgv0Nc
# Vec4X6CjdBs9thbX979XB72arKGHLOyFXqkauyL4hxppVCLtpIh3bb0aFPQTSnov
# Lbc47/T/gLn4offyct4kvFIDyE7QKt76LVbP+fT3rDB6mouyXtTP0UNEm0Mh65Zy
# oUi0mcudT6cGAxN3J0TU53/oWajwvy8LpunyNDzs9wPHh6jSTEAZNUZqaVSwuKFW
# juyk1T3osdz9HNj0d1pcVIxv76FQPfx2CWiEn2/K2yCNNWAcAgPLILCsWKAOQGPF
# mCLBsln1VWvPJ6tsds5vIy30fnFqI2si/xK4VC0nftg62fC2h5b9W9FcrBjDTZ9z
# twGpn1eqXijiuZQwggWQMIIDeKADAgECAhAFmxtXno4hMuI5B72nd3VcMA0GCSqG
# SIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMx
# GTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRy
# dXN0ZWQgUm9vdCBHNDAeFw0xMzA4MDExMjAwMDBaFw0zODAxMTUxMjAwMDBaMGIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBH
# NDCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAL/mkHNo3rvkXUo8MCIw
# aTPswqclLskhPfKK2FnC4SmnPVirdprNrnsbhA3EMB/zG6Q4FutWxpdtHauyefLK
# EdLkX9YFPFIPUh/GnhWlfr6fqVcWWVVyr2iTcMKyunWZanMylNEQRBAu34LzB4Tm
# dDttceItDBvuINXJIB1jKS3O7F5OyJP4IWGbNOsFxl7sWxq868nPzaw0QF+xembu
# d8hIqGZXV59UWI4MK7dPpzDZVu7Ke13jrclPXuU15zHL2pNe3I6PgNq2kZhAkHnD
# eMe2scS1ahg4AxCN2NQ3pC4FfYj1gj4QkXCrVYJBMtfbBHMqbpEBfCFM1LyuGwN1
# XXhm2ToxRJozQL8I11pJpMLmqaBn3aQnvKFPObURWBf3JFxGj2T3wWmIdph2PVld
# QnaHiZdpekjw4KISG2aadMreSx7nDmOu5tTvkpI6nj3cAORFJYm2mkQZK37AlLTS
# YW3rM9nF30sEAMx9HJXDj/chsrIRt7t/8tWMcCxBYKqxYxhElRp2Yn72gLD76GSm
# M9GJB+G9t+ZDpBi4pncB4Q+UDCEdslQpJYls5Q5SUUd0viastkF13nqsX40/ybzT
# QRESW+UQUOsxxcpyFiIJ33xMdT9j7CFfxCBRa2+xq4aLT8LWRV+dIPyhHsXAj6Kx
# fgommfXkaS+YHS312amyHeUbAgMBAAGjQjBAMA8GA1UdEwEB/wQFMAMBAf8wDgYD
# VR0PAQH/BAQDAgGGMB0GA1UdDgQWBBTs1+OC0nFdZEzfLmc/57qYrhwPTzANBgkq
# hkiG9w0BAQwFAAOCAgEAu2HZfalsvhfEkRvDoaIAjeNkaA9Wz3eucPn9mkqZucl4
# XAwMX+TmFClWCzZJXURj4K2clhhmGyMNPXnpbWvWVPjSPMFDQK4dUPVS/JA7u5iZ
# aWvHwaeoaKQn3J35J64whbn2Z006Po9ZOSJTROvIXQPK7VB6fWIhCoDIc2bRoAVg
# X+iltKevqPdtNZx8WorWojiZ83iL9E3SIAveBO6Mm0eBcg3AFDLvMFkuruBx8lbk
# apdvklBtlo1oepqyNhR6BvIkuQkRUNcIsbiJeoQjYUIp5aPNoiBB19GcZNnqJqGL
# FNdMGbJQQXE9P01wI4YMStyB0swylIQNCAmXHE/A7msgdDDS4Dk0EIUhFQEI6FUy
# 3nFJ2SgXUE3mvk3RdazQyvtBuEOlqtPDBURPLDab4vriRbgjU2wGb2dVf0a1TD9u
# KFp5JtKkqGKX0h7i7UqLvBv9R0oN32dmfrJbQdA75PQ79ARj6e/CVABRoIoqyc54
# zNXqhwQYs86vSYiv85KZtrPmYQ/ShQDnUBrkG5WdGaG5nLGbsQAe79APT0JsyQq8
# 7kP6OnGlyE0mpTX9iV28hWIdMtKgK1TtmlfB2/oQzxm3i0objwG2J5VT6LaJbVu8
# aNQj6ItRolb58KaAoNYes7wPD1N1KarqE3fk3oyBIa0HEEcRrYc9B9F1vM/zZn4w
# ggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBH
# NDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVT
# MRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1
# c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbS
# g9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9
# /UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXn
# HwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0
# VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4f
# sbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40Nj
# gHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0
# QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvv
# mz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T
# /jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk
# 42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5r
# mQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4E
# FgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5n
# P+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcG
# CCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQu
# Y29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8v
# Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNV
# HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIB
# AH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxp
# wc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIl
# zpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQ
# cAp876i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfe
# Kuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+j
# Sbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJsh
# IUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6
# OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDw
# N7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR
# 81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2
# VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIGsDCCBJigAwIBAgIQ
# CK1AsmDSnEyfXs2pvZOu2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAw
# MDAwWhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBT
# aWduaW5nIFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkqhkiG9w0BAQEF
# AAOCAg8AMIICCgKCAgEA1bQvQtAorXi3XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+k
# jmjYXPXrNCQH4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAkZLON4gh9
# NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXzENOLsvsI8IrgnQnAZaf6mIBJNYc9
# URnokCF4RS6hnyzhGMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRXXegY
# E2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjqGzLmysL0p6MDDnSlrzm2q2AS
# 4+jWufcx4dyt5Big2MEjR0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
# wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tvk2E0XLyTRSiDNipmKF+w
# c86LJiUGsoPUXPYVGUztYuBeM/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eR
# Gv7bUKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2FKZbS110YU0/EpF2
# 3r9Yy3IQKUHw1cVtJnZoEUETWJrcJisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCK
# ZFTBzCEa6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27IwyCQLMbDwMVhEC
# AwEAAaOCAVkwggFVMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2
# O/hfEYb7/mF7CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiuHA9P
# MA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDAzB3BggrBgEFBQcB
# AQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
# BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1
# c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwHAYDVR0gBBUwEzAH
# BgVngQwBAzAIBgZngQwBBAEwDQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6
# mdNW4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6jfCbVN7w6XUhtldU/
# SfQnuxaBRVD9nL22heB2fjdxyyL3WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzY
# gBoRGRjNYZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGVWEjVGv7XJz/9
# kNF2ht0csGBc8w2o7uCJob054ThO2m67Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ
# 9d3OVG3ZXQIUH0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m6Ri+kAew
# Q3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL6TEa/y4ZXDlx4b6cpwoG1iZnt5Lm
# Tl/eeqxJzy6kdJKt2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFSi0HA
# SIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+uKIw8y4+ICw2/O/TOHnuO77Xr
# y7fwdxPm5yg/rBKupS8ibEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
# ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+FSCH5Vzu0nAPthkX0tGFu
# v2jiJmCG6sivqf6UHedjGzqGVnhOMIIGwjCCBKqgAwIBAgIQBUSv85SdCDmmv9s/
# X+VhFjANBgkqhkiG9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGln
# aUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5
# NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4XDTIzMDcxNDAwMDAwMFoXDTM0MTAx
# MzIzNTk1OVowSDELMAkGA1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
# MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3RhbXAgMjAyMzCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAKNTRYcdg45brD5UsyPgz5/X5dLnXaEOCdwvSKOX
# ejsqnGfcYhVYwamTEafNqrJq3RApih5iY2nTWJw1cb86l+uUUI8cIOrHmjsvlmbj
# aedp/lvD1isgHMGXlLSlUIHyz8sHpjBoyoNC2vx/CSSUpIIa2mq62DvKXd4ZGIX7
# ReoNYWyd/nFexAaaPPDFLnkPG2ZS48jWPl/aQ9OE9dDH9kgtXkV1lnX+3RChG4PB
# uOZSlbVH13gpOWvgeFmX40QrStWVzu8IF+qCZE3/I+PKhu60pCFkcOvV5aDaY7Mu
# 6QXuqvYk9R28mxyyt1/f8O52fTGZZUdVnUokL6wrl76f5P17cz4y7lI0+9S769Sg
# LDSb495uZBkHNwGRDxy1Uc2qTGaDiGhiu7xBG3gZbeTZD+BYQfvYsSzhUa+0rRUG
# FOpiCBPTaR58ZE2dD9/O0V6MqqtQFcmzyrzXxDtoRKOlO0L9c33u3Qr/eTQQfqZc
# ClhMAD6FaXXHg2TWdc2PEnZWpST618RrIbroHzSYLzrqawGw9/sqhux7UjipmAmh
# cbJsca8+uG+W1eEQE/5hRwqM/vC2x9XH3mwk8L9CgsqgcT2ckpMEtGlwJw1Pt7U2
# 0clfCKRwo+wK8REuZODLIivK8SgTIUlRfgZm0zu++uuRONhRB8qUt+JQofM604qD
# y0B7AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/BAIwADAW
# BgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglg
# hkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYDVR0O
# BBYEFKW27xPn783QZKHVVqllMaPe1eNJMFoGA1UdHwRTMFEwT6BNoEuGSWh0dHA6
# Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
# NTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEBBIGDMIGAMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZT
# SEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJKoZIhvcNAQELBQADggIBAIEa1t6g
# qbWYF7xwjU+KPGic2CX/yyzkzepdIpLsjCICqbjPgKjZ5+PF7SaCinEvGN1Ott5s
# 1+FgnCvt7T1IjrhrunxdvcJhN2hJd6PrkKoS1yeF844ektrCQDifXcigLiV4JZ0q
# BXqEKZi2V3mP2yZWK7Dzp703DNiYdk9WuVLCtp04qYHnbUFcjGnRuSvExnvPnPp4
# 4pMadqJpddNQ5EQSviANnqlE0PjlSXcIWiHFtM+YlRpUurm8wWkZus8W8oM3NG6w
# QSbd3lqXTzON1I13fXVFoaVYJmoDRd7ZULVQjK9WvUzF4UbFKNOt50MAcN7MmJ4Z
# iQPq1JE3701S88lgIcRWR+3aEUuMMsOI5ljitts++V+wQtaP4xeR0arAVeOGv6wn
# LEHQmjNKqDbUuXKWfpd5OEhfysLcPTLfddY2Z1qJ+Panx+VPNTwAvb6cKmx5Adza
# ROY63jg7B145WPR8czFVoIARyxQMfq68/qTreWWqaNYiyjvrmoI1VygWy2nyMpqy
# 0tg6uLFGhmu6F/3Ed2wVbK6rr3M66ElGt9V/zLY4wNjsHPW2obhDLN9OTH0eaHDA
# dwrUAuBcYLso/zjlUlrWrBciI0707NMX+1Br/wd3H3GXREHJuEbTbDJ8WC9nR2Xl
# G3O2mflrLAZG70Ee8PBf4NvZrZCARK+AEEGKMIIG6jCCBNKgAwIBAgIQBKbQDKbn
# vxPWBgHfq1yGCTANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEXMBUGA1UE
# ChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQg
# Q29kZSBTaWduaW5nIFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIxMDYwODAw
# MDAwMFoXDTI0MDgzMDIzNTk1OVowbzELMAkGA1UEBhMCR0IxEDAOBgNVBAcTB05F
# V0JVUlkxJjAkBgNVBAoTHU1pY3JvIEZvY3VzIEludGVybmF0aW9uYWwgcGxjMSYw
# JAYDVQQDEx1NaWNybyBGb2N1cyBJbnRlcm5hdGlvbmFsIHBsYzCCAaIwDQYJKoZI
# hvcNAQEBBQADggGPADCCAYoCggGBAOoiQYhGCwyc+fGmIadVulPc2GZ9lCYQxC9K
# 1vSQ3pqs+3huyMrjKp2tXRU4HmwpY2w3tdvYr7Zpbj3wbKPR67IBp0IFvz0KNAYt
# 7AebJHbdPOqZvS01XV5ofNq59ABKvQOAcuAdmUadYLdVRYcfdMDX4D2i0WSuheIS
# NkZW3WrpBDYeRryJ2HwhoSQXbhHGtfihGEQHtz+KLpjSxJ/ntj60epd1i6a4Q2Ro
# Djgtrv0Ee4xq/ATZ9a6ukTuWB+V/WedvInFpD5etSBKqLhC93pnCPoO+jmlQkC6q
# qHApjAfSfsECnEUqVmr/DpR1LU1MyiGbHKog3LED7h0pCKUpQdeJc/R34ZRKtEW1
# e01TdMqCk7YsUZlKWz+Xbl6ZzYidPaYdPtKQaSXAPzuIzbys9LoKd5U2VBxCylLL
# vZgJEQlYhXUlu/vEBek6XzBuBUH4DZObmVIbkVS7v2sT6fXik6ibhU4PPEDdq0yG
# bjSHfze1LpXc8fIriotRukuSRIMv8QIDAQABo4ICBjCCAgIwHwYDVR0jBBgwFoAU
# aDfg67Y7+F8Rhvv+YXsIiGX0TkIwHQYDVR0OBBYEFIboW+tVaDsOcwhEgehx7b3u
# JpoxMA4GA1UdDwEB/wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0f
# BIGtMIGqMFOgUaBPhk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRU
# cnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGg
# T4ZNaHR0cDovL2NybDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29k
# ZVNpZ25pbmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcmwwPgYDVR0gBDcwNTAzBgZn
# gQwBBAEwKTAnBggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BT
# MIGUBggrBgEFBQcBAQSBhzCBhDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
# aWNlcnQuY29tMFwGCCsGAQUFBzAChlBodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
# Q0ExLmNydDAMBgNVHRMBAf8EAjAAMA0GCSqGSIb3DQEBCwUAA4ICAQBdyv7VGFvF
# 12qOAQ3bV8ae6galTCD2mxwlzEfc+6cmh/5h/h9N8f6THKy97LsrDN+wWA4AaVfL
# 4o0Pk/lZI4FXYFC5ljqwoFIA2CqcqVi2fpnWSlDcNSu+WLIrcD9StY75xkEPvgnV
# sMbLyIhQvlMwMpQh/MnMREQ40I/3nXZDVksjoRa3frEQYQLr8wEmHoS86l/tJ4R0
# P/cmK63nGMb8uc1LgREopxlCjf/NaNK8z2BjD8iJ7E0B1pGyr5viRM0/+I+OjAru
# bsmWUcYPkibX7pQ1c1shDfpu4fo3eqSz2nIMPEnKOeJrxSvi5tXfY8mIfO4uo43K
# Dg+9SZgWTLe4T2P1/jERijc3SpiF9QOYpDjktQPBla4CcF6b5HAi/ffs35T7wCNq
# mlA9hDrb+6djoCHkKaCQbbWklPujN/SxR2jpfkrEX0QuwZ76dZ9RYh9aNFeSLw4r
# qH6Lf2Bd2JjYTpw1zKN+FB6x44dZqWmVzFSrxxnzxxS1ptuwwyzL6dRpDO003y0O
# bQV2sCoUb3jeE37ifqOOfqeSrsIqueFCwlHtCnLjKM6KTR8iykYvoXop8UeADEJY
# Yyi8tqt3SbCKBF4fTEjLq82CUbpjjlEqVoN8pXyoFZyt9QoXE5bsygERRyPn07Mg
# fMhL3Zj6/nYG8FzyJe9YyjHD7smoGhwl9DGCBdMwggXPAgEBMH0waTELMAkGA1UE
# BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2Vy
# dCBUcnVzdGVkIEc0IENvZGUgU2lnbmluZyBSU0E0MDk2IFNIQTM4NCAyMDIxIENB
# MQIQBKbQDKbnvxPWBgHfq1yGCTANBglghkgBZQMEAgEFAKCBhDAYBgorBgEEAYI3
# AgEMMQowCKACgAChAoAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCAZRkEEx5s
# C9BQ08PJ59n/gLz8SdzXCHJDxukHAhR3njANBgkqhkiG9w0BAQEFAASCAYAVey48
# P+BNoc+7JSPDKp25KAh88VHDj7zTLnNJEWosB0cSFneJpayDSN/XXGkY9f0ic1dW
# xtEY6m1r+/DXmeimziz5bp5N7zJnHENNLcY+EnC+SXejtpoJzi6+/rnJdgs/+mSe
# NZCUVAdG8lgj81noPsfaXzPlonVAu2KMxCPYbX8aelERWWBzS0sgCmIpmIfXzu5/
# XOFbfkWvPSO98LY+m7Kp0tYRpCdDJq52XVNyaVnkEc1iSpOJ9QimIsY8lkSkVIle
# Qd6iNRns3Tlvhq4Qzo8lIdm3/Xax6bw8GPXAUMpb64RCtwz6laRTyxj2NOEIB+l9
# f1EAQDlA0Ng00woq7754BMaVvikbzAgvAx+1ATuHVmoNI4dp+XG2f7/r4Wt/Jfb6
# LOBAh49ctH2/st8K9gxNlqxvZ5bSYiS1N4tx6h0D5ge24f4qshST5JoLgLL2okxs
# gw1otWlEf+yDPOOh5Vjruicf/N+5MpBnkXTf6/V3RyfwTHdY8Yru1ZAh/zOhggMg
# MIIDHAYJKoZIhvcNAQkGMYIDDTCCAwkCAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUG
# A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0ZWQg
# RzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhAFRK/zlJ0IOaa/2z9f
# 5WEWMA0GCWCGSAFlAwQCAQUAoGkwGAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAc
# BgkqhkiG9w0BCQUxDxcNMjQwMTA1MTM0OTQ1WjAvBgkqhkiG9w0BCQQxIgQgVNl/
# H2XvV2i/Ydwl20N6b2mMwWlaKamZb2mE4mUTkkkwDQYJKoZIhvcNAQEBBQAEggIA
# G+zz+CEJVjMv0URVYWVU4IdDFQi2yqa8NaYqwzJ3Kiay5YK2Az1sQ1n4TENV96Wi
# K7wgLg/VimP1mQFqH7DRJxHJl7sQdT5jfThPIjD1DLBPPL9/4hwAVEuEDYqk06/N
# PT+y9zjtjYapXp2qvaWs6sz0NQxZZCvoieJIsKwXg6eAXj9zyrhzJ2b5qY2+AosR
# 3ZBoTq4Ut/25f6zQmp3pCQHpBIM4Pw11lq3RmldzzqzR3e8M30QkCGqS6/Dnk0Nl
# JswDWQo9yhGcyMlBmfN163/SGC8oGq4fVCjKFuHIaqkhEgz3KNND8qBga+i3P/aL
# FGVbvP3Mp5Is23wOhRkXLn/5ZlnD5GvBExmVQh6G5Z2nxp2tD5XePIAKVy5kq62r
# lLaVb1qssKJLUlAQPtFJ3p8P11gZvBD28Uz/wKFk4ry6/W1Xl56VDpP4rFh/KdKL
# MXQytLln0p0Qz8FAKRemtcilHCDD2tmZHuRNVD94qEWEqNwlwHJsIYpKY9Tjv4wT
# Zt9NE5kFx79z/Tm7RYJLJ/eTHkVzyZWKet1HFyt1Id07MC2ol05ZrEE168V+slrY
# Rouy8npLYHcvJY1krJ0tBvya0iqNTrwiNaNcZ3qqKUs1QolsKREfM+VOT0geY26l
# uArXxOREUEIJKFTYTXuBKMX8l79vSCgkq7Gk3seZm+U=
# SIG # End signature block
