#/bin/bash

source ${0%/*}/def_copies.sh

# Define variables
ACTIVE_MQ_DEFAULT="/opt/netiq/idm/apps/activemq"
JAVA_DEFAULT="/opt/netiq/idm/apps/jre"
POSTGRESQL_ADMIN_DEFAULT="postgres"
POSTGRESQL_PATH_DEFAULT_BIN="/opt/netiq/idm/apps/postgres/bin"
POSTGRESQL_PATH_DEFAULT_DATA="/opt/netiq/idm/apps/postgres/data"
POSTGRESQL_PORT_DEFAULT="5432"
POSTGRESQL_SERVER_DEFAULT="localhost"
TOMCAT_DEFAULT="/opt/netiq/idm/apps/tomcat"
ACTIVEMQ_REQUIRED_1=activemq-all-5.17*.jar
ACTIVEMQ_REQUIRED_2=log4j-api*.jar

VALIDITY_CODE_GOOD="0"
VALIDITY_CODE_HALT="1"

ERROR_ARG_LIST_WRONG_LENGTH="7"

COMPARE_CODE_EQ="0"
COMPARE_CODE_LT="1"
COMPARE_CODE_GT="2"

SUMMARY_REGISTRATION=()
SUMMARY_CODE_COMPLETED="0"
SUMMARY_CODE_SKIPPED="1"
SUMMARY_CODE_MISSED="2"
SUMMARY_CODE_FAILED="3"

DIR_COMPRESSED="./compressed"

FILE_ACTIVE_MQ="${DIR_COMPRESSED}/apache-activemq-${ACTIVE_MQ_VERSION_TARGET}-bin.tar.gz"
FILE_JAVA="${DIR_COMPRESSED}/zulu${JAVA_DOWNLOAD_CODE}-linux_x64.tar.gz"
FILE_TOMCAT="${DIR_COMPRESSED}/apache-tomcat-${TOMCAT_VERSION_TARGET}.tar.gz"

TIMESTAMP="$( date "+%Y%m%d-%H%M%S" )"

ALREADY_OBTAINED_JAVA_LOC="false"

get_valid_java_path() {
  if [ "${ALREADY_OBTAINED_JAVA_LOC}" == "false" ]; then
    # Assume we will obtain a valid path, since that is true unless the script is terminated
    ALREADY_OBTAINED_JAVA_LOC="true"
    # We need a valid location to proceed
    local input_invalid="true"
    while [ "${input_invalid}" == "true" ]; do
      # Ask user for the location, using default if nothing entered
      echo -n "  Where is your Java (JRE) located? [${java_loc_final}]: "
      read java_loc
      if [ -n "${java_loc}" ]; then java_loc_final="${java_loc}"; fi
      getPathValidity "${java_loc_final}" "${JAVA_VALIDITY[@]}"
      if [ "$?" == "${VALIDITY_CODE_GOOD}" ]; then
        # Path is valid; we can proceed
        input_invalid="false"
      fi
    done
  fi
}

# Define functions/subroutines
# Input ($1) or read from prompt
trimInput() {
  local input
  if [ $# -eq 0 ]; then read input; else input="${1}"; fi
  echo "$( echo "${input}" | sed "s|^\s*||g" | sed "s|\s*$||g" )"
}

# Version number($1)i; 5.15.9 or 1.8.0_122
parseVersion() {
  if [ $# -lt 1 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi
  echo "$( echo "${1}" | sed "s|\.| |g" | sed "s|_| |g" )"
}

# Actual ($1), Expected ($2)
compareTwoVersions() {
  if [ $# -lt 2 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  version_array_one=( $( parseVersion ${1} ) )
  version_array_two=( $( parseVersion ${2} ) )

  local counter=0
  local max_count="${#version_array_two[*]}"
  while (( counter < max_count )); do
    if [ "${version_array_one[${counter}]}" == "" ]; then
      return ${COMPARE_CODE_EQ}
    elif (( ${version_array_one[${counter}]} < ${version_array_two[${counter}]} )); then
      return ${COMPARE_CODE_LT}
    elif (( ${version_array_one[${counter}]} > ${version_array_two[${counter}]} )); then
      return ${COMPARE_CODE_GT}
    fi
    (( counter=counter+1 ))
  done
  return ${COMPARE_CODE_EQ}
}

# Component ($1), Actual version ($2), Expected version ($3)
compareVersionsAndPrompt() {
  if [ $# -lt 3 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  compareTwoVersions ${2} ${3}
  local comparison_result="$?"
  if [ "${comparison_result}" == "${COMPARE_CODE_LT}" ]; then
    echo "  Expected ${1} version ${3}, but instead found the older ${2}"
  elif [ "${comparison_result}" == "${COMPARE_CODE_GT}" ]; then
    echo -e "  Your ${1} version (${2}) surpasses the target (${3})\n  We cannot guarantee your version will work, but it might."
  else #if [ "${comparison_result}" == "${COMPARE_CODE_EQ}" ]; then
    echo "  Your ${1} version (${2}) appears to be valid."
  fi
}

# Base path ($1), Relative path array ($2)
getPathValidity() {
  if [ $# -lt 2 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  local valid="true"
  local array=( ${@:2} )
  for path in ${array[@]}; do
    if [ ! -r "${1}/${path}" ] &&
       [ ! -d "${1}/${path}" ]; then
      echo "    Expected to find \"${1}/${path}\""
      valid="false"
    fi
  done

  if [ "${valid}" == "true" ]; then
    return ${VALIDITY_CODE_GOOD}
  else
    return ${VALIDITY_CODE_HALT}
  fi
}

# Component ($1), Source path ($2), Destination path ($3), Relative path array ($4)
copyFilesOver() {
  if [ $# -lt 4 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  local array=( ${@:4} )
  for file in ${array[@]}; do
    if [ -r "${2}/${file}" ]; then
      if [ -f ${3}/${file} ]; then
        local backup="-backup"
        if [ "${file}" != "${file#*.}" ]; then
          backup="${backup}.${file#*.}"
        fi
        echo "    Backing up existing ${1} \"${file}\" to \"${file%%.*}${backup}\""
        mv -f ${3}/${file} ${3}/${file%%.*}${backup}
      fi
      echo "    Copying ${1} \"${file}\""
      cp ${2}/${file} ${3}/${file}
    elif [ -r ${2}/${file} ]; then
      echo "    Copying ${1} \"${file}\""
      cp ${2}/${file} ${3}/${file%/*}
    fi
  done
}

# Tomcat path ($1)
disable_tomcat_ajp_connector() {
  if [ $# -lt 1 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  echo "    Disabling Tomcat AJP connector within \"${1}/conf/server.xml\""

  rex_line_01="\(\\s\+<\!-- Define an AJP 1\\.3 Connector on port [0-9]\+ -->\)\(\\s*\)"
  rex_line_02="\(\\s\+\)\(<Connector port=\"[0-9]\+\" protocol=\"AJP/1\\.3\" redirectPort=\"[0-9]\+\"\\s*/>\)\(\\s*\)"
  regex_replace="\\1\\2\\n\\3<!--\\n\\3\\4\\5\\n\\3-->"
  sed -i "\|^${rex_line_01}\$|{
    $!{ N # Append the next line if not the last
      \|^${rex_line_01}\n${rex_line_02}\$|{
        s|^${rex_line_01}\n${rex_line_02}\$|${regex_replace}|g
      }
    }
  }" "${1}/conf/server.xml"
}

# Component ($1), Destination path ($2), Relative path array ($3)
discardFiles() {
  if [ $# -lt 3 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  local array=( ${@:3} )
  for file in ${array[@]}; do
    if [ -r "${2}/${file}" ] || [ -d "${2}/${file}" ]; then
      echo "    Removing existing ${1} \"${file}\""
      rm -rf ${2}/${file}
    fi
  done
}

# Add required jars for ActiveMQ 5.17 if not already there
addRequiredMQJar() {
  exists() {
    [ -e "$1" ]
   }
  
  if (exists $ACTIVE_MQ_DEFAULT/$ACTIVEMQ_REQUIRED_1)
   then
   if !(exists $TOMCAT_DEFAULT/lib/$ACTIVEMQ_REQUIRED_1)
        then
        echo "    Adding required jar #1 for ActiveMQ 5.17"
        cp $ACTIVE_MQ_DEFAULT/$ACTIVEMQ_REQUIRED_1 $TOMCAT_DEFAULT/lib
        chown novlua:novlua $TOMCAT_DEFAULT/lib/$ACTIVEMQ_REQUIRED_1
   fi
  fi

  if (exists $ACTIVE_MQ_DEFAULT/lib/optional/$ACTIVEMQ_REQUIRED_2)
   then
   if !(exists $TOMCAT_DEFAULT/lib/$ACTIVEMQ_REQUIRED_2)
        then
        echo "    Adding required jar #2 for ActiveMQ 5.17"
        cp $ACTIVE_MQ_DEFAULT/lib/optional/$ACTIVEMQ_REQUIRED_2 $TOMCAT_DEFAULT/lib
        chown novlua:novlua $TOMCAT_DEFAULT/lib/$ACTIVEMQ_REQUIRED_2
   fi
  fi

  if (exists $TOMCAT_DEFAULT/lib/activemq-all-5.16*.jar)
        then
        echo "    Deleting old activemq 5.16 jar"
        rm $TOMCAT_DEFAULT/lib/activemq-all-5.16*.jar
  fi

}

showDifferences() {
  if [ $# -lt 2 ]; then return ${ERROR_ARG_LIST_WRONG_LENGTH}; fi

  echo -n "  Do you want to diff/compare \"${1}\" with \"${2}\"? [Y/n]: "
  read do_please
  if [ -z "${do_please}" ] || [ "${do_please}" == "y" ] || [ "${do_please}" == "Y" ]; then
    diff --brief --recursive --side-by-side ${1} ${2}
  fi
}

registerForSummary() {
  local array=( ${@:1} )
  for name in ${array[@]}; do
    local found="false"
    for register in ${SUMMARY_REGISTRATION[@]}; do
      if [ "${register}" == "${name}" ]; then
        local found="true"
        continue;
      fi
    done
    if [ "${found}" == "false" ]; then SUMMARY_REGISTRATION=( ${SUMMARY_REGISTRATION[@]} ${name} ); fi
  done
}

printSummary() {
  echo -e "\nSummary"
  for register in ${SUMMARY_REGISTRATION[@]}; do
    if [ "${register}" == "activemq" ]; then printOneSummary "ActiveMQ" "${active_mq_summary_code}"; fi
    if [ "${register}" == "java" ]; then printOneSummary "Java" "${java_summary_code}"; fi
    if [ "${register}" == "postgresql" ]; then printOneSummary "PostgreSQL" "${postgresql_summary_code}"; fi
    if [ "${register}" == "tomcat" ]; then printOneSummary "Tomcat" "${tomcat_summary_code}"; fi
  done
}

# Component name (${1}); summary code (${2})
printOneSummary() {
  case ${2} in
    ${SUMMARY_CODE_COMPLETED})
      echo "  ${1} updated successfully"
    ;;
    ${SUMMARY_CODE_SKIPPED})
      echo "  ${1} update was possible but skipped"
    ;;
    ${SUMMARY_CODE_MISSED})
      echo "  ${1} unavailable due to missing installation file"
    ;;
    ${SUMMARY_CODE_FAILED})
      echo "  ${1} failed to update"
    ;;
    *)
      echo "Unknown code received: \"${2}\""
  esac
}
