#!/bin/bash

source ${0%/*}/def_upgrade.sh

# Assume default value first; modify as user provides input
active_mq_loc_final="${ACTIVE_MQ_DEFAULT}"
java_loc_final="${JAVA_DEFAULT}"
tomcat_loc_final="${TOMCAT_DEFAULT}"

active_mq_summary_code="${SUMMARY_CODE_FAILED}"
java_summary_code="${SUMMARY_CODE_FAILED}"
tomcat_summary_code="${SUMMARY_CODE_FAILED}"
if [ -z "${postgresql_summary_code}" ]; then ALLOW_SUMMARY_NOW="true"; else ALLOW_SUMMARY_NOW="false"; fi

## Verify that the compressed Tomcat tar.gz exists; otherwise, there is no point doing more
if [ -r ${FILE_TOMCAT} ]; then
  tomcat_summary_code="${SUMMARY_CODE_SKIPPED}"
  echo -n "It appears you are ready to upgrade Tomcat. Shall we proceed? [y/N]: "
  read do_please
  if [ "${do_please}" == "y" ] || [ "${do_please}" == "Y" ]; then
    tomcat_summary_code="${SUMMARY_CODE_FAILED}"
    # We need a valid location to proceed
    input_invalid="true"
    while [ "${input_invalid}" == "true" ]; do
      # Ask user for the location, using default if nothing entered
      echo -n "  Where is your Tomcat located? [${tomcat_loc_final}]: "
      read tomcat_loc
      if [ -n "${tomcat_loc}" ]; then tomcat_loc_final="${tomcat_loc}"; fi

      getPathValidity "${tomcat_loc_final}" "${TOMCAT_VALIDITY[@]}"
      if [ "$?" == "${VALIDITY_CODE_GOOD}" ]; then
        # Path is valid; we can proceed
        input_invalid="false"

        # Stop the Tomcat service
        echo -en "    Go ahead and stop your Tomcat service.\n      When the Tomcat service has completely stopped, press ENTER."
        read ignored

        # Get the owner and group of the existing folder
        tomcat_rename_owner="$( stat -c %U ${tomcat_loc_final} )"
        tomcat_rename_group="$( stat -c %G ${tomcat_loc_final} )"

        # Rename older tomcat parent folder
        tomcat_loc_rename="${tomcat_loc_final}-${TIMESTAMP}"
        echo -e "    Renaming existing Tomcat location as \"${tomcat_loc_rename}\""
        mv ${tomcat_loc_final} ${tomcat_loc_rename}

        # Unpack the new Tomcat tar where the old Tomcat was
        mkdir -p "${tomcat_loc_final}"
        echo -e "    Unpacking \"${FILE_TOMCAT}\" into \"${tomcat_loc_final}\""
        tar -xf "${FILE_TOMCAT}" -C "${tomcat_loc_final}" --strip-components=1

        discardFiles "Tomcat" "${tomcat_loc_final}" "${TOMCAT_DISCARD[@]}"
        copyFilesOver "Tomcat" "${tomcat_loc_rename}" "${tomcat_loc_final}" "${TOMCAT_RETAIN[@]}"
        disable_tomcat_ajp_connector "${tomcat_loc_final}"

        echo "    Recursively defining \"${tomcat_loc_final}\" owner as \"${tomcat_rename_owner}\" and group as \"${tomcat_rename_group}\""
        chown -R ${tomcat_rename_owner}.${tomcat_rename_group} ${tomcat_loc_final}

        tomcat_summary_code="${SUMMARY_CODE_COMPLETED}"
        showDifferences ${tomcat_loc_rename} ${tomcat_loc_final}
      fi
    done
  fi
else
  echo "You need to supply \"${FILE_TOMCAT}\" before Tomcat can be upgraded"
  tomcat_summary_code="${SUMMARY_CODE_MISSED}"
fi


## Verify that the compressed ActiveMQ tar.gz exists; otherwise, there is no point doing more
if [ -r ${FILE_ACTIVE_MQ} ]; then
  active_mq_summary_code="${SUMMARY_CODE_SKIPPED}"
  echo -n "It appears you are ready to upgrade ActiveMQ. Shall we proceed? [y/N]: "
  read do_please
  if [ "${do_please}" == "y" ] || [ "${do_please}" == "Y" ]; then
    active_mq_summary_code="${SUMMARY_CODE_FAILED}"
    # We need a valid location to proceed
    input_invalid="true"
    while [ "${input_invalid}" == "true" ]; do
      # Ask user for the location, using default if nothing entered
      echo -n "  Where is your ActiveMQ located? [${active_mq_loc_final}]: "
      read active_mq_loc
      if [ -n "${active_mq_loc}" ]; then active_mq_loc_final="${active_mq_loc}"; fi

      getPathValidity "${active_mq_loc_final}" "${ACTIVE_MQ_VALIDITY[@]}"
      if [ "$?" == "${VALIDITY_CODE_GOOD}" ]; then
        # Path is valid; we can proceed
        input_invalid="false"

        get_valid_java_path

        # Stop the ActiveMQ service
        echo -en "    Go ahead and stop your ActiveMQ service.\n      When the ActiveMQ service has completely stopped, press ENTER."
        read ignored

        # Get the owner and group of the existing folder or symbolic link
        active_mq_rename_owner="$( stat -c %U ${active_mq_loc_final} )"
        active_mq_rename_group="$( stat -c %G ${active_mq_loc_final} )"

        # Rename older ActiveMQ parent folder
        active_mq_loc_rename="${active_mq_loc_final}-${TIMESTAMP}"
        echo -e "    Renaming existing ActiveMQ location as \"${active_mq_loc_rename}\""
        mv ${active_mq_loc_final} ${active_mq_loc_rename}

        # Unpack the new ActiveMQ tar where the old ActiveMQ was
        mkdir -p "${active_mq_loc_final}"
        echo -e "    Unpacking \"${FILE_ACTIVE_MQ}\" into \"${active_mq_loc_final}\""
        tar -xf "${FILE_ACTIVE_MQ}" -C "${active_mq_loc_final}" --strip-components=1

        sed -i "s|#JAVA_HOME=\"\"|JAVA_HOME=\"${java_loc_final}\"|g" "${active_mq_loc_final}/bin/env" 2>&1 | ${TEE} ${FILE_LOG}

        copyFilesOver "ActiveMQ" "${active_mq_loc_rename}" "${active_mq_loc_final}" "${ACTIVE_MQ_RETAIN[@]}"

        echo "    Recursively defining \"${active_mq_loc_final}\" owner as \"${active_mq_rename_owner}\" and group as \"${active_mq_rename_group}\""
        chown -R ${active_mq_rename_owner}.${active_mq_rename_group} ${active_mq_loc_final}

        addRequiredMQJar

        active_mq_summary_code="${SUMMARY_CODE_COMPLETED}"
        showDifferences ${active_mq_loc_rename} ${active_mq_loc_final}
      fi
    done
  fi
else
  echo "You need to supply \"${FILE_ACTIVE_MQ}\" before ActiveMQ can be upgraded"
  active_mq_summary_code="${SUMMARY_CODE_MISSED}"
fi


## Verify that the compressed Java tar.gz exists; otherwise, there is no point doing more
if [ -r ${FILE_JAVA} ]; then
  java_summary_code="${SUMMARY_CODE_SKIPPED}"
  echo -n "It appears you are ready to upgrade Java. Shall we proceed? [y/N]: "
  read do_please
  if [ "${do_please}" == "y" ] || [ "${do_please}" == "Y" ]; then
    java_summary_code="${SUMMARY_CODE_FAILED}"
    get_valid_java_path

    # Get the owner and group of the existing folder or symbolic link
    java_rename_owner="$( stat -c %U ${java_loc_final} )"
    java_rename_group="$( stat -c %G ${java_loc_final} )"

    copy_link="${java_loc_final%/*}/copier-${TIMESTAMP}"

    if [ -L "${java_loc_final}" ]; then
      # Leave actual folder alone
      # Move symbolic link to copy-symbolic-link so it points to same destination as old link
      ln -s $( readlink ${java_loc_final} ) ${copy_link}
      rm ${java_loc_final}
    else
      java_loc_rename="${java_loc_final}-${TIMESTAMP}"
      # Rename the old folder
      mv ${java_loc_final} ${java_loc_rename}
      # Create a copy-symbolic-link that points to old files
      ln -s ${java_loc_rename##*/} ${copy_link}
    fi

    # Unpack Java into unique destination name (same path; jdk-timestamp)
    # Note: this is the reverse of other moves because Java is likely in use
    java_loc_jdk="${java_loc_final%/*}/jdk"
    if [ -d "${java_loc_jdk}" ] || [ -f "${java_loc_jdk}" ]; then
      java_loc_jdk="${java_loc_jdk}-${TIMESTAMP}"
    fi

    mkdir -p "${java_loc_jdk}"
    echo -e "  Unpacking \"${FILE_JAVA}\" into \"${java_loc_jdk}\""
    tar -xf "${FILE_JAVA}" -C "${java_loc_jdk}" --strip-components=1

    # Create a symbolic link with the old name but new destination
    #java1.8 ln -s "${java_loc_jdk##*/}/jre" ${java_loc_final}
    ln -s "${java_loc_jdk##*/}" ${java_loc_final}

    copyFilesOver "Java" "${copy_link}" "${java_loc_final}" "${JAVA_RETAIN[@]}"

    echo "  Recursively defining \"${java_loc_jdk}\" owner as \"${java_rename_owner}\" and group as \"${java_rename_group}\""
    chown -R ${java_rename_owner}.${java_rename_group} ${java_loc_jdk}
    chown -R ${java_rename_owner}.${java_rename_group} ${java_loc_final}

    java_summary_code="${SUMMARY_CODE_COMPLETED}"
    showDifferences $( readlink -f ${copy_link} ) ${java_loc_final}

    # Remove copy-link
    rm ${copy_link}
  fi
else
  echo "You need to supply \"${FILE_JAVA}\" before Java can be upgraded"
  java_summary_code="${SUMMARY_CODE_MISSED}"
fi

registerForSummary "activemq" "java" "tomcat"
if [ "${ALLOW_SUMMARY_NOW}" == "true" ]; then printSummary; fi
