/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES Auditing Sample Client (vdump).
 |
 | File: vdump_filter.c
 |
 | Desc: Filter file generator for vigil.ko filter API.
 +-------------------------------------------------------------------------*/

/*******************************************************************************
** Compiler setup.
*/

	/*---------------------------------------------------------------------------
	** Auto Tools
	*/
#ifdef HAVE_CONFIG_H
 #ifndef __config_h__
	#include <config.h>  /* PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT */
	#define __config_h__
 #endif 
#endif

	/*---------------------------------------------------------------------------
	** ANSI/POSIX
	*/
	#include <stdlib.h>	/* exit() */
	#include <stdio.h>	/* stderr, fprintf() */
	#include <errno.h>
	#include <time.h>
	#include <string.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <dirent.h>

	#include <sys/stat.h>
	#include <fcntl.h>

	#include <signal.h>
	#include <objalloc.h>

	/*---------------------------------------------------------------------------
	** Project
	*/
	#include <vigil.h>
	#include <libvigil.h>

/*******************************************************************************
** Get specified filter file header.
*/
int VDUMP_FILTER_FilterFileHeaderGet(
		/* I- filePath      */ const char               *filePath,
		/* -O fileHdr_Alloc */ VIGIL_FILTER_FILE_HDR_T **fileHdr_Alloc
		)
	{
	int      rCode=VIGIL_SUCCESS;
	FILE    *fp=NULL;
	size_t   bytes;
	VIGIL_FILTER_FILE_HDR_T *fileHdr=NULL;

	/*---------------------------------------------------------------------------
	** Validate caller args.
	*/
	if((NULL==filePath) || (NULL==fileHdr_Alloc))
		{
		rCode=EINVAL;
		fprintf(stderr, "%s error[VDUMP_FILTER_FilterFileHeaderGet]: Bad arg[%s%s]\n",
			PROGRAM_NAME,
			filePath ? "" : " NULL==filePath",
			fileHdr_Alloc ? "" : " NULL==fileHdr_Alloc"
			);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Allocate memory to header structure.
	*/
	fileHdr=malloc(sizeof(*fileHdr));
	if(NULL == fileHdr)
		{
		rCode=ENOMEM;
		goto END_ERR;
		}
	memset(fileHdr, 0x11, sizeof(*fileHdr));

	/*---------------------------------------------------------------------------
	** Open specified filter file.
	*/
	errno=0;
	fp=fopen(filePath, "rb");
	if(NULL == fp)
		rCode=errno;

	switch(rCode)
		{
		case VIGIL_SUCCESS:
			/** Read signature. **/
			errno=0;
			bytes=fread(fileHdr, 1, sizeof(*fileHdr), fp);
			if(0 == bytes)
				{
				rCode = errno;
				goto END_ERR;
				}

			/** Verify file headerSize. **/
			if(bytes < sizeof(*fileHdr))
				{
				rCode=EDOM;
				goto END_ERR;
				}
			break;

		case ENOENT:
			/** Reset error **/
			rCode = VIGIL_SUCCESS;

			/** Initialize structure **/
			fileHdr->signature          = VIGIL_FILTER_FILE_SIGNATURE;
			fileHdr->headerSize         = sizeof(*fileHdr);
			fileHdr->version            = VIGIL_FILTER_FILE_VERSION;
			fileHdr->firstSectionOffset = fileHdr->headerSize;

			/** Create the output file. **/
			errno=0;
			fp=fopen(filePath, "wb");
			if(NULL == fp)
				rCode=errno;
			if(rCode)
				goto END_ERR;

			errno=0;
			bytes=fwrite(fileHdr, 1, sizeof(*fileHdr), fp);
			if(0 == bytes)
				{
				rCode = errno;
				goto END_ERR;
				}
			if(bytes < sizeof(*fileHdr))
				{
				rCode=ENOSPC;
				goto END_ERR;
				}
			break;

		default:
			goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Verify file signature and headerSize.
	*/
	if(VIGIL_FILTER_FILE_SIGNATURE != fileHdr->signature)
		{
		rCode=EDOM;
		goto END_ERR;
		}

	if(fileHdr->headerSize < sizeof(*fileHdr))
		{
		rCode=ENODATA;
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Verify file version.
	*/
	if(fileHdr->version < VIGIL_FILTER_FILE_VERSION)
		{
		rCode=ESTALE;
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Return file header to caller.
	*/
	*fileHdr_Alloc = fileHdr;

END_ERR:

	if(fp)
		fclose(fp);

	if(rCode)
		{
		if(fileHdr)
			free(fileHdr);
		}

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_WriteFilterSection_MatchBitMask(
		/* I- filePath     */ const char *filePath,
		/* I- filterName   */ const char *filterName,
		/* I- filterFlags  */ uint32_t    filterFlags,
		/* I- filterOffset */ uint32_t    filterOffset,
		/* I- filterLen    */ uint32_t    filterLen,
		/* I- filterValue  */ void       *filterValue,
		/* I- filterMask   */ void       *filterMask
		)
	{
	int                      rCode         = VIGIL_SUCCESS;
	VIGIL_FILTER_FILE_HDR_T *fileHdr_Alloc = NULL;
	VIGIL_FILTER_SECTION_T  *filterSection = NULL;
	size_t                   filterSectionSize;
	FILE                    *fp            = NULL;
	size_t                   bytes;

	/*---------------------------------------------------------------------------
	** Validate caller args.
	*/
	if(      (NULL==filePath)
			|| (0 == filterLen)
			|| (NULL==filterValue)
			|| (NULL == filterMask)
			)
		{
		rCode=EINVAL;
		fprintf(stderr, "%s error[VDUMP_FILTER_FilterFileHeaderGet]: Bad arg[%s%s%s%s]\n",
			PROGRAM_NAME,
			filePath    ? "" : " NULL==filePath",
			filterLen   ? "" : " 0==filterLen",
			filterValue ? "" : " NULL==filterValue",
			filterMask  ? "" : " NULL==filterMask"
			);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Allocate & initialize filterSection.
	*/
	filterSectionSize=(sizeof(filterSection->matchBitMask) - sizeof(filterSection->matchBitMask.data)) + (filterLen * 2);
	filterSection=malloc(filterSectionSize);
	if(NULL == filterSection)
		{
		rCode=ENOMEM;
		fprintf(stderr, "%s error: VDUMP_FILTER_WriteFilterSection|malloc(filterSection) failed.\n", PROGRAM_NAME);
		goto END_ERR;
		}
	memset(filterSection, 0, filterSectionSize);

	/** Generic filter header **/
	filterSection->hdr.length = filterSectionSize;
	filterSection->hdr.type   = VIGIL_FILTER_T__MATCH_BIT_MASK;
	filterSection->hdr.flags  = filterFlags;
	snprintf(filterSection->hdr.name, sizeof(filterSection->hdr.name), "VFILTER_%s", filterName);

	/** MatchBitMask specific filter **/
	filterSection->matchBitMask.filterOffset = filterOffset;
	filterSection->matchBitMask.filterLen    = filterLen;

	filterSection->matchBitMask.valueOffset = 0;
	memcpy(&filterSection->matchBitMask.data[filterSection->matchBitMask.valueOffset], filterValue, filterLen);

	filterSection->matchBitMask.maskOffset = filterLen;
	memcpy(&filterSection->matchBitMask.data[filterSection->matchBitMask.maskOffset], filterMask, filterLen);

	/*---------------------------------------------------------------------------
	** Get (create file if required) file header.
	*/
	rCode=VDUMP_FILTER_FilterFileHeaderGet(
		/* I- filePath      */ filePath,
		/* -O fileHdr_Alloc */ &fileHdr_Alloc
		);
	switch(rCode)
		{
		case VIGIL_SUCCESS:
			break;

		case ENOTDIR:
		case EISDIR:
			fprintf(stderr, "%s error: Output path[%s] invalid.\n", PROGRAM_NAME, filePath);
			goto END_ERR;

		case ESTALE:
		case ENODATA:
		case EDOM:
			fprintf(stderr, "%s error: Specified file[%s] invalid.\n", PROGRAM_NAME, filePath);
			goto END_ERR;

		case ENOSPC:
			fprintf(stderr, "%s error: Specified file[%s] could not be created.\n", PROGRAM_NAME, filePath);
			goto END_ERR;

		case ENOMEM:
			fprintf(stderr, "%s error: Cannot allocate header memory for file[%s].\n", PROGRAM_NAME, filePath);
			goto END_ERR;

		default:
			fprintf(stderr, "%s error: VDUMP_FILTER_FilterFileHeaderGet(\"%s\") reports %d.\n", PROGRAM_NAME, filePath, rCode);
			goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Append section to end of file.
	*/
	/** Open specified filter file. **/
	errno=0;
	fp=fopen(filePath, "ab");
	if(NULL == fp)
		rCode=errno;

	/** Write new section. **/
	errno=0;
	bytes=fwrite(filterSection, 1, filterSection->hdr.length, fp);
	if(0 == bytes)
		{
		rCode = errno;
		goto END_ERR;
		}
	if(bytes < filterSection->hdr.length)
		{
		rCode=ENOSPC;
		goto END_ERR;
		}

END_ERR:

	if(fp)
		fclose(fp);

	if(fileHdr_Alloc)
		free(fileHdr_Alloc);

	if(filterSection)
		free(filterSection);

	return(rCode);
	}

/*******************************************************************************
** 
*/
int VDUMP_FILTER_FilterTypeAdd(
		/* I- filePath      */ const char *filePath,
		/* I- filterTypeStr */ const char *filterTypeStr,
		/* I- filterFlags   */ uint32_t    filterFlags
		)
	{
	int      rCode       = VIGIL_SUCCESS;
	uint32_t filterValue = 0;
	uint32_t filterMask  = 0xFFFFFFFF;
	uint32_t filterType  = VIGIL_FILTER_T__NONE;

	/*---------------------------------------------------------------------------
	** Validate caller args.
	*/
	if(      (NULL==filePath)
			|| (NULL==filterTypeStr)
			)
		{
		rCode=EINVAL;
		fprintf(stderr, "%s error: VDUMP_FILTER_FilterTypeAdd() reports: Bad arg[%s%s]\n",
			PROGRAM_NAME,
			filePath      ? "" : " NULL==filePath",
			filterTypeStr ? "" : " NULL==filterTypeStr"
			);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Initialize filterValue bits.
	*/
	if(0==strcmp("VIGIL", filterTypeStr))
		{
		filterType = VIGIL_FILTER_T__MATCH_BIT_MASK;
		filterValue = VIGIL_T_VIGIL;
		}
	else if(0==strcmp("NEB", filterTypeStr))
		{
		filterType = VIGIL_FILTER_T__MATCH_BIT_MASK;
		filterValue = VIGIL_T_NEB;
		}
	else if(0==strcmp("NCP", filterTypeStr))
		{
		filterType = VIGIL_FILTER_T__MATCH_BIT_MASK;
		filterValue = VIGIL_T_NCP;
		}
	else
		{
		rCode=ENOENT;
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Write filter section to file.
	*/
	switch(filterType)
		{
		case VIGIL_FILTER_T__MATCH_BIT_MASK:
			rCode=VDUMP_FILTER_WriteFilterSection_MatchBitMask(
				/* I- filePath     */ filePath,
				/* I- filterName   */ filterTypeStr,
				/* I- filterFlags  */ filterFlags,
				/* I- filterOffset */ offsetof(VIGIL_AUDIT_REC_HDR_T, type),
				/* I- filterLen    */ sizeof(filterValue),
				/* I- filterValue  */ &filterValue,
				/* I- filterMask   */ &filterMask
				);
			if(rCode)
				{
				fprintf(stderr, "%s error: VDUMP_FILTER_FilterTypeAdd|VDUMP_FILTER_WriteFilterSection() reports %d.\n", PROGRAM_NAME, rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_FILTER_T__NONE:
		default:
			rCode=EINVAL;
			fprintf(stderr, "%s error: VDUMP_FILTER_FilterTypeAdd() reports: Unknown filter type[%d].\n", PROGRAM_NAME, filterType);
			goto END_ERR;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_GetFilterTypeStr(
		/* I- filterType    */ uint32_t   filterType,
		/* -O filterTypeStr */ char     **filterTypeStr
		)
	{
	int rCode = VIGIL_SUCCESS;
	char *cp;

	switch(filterType)
		{
		case VIGIL_FILTER_T__MATCH_BIT_MASK:
			cp="MATCH_BIT_MASK";
			break;

		default:
			rCode=ENOENT;
			cp="[Unknown filter type]";
			break;
		}

	if(filterTypeStr)
		*filterTypeStr=cp;

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_HexDump(
		/* I- outStream */ FILE  *outStream,
		/* I- indent    */ int    indent,
		/* I- buf       */ void  *buf,
		/* I- bufLen    */ size_t bufLen
		)
	{
	int rCode = VIGIL_SUCCESS;
	char *cp;

	for(cp=buf; cp < (char *)(buf + bufLen); cp += 16)
		{
		char ascii[16+1];
		char hex[(16 * 3)+1];
		int  hexLen=0;
		int  x;

		memset(ascii, 0, sizeof(ascii));
		memset(hex, 0, sizeof(hex));
		for(x=0; x<16; ++x)
			{
			if((cp+x) < (char *)(buf+bufLen))
				{
				hexLen += snprintf(&hex[hexLen], (sizeof(hex) - hexLen), " %02X", 0x000000FF & cp[x]);
				ascii[x] = ((cp[x] < ' ') || (cp[x] > 126)) ? '.' : cp[x];
				}
			else
				{
				hexLen += snprintf(&hex[hexLen], (sizeof(hex) - hexLen), "   ");
				ascii[x] = ' ';
				}
			}

		fprintf(outStream, "%*s0x%08X | %s   %s\n",
			indent, "",
			(cp-(char *)buf),
			hex,
			ascii
			);
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_DumpFileSection_MatchBitMask(
		/* I- outStream */ FILE                   *outStream,
		/* I- section   */ VIGIL_FILTER_SECTION_T *section
		)
	{
	int rCode = VIGIL_SUCCESS;

	fprintf(outStream, "         filterOffset: 0x%08X [%d]\n", section->matchBitMask.filterOffset, section->matchBitMask.filterOffset);
	fprintf(outStream, "         filterLen...: 0x%08X [%d]\n", section->matchBitMask.filterLen, section->matchBitMask.filterLen);
	fprintf(outStream, "         valueOffset.: 0x%08X [%d]\n", section->matchBitMask.valueOffset, section->matchBitMask.valueOffset);
	rCode=VDUMP_FILTER_HexDump(
		/* I- outStream */ outStream,
		/* I- indent    */ 12,
		/* I- buf       */ &section->matchBitMask.data[section->matchBitMask.valueOffset],
		/* I- bufLen    */ section->matchBitMask.filterLen
		);
	if(rCode)
		goto END_ERR;

	fprintf(outStream, "         maskOffset..: 0x%08X [%d]\n", section->matchBitMask.maskOffset, section->matchBitMask.maskOffset);
	rCode=VDUMP_FILTER_HexDump(
		/* I- outStream */ outStream,
		/* I- indent    */ 12,
		/* I- buf       */ &section->matchBitMask.data[section->matchBitMask.maskOffset],
		/* I- bufLen    */ section->matchBitMask.filterLen
		);
	if(rCode)
		goto END_ERR;

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_DumpFileSection(
		/* I- outStream */ FILE                   *outStream,
		/* I- section   */ VIGIL_FILTER_SECTION_T *section
		)
	{
	int rCode = VIGIL_SUCCESS;
	char *filterTypeStr;

	rCode=VDUMP_FILTER_GetFilterTypeStr(section->hdr.type, &filterTypeStr);
	if(rCode)
		goto END_ERR;

	fprintf(outStream, "      length: 0x%08X [%d]\n", section->hdr.length, section->hdr.length);
	fprintf(outStream, "      type..: 0x%08X [%s]\n", section->hdr.type, filterTypeStr);
	fprintf(outStream, "      flags.: 0x%08X [%s]\n",
		section->hdr.flags,
		(VIGIL_FILTER_F__EXCLUDE & section->hdr.flags) ? "EXCLUDE" : "INCLUDE"
		);

	fprintf(outStream, "      [type specific fields]\n");
	switch(section->hdr.type)
		{
		case VIGIL_FILTER_T__MATCH_BIT_MASK:
			rCode=VDUMP_FILTER_DumpFileSection_MatchBitMask(
				/* I- outStream */ outStream,
				/* I- section   */ section
				);
			if(rCode)
				goto END_ERR;
			break;

		default:
			break;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_FILTER_DumpFile(
		/* I- filePath  */ const char *filePath,
		/* I- outStream */ FILE       *outStream 
		)
	{
	int rCode = VIGIL_SUCCESS;
	VIGIL_FILTER_FILE_HDR_T *fileHdr_Alloc=NULL;
	char *cp;
	unsigned short *sp;
	FILE *fp=NULL;
	VIGIL_FILTER_SECTION_HDR_T sectionHdr;
	size_t bytes;
	long filePos;
	VIGIL_FILTER_SECTION_T *section=NULL;
	int sectionNumber=0;

	/*---------------------------------------------------------------------------
	** Validate caller args.
	*/
	if(NULL==filePath)
		{
		rCode=EINVAL;
		goto END_ERR;
		}

	fprintf(outStream, "[Filter File: \"%s\"]\n", filePath);

	/*---------------------------------------------------------------------------
	** Parse file header.
	*/
	rCode=VDUMP_FILTER_FilterFileHeaderGet(
		/* I- filePath      */ filePath,
		/* -O fileHdr_Alloc */ &fileHdr_Alloc
		);
	if(rCode)
		goto END_ERR;

	cp = (char *)&fileHdr_Alloc->signature;
	fprintf(outStream, "   File signature......: 0x%08X [%c%c%c%c]\n",
		fileHdr_Alloc->signature,
		cp[0], cp[1], cp[2], cp[3]
		);
	sp=(unsigned short *)&fileHdr_Alloc->version;
	fprintf(outStream, "   File version........: 0x%08X [%d.%02d]\n",
		fileHdr_Alloc->version,
		sp[1], sp[0]
		);
	fprintf(outStream, "   File header size....: 0x%08X [%d]\n", fileHdr_Alloc->headerSize, fileHdr_Alloc->headerSize);
	fprintf(outStream, "   First section offset: 0x%08X [%d]\n", fileHdr_Alloc->firstSectionOffset, fileHdr_Alloc->firstSectionOffset);

	/*---------------------------------------------------------------------------
	** Parse file sections.
	*/
	/** Open target file. **/
	errno=0;
	fp=fopen(filePath, "rb");
	if(NULL == fp)
		{
		rCode=errno;
		goto END_ERR;
		}

	/** Seek to the first section offset. **/
	errno=0;
	if((-1) == fseek(fp, fileHdr_Alloc->firstSectionOffset, SEEK_SET))
		{
		rCode=errno;
		goto END_ERR;
		}

	while(1)
		{
		/** Save current position **/
		errno=0;
		if((-1) == (filePos=ftell(fp)))
			{
			rCode = errno;
			goto END_ERR;
			}

		/** Read section Header**/
		errno=0;
		bytes=fread(&sectionHdr, 1, sizeof(sectionHdr), fp);
		if(0 == bytes)
			{
			rCode = errno;
			if(0 == rCode)
				break;

			goto END_ERR;
			}

		/** Verify file headerSize read. **/
		if(sizeof(sectionHdr) != bytes)
			{
			rCode=EDOM;
			goto END_ERR;
			}

		/** Allocate section buffer **/
		section=malloc(sectionHdr.length);
		if(NULL == section)
			{
			rCode=ENOMEM;
			goto END_ERR;
			}
		memset(section, 0, sizeof(*section));

		/** Restore position. **/
		errno=0;
		if((-1) == fseek(fp, filePos, SEEK_SET))
			{
			rCode=errno;
			goto END_ERR;
			}

		/** Read section **/
		errno=0;
		bytes=fread(section, 1, sectionHdr.length, fp);
		if(0 == bytes)
			{
			rCode = errno;
			goto END_ERR;
			}

		/** Verify file section size read. **/
		if(sectionHdr.length != bytes)
			{
			rCode=EDOM;
			goto END_ERR;
			}

		/** Display section. **/
		fprintf(outStream, "   [Section #%d]\n", ++sectionNumber);
		rCode=VDUMP_FILTER_DumpFileSection(
			/* I- outStream */ outStream,
			/* I- section   */ section
			);
		if(rCode)
			goto END_ERR;

		/** Free section buffer **/
		free(section);
		section=NULL;
		}

END_ERR:

	if(section)
		free(section);

	if(fp)
		fclose(fp);

	if(fileHdr_Alloc)
		free(fileHdr_Alloc);

	return(rCode);
	}







