/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES Auditing Sample Client (vdump).
 |
 | File: vdump_out_std.c
 |
 | Desc: Parses records and displays them in default format.
 +-------------------------------------------------------------------------*/

/*******************************************************************************
** Compiler setup.
*/

	/*---------------------------------------------------------------------------
	** Auto Tools
	*/
#ifdef HAVE_CONFIG_H
 #ifndef __config_h__
	#include <config.h>  /* PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT */
	#define __config_h__
 #endif 
#endif

	/*---------------------------------------------------------------------------
	** ANSI/POSIX
	*/
	#include <stdlib.h>	/* exit() */
	#include <stdio.h>	/* stderr, fprintf() */
	#include <errno.h>
	#include <time.h>
	#include <string.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <dirent.h>

	#include <sys/stat.h>
	#include <fcntl.h>

	#include <signal.h>

       #include <sys/types.h>
       #include <pwd.h>


	/*---------------------------------------------------------------------------
	** Project
	*/
	#include <vigil.h>
	#include <libvigil.h>
	#include "vdump_out_csv.h"

	#include <zParams.h>

	#include <ncpID.h>

/*******************************************************************************
** Global storage.
*/

/*******************************************************************************
** Print a unicode as an element.
*/
int VDUMP_OUT_STD_Print_UnicodeStr(
		/* -O outStream  */ FILE     *outStream,
		/* I- label      */ char     *label,
		/* I- unicodeStr */ uint16_t *unicodeStr
		)
	{
	int rCode=VIGIL_SUCCESS;

	fprintf(outStream, "%s%s[", label ? " " : "", label ? label : "Path");

	while(*unicodeStr)
		{
		fprintf(outStream, "%c", (char)(0x00FF & *unicodeStr));
		++unicodeStr;
		}

	fprintf(outStream, "]");

	return(rCode);
	}

int VDUMP_OUT_STD_StrUnicode2Ascii(
		/* I- unicodeStr */ uint16_t *unicodeStr,
		/* -O asciiStr   */ uint8_t  *asciiStr
		)
	{
	int rCode=VIGIL_SUCCESS;

	if((NULL==unicodeStr) || (NULL==asciiStr))
		{
		rCode=EINVAL;
		goto END_ERR;
		}

	while(*unicodeStr)
		{
		*asciiStr = (uint8_t)(0x00FF & *unicodeStr);
		++asciiStr;
		++unicodeStr;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_PrintDate_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- date      */ uint16_t date
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char *month[] =
		{
		/* 0 */ "NULL-uary",
		/* 1 */ "January",
		/* 2 */ "February",
		/* 3 */ "March",
		/* 4 */ "April",
		/* 5 */ "May",
		/* 6 */ "June",
		/* 7 */ "July",
		/* 8 */ "August",
		/* 9 */ "September",
		/* 10 */ "October",
		/* 11 */ "November",
		/* 12 */ "December",
		/* 13 */ "13-uary",
		/* 14 */ "14-uary",
		/* 15 */ "15-uary"
		};

	fprintf(outStream, "%s%s%s%2d %3.3s %04d%s",
		label ? " " : "",
		label ? label : "",
		label ? "[" : "",
		/* Day   */ (0x001F & date),
		/* Month */ month[(0x01E0 & date) >> 5],
		/* Year  */ ((0xFE00 & date) >> 9) + 1980,
		label ? "]" : ""
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_PrintTime_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- time      */ uint16_t time
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, "%s%s%s%2d:%02d:%02d%s",
		label ? " " : "",
		label ? label : "",
		label ? "[" : "",
		/* Hours   */ (0xF800 & time) >> 11,
		/* Minutes */ (0x07E0 & time) >> 5,
		/* Seconds */ (0x001F & time) * 2,
		label ? "]" : ""
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintTimeStamp(
		/* -O outStream */ FILE           *outStream,
		/* I- timeVal   */ struct timeval *timeVal
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm timeDate;
	char timeDateStr[63+1];

	localtime_r(&timeVal->tv_sec, &timeDate);
	strftime(timeDateStr, sizeof(timeDateStr), "%Y-%02m-%02d %02H:%02M:%02S", &timeDate);
	fprintf(outStream, " %s.%06lu", timeDateStr, timeVal->tv_usec);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintTime_64()
**                   VDUMP_OUT_STD_PrintTimes()
*/
int VDUMP_OUT_STD_PrintTime(
		/* -O outStream */ FILE   *outStream,
		/* I- label     */ char   *label,
		/* I- timep     */ time_t *timep
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm tm;
	char tmStr[63+1];
	
	localtime_r(timep, &tm);
	strftime(tmStr, sizeof(tmStr), "%Y-%02m-%02d %02H:%02M:%02S", &tm);
	fprintf(outStream, "%s%s%s%s%s",
		label ? " " : "",
		label ? label : "",
		label ? "[" : "",
		tmStr,
		label ? "]" : ""
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_PrintTime_64(
		/* -O outStream */ FILE     *outStream,
		/* I- label     */ char     *label,
		/* I- timep     */ uint64_t *timep_64
		)
	{
	int  rCode=VIGIL_SUCCESS;

	rCode=VDUMP_OUT_STD_PrintTime(
		/* -O outStream */ outStream,
		/* I- label     */ label,
		/* I- timep     */ (time_t *)timep_64
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_PrintTimes(
		/* -O outStream */ FILE              *outStream,
		/* I- times     */ VIGIL_TIME_INFO_T *times
		)
	{
	int  rCode=VIGIL_SUCCESS;
	
	rCode=VDUMP_OUT_STD_PrintTime_64(outStream, "accessed", &times->accessedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_PrintTime(accessedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_STD_PrintTime_64(outStream, "created", &times->createdTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_PrintTime(createdTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_STD_PrintTime_64(outStream, "modified", &times->modifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_PrintTime(times->modifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_STD_PrintTime_64(outStream, "metaDataModified", &times->metaDataModifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_PrintTime(metaDataModifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_ElementPrint_Guid(
		/* -O outStream */ FILE         *o_outStream,
		/* I- label     */ char         *i_label,
		/* I- guid      */ VIGIL_GUID_T *i_guid
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char *cp = (char *)i_guid;
	char *guidStr=NULL;
	uint8_t *guidStr_ALLOC = NULL;

	/*-------------------------------------------------------------------------
	** Validate caller arg(s).
	*/
	if((NULL==o_outStream) || (NULL == i_guid))
		{
		rCode=EINVAL;
		LIBVIGIL_ERR("Bad arg[%s%s]\n",
			o_outStream ? "" : " NULL=o_outStream",
			i_guid      ? "" : " NULL=i_guid"
			);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Discover if the i_guid is a magic (nds/nss) constant.
	*/

	/** First, resolve NSS guid constants. (Root/supervisor, etc.) **/
	rCode=LIBVIGIL_GUID_ConstStr(
		/* I- guid    */ i_guid,
		/* -O guidStr */ &guidStr
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_GUID_ConstChar() reports %d\n", rCode);
		goto END_ERR;
		}

	/** Second, resolve NDS guids.  (NDS users, groups, etc.) **/
	if(NULL == guidStr)
		{
		uint16_t strDn[((255 + 1) * 2)];

		rCode=NCPMapGUIDToDN(
			/* I- guid   */ (char *)i_guid,
			/* I- dnSize */ sizeof(strDn),
			/* I- dn     */ strDn
			);
		switch(rCode)
			{
			case VIGIL_SUCCESS:
				VDUMP_OUT_STD_StrUnicode2Ascii(
					/* I- unicodeStr */ strDn,
					/* -O asciiStr   */ (uint8_t *)strDn
					);
	
				guidStr=(uint8_t*)strDn;
				break;

			case (-765): /* eDirectory error code: FFFFFD03 ERR EOF HIT */
				rCode=VIGIL_SUCCESS;
				break;

			default:
				LIBVIGIL_ERR("LIBVIGIL_EDIR_MapGuidToDn_ALLOC() reports %d\n", rCode);
				goto END_ERR;
			}
		}

	/** Third, Resolve NSS guids. (VolumeIDs, PoolIDs, etc.)**/
	if(NULL == guidStr)
		{
		rCode=LIBVIGIL_NSS_MapNssGUIDToNssObjectName(
			/* I- objectGuid       */ i_guid,
			/* -O objectName_ALLOC */ &guidStr_ALLOC
			);
		if(rCode)
			rCode=VIGIL_SUCCESS;
		else
			guidStr=guidStr_ALLOC;
		}

	/*-------------------------------------------------------------------------
	** Output the guid string.
	*/
	fprintf(o_outStream, " %s[%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%s%s%s]",
		i_label?i_label:"guid",
		0xFF & cp[0],
		0xFF & cp[1],
		0xFF & cp[2],
		0xFF & cp[3],
		0xFF & cp[4],
		0xFF & cp[5],
		0xFF & cp[6],
		0xFF & cp[7],
		0xFF & cp[8],
		0xFF & cp[9],
		0xFF & cp[10],
		0xFF & cp[11],
		0xFF & cp[12],
		0xFF & cp[13],
		0xFF & cp[14],
		0xFF & cp[15],
		guidStr ? " \"" : "",
		guidStr ? guidStr : "",
		guidStr ? "\"" : ""
		);

END_ERR:

	if(guidStr_ALLOC)
		LIBVIGIL_MEM_Free((void **)&guidStr_ALLOC);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecordElement()
*/
int VDUMP_OUT_STD_PrintHex(
		/* -O outStream */ FILE  *outStream,
		/* I- buf       */ void  *buf,
		/* I- bufLen    */ size_t bufLen,
		/* I- label     */ char  *label
		)
	{
	int  rCode=VIGIL_SUCCESS;
	size_t nCnt;
	char *b=(char *)buf;

	fprintf(outStream, " %s[", label?label:"");

	for(nCnt=0; nCnt < bufLen; ++nCnt)
		fprintf(outStream, "%02X", 0xFF & b[nCnt]);

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintFhState(
		/* -O outStream */ FILE    *outStream,
		/* I- fhState   */ uint32_t fhState
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " fhState[0x%08X", fhState);

	if(/*FS_CFS_ASYNC_CLOSE */0x20 & fhState) fprintf(outStream, " 6-CFS_ASYNC_CLOSE");  /* The grantedRights counts as been done synchronously, now do the async part of the close. */
	if(/*FS_CFS_OPEN        */0x10 & fhState) fprintf(outStream, " 5-CFS_OPEN");  /* The open was done for a CFS slave on the master. */
	if(/*FS_NDS_OPEN        */0x08 & fhState) fprintf(outStream, " 4-NDS_OPEN");
	if(/*FS_READ_BACKUP     */0x04 & fhState) fprintf(outStream, " 3-READ_SNAPSHOT");
	if(/*FS_WRITE_SNAPSHOT  */0x02 & fhState) fprintf(outStream, " 2-WRITE_SNAPSHOT");
	if(/*FH_MODIFIED        */0x01 & fhState) fprintf(outStream, " 1-MODIFIED");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintFileType(
		/* -O outStream */ FILE    *outStream,
		/* I- fileType  */ uint32_t fileType
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " fileType[%d", fileType);

	switch(fileType)
		{
		case zFILE_UNKNOWN:            fprintf(outStream, " UNKNOWN"); break;
		case zFILE_REGULAR:            fprintf(outStream, " REGULAR_FILE/DIRECTORY"); break;
		case zFILE_EXTENDED_ATTRIBUTE: fprintf(outStream, " EXTENDED_ATTRIBUTE"); break;
		case zFILE_NAMED_DATA_STREAM:  fprintf(outStream, " NAMED_DATA_STREAM"); break;
		case zFILE_PIPE:               fprintf(outStream, " NAMED_PIPE"); break;
		case zFILE_VOLUME:             fprintf(outStream, " VOLUME"); break;
		case zFILE_POOL:               fprintf(outStream, " POOL"); break;
		default:                       fprintf(outStream, " ???"); break;
		}

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintFileAttributes(
		/* -O outStream      */ FILE    *outStream,
		/* I- fileAttributes */ uint64_t fileAttributes
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " fileAttributes[0x%08X", fileAttributes);

	if(zFA_VOLATILE                  & fileAttributes) fprintf(outStream, " 32-ATTR_VOLATILE");
	if(zFA_ATTR_ARCHIVE              & fileAttributes) fprintf(outStream, " 31-ATTR_ARCHIVE");
	if(zFA_CANT_COMPRESS_DATA_STREAM & fileAttributes) fprintf(outStream, " 30-CANT_COMPRESS_DATA_STREAM");
	if(zFA_HARDLINK                  & fileAttributes) fprintf(outStream, " 29-HARDLINK");
	if(zFA_DO_NOT_COMPRESS_FILE      & fileAttributes) fprintf(outStream, " 28-DO_NOT_COMPRESS_FILE");
	if(zFA_DATA_STREAM_IS_COMPRESSED & fileAttributes) fprintf(outStream, " 27-DATA_STREAM_IS_COMPRESSED");
	if(zFA_COMPRESS_FILE_IMMEDIATELY & fileAttributes) fprintf(outStream, " 26-COMPRESS_FILE_IMMEDIATELY");
	if(0x01000000                    & fileAttributes) fprintf(outStream, " 25-???");
	if(zFA_REMOTE_DATA_INHIBIT       & fileAttributes) fprintf(outStream, " 24-REMOTE_DATA_INHIBIT");
	if(zFS_REMOTE_DATA_ACCESS        & fileAttributes) fprintf(outStream, " 23-REMOTE_DATA_ACCESS");
	if(zFA_IS_LINK                   & fileAttributes) fprintf(outStream, " 22-IS_LINK");
	if(zFA_IS_ADMIN_LINK             & fileAttributes) fprintf(outStream, " 21-IS_ADMIN_LINK");
	if(zFA_COPY_INHIBIT              & fileAttributes) fprintf(outStream, " 20-COPY_INHIBIT");
	if(zFA_DELETE_INHIBIT            & fileAttributes) fprintf(outStream, " 19-DELETE_INHIBIT");
	if(zFA_RENAME_INHIBIT            & fileAttributes) fprintf(outStream, " 18-RENAME_INHIBIT");
	if(zFA_IMMEDIATE_PURGE           & fileAttributes) fprintf(outStream, " 17-IMMEDIATE_PURGE");
	if(0x00008000                    & fileAttributes) fprintf(outStream, " 16-???");
	if(0x00004000                    & fileAttributes) fprintf(outStream, " 15-???");
	if(zFA_NOT_VIRTUAL_FILE          & fileAttributes) fprintf(outStream, " 14-NOT_VIRTUAL_FILE");
	if(zFA_TRANSACTION               & fileAttributes) fprintf(outStream, " 13-TRANSACTION");
	if(zFA_NO_SUBALLOC               & fileAttributes) fprintf(outStream, " 12-NO_SUBALLOC");
	if(0x00000400                    & fileAttributes) fprintf(outStream, " 11-SMODE_BITS_R");
	if(0x00000200                    & fileAttributes) fprintf(outStream, " 10-SMODE_BITS_W");
	if(0x00000100                    & fileAttributes) fprintf(outStream, " 9-SMODE_BITS_X");
	if(zFA_SHAREABLE                 & fileAttributes) fprintf(outStream, " 8-SHAREABLE");
	if(0x00000040                    & fileAttributes) fprintf(outStream, " 7-???");
	if(zFA_ARCHIVE                   & fileAttributes) fprintf(outStream, " 6-ARCHIVE");
	if(zFA_SUBDIRECTORY              & fileAttributes) fprintf(outStream, " 5-SUBDIRECTORY");
	if(zFA_EXECUTE                   & fileAttributes) fprintf(outStream, " 4-EXECUTE");
	if(zFA_SYSTEM                    & fileAttributes) fprintf(outStream, " 3-SYSTEM");
	if(zFA_HIDDEN                    & fileAttributes) fprintf(outStream, " 2-HIDDEN");
	if(zFA_READ_ONLY                 & fileAttributes) fprintf(outStream, " 1-READ_ONLY");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintRequestedRights(
		/* -O outStream       */ FILE    *outStream,
		/* I- requestedRights */ uint32_t requestedRights
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " requestedRights[0x%08X", requestedRights);

	if(zRR_DONT_UPDATE_ACCESS_TIME        & requestedRights) fprintf(outStream, " 32-DONT_UPDATE_ACCESS_TIME");
	if(zRR_CANT_DELETE_WHILE_OPEN         & requestedRights) fprintf(outStream, " 31-CANT_DELETE_WHILE_OPEN");
	if(zRR_OPENER_CAN_DELETE_WHILE_OPEN   & requestedRights) fprintf(outStream, " 30-OPENER_CAN_DELETE_WHILE_OPEN");
	if(zRR_CREATE_WITHOUT_READ_ACCESS     & requestedRights) fprintf(outStream, " 29-CREATE_WITHOUT_READ_ACCESS");
	if(zRR_DENY_RW_OPENER_CAN_REOPEN      & requestedRights) fprintf(outStream, " 28-DENY_RW_OPENER_CAN_REOPEN");
	if(zRR_READ_ACCESS_TO_SNAPSHOT        & requestedRights) fprintf(outStream, " 27-READ_ACCESS_TO_SNAPSHOT");
	if(zRR_MASK_READ_IN_DROP_BOXES        & requestedRights) fprintf(outStream, " 26-MASK_READ_IN_DROP_BOXES");
	if(zRR_PREVENT_FS_HOOKS               & requestedRights) fprintf(outStream, " 25-PREVENT_FS_HOOKS");
	if(0x00800000                         & requestedRights) fprintf(outStream, " 24-???");
	if(0x00400000                         & requestedRights) fprintf(outStream, " 23-???");
	if(zRR_PSA_CACHE                      & requestedRights) fprintf(outStream, " 22-PSA_CACHE");
	if(zRR_TRANSACTION_ACTIVE             & requestedRights) fprintf(outStream, " 21-TRANSACTION_ACTIVE");
	if(0x00080000                         & requestedRights) fprintf(outStream, " 20-???");
	if(0x00040000                         & requestedRights) fprintf(outStream, " 19-???");
	if(zRR_ALLOW_SECURE_DIRECTORY_ACCESS  & requestedRights) fprintf(outStream, " 18-ALLOW_SECURE_DIRECTORY_ACCESS");
	if(0x00010000                         & requestedRights) fprintf(outStream, " 17-???");
	if(0x00008000                         & requestedRights) fprintf(outStream, " 16-???");
	if(0x00004000                         & requestedRights) fprintf(outStream, " 15-???");
	if(zRR_DIO_MODE                       & requestedRights) fprintf(outStream, " 14-DIO_MODE");
	if(zRR_PURGE_IMMEDIATE_ON_CLOSE       & requestedRights) fprintf(outStream, " 13-PURGE_IMMEDIATE_ON_CLOSE");
	if(zRR_FLUSH_ON_CLOSE                 & requestedRights) fprintf(outStream, " 12-FLUSH_ON_CLOSE");
	if(zRR_DELETE_FILE_ON_CLOSE           & requestedRights) fprintf(outStream, " 11-DELETE_FILE_ON_CLOSE");
	if(zRR_LEAVE_FILE_COMPRESSED          & requestedRights) fprintf(outStream, " 10-LEAVE_FILE_COMPRESSED");
	if(zRR_ENABLE_IO_ON_COMPRESSED_DATA   & requestedRights) fprintf(outStream, " 9-ENABLE_IO_ON_COMPRESSED_DATA");
	if(0x00000080                         & requestedRights) fprintf(outStream, " 8-???");
	if(0x00000040                         & requestedRights) fprintf(outStream, " 7-???");
	if(0x00000020                         & requestedRights) fprintf(outStream, " 6-???");
	if(zRR_SCAN_ACCESS                    & requestedRights) fprintf(outStream, " 5-SCAN_ACCESS");
	if(zRR_DENY_WRITE                     & requestedRights) fprintf(outStream, " 4-DENY_WRITE");
	if(zRR_DENY_READ                      & requestedRights) fprintf(outStream, " 3-DENY_READ");
	if(zRR_WRITE_ACCESS                   & requestedRights) fprintf(outStream, " 2-WRITE_ACCESS");
	if(zRR_READ_ACCESS                    & requestedRights) fprintf(outStream, " 1-READ_ACCESS");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintCreateFlags(
		/* -O outStream   */ FILE    *outStream,
		/* I- CreateFlags */ uint32_t createFlags
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " createFlags[0x%08X", createFlags);

	if(0x80000000                & createFlags) fprintf(outStream, " 32-???");
	if(0x40000000                & createFlags) fprintf(outStream, " 31-???");
	if(0x20000000                & createFlags) fprintf(outStream, " 30-???");
	if(0x10000000                & createFlags) fprintf(outStream, " 29-???");
	if(0x08000000                & createFlags) fprintf(outStream, " 28-???");
	if(0x04000000                & createFlags) fprintf(outStream, " 27-???");
	if(0x02000000                & createFlags) fprintf(outStream, " 26-???");
	if(0x01000000                & createFlags) fprintf(outStream, " 25-???");
	if(0x00800000                & createFlags) fprintf(outStream, " 24-???");
	if(0x00400000                & createFlags) fprintf(outStream, " 23-???");
	if(0x00200000                & createFlags) fprintf(outStream, " 22-???");
	if(0x00100000                & createFlags) fprintf(outStream, " 21-???");
	if(0x00080000                & createFlags) fprintf(outStream, " 20-???");
	if(0x00040000                & createFlags) fprintf(outStream, " 19-???");
	if(0x00020000                & createFlags) fprintf(outStream, " 18-???");
	if(0x00010000                & createFlags) fprintf(outStream, " 17-???");
	if(0x00008000                & createFlags) fprintf(outStream, " 16-???");
	if(0x00004000                & createFlags) fprintf(outStream, " 15-???");
	if(0x00002000                & createFlags) fprintf(outStream, " 14-???");
	if(0x00001000                & createFlags) fprintf(outStream, " 13-???");
	if(0x00000800                & createFlags) fprintf(outStream, " 12-???");
	if(0x00000400                & createFlags) fprintf(outStream, " 11-???");
	if(0x00000200                & createFlags) fprintf(outStream, " 10-???");
	if(zCREATE_KEEP_VFS_CACHE    & createFlags) fprintf(outStream, " 9-KEEP_VFS_CACHE");
	if(0x00000080                & createFlags) fprintf(outStream, " 8-???");
	if(0x00000040                & createFlags) fprintf(outStream, " 7-???");
	if(0x00000020                & createFlags) fprintf(outStream, " 6-???");
	if(0x00000010                & createFlags) fprintf(outStream, " 5-???");
	if(0x00000008                & createFlags) fprintf(outStream, " 4-???");
	if(zCREATE_DELETE_IF_THERE   & createFlags) fprintf(outStream, " 3-DELETE_IF_THERE");
	if(zCREATE_TRUNCATE_IF_THERE & createFlags) fprintf(outStream, " 2-TRUNCATE_IF_THERE");
	if(zCREATE_OPEN_IF_THERE     & createFlags) fprintf(outStream, " 1-OPEN_IF_THERE");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintOpenCreateActionFlags(
		/* -O outStream             */ FILE    *outStream,
		/* I- OpenCreateActionFlags */ uint32_t openCreateActionFlags
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " openCreateActionFlags[0x%08X", openCreateActionFlags);

	if(0x80000000       & openCreateActionFlags) fprintf(outStream, " 32-???");
	if(0x40000000       & openCreateActionFlags) fprintf(outStream, " 31-???");
	if(0x20000000       & openCreateActionFlags) fprintf(outStream, " 30-???");
	if(0x10000000       & openCreateActionFlags) fprintf(outStream, " 29-???");
	if(0x08000000       & openCreateActionFlags) fprintf(outStream, " 28-???");
	if(0x04000000       & openCreateActionFlags) fprintf(outStream, " 27-???");
	if(0x02000000       & openCreateActionFlags) fprintf(outStream, " 26-???");
	if(0x01000000       & openCreateActionFlags) fprintf(outStream, " 25-???");
	if(0x00800000       & openCreateActionFlags) fprintf(outStream, " 24-???");
	if(0x00400000       & openCreateActionFlags) fprintf(outStream, " 23-???");
	if(0x00200000       & openCreateActionFlags) fprintf(outStream, " 22-???");
	if(0x00100000       & openCreateActionFlags) fprintf(outStream, " 21-???");
	if(0x00080000       & openCreateActionFlags) fprintf(outStream, " 20-???");
	if(0x00040000       & openCreateActionFlags) fprintf(outStream, " 19-???");
	if(0x00020000       & openCreateActionFlags) fprintf(outStream, " 18-???");
	if(0x00010000       & openCreateActionFlags) fprintf(outStream, " 17-???");
	if(0x00008000       & openCreateActionFlags) fprintf(outStream, " 16-???");
	if(0x00004000       & openCreateActionFlags) fprintf(outStream, " 15-???");
	if(0x00002000       & openCreateActionFlags) fprintf(outStream, " 14-???");
	if(0x00001000       & openCreateActionFlags) fprintf(outStream, " 13-???");
	if(0x00000800       & openCreateActionFlags) fprintf(outStream, " 12-???");
	if(0x00000400       & openCreateActionFlags) fprintf(outStream, " 11-???");
	if(0x00000200       & openCreateActionFlags) fprintf(outStream, " 10-???");
	if(0x00000100       & openCreateActionFlags) fprintf(outStream, " 9-???");
	if(0x00000080       & openCreateActionFlags) fprintf(outStream, " 8-???");
	if(0x00000040       & openCreateActionFlags) fprintf(outStream, " 7-???");
	if(0x00000020       & openCreateActionFlags) fprintf(outStream, " 6-???");
	if(0x00000010       & openCreateActionFlags) fprintf(outStream, " 5-???");
	if(0x00000008       & openCreateActionFlags) fprintf(outStream, " 4-???");
	if(0x00000004 /*zBEAST_TRUNCATED*/ & openCreateActionFlags) fprintf(outStream, " 3-TRUNCATED");
	if(0x00000002 /*zBEAST_CREATED*/   & openCreateActionFlags) fprintf(outStream, " 2-CREATED");
	if(0x00000001 /*zBEAST_EXISTED*/   & openCreateActionFlags) fprintf(outStream, " 1-EXISTED");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintRenameFlags(
		/* -O outStream   */ FILE       *outStream,
		/* I- label       */ const char *label,
		/* I- renameFlags */ uint32_t    renameFlags
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s[0x%08X", label?label:"renameFlags", renameFlags);

	if(0x80000000                      & renameFlags) fprintf(outStream, " 32-???");
	if(0x40000000                      & renameFlags) fprintf(outStream, " 31-???");
	if(0x20000000                      & renameFlags) fprintf(outStream, " 30-???");
	if(0x10000000                      & renameFlags) fprintf(outStream, " 29-???");
	if(0x08000000                      & renameFlags) fprintf(outStream, " 28-???");
	if(0x04000000                      & renameFlags) fprintf(outStream, " 27-???");
	if(0x02000000                      & renameFlags) fprintf(outStream, " 26-???");
	if(0x01000000                      & renameFlags) fprintf(outStream, " 25-???");
	if(0x00800000                      & renameFlags) fprintf(outStream, " 24-???");
	if(0x00400000                      & renameFlags) fprintf(outStream, " 23-???");
	if(0x00200000                      & renameFlags) fprintf(outStream, " 22-???");
	if(0x00100000                      & renameFlags) fprintf(outStream, " 21-???");
	if(0x00080000                      & renameFlags) fprintf(outStream, " 20-???");
	if(0x00040000                      & renameFlags) fprintf(outStream, " 19-???");
	if(0x00020000                      & renameFlags) fprintf(outStream, " 18-???");
	if(0x00010000                      & renameFlags) fprintf(outStream, " 17-???");
	if(0x00008000                      & renameFlags) fprintf(outStream, " 16-???");
	if(0x00004000                      & renameFlags) fprintf(outStream, " 15-???");
	if(0x00002000                      & renameFlags) fprintf(outStream, " 14-???");
	if(0x00001000                      & renameFlags) fprintf(outStream, " 13-???");
	if(0x00000800                      & renameFlags) fprintf(outStream, " 12-???");
	if(0x00000400                      & renameFlags) fprintf(outStream, " 11-???");
	if(0x00000200                      & renameFlags) fprintf(outStream, " 10-???");
	if(zRENAME_KEEP_VFS_CACHE          & renameFlags) fprintf(outStream, " 9-KEEP_VFS_CACHE");
	if(0x00000080                      & renameFlags) fprintf(outStream, " 8-???");
	if(0x00000040 /*zRENAME_TARGET_IS_PATTERN*/       & renameFlags) fprintf(outStream, " 7-TARGET_IS_PATTERN");
	if(0x00000020 /*zRENAME_DONT_RENAME_DIRECTORIES*/ & renameFlags) fprintf(outStream, " 6-DONT_RENAME_DIRECTORIES");
	if(0x00000010 /*zRENAME_DONT_RENAME_FILES*/       & renameFlags) fprintf(outStream, " 5-DONT_RENAME_FILES");
	if(zRENAME_DELETE_FILE_IF_THERE    & renameFlags) fprintf(outStream, " 4-DELETE_FILE_IF_THERE");
	if(zRENAME_THIS_NAME_SPACE_ONLY    & renameFlags) fprintf(outStream, " 3-THIS_NAME_SPACE_ONLY");
	if(0x00000002 /*zRENAME_COMPATABILITY*/           & renameFlags) fprintf(outStream, " 2-COMPATABILITY");
	if(zRENAME_ALLOW_RENAMES_TO_MYSELF & renameFlags) fprintf(outStream, " 1-ALLOW_RENAMES_TO_MYSELF");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintLinkFlags(
		/* -O outStream */ FILE       *outStream,
		/* I- label     */ const char *label,
		/* I- linkFlags */ uint32_t    linkFlags
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s[0x%08X", label?label:"linkFlags", linkFlags);

	if(0x80000000         & linkFlags) fprintf(outStream, " 32-???");
	if(0x40000000         & linkFlags) fprintf(outStream, " 31-???");
	if(0x20000000         & linkFlags) fprintf(outStream, " 30-???");
	if(0x10000000         & linkFlags) fprintf(outStream, " 29-???");
	if(0x08000000         & linkFlags) fprintf(outStream, " 28-???");
	if(0x04000000         & linkFlags) fprintf(outStream, " 27-???");
	if(0x02000000         & linkFlags) fprintf(outStream, " 26-???");
	if(0x01000000         & linkFlags) fprintf(outStream, " 25-???");
	if(0x00800000         & linkFlags) fprintf(outStream, " 24-???");
	if(0x00400000         & linkFlags) fprintf(outStream, " 23-???");
	if(0x00200000         & linkFlags) fprintf(outStream, " 22-???");
	if(0x00100000         & linkFlags) fprintf(outStream, " 21-???");
	if(0x00080000         & linkFlags) fprintf(outStream, " 20-???");
	if(0x00040000         & linkFlags) fprintf(outStream, " 19-???");
	if(0x00020000         & linkFlags) fprintf(outStream, " 18-???");
	if(0x00010000         & linkFlags) fprintf(outStream, " 17-???");
	if(0x00008000         & linkFlags) fprintf(outStream, " 16-???");
	if(0x00004000         & linkFlags) fprintf(outStream, " 15-???");
	if(0x00002000         & linkFlags) fprintf(outStream, " 14-???");
	if(0x00001000         & linkFlags) fprintf(outStream, " 13-???");
	if(0x00000800         & linkFlags) fprintf(outStream, " 12-???");
	if(0x00000400         & linkFlags) fprintf(outStream, " 11-???");
	if(0x00000200         & linkFlags) fprintf(outStream, " 10-???");
	if(zLF_KEEP_VFS_CACHE & linkFlags) fprintf(outStream, " 9-KEEP_VFS_CACHE");
	if(0x00000080         & linkFlags) fprintf(outStream, " 8-???");
	if(0x00000040         & linkFlags) fprintf(outStream, " 7-???");
	if(0x00000020         & linkFlags) fprintf(outStream, " 6-???");
	if(0x00000010         & linkFlags) fprintf(outStream, " 5-???");
	if(0x00000008         & linkFlags) fprintf(outStream, " 4-???");
	if(0x00000004         & linkFlags) fprintf(outStream, " 3-???");
	if(0x00000002         & linkFlags) fprintf(outStream, " 2-???");
	if(zLF_HARD_LINK      & linkFlags) fprintf(outStream, " 1-HARD_LINK");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintRights(
		/* -O outStream */ FILE       *outStream,
		/* I- label     */ const char *label,
		/* I- rights    */ uint32_t    rights
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s[0x%08X", label ? label : "rights", rights);

	if(0x80000000                 & rights) fprintf(outStream, " 32-???");
	if(0x40000000                 & rights) fprintf(outStream, " 31-???");
	if(0x20000000                 & rights) fprintf(outStream, " 30-???");
	if(0x10000000                 & rights) fprintf(outStream, " 29-???");
	if(0x08000000                 & rights) fprintf(outStream, " 28-???");
	if(0x04000000                 & rights) fprintf(outStream, " 27-???");
	if(0x02000000                 & rights) fprintf(outStream, " 26-???");
	if(0x01000000                 & rights) fprintf(outStream, " 25-???");
	if(0x00800000                 & rights) fprintf(outStream, " 24-???");
	if(0x00400000                 & rights) fprintf(outStream, " 23-???");
	if(0x00200000                 & rights) fprintf(outStream, " 22-???");
	if(0x00100000                 & rights) fprintf(outStream, " 21-???");
	if(0x00080000                 & rights) fprintf(outStream, " 20-???");
	if(0x00040000                 & rights) fprintf(outStream, " 19-???");
	if(0x00020000                 & rights) fprintf(outStream, " 18-???");
	if(0x00010000                 & rights) fprintf(outStream, " 17-???");
	if(0x00008000 /*zAUTHORIZE_INHERIT_DOWN*/    & rights) fprintf(outStream, " 16-INHERIT_DOWN");
	if(0x00004000 /*zAUTHORIZE_INHERIT_UP*/      & rights) fprintf(outStream, " 15-INHERIT_UP");
	if(0x00002000 /*zAUTHORIZE_NEGATIVE_RIGHTS*/ & rights) fprintf(outStream, " 14-NEGATIVE_RIGHTS");
	if(0x00001000                 & rights) fprintf(outStream, " 13-???");
	if(0x00000800                 & rights) fprintf(outStream, " 12-???");
	if(zAUTHORIZE_SECURE          & rights) fprintf(outStream, " 11-SECURE");
	if(zAUTHORIZE_SALVAGE         & rights) fprintf(outStream, " 10-SALVAGE");
	if(zAUTHORIZE_SUPERVISOR      & rights) fprintf(outStream, " 9-SUPERVISOR");
	if(zAUTHORIZE_MODIFY_METADATA & rights) fprintf(outStream, " 8-MODIFY_METADATA");
	if(zAUTHORIZE_SEE_FILES       & rights) fprintf(outStream, " 7-SEE_FILES");
	if(zAUTHORIZE_ACCESS_CONTROL  & rights) fprintf(outStream, " 6-ACCESS_CONTROL");
	if(zAUTHORIZE_DELETE_ENTRY    & rights) fprintf(outStream, " 5-DELETE_ENTRY");
	if(zAUTHORIZE_CREATE_ENTRY    & rights) fprintf(outStream, " 4-CREATE_ENTRY");
	if(0x00000004                 & rights) fprintf(outStream, " 3-???");
	if(zAUTHORIZE_WRITE_CONTENTS  & rights) fprintf(outStream, " 2-WRITE_CONTENTS");
	if(zAUTHORIZE_READ_CONTENTS   & rights) fprintf(outStream, " 1-READ_CONTENTS");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintModifyInfoMask(
		/* -O outStream      */ FILE    *outStream,
		/* I- modifyInfoMask */ uint32_t modifyInfoMask
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " modifyInfoMask[0x%08X", modifyInfoMask);

	if(0x80000000                    & modifyInfoMask) fprintf(outStream, " 32-???");
	if(0x40000000                    & modifyInfoMask) fprintf(outStream, " 31-???");
	if(0x20000000                    & modifyInfoMask) fprintf(outStream, " 30-???");
	if(zMOD_ALL_TRUSTEES             & modifyInfoMask) fprintf(outStream, " 29-ALL_TRUSTEES");
	if(zMOD_INH_RIGHTS_MASK          & modifyInfoMask) fprintf(outStream, " 28-INH_RIGHTS_MASK");
	if(zMOD_READ_AHEAD_BLOCKS        & modifyInfoMask) fprintf(outStream, " 27-READ_AHEAD_BLOCKS");
	if(zMOD_DIR_QUOTA                & modifyInfoMask) fprintf(outStream, " 26-DIR_QUOTA");
	if(zMOD_VOL_QUOTA                & modifyInfoMask) fprintf(outStream, " 25-VOL_QUOTA");
	if(zMOD_VOL_DATA_SHREDDING_COUNT & modifyInfoMask) fprintf(outStream, " 24-VOL_DATA_SHREDDING_COUNT");
	if(zMOD_POOL_NDS_OBJECT_ID       & modifyInfoMask) fprintf(outStream, " 23-POOL_NDS_OBJECT_ID");
	if(zMOD_POOL_ATTRIBUTES          & modifyInfoMask) fprintf(outStream, " 22-POOL_ATTRIBUTES");
	if(zMOD_VOL_HIGH_WATER_MARK      & modifyInfoMask) fprintf(outStream, " 21-VOL_HIGH_WATER_MARK");
	if(zMOD_VOL_LOW_WATER_MARK       & modifyInfoMask) fprintf(outStream, " 20-VOL_LOW_WATER_MARK");
	if(zMOD_VOL_MAX_KEEP_SECONDS     & modifyInfoMask) fprintf(outStream, " 19-VOL_MAX_KEEP_SECONDS");
	if(zMOD_VOL_MIN_KEEP_SECONDS     & modifyInfoMask) fprintf(outStream, " 18-VOL_MIN_KEEP_SECONDS");
	if(zMOD_VOL_NDS_OBJECT_ID        & modifyInfoMask) fprintf(outStream, " 17-VOL_NDS_OBJECT_ID");
	if(zMOD_VOL_ATTRIBUTES           & modifyInfoMask) fprintf(outStream, " 16-VOL_ATTRIBUTES");
	if(zMOD_EXTATTR_FLAGS            & modifyInfoMask) fprintf(outStream, " 15-EXTATTR_FLAGS");
	if(zMOD_UNIX_METADATA            & modifyInfoMask) fprintf(outStream, " 14-UNIX_METADATA");
	if(zMOD_MAC_METADATA             & modifyInfoMask) fprintf(outStream, " 13-MAC_METADATA");
	if(zMOD_DELETED_INFO             & modifyInfoMask) fprintf(outStream, " 12-DELETED_INFO");
	if(zMOD_PRIMARY_NAMESPACE        & modifyInfoMask) fprintf(outStream, " 11-PRIMARY_NAMESPACE");
	if(zMOD_METADATA_MODIFIER_ID     & modifyInfoMask) fprintf(outStream, " 10-METADATA_MODIFIER_ID");
	if(zMOD_MODIFIER_ID              & modifyInfoMask) fprintf(outStream, " 9-MODIFIER_ID");
	if(zMOD_ARCHIVER_ID              & modifyInfoMask) fprintf(outStream, " 8-ARCHIVER_ID");
	if(zMOD_OWNER_ID                 & modifyInfoMask) fprintf(outStream, " 7-OWNER_ID");
	if(zMOD_METADATA_MODIFIED_TIME   & modifyInfoMask) fprintf(outStream, " 6-METADATA_MODIFIED_TIME");
	if(zMOD_ACCESSED_TIME            & modifyInfoMask) fprintf(outStream, " 5-ACCESSED_TIME");
	if(zMOD_MODIFIED_TIME            & modifyInfoMask) fprintf(outStream, " 4-MODIFIED_TIME");
	if(zMOD_ARCHIVED_TIME            & modifyInfoMask) fprintf(outStream, " 3-ARCHIVED_TIME");
	if(zMOD_CREATED_TIME             & modifyInfoMask) fprintf(outStream, " 2-CREATED_TIME");
	if(zMOD_FILE_ATTRIBUTES          & modifyInfoMask) fprintf(outStream, " 1-FILE_ATTRIBUTES");

	fprintf(outStream, "]");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintBooleanElement(
		/* -O outStream */ FILE       *outStream,
		/* I- label     */ const char *label,
		/* I- value     */ uint32_t    value
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s[%d %s]", label?label:"???", value, value?"TRUE":"FALSE");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintUid(
		/* -O outStream */ FILE       *outStream,
		/* I- label     */ const char *label,
		/* I- uid       */ uid_t       uid
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char strings[_SC_GETPW_R_SIZE_MAX];
	char *name=NULL;
	struct passwd *pwBufP = NULL;
	struct passwd pwBuf =
		{
		.pw_name = NULL,
		.pw_passwd = NULL,
		.pw_uid = 0,
		.pw_gid = 0,
		.pw_gecos = NULL,
		.pw_dir = NULL,
		.pw_shell = NULL
		};

	if(0 == uid)
		name="root";
	else
		{
		if(0 == getpwuid_r(
			/* I- uid    */ uid,
			/* I- pwbuf  */ &pwBuf,
			/* I- buf    */ strings,
			/* I- buflen */ sizeof(name),
			/* -O pwbufp */ &pwBufP
			));
			name=pwBuf.pw_name;
		}

	fprintf(outStream, " %s[%d%s%s%s]",
			label ? label : "uid",
			uid,
			name ? " \"" : "",
			name ? name : "",
			name ? "\"" : ""
			);

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_STD_PrintGid(
		/* -O outStream */ FILE       *outStream,
		/* I- label     */ const char *label,
		/* I- gid       */ gid_t       gid
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char *name=NULL;

	if(0 == gid)
		name="root";

	fprintf(outStream, " %s[%d%s%s%s]",
			label ? label : "gid",
			gid,
			name ? " \"" : "",
			name ? name : "",
			name ? "\"" : ""
			);

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintNcpRecord
**                   VDUMP_OUT_STD_PrintNebRecord
**                   VDUMP_OUT_STD_PrintVigilRecord
*/
int VDUMP_OUT_STD_PrintRecordElement(
		/* -O outStream     */ FILE  *outStream,
		/* I- elementHeader */ char **element
		)
	{
	int  rCode=VIGIL_SUCCESS;
	VIGIL_ELEMENT_HDR_T *elementHeader = (VIGIL_ELEMENT_HDR_T *)*element;

	switch(elementHeader->type)
		{
		/*-----------------------------------------------------------------------
		** VIGIL_ELEMENT_PATH_T
		*/
		case VIGIL_ET_PATH:
			{
			VIGIL_ELEMENT_PATH_T *e=(void *)elementHeader;
			char *pathTypeStr;

			switch(e->pathType)
				{
				default:
				case VIGIL_ELEMENT_PATH_TYPE_ANONYMOUS:   pathTypeStr="path";   break;
				case VIGIL_ELEMENT_PATH_TYPE_TARGET:      pathTypeStr="target"; break;
				case VIGIL_ELEMENT_PATH_TYPE_SOURCE:      pathTypeStr="source"; break;
				case VIGIL_ELEMENT_PATH_TYPE_DESTINATION: pathTypeStr="destination"; break;
				}

			switch(e->nameSpace)
				{
				default:
				case VIGIL_ELEMENT_PATH_NS_ANONYMOUS:
					fprintf(outStream, " %s[??? namespace=%u]", pathTypeStr, e->nameSpace);
					break;

				case VIGIL_ELEMENT_PATH_NS_FAMILIAR:
					fprintf(outStream, " %s[%s]", pathTypeStr, e->data.familiar);
					break;

				case VIGIL_ELEMENT_PATH_NS_UNICODE:
					VDUMP_OUT_STD_Print_UnicodeStr(outStream, pathTypeStr, e->data.unicode);
					break;
				}

			break;
			}

		case VIGIL_ET_NEB_EXIT:
			{
			VIGIL_ELEMENT_NEB_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, " enterRetStatus[%d]", e->enterRetStatus);
			fprintf(outStream, " opRetCode[%d]", e->opRetCode);
			break;
			}

		case VIGIL_ET_NEB_CREATE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CREATE_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintFileType(outStream, e->fileType);
			VDUMP_OUT_STD_PrintFileAttributes(outStream, e->fileAttributes);
			VDUMP_OUT_STD_PrintCreateFlags(outStream, e->createFlags);
			VDUMP_OUT_STD_PrintRequestedRights(outStream, e->requestedRights);
			fprintf(outStream, " createAndOpen[0x%08X]", e->createAndOpen);
			break;
			}

		case VIGIL_ET_NEB_CREATE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CREATE_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, " retOpenCreateAction[0x%08X]", e->retOpenCreateAction);
			fprintf(outStream, " retKey[%llX]", e->retKey);
			fprintf(outStream, " retZid[%llX]", e->retZid);
			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "retVolID", &e->retVolID);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_OPEN_ENTER:
			{
			VIGIL_ELEMENT_NEB_OPEN_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRequestedRights(outStream, e->requestedRights);
			break;
			}

		case VIGIL_ET_NEB_OPEN_EXIT:
			{
			VIGIL_ELEMENT_NEB_OPEN_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, " retKey[%llX]", e->retKey);
			fprintf(outStream, " retZid[%llX]", e->retZid);
			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "retVolID", &e->retVolID);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_CLOSE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CLOSE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, " key[%llX]", e->key);
			fprintf(outStream, " fhState[0x%08X]", e->fhState);
			VDUMP_OUT_STD_PrintFhState(outStream, e->fhState);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_CLOSE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CLOSE_EXIT_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintBooleanElement(outStream, "fileDeleted", e->fileDeleted);;
			break;
			}

		case VIGIL_ET_NEB_RENAME_ENTER:
			{
			VIGIL_ELEMENT_NEB_RENAME_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRenameFlags(outStream, "renameFlags", e->renameFlags);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER:
			{
			VIGIL_ELEMENT_MODIFY_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintModifyInfoMask(outStream, e->modifyInfoMask);
			fprintf(outStream, " modifyTypeInfoMask[0x%08X]", e->modifyTypeInfoMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_FILE_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintFileAttributes(outStream, e->fileAttributes);
			fprintf(outStream, " fileAttributesModMask[0x%08X]", e->fileAttributesModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_CREATED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "created", &e->created);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "archived", &e->archived);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "modified", &e->modified);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ACCESSED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "accessed", &e->accessed);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "metaDataModified", &e->metaDataModified);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_OWNER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "owner", &e->owner);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "archiver", &e->archiver);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "modifier", &e->modifier);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "metaDataModifier", &e->metaDataModifier);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_PRIMARY_NAMESPACE:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T *e=(void *)elementHeader;

			fprintf(outStream, " primaryNameSpaceID[%d]", e->primaryNameSpaceID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DELETED_INFO:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintTime_64(outStream, "time", &e->time);
			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "id", &e->id);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MAC_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.finderInfo,
				/* I- bufLen    */ sizeof(e->info.finderInfo),
				/* I- label     */ "finderInfo"
				);

			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.proDOSInfo,
				/* I- bufLen    */ sizeof(e->info.proDOSInfo),
				/* I- label     */ "proDOSInfo"
				);

			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.filler,
				/* I- bufLen    */ sizeof(e->info.filler),
				/* I- label     */ "filler"
				);

			fprintf(outStream, " dirRightsMask[0x%08X]", e->info.dirRightsMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_UNIX_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T *e=(void *)elementHeader;

			fprintf(outStream, " fMode[0x%08X]",            e->info.fMode);
			fprintf(outStream, " rDev[0x%08X]",             e->info.rDev);
			fprintf(outStream, " myFlags[0x%08X]",          e->info.myFlags);
			fprintf(outStream, " nfsUID[0x%08X]",           e->info.nfsUID);
			fprintf(outStream, " nfsGID[0x%08X]",           e->info.nfsGID);
			fprintf(outStream, " nwUID[0x%08X]",            e->info.nwUID);
			fprintf(outStream, " nwGID[0x%08X]",            e->info.nwGID);
			fprintf(outStream, " nwEveryone[0x%08X]",       e->info.nwEveryone);
			fprintf(outStream, " nwUIDRights[0x%08X]",      e->info.nwUIDRights);
			fprintf(outStream, " nwGIDRights[0x%08X]",      e->info.nwGIDRights);
			fprintf(outStream, " nwEveryoneRights[0x%08X]", e->info.nwEveryoneRights);
			fprintf(outStream, " acsFlags[%08X]",           e->info.acsFlags);
			fprintf(outStream, " firstCreated[0x%08X]",     e->info.firstCreated);
			fprintf(outStream, " variableSize[0x%08X]",     e->info.variableSize);
			if(e->info.variableSize)
				{
				VDUMP_OUT_STD_PrintHex(
					/* -O outStream */ outStream,
					/* I- buf       */ e->info.variableData,
					/* I- bufLen    */ e->info.variableSize,
					/* I- label     */ "variableData"
					);
				}
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_EXTATTR_FLAGS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T *e=(void *)elementHeader;

			fprintf(outStream, " extAttrUserFlags[0x%08X]", e->extAttrUserFlags);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.features.enabled[0x%08X]", e->enabled);
			fprintf(outStream, " vol.features.enableModMask[0x%08X]", e->enableModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "vol.ndsObjectID", &e->ndsObjectID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MIN_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.salvage.minKeepSeconds[0x%08X]", e->minKeepSeconds);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MAX_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.salvage.maxKeepSeconds[0x%08X]", e->maxKeepSeconds);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_LOW_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.salvage.lowWaterMark[0x%08X]", e->lowWaterMark);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_HIGH_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.salvage.highWaterMark[0x%08X]", e->highWaterMark);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream, " pool.features.enabled[0x%08X]", e->enabled);
			fprintf(outStream, " pool.features.enableModMask[0x%08X]", e->enableModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "pool.ndsObjectID", &e->ndsObjectID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_DATA_SHREDDING_COUNT:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.dataShreddingCount[0x%08X]", e->dataShreddingCount);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream, " vol.totalSpaceQuota[%llu]", e->totalSpaceQuota);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DIR_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream, " dirQuota.quota[%lld]", e->quota);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_READ_AHEAD_BLOCKS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T *e=(void *)elementHeader;

			fprintf(outStream, " readAheadBlocks[]");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_INH_RIGHTS_MASK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRights(outStream, "inheritedRightsMask", e->inheritedRightsMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ALL_TRUSTEES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T *e=(void *)elementHeader;
			uint32_t entry;

			fprintf(outStream, " numEntries[%d]", e->numEntries);
			for(entry=0; entry < e->numEntries; ++entry)
				{
				fprintf(outStream, "{#%d", entry);
				VDUMP_OUT_STD_ElementPrint_Guid(outStream, "trustee", (VIGIL_GUID_T *)(&e->trusteeTable[entry].trustee));
				VDUMP_OUT_STD_PrintRights(outStream, "rights", e->trusteeTable[entry].rights);
				fprintf(outStream, "}");
				}
			break;
			}

		case VIGIL_ET_WHO__ANONYMOUS:
			{
			VIGIL_ELEMENT_WHO_ANONYMOUS_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->data,
				/* I- bufLen    */ e->hdr.length - sizeof(e->hdr),
				/* I- label     */ "anonymous"
				);
			break;
			}

		case VIGIL_ET_WHO__NCP_POSIX:
			{
			VIGIL_ELEMENT_WHO_NCP_POSIX_T *e=(void *)elementHeader;

			fprintf(outStream, " connID[%d]", e->connID);
			fprintf(outStream, " taskID[%d]", e->taskID);
			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "guid", &e->guid);
			break;
			}

		case VIGIL_ET_WHO__LINUX:
			{
			VIGIL_ELEMENT_WHO_LINUX_T *e=(void *)elementHeader;

			fprintf(outStream, " uid[%d]", e->uid);
			fprintf(outStream, " euid[%d]", e->euid);
			fprintf(outStream, " suid[%d]", e->suid);
			fprintf(outStream, " fsuid[%d]", e->fsuid);
			fprintf(outStream, " gid[%d]", e->gid);
			fprintf(outStream, " egid[%d]", e->egid);
			fprintf(outStream, " sgid[%d]", e->sgid);
			fprintf(outStream, " fsgid[%d]", e->fsgid);
			fprintf(outStream, " comm[%s]", e->comm);
			break;
			}

		case VIGIL_ET_NEB_ADDTRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_ADDTRUSTEE_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "trusteeID", &e->trusteeID);
			VDUMP_OUT_STD_PrintRights(outStream, "rights", e->rights);
			VDUMP_OUT_STD_PrintRights(outStream, "attributes", e->attributes);
			break;
			}

		case VIGIL_ET_NEB_REMOVETRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_REMOVETRUSTEE_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "trusteeID", &e->trusteeID);
			VDUMP_OUT_STD_PrintBooleanElement(outStream, "purgedFileFlag", e->purgedFileFlag);
			break;
			}

		case VIGIL_ET_NEB_SETINHERITEDRIGHTS_ENTER:
			{
			VIGIL_ELEMENT_NEB_SETINHERITEDRIGHTS_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRights(outStream, "inheritedRights", e->inheritedRights);
			fprintf(outStream, " authorizeFlag[0x%08X]",   e->authorizeFlag);
			break;
			}

		case VIGIL_ET_NAME:
			{
			VIGIL_ELEMENT_NAME_T *e=(void *)elementHeader;
			char *nameTypeStr;

			switch(e->nameType)
				{
				case VIGIL_ELEMENT_NAMETYPE_CLIENT:     nameTypeStr="Client"; break;
				case VIGIL_ELEMENT_NAMETYPE_CLIENTUSER: nameTypeStr="ClientUser"; break;
				default:                                nameTypeStr="(unknown)"; break;
				}

			fprintf(outStream, " %s[%s]", nameTypeStr, e->name);
			break;
			}

		case VIGIL_ET_NCP_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_NCP_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream, " searchAttributes[%d]", 0xFF & e->searchAttributes);
			fprintf(outStream, " desiredAccessRights[%d]", 0xFF & e->desiredAccessRights);
			fprintf(outStream, " linuxPath[%s]", e->linuxPath);
			fprintf(outStream, " status[%d]", e->status);
			/** Check e->hdr.length to verify that the values below were passed. **/
			if(0==e->status)
				{
				fprintf(outStream, " fileHandle[%02X%02X%02X%02X%02X%02X]",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				VDUMP_OUT_STD_PrintFileAttributes(outStream, e->fileAttributes);
				fprintf(outStream, " fileExecuteType[%02X]", 0xFF & e->fileExecuteType);
				fprintf(outStream, " fileLen[%lu]", e->fileLen);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "creationDate", e->creationDate);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "lastAccessDate", e->lastAccessDate);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "lastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_STD_PrintTime_MSDOS(outStream, "lastUpdateTime", e->lastUpdateTime);
				}
			break;
			}

		case VIGIL_ET_NCP_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_NCP_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream, " fileHandle[%02X%02X%02X%02X%02X%02X]",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5]
				);
			fprintf(outStream, " status[%d]", e->status);
			break;
			}

#if 0   // TODO:
		case VIGIL_ET_WHO__EDIR_GUID:
			break;
#endif

		case VIGIL_ET_NSS_CREATE:
			{
			VIGIL_ELEMENT_NSS_Create_T *e=(void *)elementHeader;

			fprintf(outStream, " key[0x%llX]", e->key);
			VDUMP_OUT_STD_PrintRequestedRights(outStream, e->requestedRights);
			VDUMP_OUT_STD_PrintCreateFlags(outStream, e->createFlags);
			VDUMP_OUT_STD_PrintBooleanElement(outStream, "createAndOpen", e->createAndOpen);
			VDUMP_OUT_STD_PrintOpenCreateActionFlags(outStream, e->retOpenCreateAction);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_OPEN:
			{
			VIGIL_ELEMENT_NSS_Open_T *e=(void *)elementHeader;

			fprintf(outStream, " key[0x%llX]", e->key);
			VDUMP_OUT_STD_PrintRequestedRights(outStream, e->requestedRights);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_CLOSE:
			{
			VIGIL_ELEMENT_NSS_Close_T *e=(void *)elementHeader;

			fprintf(outStream, " key[0x%llX]", e->key);
			VDUMP_OUT_STD_PrintFhState(outStream, e->fhState);
			VDUMP_OUT_STD_PrintBooleanElement(outStream, "fileDeleted", e->fileDeleted);
			VDUMP_OUT_STD_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_RENAME:
			{
			VIGIL_ELEMENT_NSS_Rename_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRenameFlags(outStream, "renameFlags", e->renameFlags);
			break;
			}

		case VIGIL_ET_NSS_LINK:
			{
			VIGIL_ELEMENT_NSS_Link_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintLinkFlags(outStream, "linkFlags", e->linkFlags);
			break;
			}

		case VIGIL_ET_NSS_MODIFYINFO:
			{
			VIGIL_ELEMENT_NSS_ModifyInfo_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintModifyInfoMask(outStream, e->modifyInfoMask);
			fprintf(outStream, " modifyTypeInfoMask[0x%llX]", e->modifyTypeInfoMask);
			break;
			}

		case VIGIL_ET_NSS_ADDTRUSTEE:
			{
			VIGIL_ELEMENT_NSS_AddTrustee_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "trusteeId", &e->trusteeID);
			VDUMP_OUT_STD_PrintRights(outStream, "rights", e->rights);
			VDUMP_OUT_STD_PrintRights(outStream, "previousRights", e->previousRights);
			VDUMP_OUT_STD_PrintRights(outStream, "attributes", e->attributes);
			break;
			}

		case VIGIL_ET_NSS_REMOVETRUSTEE:
			{
			VIGIL_ELEMENT_NSS_RemoveTrustee_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "trusteeId", &e->trusteeID);
			VDUMP_OUT_STD_PrintRights(outStream, "rights", e->rights);
			VDUMP_OUT_STD_PrintBooleanElement(outStream, "purgedFileFlag", e->purgedFileFlag);
			break;
			}

		case VIGIL_ET_NSS_SETINHERITEDRIGHTS:
			{
			VIGIL_ELEMENT_NSS_SetInheritedRights_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_PrintRights(outStream, "inheritedRights", e->inheritedRights);
			VDUMP_OUT_STD_PrintRights(outStream, "previousInheritedRights", e->previousInheritedRights);
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_CIFS_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream, " linuxPath[%s]", e->linuxPath);
			fprintf(outStream, " desiredAccessRights[%02X]", 0xFF & e->desiredAccessRights);
			fprintf(outStream, " status[%u]", 0xFF & e->status);
			if(0 == e->status)
				{
				fprintf(outStream, " fileHandle[%02X%02X%02X%02X%02X%02X]",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				VDUMP_OUT_STD_PrintFileAttributes(outStream, e->fileAttributes);
				fprintf(outStream, " fileExectueType[%02X]", 0xFF & e->fileExecuteType);
				fprintf(outStream, " fileLen[%u]", e->fileLen);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "creationDate", e->creationDate);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "lastAccessDate", e->lastAccessDate);
				VDUMP_OUT_STD_PrintDate_MSDOS(outStream, "lastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_STD_PrintTime_MSDOS(outStream, "lastUpdateTime", e->lastUpdateTime);
				}
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_CIFS_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream, " fileHandle[%02X%02X%02X%02X%02X%02X]",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5]
				);

			fprintf(outStream, " status[%u]", 0xFF & e->status);
			break;
			}

		case VIGIL_ET_PMD_NCP:
			{
			VIGIL_ELEMENT_PMD_NCP_T *e=(void *)elementHeader;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "guid", &e->guid);
			fprintf(outStream, " connID[%u]", e->connID);
			fprintf(outStream, " taskID[%u]", e->taskID);

			break;
			}

		case VIGIL_ET_PMD_CIFS:
			{
			VIGIL_ELEMENT_PMD_CIFS_T *e=(void *)elementHeader;
			char *elementTmp = (char *)&e->netAddr;

			VDUMP_OUT_STD_ElementPrint_Guid(outStream, "guid", &e->guid);
			fprintf(outStream, " connID[%u]", e->connID);

			rCode=VDUMP_OUT_STD_PrintRecordElement(
				/* -O outStream     */ outStream,
				/* I- elementHeader */ &elementTmp
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VIGIL_ET_PMD_CIFS:VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
				rCode=VIGIL_SUCCESS;
//				goto END_ERR;
				}

			break;
			}

		case VIGIL_ET_NET_ADDR_IPv4:
			{
			VIGIL_ELEMENT_NET_ADDR_T *e=(void *)elementHeader;

			fprintf(outStream, " netAddr_IPv4[%02X.%02X.%02X.%02X]",
				0xFF & e->addr.octet[0],
				0xFF & e->addr.octet[1],
				0xFF & e->addr.octet[2],
				0xFF & e->addr.octet[3]
				);

			break;
			}

		case VIGIL_ET_NET_ADDR_IPv6:
			{
			VIGIL_ELEMENT_NET_ADDR_T *e=(void *)elementHeader;

			fprintf(outStream, " netAddr_IPv6[%4X:%4X:%4X:%4X:%4X:%4X:%4X:%4X]",
				0xFFFF & e->addr.hextet[0],
				0xFFFF & e->addr.hextet[1],
				0xFFFF & e->addr.hextet[2],
				0xFFFF & e->addr.hextet[3],
				0xFFFF & e->addr.hextet[4],
				0xFFFF & e->addr.hextet[5],
				0xFFFF & e->addr.hextet[6],
				0xFFFF & e->addr.hextet[7]
				);

			break;
			}

		default:
			fprintf(outStream, " Unknown element type[0x%04X]", elementHeader->type);
			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ (void *)elementHeader,
				/* I- bufLen    */ elementHeader->length,
				/* I- label     */ "element"
				);
			break;
		}

	(*element) += elementHeader->length;

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintNcpRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NCP_T *ncp
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = ncp->data;

	fprintf(outStream, " NCP");

	switch(ncp->event)
		{
		case VIGIL_E_NCP_OPEN:
			fprintf(outStream, " OPEN");
			break;

		case VIGIL_E_NCP_CLOSE:
			fprintf(outStream, " CLOSE");
			break;

		default:
			fprintf(outStream, " Unknown NCP event[%d]\n", ncp->event);
			break;
		}

	for(curElement=0; curElement < ncp->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_STD_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintCifsRecord(
		/* -O outStream */ FILE                   *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_CIFS_T *cifs
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = cifs->data;

	fprintf(outStream, " CIFS");

	switch(cifs->event)
		{
		case VIGIL_E_CIFS_OPEN:
			fprintf(outStream, " OPEN");
			break;

		case VIGIL_E_CIFS_CLOSE:
			fprintf(outStream, " CLOSE");
			break;

		default:
			fprintf(outStream, " Unknown CIFS event[%d]\n", cifs->event);
			break;
		}

	for(curElement=0; curElement < cifs->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_STD_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintNebRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NEB_T *neb
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = neb->data;

	fprintf(outStream, " NEB  #%u", neb->enterExitID);
	switch(neb->event)
		{
		case VIGIL_E_NEB_DELETE_ENTER:
			fprintf(outStream, " DELETE_ENTER");
			break;

		case VIGIL_E_NEB_DELETE_EXIT:
			fprintf(outStream, " DELETE_EXIT");
			break;

		case VIGIL_E_NEB_CREATE_ENTER:
			fprintf(outStream, " CREATE_ENTER");
			break;

		case VIGIL_E_NEB_CREATE_EXIT:
			fprintf(outStream, " CREATE_EXIT");
			break;

		case VIGIL_E_NEB_OPEN_ENTER:
			fprintf(outStream, " OPEN_ENTER");
			break;

		case VIGIL_E_NEB_OPEN_EXIT:
			fprintf(outStream, " OPEN_EXIT");
			break;

		case VIGIL_E_NEB_CLOSE_ENTER:
			fprintf(outStream, " CLOSE_ENTER");
			break;

		case VIGIL_E_NEB_CLOSE_EXIT:
			fprintf(outStream, " CLOSE_EXIT");
			break;

		case VIGIL_E_NEB_RENAME_ENTER:
			fprintf(outStream, " RENAME_ENTER");
			break;

		case VIGIL_E_NEB_RENAME_EXIT:
			fprintf(outStream, " RENAME_EXIT");
			break;

		case VIGIL_E_NEB_MODIFY_ENTER:
			fprintf(outStream, " MODIFY_ENTER");
			break;

		case VIGIL_E_NEB_MODIFY_EXIT:
			fprintf(outStream, " MODIFY_EXIT");
			break;

		case VIGIL_E_NEB_ADDTRUSTEE_ENTER:
			fprintf(outStream, " TRUSTEE_ADD_ENTER ");
			break;

		case VIGIL_E_NEB_ADDTRUSTEE_EXIT:
			fprintf(outStream, " TRUSTEE_ADD_EXIT");
			break;

		case VIGIL_E_NEB_REMOVETRUSTEE_ENTER:
			fprintf(outStream, " TRUSTEE_REMOVE_ENTER");
			break;

		case VIGIL_E_NEB_REMOVETRUSTEE_EXIT:
			fprintf(outStream, " TRUSTEE_REMOVE_EXIT");
			break;

		case VIGIL_E_NEB_SETINHERITEDRIGHTS_ENTER:
			fprintf(outStream, " INHERITED_RIGHTS_SET_ENTER");
			break;

		case VIGIL_E_NEB_SETINHERITEDRIGHTS_EXIT:
			fprintf(outStream, " INHERITED_RIGHTS_SET_EXIT");
			break;

		default:
			fprintf(outStream, "  Unknown NEB event[%d]\n", neb->event);
			break;
		}

	for(curElement=0; curElement < neb->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_STD_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintNssRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NSS_T *nss
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = nss->data;

	/*---------------------------------------------------------------------------
	** NSS
	*/
	fprintf(outStream, " NSS");

	/*---------------------------------------------------------------------------
	** Event
	*/
	switch(nss->event)
		{
		case VIGIL_E_NSS_DELETE:
			fprintf(outStream, " DELETE");
			break;

		case VIGIL_E_NSS_CREATE:
			fprintf(outStream, " CREATE");
			break;

		case VIGIL_E_NSS_OPEN:
			fprintf(outStream, " OPEN");
			break;

		case VIGIL_E_NSS_CLOSE:
			fprintf(outStream, " CLOSE");
			break;

		case VIGIL_E_NSS_RENAME:
			fprintf(outStream, " RENAME");
			break;

		case VIGIL_E_NSS_LINK:
			fprintf(outStream, " LINK");
			break;

		case VIGIL_E_NSS_MODIFY:
			fprintf(outStream, " MODIFY");
			break;

		case VIGIL_E_NSS_ADDTRUSTEE:
			fprintf(outStream, " TRUSTEE_ADD");
			break;

		case VIGIL_E_NSS_REMOVETRUSTEE:
			fprintf(outStream, " TRUSTEE_REMOVE");
			break;

		case VIGIL_E_NSS_SETINHERITEDRIGHTS:
			fprintf(outStream, " INHERITED_RIGHTS_SET");
			break;

		default:
			fprintf(outStream, "  Unknown NSS event[%d]\n", nss->event);
			break;
		}

	/*---------------------------------------------------------------------------
	** Event metadata
	*/
	fprintf(outStream,
		" TaskID[%lu]"
		" Zid[%llX]"
		" ParentZid[%llX]"
		"",
		nss->taskID,
		nss->zid,
		nss->parentZid
		);

	VDUMP_OUT_STD_PrintFileType(outStream, nss->fileType);
	VDUMP_OUT_STD_PrintFileAttributes(outStream, nss->fileAttributes);
	fprintf(outStream, " OpRetCode[%d]", nss->opRetCode);

	rCode=VDUMP_OUT_STD_ElementPrint_Guid(
		/* -O outStream */ outStream,
		/* I- label     */ "VolID",
		/* I- ndsGuid   */ &nss->volID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_ElementPrint_Guid(VolID) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_STD_ElementPrint_Guid(
		/* -O outStream */ outStream,
		/* I- label     */ "UserID",
		/* I- ndsGuid   */ &nss->userID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_ElementPrint_Guid(UserID) reports: %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Linux metadata
	*/
	VDUMP_OUT_STD_PrintUid(outStream, "uid",   nss->uid);
	VDUMP_OUT_STD_PrintUid(outStream, "euid",  nss->euid);
	VDUMP_OUT_STD_PrintUid(outStream, "suid",  nss->suid);
	VDUMP_OUT_STD_PrintUid(outStream, "fsuid", nss->fsuid);

	VDUMP_OUT_STD_PrintGid(outStream, "gid",   nss->gid);
	VDUMP_OUT_STD_PrintGid(outStream, "egid",  nss->egid);
	VDUMP_OUT_STD_PrintGid(outStream, "sgid",  nss->sgid);
	VDUMP_OUT_STD_PrintGid(outStream, "fsgid", nss->fsgid);

	fprintf(outStream, " comm[%s]",  nss->comm);

	/*---------------------------------------------------------------------------
	** Elements
	*/
	for(curElement=0; curElement < nss->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_STD_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_STD_PrintRecord()
*/
int VDUMP_OUT_STD_PrintVigilRecord(
		/* -O outStream */ FILE                    *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_VIGIL_T *rec
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = rec->data;

	fprintf(outStream, " VIGIL");
	switch(rec->event)
		{
		case VIGIL_E_VIGIL_ROLL:
			{
			fprintf(outStream, " Roll:");
			break;
			}

		case VIGIL_E_VIGIL_START:
			fprintf(outStream, " Start: vigil.ko");
			break;

		case VIGIL_E_VIGIL_STOP:
			fprintf(outStream, " Stop: vigil.ko");
			break;

		case VIGIL_E_VIGIL_NEB_START:
			fprintf(outStream, " Start: vigil_neb.ko");
			break;

		case VIGIL_E_VIGIL_NEB_STOP:
			fprintf(outStream, " Stop: vigil_neb.ko");
			break;

		case VIGIL_E_VIGIL_NCP_START:
			fprintf(outStream, " Start: vigil_ncp.ko");
			break;

		case VIGIL_E_VIGIL_NCP_STOP:
			fprintf(outStream, " Stop: vigil_ncp.ko");
			break;

		case VIGIL_E_VIGIL_CLIENT_START:
			fprintf(outStream, " Start: VigilAuditingClient");
			break;

		case VIGIL_E_VIGIL_CLIENT_STOP:
			fprintf(outStream, " Stop: VigilAuditingClient");
			break;

		case VIGIL_E_VIGIL_USER_START:
			fprintf(outStream, " Start: VigilAuditingClientUser");
			break;

		case VIGIL_E_VIGIL_USER_STOP:
			fprintf(outStream, " Stop: VigilAuditingClientUser");
			break;

		default:
			fprintf(outStream, " Unknown vigil event[%d] length[%d] dataElements[%hd]\n", rec->event, rec->hdr.length, rec->dataElements);
			break;
		}

	for(curElement=0; curElement < rec->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_STD_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_STD_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_MAIN_ProcessAuditLog()
*/
int VDUMP_OUT_STD_PrintRecord(
		/* -O outStream */ FILE              *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_T *rec,
		/* I- recLen    */ size_t             recLen
		)
	{
	int  rCode=VIGIL_SUCCESS;

	rCode=LIBVIGIL_PARSE_RecValidate(rec, recLen);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_RecValidate() reports: %d\n", rCode);
		goto END_ERR;
		}

	fprintf(outStream,
		"%-10llu"
		" pid[%u]",
		rec->hdr.recNo,
		rec->hdr.pid
		);

	rCode=VDUMP_OUT_STD_PrintTimeStamp(
		/* -O outStream */ outStream,
		/* I- record    */ &rec->hdr.time
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_STD_PrintTimeStamp() reports: %d\n", rCode);
		goto END_ERR;
		}

	switch(rec->hdr.type)
		{
		case VIGIL_T_VIGIL:
			rCode=VDUMP_OUT_STD_PrintVigilRecord(
				/* -O outStream */ outStream,
				/* I- vigilRec  */ &rec->vigil
				);
			switch(rCode)
				{
				case VIGIL_SUCCESS:
					break;

				case ESTALE:
					break;

				default:
					LIBVIGIL_ERR("VDUMP_OUT_STD_PrintVigilRecord() reports: %d\n", rCode);
					goto END_ERR;
				}
			break;

		case VIGIL_T_NEB:
			rCode=VDUMP_OUT_STD_PrintNebRecord(
				/* -O outStream */ outStream,
				/* I- nebRec    */ &rec->neb
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_STD_PrintNebRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NCP:
			rCode=VDUMP_OUT_STD_PrintNcpRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->ncp
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_STD_PrintNcpRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_CIFS:
			rCode=VDUMP_OUT_STD_PrintCifsRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->cifs
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_STD_PrintCifsRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NSS:
			rCode=VDUMP_OUT_STD_PrintNssRecord(
				/* -O outStream */ outStream,
				/* I- nssRec    */ &rec->nss
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_STD_PrintNssRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		default:
			fprintf(outStream, " Unknown record type[%d]\n", rec->hdr.type);
			VDUMP_OUT_STD_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf    */ (void *)rec,
				/* I- bufLen */ rec->hdr.length,
				/* I- label  */ "rec"
				);
			break;
		}

	fprintf(outStream, "\n");

END_ERR:

	return(rCode);
	}






