#ifndef __INCLUDE_VIGIL_H__
	#define __INCLUDE_VIGIL_H__

/*******************************************************************************
** Compiler setup.
*/
#ifdef __KERNEL__
	#include <linux/module.h>
	#include <linux/types.h>
#else
	#include <stdint.h>
	#include <time.h>
	#include <sys/time.h>
	#include <unistd.h>
#endif

#include <linux/limits.h>

/*******************************************************************************
** Define macros.
*/
	/*---------------------------------------------------------------------------
	** Vigil specific standard values.
	*/
	/* Boolean values */
	#define VIGIL_TRUE                           (-1)
	#define VIGIL_FALSE                          (0)
	
	/* Return codes */
	#define VIGIL_SUCCESS                        0

	/*---------------------------------------------------------------------------
	** Misc. global definitions.
	*/
	#define VIGIL_DIR_NAME_AUDIT                 "audit"
	#define VIGIL_DIR_NAME                       "vigil"
	#define VIGIL_DIR_NAME_NCP                   "ncp"
	#define VIGIL_SIGN                           "\nVIGIL"
	#define VIGIL_CONTROL_FILE_NAME              "CONTROL"
	#define VIGIL_CONTROL_FILE_PATH              "/sys/audit/vigil/CONTROL"
	#define VIGIL_CLIENT_DIR_PREFIX              "CLIENT_"
	#define VIGIL_CLIENT_CONTROL_PATH_PREFIX     "/sys/audit/vigil/CLIENT_"
	#define VIGIL_USER_DIR_PREFIX                "USER_"
	#define VIGIL_NCP_CONTROL_FILE_PATH          "/sys/audit/vigil/ncp/CONTROL"
	#define VIGIL_CIFS_CONTROL_FILE_PATH         "/sys/audit/vigil/cifs/CONTROL"
	#define VIGIL_SYSFS_DIR_PREFIX               "/sys/audit/vigil"
	#define VIGIL_SYS_FILENAME_CPUQ              "cpu_queue_"
	#define VIGIL_FILTER_FILENAME_PREFIX         "FILTER_"
	#define VIGIL_DIR_NAME_CIFS                  "cifs"
	#define VIGIL_CLIENT_CONF_PATH               "/etc/audit/vigil/client"
	#define VIGIL_PERSIST_CONF_FILE_PREFIX       "CLIENT_"
	#define VIGIL_PERSIST_CONF_FILE_POSTFIX      ".conf"

	/*---------------------------------------------------------------------------
	** Command tag (suffix) used by user-space to request actions in vigil.ko.
	*/
	#define VIGIL_SYS_COMMAND_OPEN           "OPEN "
	#define VIGIL_SYS_COMMAND_CLOSE          "CLOSE "
	#define VIGIL_SYS_COMMAND_NCPLOCALAUDIT  "ncp"
	#define VIGIL_SYS_COMMAND_CLIENTOPEN     "CLIENT_OPEN"
	#define VIGIL_SYS_COMMAND_USEROPEN       "USER_OPEN"
	#define VIGIL_SYS_COMMAND_FILTERIMPORT   "FILTER_IMPORT"
	#define VIGIL_SYS_COMMAND_CIFSLOCALAUDIT "cifs"

	/*---------------------------------------------------------------------------
	** Client command key labels.
	*/
	#define VIGIL_CLIENT_K_NAME            "ClientName"
	#define VIGIL_CLIENT_K_KEY             "ClientKey"
	#define VIGIL_CLIENT_K_FLAGS           "ClientFlags"
	#define VIGIL_CLIENT_K_DIRPATH         "ClientDirectory"
	#define VIGIL_CLIENT_K_SIZE_MAX        "ClientSizeMax"
	#define VIGIL_CLIENT_K_RECS_LIMIT      "ClientRecordsLimit"
	#define VIGIL_CLIENT_K_EVENTMASK_VIGIL "ClientEventMask_Vigil"
	#define VIGIL_CLIENT_K_EVENTMASK_NEB   "ClientEventMask_Neb"
	#define VIGIL_CLIENT_K_EVENTMASK_NCP   "ClientEventMask_Ncp"

	/*---------------------------------------------------------------------------
	** User command key labels.
	*/
	#define VIGIL_USER_K_NAME       "UserName"
	#define VIGIL_USER_K_KEY        "UserKey"
	#define VIGIL_USER_K_FLAGS      "UserFlags"
	#define VIGIL_USER_K_SIGIO_PID  "UserSigioPid"

	/*---------------------------------------------------------------------------
	** Filter command key labels.
	*/
	#define VIGIL_FILTER_K_PATH     "FilterFilePath"

	/*---------------------------------------------------------------------------
	** (bit) flags for:
	**    VIGIL_CLIENT_NODE.flags
	*/
	#define VIGIL_CLIENT_F_NONE               0x00000000
	#define VIGIL_CLIENT_F_ROLLOUTPUTFILE     0x00000001   /* Roll output file when reaches it limited size or rec count. */
	#define VIGIL_CLIENT_F_EXCLUDE_SIGIO_PIDS 0x00000002   /* Do not audit registered SIGIO PIDs for this client. */
	#define VIGIL_CLIENT_F_SHARE              0x00000004   /* Allow multiple user-space clients to share the same auditing client. */
	#define VIGIL_CLIENT_F_KEEP               0x00000008   /* Override client distruction when the last user closes. */

	/** Partially implemented flags: **/
	#define VIGIL_CLIENT_F_PERSIST            0x00000010   /* Saves client parameters to .conf file so that client can be automatically reloaded if the audit engine is restarted. */

	#define VIGIL_USER_F_NONE                 0x00000000

	/*---------------------------------------------------------------------------
	** Generic states for various structures:
	*/
	#define VIGIL_STATE_HALTED      0x00
	#define VIGIL_STATE_RUNNING     0x01
	#define VIGIL_STATE_CLOSING     0x02
	#define VIGIL_STATE_CLOSED      0x04

	/** Client internal states (per bit) **/
	#define VIGIL_CLIENT_STATE_INITIAL      0x00000000
	#define VIGIL_CLIENT_STATE_SIGIO_NEEDED 0x00000001

	/*---------------------------------------------------------------------------
	** Client file tags used to identify data passed by the client control file
	** to user-space.
	*/
	#define VIGIL_CLIENT_TAG_STATE           " State................: "
	#define VIGIL_CLIENT_TAG_FLAGS           " Flags................: "
	#define VIGIL_CLIENT_TAG_NAME            " ClientName...........: "
	#define VIGIL_CLIENT_TAG_OWNERPID        " ClientOwnerPid.......: "
	#define VIGIL_CLIENT_TAG_DIRPATH         " OutputDirectory......: "
	#define VIGIL_CLIENT_TAG_FILEPATHINITIAL " OutputFilePathInitial: "
	#define VIGIL_CLIENT_TAG_FILEPATH        " OutputFilePath.......: "
	#define VIGIL_CLIENT_TAG_OUTPUTSIZE      " OutputSize...........: "
	#define VIGIL_CLIENT_TAG_OUTPUTSIZE_MAX  " OutputSizeMax........: "
	#define VIGIL_CLIENT_TAG_OUTPUTRECS      " OutputRecordsLimit...: "
	#define VIGIL_CLIENT_TAG_OUTPUTRECS_MAX  " OutputRecordsTotal...: "

	/*---------------------------------------------------------------------------
	** User file tags used to identify data passed by the user control file
	** to user-space.
	*/
	#define VIGIL_USER_TAG_STATE            " State.......: "
	#define VIGIL_USER_TAG_FLAGS            " Flags.......: "
	#define VIGIL_USER_TAG_NAME             " userName....: "
	#define VIGIL_USER_TAG_OWNERPID         " userOwnerPid: "

	#define VIGIL_FILTER_TAG_OFFSET         " filterOffset: "
	#define VIGIL_FILTER_TAG_LENGTH         " filterLength: "
	#define VIGIL_FILTER_TAG_VALUE          " value.......: "
	#define VIGIL_FILTER_TAG_MASK           " mask........: "

	/*---------------------------------------------------------------------------
	** Vigil PERSIST xml .conf file values.
	*/
	#define VIGIL_PERSIST_TAG_CLIENT        "VIGIL_CLIENT"
	#define VIGIL_PERSIST_ATTR_TYPE         "Type"
	#define VIGIL_PERSIST_ATTR_NAME         "Name"
	#define VIGIL_PERSIST_ATTR_KEY          "Key"
	#define VIGIL_PERSIST_ATTR_FLAGS        "Flags"
	#define VIGIL_PERSIST_ATTR_DIRPATH      "OutputDirPath"
	#define VIGIL_PERSIST_ATTR_FILEPATHINIT "OutputFilePathInitial"
	#define VIGIL_PERSIST_ATTR_FILEPATHCURR "OutputFilePath"
	#define VIGIL_PERSIST_ATTR_FILESIZECUR  "OutputSize"
	#define VIGIL_PERSIST_ATTR_FILESIZEMAX  "OutputSizeMax"
	#define VIGIL_PERSIST_ATTR_FILERECSCUR  "OutputRecordsTotal"
	#define VIGIL_PERSIST_ATTR_FILERECSMAX  "OutputRecordsLimit"
	#define VIGIL_PERSIST_ATTR_FILERECCUR   "RecNo"

	#define VIGIL_PERSIST_TAG_EVENTMASK     "EVENT_MASK"
	#define VIGIL_PERSIST_ATTR_MASKNO       "MaskNumber"
	#define VIGIL_PERSIST_ATTR_MASK         "Mask"

	#define VIGIL_PERSIST_TAG_FILTER        "FILTER"

	#define VIGIL_PERSIST_TAG_FILTER_DATA   "FILTER_DATA"
	#define VIGIL_PERSIST_ATTR_OFFSET       "Offset"
	#define VIGIL_PERSIST_ATTR_LENGTH       "Length"

	/*---------------------------------------------------------------------------
	** Vigil Archive Client values.
	*/
	#define VIGIL_ARCHIVE_CLIENT_NAME           "vigil"
	#define VIGIL_ARCHIVE_CLIENT_KEY            "clientKey"
	#define VIGIL_ARCHIVE_CLIENT_FLAGS          (VIGIL_CLIENT_F_ROLLOUTPUTFILE|VIGIL_CLIENT_F_EXCLUDE_SIGIO_PIDS|VIGIL_CLIENT_F_SHARE)
	#define VIGIL_ARCHIVE_CLIENT_DIRPATH        "/var/log/audit/vigil/archive"
	#define VIGIL_ARCHIVE_CLIENT_OUTPUTSIZEMAX   (2 * 1024 * 1024) /* 2MB */
	#define VIGIL_ARCHIVE_CLIENT_OUTPUTRECSLIMIT 0
	#define VIGIL_ARCHIVE_CLIENT_USERNAME        "archive"
	#define VIGIL_ARCHIVE_CLIENT_USERKEY         "userKey"
	#define VIGIL_ARCHIVE_CLIENT_USERFLAGS       VIGIL_USER_F_NONE
	#define VIGIL_ARCHIVE_CLIENT_OUTPUTSIGIOPID  0

	/*---------------------------------------------------------------------------
	** Audit record types.
	*/
	
	/* Audit record integer-types */
	#define VIGIL_T_VIGIL                        0   /* Internal vigil events. */
	#define VIGIL_T_NEB                          1   /* Novell Event Bus Events. */
	#define VIGIL_T_NCP                          2   /* NetWare Core Protocol Local Events. */
	#define VIGIL_T_NSS                          3   /* Novell Storage Services Events. */
	#define VIGIL_T_CIFS                         4   /* Common Internet File System Events */
	#define VIGIL_T__CNT                         5

	/* Generic bit-event definitions */
	#define VIGIL_E__BITS_NONE                   0x00000000
	#define VIGIL_E__BITS_ALL                    0xFFFFFFFF

	/* Vigil (internal) bit-events. */
	#define VIGIL_E_VIGIL_START                  0x00000001   /* vigil started */
	#define VIGIL_E_VIGIL_STOP                   0x00000002   /* vigil stopped */
	#define VIGIL_E_VIGIL_NEB_START              0x00000004   /* vigil_neb started */
	#define VIGIL_E_VIGIL_NEB_STOP               0x00000008   /* vigil_neb stopped */
	#define VIGIL_E_VIGIL_NCP_START              0x00000010   /* vigil_ncp started */
	#define VIGIL_E_VIGIL_NCP_STOP               0x00000020   /* vigil_ncp stopped */
	#define VIGIL_E_VIGIL_CLIENT_START           0x00000040   /* vigil client started. */
	#define VIGIL_E_VIGIL_CLIENT_STOP            0x00000080   /* vigil client stopped. */
	#define VIGIL_E_VIGIL_USER_START             0x00000100   /* vigil client/user started. */
	#define VIGIL_E_VIGIL_USER_STOP              0x00000200   /* vigil client/user stopped. */
	#define VIGIL_E_VIGIL_ROLL                   0x00000400   /* Indicates end-of-file, and specifies next file path. */
	#define VIGIL_E_VIGIL_NSS_START              0x00000800   /* vigil_nss started */
	#define VIGIL_E_VIGIL_NSS_STOP               0x00001000   /* vigil_nss stopped */
	#define VIGIL_E_VIGIL_CIFS_START             0x00002000   /* vigil_cifs started. */
	#define VIGIL_E_VIGIL_CIFS_STOP              0x00004000   /* vigil_cifs stopped. */

	/* Novell Event Buss(NEB) bit-events */
	#define VIGIL_E_NEB_DELETE_ENTER             0x00000001
	#define VIGIL_E_NEB_DELETE_EXIT              0x00000002
	#define VIGIL_E_NEB_CREATE_ENTER             0x00000004
	#define VIGIL_E_NEB_CREATE_EXIT              0x00000008
	#define VIGIL_E_NEB_OPEN_ENTER               0x00000010
	#define VIGIL_E_NEB_OPEN_EXIT                0x00000020
	#define VIGIL_E_NEB_CLOSE_ENTER              0x00000040
	#define VIGIL_E_NEB_CLOSE_EXIT               0x00000080
	#define VIGIL_E_NEB_RENAME_ENTER             0x00000100
	#define VIGIL_E_NEB_RENAME_EXIT              0x00000200
	#define VIGIL_E_NEB_MODIFY_ENTER             0x00000400
	#define VIGIL_E_NEB_MODIFY_EXIT              0x00000800
	#define VIGIL_E_NEB_ADDTRUSTEE_ENTER         0x00001000
	#define VIGIL_E_NEB_ADDTRUSTEE_EXIT          0x00002000
	#define VIGIL_E_NEB_REMOVETRUSTEE_ENTER      0x00004000
	#define VIGIL_E_NEB_REMOVETRUSTEE_EXIT       0x00008000
	#define VIGIL_E_NEB_SETINHERITEDRIGHTS_ENTER 0x00010000
	#define VIGIL_E_NEB_SETINHERITEDRIGHTS_EXIT  0x00020000
	#define VIGIL_E_NEB__CNT                     18
	
	/* NetWare Core Protocol(NCP) bit-events */
	#define VIGIL_E_NCP_OPEN                     0x00000010
	#define VIGIL_E_NCP_CLOSE                    0x00000020

	/* Novell Storage Services (NSS) bit-events */
	#define VIGIL_E_NSS_DELETE                   0x00000001
	#define VIGIL_E_NSS_CREATE                   0x00000002
	#define VIGIL_E_NSS_OPEN                     0x00000004
	#define VIGIL_E_NSS_CLOSE                    0x00000008
	#define VIGIL_E_NSS_RENAME                   0x00000010
	#define VIGIL_E_NSS_LINK                     0x00000020
	#define VIGIL_E_NSS_MODIFY                   0x00000040
	#define VIGIL_E_NSS_ADDTRUSTEE               0x00000080
	#define VIGIL_E_NSS_REMOVETRUSTEE            0x00000100
	#define VIGIL_E_NSS_SETINHERITEDRIGHTS       0x00000200
	#define VIGIL_E_NSS__CNT                     10

	/* Common Internet File System (CIFS) services bit-events */
	#define VIGIL_E_CIFS_OPEN                    0x00000001
	#define VIGIL_E_CIFS_CLOSE                   0x00000002

	/*---------------------------------------------------------------------------
	** Event data element structure types.
	*/
	//deprecated #define VIGIL_ET_WHO                                0x0001
	#define VIGIL_ET_PATH                                            0x0002
	#define VIGIL_ET_NEB_EXIT                                        0x0003
	#define VIGIL_ET_NEB_CREATE_ENTER                                0x0004
	#define VIGIL_ET_NEB_CREATE_EXIT                                 0x0005
	#define VIGIL_ET_NEB_OPEN_ENTER                                  0x0006
	#define VIGIL_ET_NEB_OPEN_EXIT                                   0x0007
	#define VIGIL_ET_NEB_CLOSE_ENTER                                 0x0008
	#define VIGIL_ET_NEB_CLOSE_EXIT                                  0x0009
	#define VIGIL_ET_NEB_RENAME_ENTER                                0x000A
	#define VIGIL_ET_NEB_MODIFY_ENTER                                0x000B
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_FILE_ATTRIBUTES          0x000C
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_CREATED_TIME             0x000D
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVED_TIME            0x000E
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIED_TIME            0x000F
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ACCESSED_TIME            0x0010
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIED_TIME   0x0011
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_OWNER_ID                 0x0012
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVER_ID              0x0013
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIER_ID              0x0014
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIER_ID     0x0015
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_PRIMARY_NAMESPACE        0x0016
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DELETED_INFO             0x0017
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MAC_METADATA             0x0018
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_UNIX_METADATA            0x0019
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_EXTATTR_FLAGS            0x001A
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_ATTRIBUTES           0x001B
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_NDS_OBJECT_ID        0x001C
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MIN_KEEP_SECONDS     0x001D
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MAX_KEEP_SECONDS     0x001E
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_LOW_WATER_MARK       0x001F
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_HIGH_WATER_MARK      0x0020
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_ATTRIBUTES          0x0021
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_NDS_OBJECT_ID       0x0022
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_DATA_SHREDDING_COUNT 0x0023
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_QUOTA                0x0024
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DIR_QUOTA                0x0025
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_READ_AHEAD_BLOCKS        0x0026
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_INH_RIGHTS_MASK          0x0027
	#define VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ALL_TRUSTEES             0x0028
	#define VIGIL_ET_WHO__ANONYMOUS                                  0x0029
	#define VIGIL_ET_WHO__NCP_POSIX                                  0x002A
	//#define VIGIL_ET_WHO__NCP_LOCAL                                0x002B
	#define VIGIL_ET_WHO__LINUX                                      0x0030
	#define VIGIL_ET_NEB_ADDTRUSTEE_ENTER                            0x0031
	#define VIGIL_ET_NEB_REMOVETRUSTEE_ENTER                         0x0032
	#define VIGIL_ET_NEB_SETINHERITEDRIGHTS_ENTER                    0x0033
	#define VIGIL_ET_NAME                                            0x0034
	#define VIGIL_ET_NCP_LOCAL__OPENFILE                             0x0035
	#define VIGIL_ET_NCP_LOCAL__CLOSEFILE                            0x0036

	#define VIGIL_ET_WHO__EDIR_GUID                                  0x0037

	#define VIGIL_ET_NSS_CREATE                                      0x0038
	#define VIGIL_ET_NSS_OPEN                                        0x0039
	#define VIGIL_ET_NSS_CLOSE                                       0x003A
	#define VIGIL_ET_NSS_RENAME                                      0x003B
//	#define VIGIL_ET_NSS_LINK     (DEFINED below in numerical order) 0x0046
	#define VIGIL_ET_NSS_MODIFYINFO                                  0x003C
	#define VIGIL_ET_NSS_ADDTRUSTEE                                  0x003D
	#define VIGIL_ET_NSS_REMOVETRUSTEE                               0x003E
	#define VIGIL_ET_NSS_SETINHERITEDRIGHTS                          0x003F

	#define VIGIL_ET_CIFS_LOCAL__OPENFILE                            0x0040
	#define VIGIL_ET_CIFS_LOCAL__CLOSEFILE                           0x0041

	#define VIGIL_ET_PMD_NCP                 /** DO NOT MODIFY!! **/ 0x0042
	#define VIGIL_ET_PMD_CIFS                /** DO NOT MODIFY!! **/ 0x0043

	#define VIGIL_ET_NET_ADDR_IPv4                                   0x0044
	#define VIGIL_ET_NET_ADDR_IPv6                                   0x0045

	#define VIGIL_ET_NSS_LINK                                        0x0046

	/*---------------------------------------------------------------------------
	** Event data element VIGIL_ET_PATH structure attributes.
	*/

	/* Path namespace (NS) attribute */
	#define VIGIL_ELEMENT_PATH_NS_ANONYMOUS      0x0000
	#define VIGIL_ELEMENT_PATH_NS_FAMILIAR       0x0001
	#define VIGIL_ELEMENT_PATH_NS_UNICODE        0x0002

	/* Path type attribute */
	#define VIGIL_ELEMENT_PATH_TYPE_ANONYMOUS    0x0000
	#define VIGIL_ELEMENT_PATH_TYPE_TARGET       0x0001
	#define VIGIL_ELEMENT_PATH_TYPE_SOURCE       0x0002
	#define VIGIL_ELEMENT_PATH_TYPE_DESTINATION  0x0003

	/*---------------------------------------------------------------------------
	** Event data element VIGIL_ET_NAME structure attributes.
	*/
	#define VIGIL_ELEMENT_NAME_MAXLEN            (255+1)
	#define VIGIL_ELEMENT_NAMETYPE__NONE         0x00000000
	#define VIGIL_ELEMENT_NAMETYPE_CLIENT        0x00000001
	#define VIGIL_ELEMENT_NAMETYPE_CLIENTUSER    0x00000002

	/*---------------------------------------------------------------------------
	** GUID constants.
	*/
	#define VIGIL_GUID_ZERO               {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
	#define VIGIL_GUID_USERID_SECURE_CONN {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01}
	#define VIGIL_GUID_USERID_ANYONE      {0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
	#define VIGIL_GUID_USERID_SUPERVISOR  {0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}
	#define VIGIL_GUID_TREENAME_ID        {0xE7, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00} /* 999=0x03E7 */
	#define VIGIL_GUID_VOLUME_ADMIN       {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF}

/*******************************************************************************
** Memory structures.
*/

#pragma pack(1)

	/*---------------------------------------------------------------------------
	** Common vigil types.
	*/

	/** *********************** GUID type (16 bytes) ************************* **/
#ifdef GUID_t
	typedef GUID_t VIGIL_GUID_T
#else
	typedef struct
		{
		uint32_t timeLow;
		uint16_t timeMid;
		uint16_t timeHighAndVersion;
		uint8_t  clockSeqHighAndReserved;
		uint8_t  clockSeqLow;
		uint8_t  node[6];
		} VIGIL_GUID_T;
#endif

	/** ************************ zTrusteeInfo_s type ************************** **/
	typedef struct
		{
		VIGIL_GUID_T trustee;
		uint32_t     rights;
		} VIGIL_TRUSTEE_INFO_T;

	/** ************************* TimeInfo_s type *************************** **/
	typedef struct
		{
		uint64_t accessedTime;            /* Last time file was accessed*/
		uint64_t createdTime;             /* Time file was created */
		uint64_t modifiedTime;            /* Last time data was changed */
		uint64_t metaDataModifiedTime;    /* Last time metadata was changed. */
		} VIGIL_TIME_INFO_T;

	/*---------------------------------------------------------------------------
	** Audit event element types.
	*/

	/** ************************* Generic elements *************************** **/

	/* Common element header.  (First field of every audit event element). */
	typedef struct
		{
		uint16_t length;
		uint16_t type;
		} VIGIL_ELEMENT_HDR_T;

	/* Network Address element **/
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		union
			{
			uint8_t  octet[16];  /* IPv4 uses octet 0-3 only, and length will be 8.  IPv6 uses 0-15, and length will be 20. */
			uint16_t hextet[8];  /* IPv6 is generally displayed using hextets. */
			} addr;              /* Being that addr is last in the structure, the octet element may be larger than 16 bytes, as long as the true structure length is reflected in the length element. */
		} VIGIL_ELEMENT_NET_ADDR_T;

	/* Elements used to indicate who the audit event should be attributed to. */
	typedef struct                               /* Allows for future who element       */
		{                                         /* extensibility, and sets minimum     */
		VIGIL_ELEMENT_HDR_T hdr;                  /* size of record to 64 bytes.         */
		uint8_t             data[64];
		} VIGIL_ELEMENT_WHO_ANONYMOUS_T;

	typedef struct                               /* Used when NCP makes POSIX calls     */
		{                                         /* as root on behalf of an NCP client. */
		VIGIL_ELEMENT_HDR_T     hdr;
		uint32_t                connID;
		uint32_t                taskID;
		VIGIL_GUID_T /*GUID_t*/ guid;
		} VIGIL_ELEMENT_WHO_NCP_POSIX_T;

	typedef struct                               /* Used when no information is evident */
		{                                         /* in the PID metadata table.  These   */
		VIGIL_ELEMENT_HDR_T     hdr;              /* values come from the Linux process/ */
		uid_t                   uid;              /* thread structure itself.            */
		uid_t                   euid;
		uid_t                   suid;
		uid_t                   fsuid;
		gid_t                   gid;
		gid_t                   egid;
		gid_t                   sgid;
		gid_t                   fsgid;
		char                    comm[16 /* TASK_COMM_LEN */];
		} VIGIL_ELEMENT_WHO_LINUX_T;

	typedef struct                               /* Used when NSS has eDir client guid data. */
		{
		VIGIL_ELEMENT_HDR_T     hdr;
		VIGIL_GUID_T /*GUID_t*/ guid;
		} VIGIL_ELEMENT_WHO_NSS_EDIR_T;

	typedef union                                 /* Union of all the WHO structures,    */
		{                                          /* used primarily to determine the PID */
		VIGIL_ELEMENT_WHO_ANONYMOUS_T anonymous;   /* metadata table row size.            */
		VIGIL_ELEMENT_WHO_NCP_POSIX_T ncpPosix;
		VIGIL_ELEMENT_WHO_LINUX_T     linuxTask;
		VIGIL_ELEMENT_WHO_NSS_EDIR_T  nssEdir;
		} VIGIL_ELEMENT_WHO_T;

	/** ************** Elements identifying process metadata ************** **/

	typedef struct                               /* Used when NSS has eDir client guid data. */
		{
		VIGIL_ELEMENT_HDR_T      hdr;
		VIGIL_GUID_T /*GUID_t*/  guid;
		uint32_t                 connID;
		uint32_t                 taskID;
		} VIGIL_ELEMENT_PMD_NCP_T;

	typedef struct                               /* Used when NSS has eDir client guid data. */
		{
		VIGIL_ELEMENT_HDR_T      hdr;
		VIGIL_GUID_T /*GUID_t*/  guid;
		uint32_t                 connID;           /* CIFS clientID */
		VIGIL_ELEMENT_NET_ADDR_T netAddr;          /* WARNING: Must be last!  (Variable length field). */
		} VIGIL_ELEMENT_PMD_CIFS_T;

	typedef union
		{
		VIGIL_ELEMENT_HDR_T      hdr;
		VIGIL_ELEMENT_PMD_NCP_T  ncp;
		VIGIL_ELEMENT_PMD_CIFS_T cifs;
		} VIGIL_ELEMENT_PMD_UNION_T;

	/** ************** Element identifying a filesystem path ************** **/
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint16_t            nameSpace;
		uint16_t            pathType;
		union
			{
			uint8_t  familiar[PATH_MAX];
			uint16_t unicode[PATH_MAX];
			}                data;
		} VIGIL_ELEMENT_PATH_T;

	/** ************** Novell Event Bus (NEB) specific elements ************** **/

	/* Common NEB exit event data. */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            enterRetStatus;
		uint32_t            opRetCode;
		} VIGIL_ELEMENT_NEB_EXIT_T;

	/* CreateEnter event data. */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            fileType;
		uint32_t            fileAttributes;
		uint32_t            createFlags;
		uint32_t            requestedRights;
		uint32_t            createAndOpen;
		} VIGIL_ELEMENT_NEB_CREATE_ENTER_T;

	/* CreateExit event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T         hdr;
		uint32_t                    retOpenCreateAction;
		uint64_t /* Key_t */        retKey;
		uint64_t /* Zid_t */        retZid;
		VIGIL_GUID_T /*VolumeID_t*/ retVolID;
		VIGIL_TIME_INFO_T           times;
		} VIGIL_ELEMENT_NEB_CREATE_EXIT_T;

	/* OpenEnter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            requestedRights;
		} VIGIL_ELEMENT_NEB_OPEN_ENTER_T;

	/* OpenExit event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T         hdr;
		uint64_t /* Key_t */        retKey;
		uint64_t /* Zid_t */        retZid;
		VIGIL_GUID_T /*VolumeID_t*/ retVolID;
		VIGIL_TIME_INFO_T           times;
		} VIGIL_ELEMENT_NEB_OPEN_EXIT_T;

	/* CloseEnter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T  hdr;
		uint64_t /* Key_t */ key;
		uint32_t             fhState;
		VIGIL_TIME_INFO_T    times;
		} VIGIL_ELEMENT_NEB_CLOSE_ENTER_T;

	/* CloseExit event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            fileDeleted;
		} VIGIL_ELEMENT_NEB_CLOSE_EXIT_T;

	/* RenameEnter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            renameFlags;
		} VIGIL_ELEMENT_NEB_RENAME_ENTER_T;

	/* ModifyEnter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            modifyInfoMask;
		uint32_t            modifyTypeInfoMask;
		} VIGIL_ELEMENT_MODIFY_T;

	/* ModifyEnter event data zMOD_FILE_ATTRIBUTES (BIT 1) 0x00000001 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            fileAttributes;
		uint32_t            fileAttributesModMask;
		} VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T;

	/* ModifyEnter event data zMOD_CREATED_TIME (BIT 2) 0x00000002 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   created;
		} VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T;

	/* ModifyEnter event data zMOD_ARCHIVED_TIME (BIT 3) 0x00000004 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   archived;
		} VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T;

	/* ModifyEnter event data zMOD_MODIFIED_TIME (BIT 4) 0x00000008 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   modified;
		} VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T;

	/* ModifyEnter event data zMOD_ACCESSED_TIME (BIT 5) 0x00000010 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   accessed;
		} VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T;

	/* ModifyEnter event data zMOD_METADATA_MODIFIED_TIME (BIT 6) 0x00000020 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   metaDataModified;
		} VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T;

	/* ModifyEnter event data zMOD_OWNER_ID (BIT 7) 0x00000040 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ owner;
		} VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T;

	/* ModifyEnter event data zMOD_ARCHIVER_ID (BIT 8) 0x00000080 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ archiver;
		} VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T;

	/* ModifyEnter event data zMOD_MODIFIER_ID (BIT 9) 0x00000100 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ modifier;
		} VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T;

	/* ModifyEnter event data zMOD_METADATA_MODIFIER_ID (BIT 10) 0x00000200 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ metaDataModifier;
		} VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T;

	/* ModifyEnter event data zMOD_PRIMARY_NAMESPACE (BIT 11) 0x00000400 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            primaryNameSpaceID;
		} VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T;

	/* ModifyEnter event data zMOD_DELETED_INFO (BIT 12) 0x00000800 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		uint64_t /*QUAD*/         time;
		VIGIL_GUID_T /*UserID_t*/ id;
		} VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T;

	/* ModifyEnter event data zMOD_MAC_METADATA (BIT 13) 0x00001000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		struct
			{
			uint8_t  finderInfo[32];
			uint8_t  proDOSInfo[6];
			uint8_t  filler[2];
			uint32_t dirRightsMask;
			}                info;
		} VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T;

	/* ModifyEnter event data zMOD_UNIX_METADATA (BIT 14) 0x00002000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		struct
			{
			uint32_t    fMode;
			uint32_t    rDev;
			uint32_t    myFlags;
			uint32_t    nfsUID;
			uint32_t    nfsGID;
			uint32_t    nwUID;
			uint32_t    nwGID;
			uint32_t    nwEveryone;
			uint32_t    nwUIDRights;
			uint32_t    nwGIDRights;
			uint32_t    nwEveryoneRights;
			uint8_t     acsFlags;
			uint8_t     firstCreated;
			int16_t     variableSize;
			/* If variableSize is non-zero, there will be "variableSize" number of
			* additional bytes of meta data associated with this file. The format
			* of this variable data is unknown to NSS
			*/
			uint8_t     variableData[1];
			}                info;
		} VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T;

	/* ModifyEnter event data zMOD_EXTATTR_FLAGS (BIT 15) 0x00004000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            extAttrUserFlags;
		} VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T;

	/* ModifyEnter event data zMOD_VOL_ATTRIBUTES (BIT 16) 0x00008000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   enabled;       /* Features currently enabled for the volume. */
		uint64_t /*QUAD*/   enableModMask; /* Mask of features that should be changed. */
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T;

	/* ModifyEnter event data zMOD_VOL_NDS_OBJECT_ID (BIT 17) 0x00010000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T      hdr;
		VIGIL_GUID_T /*NDSid_t*/ ndsObjectID;
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T;

	/* ModifyEnter event data zMOD_VOL_MIN_KEEP_SECONDS (BIT 18) 0x00020000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            minKeepSeconds; /* NOT SUPPORTED on ZLSS -- minimum number of seconds a file in kept in the salvage system. */
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T;

	/* ModifyEnter event data zMOD_VOL_MAX_KEEP_SECONDS (BIT 19) 0x00040000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            maxKeepSeconds; /* Maximum time to keep deleted files. */
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T;

	/* ModifyEnter event data zMOD_VOL_LOW_WATER_MARK (BIT 20) 0x00080000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            lowWaterMark;
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T;

	/* ModifyEnter event data zMOD_VOL_HIGH_WATER_MARK (BIT 21) 0x00100000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            highWaterMark;
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T;

	/* ModifyEnter event data zMOD_POOL_ATTRIBUTES (BIT 22) 0x00200000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   enabled;       /* Features that are currently enabled for the pool. */
		uint64_t /*QUAD*/   enableModMask; /* Mask of features that should be changed. */
		} VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T;

	/* ModifyEnter event data zMOD_POOL_NDS_OBJECT_ID (BIT 23) 0x00400000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T      hdr;
		VIGIL_GUID_T /*NDSid_t*/ ndsObjectID;
		} VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T;

	/* ModifyEnter event data zMOD_VOL_DATA_SHREDDING_COUNT (BIT 24) 0x00800000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            dataShreddingCount; /* How many writes are done to the disk to obscure the original data.  Up to 7 passes can be specified. */
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T;

	/* ModifyEnter event data zMOD_VOL_QUOTA (BIT 25) 0x01000000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint64_t /*QUAD*/   totalSpaceQuota; /* Maximum size of this volume (may over book pool) */
		} VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T;

	/* ModifyEnter event data zMOD_DIR_QUOTA (BIT 26) 0x02000000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		int64_t /*SQUAD*/   quota;
		} VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T;

	/* ModifyEnter event data zMOD_READ_AHEAD_BLOCKS (BIT 27) 0x04000000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		} VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T;

	/* ModifyEnter event data zMOD_INH_RIGHTS_MASK (BIT 28) 0x08000000 */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            inheritedRightsMask;
		} VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T;

	/* ModifyEnter event data zMOD_ALL_TRUSTEES (BIT 29) 0x10000000 */
	#define VIGIL__zMOD_ALL_TRUSTEES__MAX ((2 * 256 /*zMAX_COMPONENT_NAME*/) / sizeof(VIGIL_TRUSTEE_INFO_T))
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t             numEntries;
		VIGIL_TRUSTEE_INFO_T trusteeTable[VIGIL__zMOD_ALL_TRUSTEES__MAX];
		} VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T;

	/* AddTrustee event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ trusteeID;
		uint32_t                  rights;
		uint32_t                  attributes;
		} VIGIL_ELEMENT_NEB_ADDTRUSTEE_ENTER_T;

	/* RemoveTrustee enter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T       hdr;
		VIGIL_GUID_T /*UserID_t*/ trusteeID;
		uint32_t                  purgedFileFlag;
		} VIGIL_ELEMENT_NEB_REMOVETRUSTEE_ENTER_T;

	/* SetInheritedRights enter event data */
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t /* NINT */ inheritedRights;
		uint32_t /* BOOL */ authorizeFlag;
		} VIGIL_ELEMENT_NEB_SETINHERITEDRIGHTS_ENTER_T;

	/** *********** NetWare Core Protocol (NCP) specific elements ************ **/

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint8_t             searchAttributes;
		uint8_t             desiredAccessRights;
		uint8_t             status;
		uint8_t             fileHandle[6];   /* (Hi-Lo) */
		uint8_t             fileAttributes;
		uint8_t             fileExecuteType;
		uint32_t            fileLen;         /* (Hi-Lo) */
		uint16_t            creationDate;    /* (Hi-Lo) */
		uint16_t            lastAccessDate;  /* (Hi-Lo) */
		uint16_t            lastUpdateDate;  /* (Hi-Lo) */
		uint16_t            lastUpdateTime;  /* (Hi-Lo) */
		uint8_t             linuxPath[PATH_MAX];
		} VIGIL_ELEMENT_NCP_OPENFILE_T;

	typedef struct 
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint8_t             fileHandle[6];   /* (Hi-Lo) */
		uint8_t             status;
		} VIGIL_ELEMENT_NCP_CLOSEFILE_T;

	/** *********** VIGIL internal elements ************ **/
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint32_t            nameType; /* See: VIGIL_ELEMENT_NAMETYPE_* */
		uint8_t             name[VIGIL_ELEMENT_NAME_MAXLEN];
		} VIGIL_ELEMENT_NAME_T;

	/** *********** NSS elements *********************** **/
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T                hdr;
		uint32_t          /* LONG       */ requestedRights;     /** zRR_READ_ACCESS, etc. **/
		int64_t           /* SQUAD/Key_t*/ key;                 /** if {ender_createAndOpen != 0), opened object key (or handle). **/
		VIGIL_TIME_INFO_T /* TimeInfo_s */ times;               /** Object times: accessed, created, modified, metadataModified. **/
		uint32_t          /* LONG       */ createFlags;         /** zCREATE_OPEN_IF_THERE, etc. **/
		uint32_t          /* LONG       */ createAndOpen;       /** If non-zero, file will be opened too. **/
		uint32_t          /* LONG       */ retOpenCreateAction; /** zBEAST_CREATED, zBEAST_TRUNCATED, zBEAST_EXISTED, etc. **/
		} VIGIL_ELEMENT_NSS_Create_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T                hdr;
		uint32_t          /* LONG       */ requestedRights; /** zRR_READ_ACCESS, etc. **/
		int64_t           /* SQUAD/Key_t*/ key;             /** Opened object key (or handle). **/
		VIGIL_TIME_INFO_T /* TimeInfo_s */ times;           /** Object times: accessed, created, modified, metadataModified. **/
		} VIGIL_ELEMENT_NSS_Open_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T                hdr;
		int64_t           /* SQUAD/Key_t*/ key;         /** Object key (or handle), as assigned by create or open events. **/
		VIGIL_TIME_INFO_T /* TimeInfo_s */ times;       /** Object times: accessed, created, modified, metadataModified. **/
		uint32_t          /* LONG       */ fhState;     /** FH_NEW, FH_MODIFIED, FH_WRITE_SNAPSHOT, FH_READ_BACKUP, ... **/
		uint32_t          /* LONG       */ fileDeleted; /** TRUE|FALSE: Indicates that file was deleted on close. **/
		} VIGIL_ELEMENT_NSS_Close_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		uint32_t     /* LONG       */ renameFlags;     /** zRENAME_ALLOW_RENAMES_TO_MYSELF, zRENAME_THIS_NAMESPACE_ONLY, zRENAME_DELETE_FILE_IF_THERE, ... **/
		} VIGIL_ELEMENT_NSS_Rename_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		uint32_t     /* LONG       */ linkFlags;     /** **/
		} VIGIL_ELEMENT_NSS_Link_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		uint64_t     /* QUAD       */ modifyInfoMask;     /** What changed (detailed in enter_modifyInfo): zMOD_FILE_ATTRIBUTES, zMOD_CREATED_TIME, ... **/
		uint64_t     /* QUAD       */ modifyTypeInfoMask; /** Z_MOD_EXTATTR_FLAGS   **/
		} VIGIL_ELEMENT_NSS_ModifyInfo_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		VIGIL_GUID_T /* UserID_t   */ trusteeID;       /** eDirectory GUID of trustee to remove. **/
		uint32_t     /* LONG       */ rights;          /** zAUTHORIZE_READ_CONTENTS, zAUTHORIZE_WRITE_CONTENTS, zAUTHORIZE_CREATE_ENTRY, ... **/
		uint32_t     /* LONG       */ previousRights;  /** zAUTHORIZE_READ_CONTENTS, zAUTHORIZE_WRITE_CONTENTS, zAUTHORIZE_CREATE_ENTRY, ... **/
		uint32_t     /* LONG       */ attributes;      /** zAUTHORIZE_NAGATIVE_RIGHTS, zAUTHORIZE_INHERIT_DOWN, zAUTHORIZE_INHERIT_UP **/
		} VIGIL_ELEMENT_NSS_AddTrustee_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		VIGIL_GUID_T /* UserID_t   */ trusteeID;       /** eDirectory GUID of trustee to remove. **/
		uint32_t     /* NINT       */ rights;          /** zAUTHORIZE_READ_CONTENTS, zAUTHORIZE_WRITE_CONTENTS, zAUTHORIZE_CREATE_ENTRY, ... **/
		uint32_t     /* LONG       */ purgedFileFlag;  /** TRUE=ACL(ie:all trustees) cleared due to file deletion(purge).  FALSE=Atomic trustee removal. **/
		} VIGIL_ELEMENT_NSS_RemoveTrustee_T;

	typedef struct
		{
		VIGIL_ELEMENT_HDR_T           hdr;
		uint32_t                      inheritedRights; /** zAUTHORIZE_READ_CONTENTS, zAUTHORIZE_WRITE_CONTENTS, zAUTHORIZE_CREATE_ENTRY, ... **/
		uint32_t                      previousInheritedRights; /** zAUTHORIZE_READ_CONTENTS, zAUTHORIZE_WRITE_CONTENTS, zAUTHORIZE_CREATE_ENTRY, ... **/
		} VIGIL_ELEMENT_NSS_SetInheritedRights_T;

	/** CIFS Elements **/
	typedef struct
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint8_t             desiredAccessRights;
		uint8_t             status;
		uint8_t             fileHandle[6];   /* (Hi-Lo) */
		uint8_t             fileAttributes;
		uint8_t             fileExecuteType;
		uint32_t            fileLen;         /* (Hi-Lo) */
		uint16_t            creationDate;    /* (Hi-Lo) */
		uint16_t            lastAccessDate;  /* (Hi-Lo) */
		uint16_t            lastUpdateDate;  /* (Hi-Lo) */
		uint16_t            lastUpdateTime;  /* (Hi-Lo) */
		uint8_t             linuxPath[PATH_MAX];
		} VIGIL_ELEMENT_CIFS_OPENFILE_T;

	typedef struct 
		{
		VIGIL_ELEMENT_HDR_T hdr;
		uint8_t             fileHandle[6];   /* (Hi-Lo) */
		uint8_t             status;
		} VIGIL_ELEMENT_CIFS_CLOSEFILE_T;

	/** Generic union used to determine (worst-case) size of audit record variable-length data. **/
	typedef union
		{
		/** NEB Auditing Structures. **/
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_PATH_T             path;
			} deleteEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			} deleteExit;
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_PATH_T             path;
			VIGIL_ELEMENT_NEB_CREATE_ENTER_T createEnter;
			} createEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			VIGIL_ELEMENT_NEB_CREATE_EXIT_T  createExit;
			} createExit;
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_PATH_T             path;
			VIGIL_ELEMENT_NEB_OPEN_ENTER_T   openEnter;
			} openEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			VIGIL_ELEMENT_NEB_OPEN_EXIT_T    openExit;
			} openExit;
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_NEB_CLOSE_ENTER_T  closeEnter;
			} closeEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			VIGIL_ELEMENT_NEB_CLOSE_EXIT_T   closeExit;
			} closeExit;
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_PATH_T             sourcePath;
			VIGIL_ELEMENT_PATH_T             destinationPath;
			VIGIL_ELEMENT_NEB_RENAME_ENTER_T renameEnter;
			} renameEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			} renameExit;
		struct
			{
			VIGIL_ELEMENT_WHO_T              who;
			VIGIL_ELEMENT_PATH_T             path;
			VIGIL_ELEMENT_MODIFY_T modifyEnter;
			VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T          modifyEnter_zModFileAttributs;
			VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T             modifyEntry_zModCreatedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T            modifyEntry_zModArchivedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T            modifyEntry_zModModifiedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T            modifyEntry_zModAccessedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T   modifyEntry_zMmodMetadataModifiedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T                 modifyEntry_zModOwnerID;
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T              modifyEntry_zModArchiverID;
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T              modifyEntry_zModModifierID;
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T     modifyEntry_zModMetadataModifierID;
			VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T        modifyEntry_zModPrimaryNamespace;
			VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T             modifyEntry_zModDeletedInfo;
			VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T             modifyEntry_zModMacMetadata;
			VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T            modifyEntry_zModUnixMetadata;
			VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T            modifyEntry_zModExtattrFlags;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T           modifyEntry_zModVolAttributes;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T        modifyEntry_zModVolNdsObjectID;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T     modifyEntry_zModVolMinKeepSeconds;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T     modifyEntry_zModVolMaxKeepSeconds;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T       modifyEntry_zModVolLowWaterMark;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T      modifyEntry_zModVolHighWaterMark;
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T          modifyEntry_zModPoolAttributes;
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T       modifyEntry_zModPoolNdsObjectID;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T modifyEntry_zModVolDataShreddingCount;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T                modifyEntry_zModVolQuota;
			VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T                modifyEntry_zModDirQuota;
			VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T        modifyEntry_zModReadAheadBlocks;
			VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T          modifyEntry_zModInhRightsMask;
			VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T             modifyEntry_zModAllTrustees;
			} modifyEnter;
		struct
			{
			VIGIL_ELEMENT_NEB_EXIT_T         exit;
			} modifyExit;

		/** NCP Auditing Structures **/
		struct
			{
			VIGIL_ELEMENT_WHO_T           who;
			VIGIL_ELEMENT_NCP_OPENFILE_T  ncpOpenFile;
			} ncpOpenFile;
		struct
			{
			VIGIL_ELEMENT_WHO_T           who;
			VIGIL_ELEMENT_NCP_CLOSEFILE_T ncpCloseFile;
			} ncpCloseFile;

		/** CIFS Auditing Structures **/
		struct
			{
			VIGIL_ELEMENT_WHO_LINUX_T     who_linux;
			VIGIL_ELEMENT_PMD_CIFS_T      who_pmd;
			VIGIL_ELEMENT_CIFS_OPENFILE_T open;
			} cifsOpen;

		struct
			{
			VIGIL_ELEMENT_WHO_LINUX_T     who_linux;
			VIGIL_ELEMENT_PMD_CIFS_T      who_pmd;
			VIGIL_ELEMENT_CIFS_CLOSEFILE_T close;
			} cifsClose;

		/** NSS Auditing Structures **/
		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       path;
			} nssDelete;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       path;
			VIGIL_ELEMENT_NSS_Create_T create;
			} nssCreate;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       path;
			VIGIL_ELEMENT_NSS_Open_T   open;
			} nssOpen;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       path;
			VIGIL_ELEMENT_NSS_Close_T  close;
			} nssClose;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       src;
			VIGIL_ELEMENT_PATH_T       dst;
			VIGIL_ELEMENT_NSS_Rename_T rename;
			} nssRename;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T  pmd;
			VIGIL_ELEMENT_PATH_T       src;
			VIGIL_ELEMENT_PATH_T       dst;
			VIGIL_ELEMENT_NSS_Link_T   link;
			} nssLink;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T                             pmd;
			VIGIL_ELEMENT_PATH_T                                  path;
			VIGIL_ELEMENT_NSS_ModifyInfo_T                        modifyInfo;
			VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T          modifyEnter_zModFileAttributs;
			VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T             modifyEntry_zModCreatedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T            modifyEntry_zModArchivedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T            modifyEntry_zModModifiedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T            modifyEntry_zModAccessedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T   modifyEntry_zMmodMetadataModifiedTime;
			VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T                 modifyEntry_zModOwnerID;
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T              modifyEntry_zModArchiverID;
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T              modifyEntry_zModModifierID;
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T     modifyEntry_zModMetadataModifierID;
			VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T        modifyEntry_zModPrimaryNamespace;
			VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T             modifyEntry_zModDeletedInfo;
			VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T             modifyEntry_zModMacMetadata;
			VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T            modifyEntry_zModUnixMetadata;
			VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T            modifyEntry_zModExtattrFlags;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T           modifyEntry_zModVolAttributes;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T        modifyEntry_zModVolNdsObjectID;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T     modifyEntry_zModVolMinKeepSeconds;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T     modifyEntry_zModVolMaxKeepSeconds;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T       modifyEntry_zModVolLowWaterMark;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T      modifyEntry_zModVolHighWaterMark;
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T          modifyEntry_zModPoolAttributes;
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T       modifyEntry_zModPoolNdsObjectID;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T modifyEntry_zModVolDataShreddingCount;
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T                modifyEntry_zModVolQuota;
			VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T                modifyEntry_zModDirQuota;
			VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T        modifyEntry_zModReadAheadBlocks;
			VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T          modifyEntry_zModInhRightsMask;
			VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T             modifyEntry_zModAllTrustees;
			} nssModifyInfo;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T      pmd;
			VIGIL_ELEMENT_PATH_T           path;
			VIGIL_ELEMENT_NSS_AddTrustee_T addTrustee;
			} nssAddTrustee;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T         pmd;
			VIGIL_ELEMENT_PATH_T              path;
			VIGIL_ELEMENT_NSS_RemoveTrustee_T removeTrustee;
			} nssRemoveTrustee;

		struct
			{
			VIGIL_ELEMENT_PMD_UNION_T              pmd;
			VIGIL_ELEMENT_PATH_T                   path;
			VIGIL_ELEMENT_NSS_SetInheritedRights_T removeTrustee;
			} nssSetInheritedRights;

		} VIGIL_ELEMENT_DATA_T;

	/*---------------------------------------------------------------------------
	** Event audit records.
	*/
	/* All event audit records contain this header. */
	typedef struct
		{
		char           sign[6]; /* "\nVIGIL" */
		uint32_t       length;  /* Length of entire audit record (in Bytes). */
		uint32_t       type;    /* Type of audit record.  (Well known types in vigil.h). */
		uint64_t       recNo;   /* Record sequence number, specific to each auditing client. */
		uint32_t       pid;     /* Process ID of thread that submitted the audit record. */
		struct timeval time;    /* Time audit record was logged. */
		} VIGIL_AUDIT_REC_HDR_T;

	/* Vigil self-auditing record. */
	typedef struct
		{
		VIGIL_AUDIT_REC_HDR_T hdr;
		uint32_t              event;
		uint16_t              dataElements;
		uint16_t              dataLength;
		uint8_t               data[sizeof(VIGIL_ELEMENT_DATA_T)];
		} VIGIL_AUDIT_REC_VIGIL_T;
	
	/* Novell Event Bus (NEB) records. */
	typedef struct
		{
		VIGIL_AUDIT_REC_HDR_T hdr;
		uint32_t              event;
		uint32_t              enterExitID;
		uint16_t              dataElements;
		uint16_t              dataLength;
		uint8_t               data[sizeof(VIGIL_ELEMENT_DATA_T)];
		} VIGIL_AUDIT_REC_NEB_T;

	/* NetWare Core Protocol (NCP) records. */
	typedef struct
		{
		VIGIL_AUDIT_REC_HDR_T hdr;
		uint32_t              event;
		uint16_t              dataElements;
		uint16_t              dataLength;
		uint8_t               data[sizeof(VIGIL_ELEMENT_DATA_T)];
		} VIGIL_AUDIT_REC_NCP_T;

	/* Novell Storage Services (NSS) records. */
	typedef struct
		{
		VIGIL_AUDIT_REC_HDR_T         hdr;

		/** Common NSS event values. **/
		uint32_t                      event;
		uint32_t                      taskID;
		VIGIL_GUID_T /* VolumeID_t */ volID;
		uint64_t     /* Zid_t      */ zid;
		uint64_t     /* Zid_t      */ parentZid;
		VIGIL_GUID_T /* GUID_t     */ userID;
		uint32_t     /* NINT       */ fileType;
		uint64_t     /* QUAD       */ fileAttributes;
		uint32_t     /* STATUS     */ opRetCode;

		/** Linux userID values. **/
		uid_t                         uid;
		uid_t                         euid;
		uid_t                         suid;
		uid_t                         fsuid;
		gid_t                         gid;
		gid_t                         egid;
		gid_t                         sgid;
		gid_t                         fsgid;
		char                          comm[16 /* TASK_COMM_LEN */];

		/** Variable-length portion of audit record. **/
		uint16_t                      dataElements;
		uint16_t                      dataLength;
		uint8_t                       data[sizeof(VIGIL_ELEMENT_DATA_T)];
		} VIGIL_AUDIT_REC_NSS_T;

	/* Common Internet File System (CIFS) records. */
	typedef struct
		{
		VIGIL_AUDIT_REC_HDR_T hdr;
		uint32_t              event;
		uint16_t              dataElements;
		uint16_t              dataLength;
		uint8_t               data[sizeof(VIGIL_ELEMENT_DATA_T)];
		} VIGIL_AUDIT_REC_CIFS_T;

	/* Union of all the above event audit record types, used to size pre-allocated working buffers. (See: kern/vigil_memq.c) */
	typedef union
		{
		VIGIL_AUDIT_REC_HDR_T   hdr;
		VIGIL_AUDIT_REC_VIGIL_T vigil;
		VIGIL_AUDIT_REC_NEB_T   neb;
		VIGIL_AUDIT_REC_NCP_T   ncp;
		VIGIL_AUDIT_REC_NSS_T   nss;
		VIGIL_AUDIT_REC_CIFS_T  cifs;
		} VIGIL_AUDIT_REC_T;

	/*---------------------------------------------------------------------------
	** Filters
	*/
	/** Flags for: VIGIL_CLIENT_filterNode.flags **/
	#define VIGIL_FILTER_F_NONE            0x00000000
	#define VIGIL_FILTER_F__INCLUDE        0x00000000
	#define VIGIL_FILTER_F__EXCLUDE        0x00000001

	/** Types for: VIGIL_CLIENT_filterNode.type **/
	#define VIGIL_FILTER_T__NONE           0x00000000
	#define VIGIL_FILTER_T__MATCH_BIT_MASK 0x00000001

	/** Filter file values **/
	#define VIGIL_FILTER_FILE_SIGNATURE    0x544C4656    /* longswap("VFLT") */
	#define VIGIL_FILTER_FILE_VERSION      0x00010000    /* v01.00 */
	#define VIGIL_FILTER_NAME_SIZE         63

	/** Filter file structures **/
	typedef struct
		{
		uint32_t length;
		uint32_t type;
		uint32_t flags;
		uint8_t  name[VIGIL_FILTER_NAME_SIZE+1];
		} VIGIL_FILTER_SECTION_HDR_T;

	typedef struct
		{
		VIGIL_FILTER_SECTION_HDR_T hdr;
		uint32_t                   filterOffset; /* Offset where to apply the value/mask within the audit record. */
		uint32_t                   filterLen;    /* Length of value & valueMask (in bytes). */
		uint32_t                   valueOffset;  /* Record data offset where the bit-value is located. */
		uint32_t                   maskOffset;   /* Record data offset where the bit-mask is located. */
		uint8_t                    data[1];
		} VIGIL_FILTER_SECTION__MATCH_BIT_MASK_T;

	typedef union
		{
		VIGIL_FILTER_SECTION_HDR_T             hdr;
		VIGIL_FILTER_SECTION__MATCH_BIT_MASK_T matchBitMask;
		} VIGIL_FILTER_SECTION_T;

	typedef struct
		{
		uint32_t signature;
		uint32_t version;
		uint32_t headerSize;
		uint32_t firstSectionOffset;
		} VIGIL_FILTER_FILE_HDR_T;

#pragma pack()

#endif   /* __INCLUDE_VIGIL_H__ */
