/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES Auditing Sample Client (vdump).
 |
 | File: vdump_main.c
 |
 | Desc: Dump vigil audit records to stdout.
 +-------------------------------------------------------------------------*/

/*******************************************************************************
** Compiler setup.
*/

	/*---------------------------------------------------------------------------
	** Auto Tools
	*/
#ifdef HAVE_CONFIG_H
 #ifndef __config_h__
	#include <config.h>  /* PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT */
	#define __config_h__
 #endif 
#endif

	/*---------------------------------------------------------------------------
	** ANSI/POSIX
	*/
	#include <stdlib.h>	/* exit() */
	#include <stdio.h>	/* stderr, fprintf() */
	#include <errno.h>
	#include <time.h>
	#include <string.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <dirent.h>

	#include <sys/stat.h>
	#include <fcntl.h>

	#include <signal.h>

	/*---------------------------------------------------------------------------
	** Project
	*/
	#include <vigil.h>
	#include <libvigil.h>

	#include <ncpID.h>

	#include "vdump_filter.h"
	#include "vdump_out_csv.h"
	#include "vdump_out_std.h"
	#include "vdump_out_xml.h"

	#define VDUMP_MAIN_MAX_RETRYS   10

/*******************************************************************************
** Global storage.
*/
int     VDUMP_MAIN_exiting            = VIGIL_FALSE;
int     VDUMP_MAIN_verbose            = 0;
char   *VDUMP_MAIN_auditLogPath_ALLOC = NULL;
size_t  VDUMP_MAIN_auditLogFileOffset = 0;
char   *VDUMP_MAIN_clientName_ALLOC   = NULL;
int   (*VDUMP_MAIN_OUT_PrintRecord)(
			/* I- outStream */ FILE              *outStream,
			/* I- record    */ VIGIL_AUDIT_REC_T *rec,
			/* I- recLen    */ size_t             recLen
			) = VDUMP_OUT_STD_PrintRecord;
FILE    *VDUMP_MAIN_outStream = NULL;

/*******************************************************************************
**
*/
int VDUMP_MAIN_RollAuditFile(
		/* I- rollAuditRec   */ VIGIL_AUDIT_REC_T *rollAuditRec,
		/* IO auditFile_fd   */ int               *auditFile_fd,
		/* IO auditFile_path */ char             **auditFile_path
		)
	{
	int rCode=VIGIL_SUCCESS;
	VIGIL_ELEMENT_PATH_T *pathElement;

	rCode=LIBVIGIL_PARSE_RecElement_ByTypeInstance(
		/* I- recData         */ rollAuditRec->vigil.data,
		/* I- recDataElements */ rollAuditRec->vigil.dataElements,
		/* I- elementType     */ VIGIL_ET_PATH,
		/* I- instance        */ 0,
		/* -O elementPtr      */ (void **)&pathElement
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_MAIN_RecGetElement() reports: %d\n", rCode);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Free old filename buffer.
	*/
	rCode=LIBVIGIL_MEM_Free((void **)auditFile_path);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_MEM_Free() reports: %d\n", rCode);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Allocate new path.
	*/
	rCode=LIBVIGIL_MEM_StrDup(pathElement->data.familiar, auditFile_path);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_MEM_StrDup() reports: %d\n", rCode);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Reset current file offset.
	*/
	VDUMP_MAIN_auditLogFileOffset = 0;

	/*---------------------------------------------------------------------------
	** Close the old audit file.
	*/
	errno=0;
	if((-1) == close(*auditFile_fd))
		{
		rCode=errno;
		LIBVIGIL_ERR("close() failed.  errno[%d]\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Open the new audit file.
	*/
	errno=0;
	*auditFile_fd=open(*auditFile_path, O_RDONLY|O_ASYNC);
	if((-1) == *auditFile_fd)
		{
		rCode=errno;
		LIBVIGIL_ERR("open(\"%s\"...) failed.  errno[%d]\n",
			auditFile_path,
			rCode
			);
		goto END_ERR;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** SIGIO Process.
**
** Local references: VDUMP_MAIN_Follow()
*/
int VDUMP_MAIN_ProcessAuditLog(
		/* I- outStream */ FILE *outStream
		)
	{
	int rCode=VIGIL_SUCCESS;
	int fd=(-1);

	/*-------------------------------------------------------------------------
	** The first time in, the path may be uninitialized.
	*/
	if(NULL == VDUMP_MAIN_auditLogPath_ALLOC)
		{
		/** Get current outputFilePath_ALLOC. **/
		rCode=LIBVIGIL_SYS_CurrentFileAndOffset(
			/* I- clientName       */ VDUMP_MAIN_clientName_ALLOC,
			/* -O outputFile_ALLOC */ &VDUMP_MAIN_auditLogPath_ALLOC,
			/* -O outputFileOffset */ &VDUMP_MAIN_auditLogFileOffset
			);
		switch(rCode)
			{
			case VIGIL_SUCCESS:
				break;
	
			case ENOENT:
				LIBVIGIL_ERR("LIBVIGIL_SYS_CurrentFileAndOffset() reports: ENOENT. \n(Make sure vigil.ko has been loaded before running this program)\n");
				goto END_ERR;
	
			default:
				LIBVIGIL_ERR("LIBVIGIL_ClientInfo() reports: %d\n", rCode);
				goto END_ERR;
			}
		}

	/*---------------------------------------------------------------------------
	** Open the audit file.
	*/
	errno=0;
	fd=open(VDUMP_MAIN_auditLogPath_ALLOC, O_RDONLY|O_ASYNC);
	if((-1) == fd)
		{
		rCode=errno;
		LIBVIGIL_ERR("open(\"%s\"...) failed.  errno[%d]\n",
			VDUMP_MAIN_auditLogPath_ALLOC,
			rCode
			);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Process the file.
	*/
	while(VIGIL_FALSE == VDUMP_MAIN_exiting)
		{
		VIGIL_AUDIT_REC_T rec;
		ssize_t           bytesRead;

		/*---------------------------------------------------------------------------
		** Position to next record.
		*/
		if(((off_t)(-1)) == lseek(fd, VDUMP_MAIN_auditLogFileOffset, SEEK_SET))
			{
			rCode=errno;
			LIBVIGIL_ERR("lseek() failed.  errno[%d] VDUMP_MAIN_auditLogFileOffset[%d]\n", rCode, VDUMP_MAIN_auditLogFileOffset);
			goto END_ERR;
			}

		/*---------------------------------------------------------------------------
		** Read record from file.
		*/
		memset(&rec, 0, sizeof(rec));
		bytesRead=read(fd, &rec, sizeof(rec));
		if((-1) == bytesRead)
			{
			rCode=errno;
			LIBVIGIL_ERR("read() failed.  errno[%d]\n", rCode);
			goto END_ERR;
			}

		if(0 == bytesRead)
			break;

		/*---------------------------------------------------------------------------
		** Print the record.
		*/
		if(VDUMP_MAIN_OUT_PrintRecord)
			{
			rCode=(*VDUMP_MAIN_OUT_PrintRecord)(
				/* I- outStream */ outStream,
				/* I- record    */ &rec,
				/* I- recLen    */ bytesRead
				);
			if(rCode)
				{
				fprintf(outStream, "\n");
				LIBVIGIL_ERR("(*VDUMP_MAIN_OUT_PrintRecord)() reports: %d\n", rCode);
				goto END_ERR;
				}
			}

		/*---------------------------------------------------------------------------
		** Handle roll records.
		*/
		if((VIGIL_T_VIGIL == rec.hdr.type) && (VIGIL_E_VIGIL_ROLL==rec.vigil.event))
			{
			rCode=VDUMP_MAIN_RollAuditFile(
				/* I- rollAuditRec   */ &rec,
				/* IO auditFile_fd   */ &fd,
				/* IO auditFile_path */ &VDUMP_MAIN_auditLogPath_ALLOC
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_MAIN_RollAuditFile() reports: %d\n", rCode);
				goto END_ERR;
				}
			}
		else
			VDUMP_MAIN_auditLogFileOffset += rec.hdr.length;
		}

END_ERR:

	/*---------------------------------------------------------------------------
	** Close the audit file.
	*/
	if((-1) != fd)
   	{
		if((-1) == close(fd))
			LIBVIGIL_ERR("close() failed.  errno[%d]\n", errno);
		}

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_MAIN_Follow()
*/
void VDUMP_MAIN_SignalHandler(
		/* I- sig */ int sig
		)
	{
	int rCode=VIGIL_SUCCESS;

	switch(sig)
		{
		case SIGIO:
			if(VDUMP_MAIN_verbose)
				printf("\n***SIGIO***\n");

			rCode=VDUMP_MAIN_ProcessAuditLog(
				/* I- outStream */ VDUMP_MAIN_outStream
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_MAIN_ProcessAuditLog() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case SIGINT:
			if(VDUMP_MAIN_verbose)
				printf("\n***SIGINT***\n");

			VDUMP_MAIN_exiting = VIGIL_TRUE;
		 	signal(SIGINT, VDUMP_MAIN_SignalHandler);
			break;

		default:
			break;
		}

END_ERR:

	return;
	}

/*******************************************************************************
** Local references: main()
*/
int VDUMP_MAIN_Follow(
		/* I- clientKey         */ char    *clientKey,
		/* I- outputDirPath     */ char    *outputDirPath,
		/* I- filterFilePath    */ char    *filterFilePath,   /* NULL=noFilter. */
		/* I- outputFileSizeMax */ size_t   outputFileSizeMax,
		/* I- outputRecsLimit   */ uint64_t outputRecordsLimit
		)
	{
	int    rCode=VIGIL_SUCCESS;
	size_t outputRecsTotal;
	int    auditFileNodificationOn = VIGIL_FALSE;
	int    clientFlags
				= VIGIL_CLIENT_F_ROLLOUTPUTFILE
				| VIGIL_CLIENT_F_EXCLUDE_SIGIO_PIDS
				| VIGIL_CLIENT_F_SHARE;
	char   *userName=NULL;
	int     clientOpen=VIGIL_FALSE;

	/*-------------------------------------------------------------------------
	** Validate caller arg(s).
	*/

	/*---------------------------------------------------------------------------
	** Get a unique username.
	*/
	rCode=LIBVIGIL_SYS_GenerateUsername(&userName);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_SYS_GenerateUsername() reports: %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Register SIGIO handler.
	*/
 	if(SIG_ERR == signal(SIGIO, VDUMP_MAIN_SignalHandler))
		{
		rCode=(-1);
		LIBVIGIL_ERR("signal(SIGIO) failed.\n");
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Register SIGINT handler.
	*/
 	if(SIG_ERR == signal(SIGINT, VDUMP_MAIN_SignalHandler))
		{
		rCode=(-1);
		LIBVIGIL_ERR("signal(SIGINT) failed.\n");
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Open an vigil auditing client.
	*/
	rCode=LIBVIGIL_SYS_ClientOpen(
		/* I- clientName        */ VDUMP_MAIN_clientName_ALLOC,
		/* I- clientKey         */ clientKey,
		/* I- clientFlags       */ clientFlags,
		/* I- outputDirPath     */ outputDirPath,
		/* I- outputFileSizeMax */ outputFileSizeMax,
		/* I- recordsLimit      */ outputRecordsLimit,
		/* I- userName          */ userName,
		/* I- userKey           */ userName,
		/* I- userFlags         */ VIGIL_USER_F_NONE,
		/* I- sigioPid          */ getpid()
		);
	switch(rCode)
		{
		case VIGIL_SUCCESS:
			clientOpen=VIGIL_TRUE;
			break;

		case ENOENT:
			goto END_ERR;

		default:
			LIBVIGIL_ERR("LIBVIGIL_SYS_ClientOpen() reports: %d\n", rCode);
			goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Implement filter file.
	*/
	if(filterFilePath)
		{
		rCode=LIBVIGIL_SYS_FilterFileImport(
			/* I- clientName     */ VDUMP_MAIN_clientName_ALLOC,
			/* I- clientKey      */ clientKey,
			/* I- filterFilePath */ filterFilePath
			);
		if(rCode)
			{
			LIBVIGIL_ERR("LIBVIGIL_SYS_FilterFileImport() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	/*---------------------------------------------------------------------------
	** Main loop.
	*/
	while(VIGIL_FALSE == VDUMP_MAIN_exiting)
		{
		rCode=LIBVIGIL_PROC_Pause();
		if(rCode)
			{
			LIBVIGIL_ERR("LIBVIGIL_PROC_Pause() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	if(clientOpen)
		{
		int rCode;
	
		rCode=LIBVIGIL_SYS_UserClose(
			/* I- clientName */ VDUMP_MAIN_clientName_ALLOC,
			/* I- userName   */ userName,
			/* I- userKey    */ clientKey
			);
		if(rCode)
			LIBVIGIL_ERR("LIBVIGIL_SYS_UserClose() reports: %d\n", rCode);
		}

	if(userName)
		LIBVIGIL_MEM_Free((void **)&userName);

	return(rCode);
	}

/*******************************************************************************
** Local references: main()
*/
int VDUMP_MAIN_FilterPathCreate(
		/* -O filterFilePath */ char **filterFilePath
		)
	{
	int   rCode       = VIGIL_SUCCESS;
	size_t filterFilePathSize;

	/*---------------------------------------------------------------------------
	** Validate caller arg(s).
	*/
	if(NULL == filterFilePath)
		{
		rCode=EINVAL;
		LIBVIGIL_ERR("Bad arg[NULL==filterFilePath]\n");
		goto END_ERR;
		}

	if(NULL != *filterFilePath)
		{
		rCode=EALREADY;
		LIBVIGIL_ERR("filterFilePath already initialized.\n");
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Allocate path buffer memory.
	*/
	filterFilePathSize=strlen("/tmp/vdump_filter") + 1 + 10 + 1;
	rCode=LIBVIGIL_MEM_Alloc(
		/* I- ptr  */ (void **)filterFilePath,
		/* I- size */ filterFilePathSize
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_MEM_Alloc() reports: %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Assemble path in buffer memory.
	*/
	snprintf((*filterFilePath), filterFilePathSize, "%s_0x%08X", "/tmp/vdump_filter", getpid());

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Program start.
*/
int main(
		/* I- argc */ int argc, 
		/* I- argv */ char *argv[]
		)
	{
	int      rCode             = VIGIL_SUCCESS;
	char    *cmdLineOpts        = "c:fF:hi:l:o:r:vV:x:";
	int      displayAll         = VIGIL_FALSE;
	int      arg;
	int      follow             = VIGIL_FALSE;
	char    *clientKey          = NULL;
	char    *outputDir          = NULL;
	char    *filterFilePath     = NULL;
	size_t   outputFileSizeMax  = (256 * 1024);
	uint64_t outputRecordsLimit = 0LL;

	/*---------------------------------------------------------------------------
	** Sign-on.
	*/
	LIBVIGIL_NOTE("STARTED: %s %s %s\n", PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT);

	/*---------------------------------------------------------------------------
	** Initialize globals.
	*/
	VDUMP_MAIN_outStream = stdout;

	/*---------------------------------------------------------------------------
	** Ensure root user only.
	*/
	if(getuid())
		{
		rCode=EPERM;
		LIBVIGIL_ERR("Execution of this program is limited to the root user.\n");
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Initialize library services.
	*/
	rCode=OpenNCPServLib();
	if(rCode)
		{
		LIBVIGIL_ERR("OpenNCPServLib() reports %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Parse command line args
	*/
	while((arg = getopt(argc, argv, cmdLineOpts)) != -1)
		{
		switch(arg)
			{
			case 'c':   /** Client name **/
				VDUMP_MAIN_clientName_ALLOC = strdup(optarg);
				if(NULL == VDUMP_MAIN_clientName_ALLOC)
					{
					rCode=ENOMEM;
					LIBVIGIL_ERR("clientName=strdup(\"%s\") failed.\n", optarg);
					goto END_ERR;
					}
				clientKey  = NULL;
				outputDir  = NULL;
				break;

			case 'f':   /** Follow (list events as the occur). **/
				follow=VIGIL_TRUE;
				break;

			case 'F':  /** Output format. **/
				if(0==strcasecmp("CSV", optarg))
					{
					VDUMP_MAIN_OUT_PrintRecord = VDUMP_OUT_CSV_PrintRecord;
					LIBVIGIL_NOTE("   Output format: %s\n", optarg);
					}
				else if(0==strcasecmp("NUL", optarg))
					{
					VDUMP_MAIN_OUT_PrintRecord = NULL;
					LIBVIGIL_NOTE("   Output format: %s\n", optarg);
					}
				else if(0==strcasecmp("STD", optarg))
					{
					VDUMP_MAIN_OUT_PrintRecord = VDUMP_OUT_STD_PrintRecord;;
					LIBVIGIL_NOTE("   Output format: %s\n", optarg);
					}
				else if(0==strcasecmp("XML", optarg))
					{
					VDUMP_MAIN_OUT_PrintRecord = VDUMP_OUT_XML_PrintRecord;;
					LIBVIGIL_NOTE("   Output format: %s\n", optarg);
					}
				else 
					LIBVIGIL_ERR("Specified output format[%s] is invalid.\n", optarg);
				break;

			case 'h':
				printf("Usage:   %s [OPTION]...\n", argv[0]);
				printf("Dump audit log.\n");
				printf(	"\n"
							"   -c{clientName} Target vigil client {clientName}.\n"
							"   -f             Follow audit events (live).\n"
							"   -F{formatType} Sets the output format to {formatType}.\n"
							"   -h             Displays this help screen.\n"
							"   -i{filterType} Include records matching {filterType}\n"
							"                     (Default: Include all records)\n"
							"   -l{size}       Limit the vigil auditing client's audit file to {size} bytes (before rolling).\n"
							"   -o{filePath}   Send output to file {filePath} instead of stdout.\n"
							"   -r{recCnt}     Limit the vigil auditing client's audit file to {recCnt} records (before rolling).\n"
							"   -v             Increase verboseness by one.\n"
							"   -V{n}          Set verboseness to {n}.\n"
							"   -x{filterType} Exclude records matching {filterType}\n"
							"                     (Default: Exclude no records).\n"
							"\n");
				printf(  "Filter Types:\n"
							"    VIGIL         Matches audit records generated by\n"
							"                  processes internal to the audit engine.\n"
							"    NEB           Matches audit records generated by the\n"
							"                  NSS/Novell Event Bus engine.\n"
							"    NCP           Matches audit records generated by the\n"
							"                  operations handled internally by the\n"
							"                  user-space NCP engine.\n"
							"\n");
				printf(  "Format Types:\n"
							"    CSV           Comma-separated values format.\n"
							"    NUL           No format (no output).\n"
							"    STD           Standard (legacy) format [default].\n"
							"    XML           XML format.\n"
							"\n");
				goto END_ERR;

			case 'i':
				if(NULL == filterFilePath)
					{
					rCode=VDUMP_MAIN_FilterPathCreate(&filterFilePath);
					if(rCode)
						{
						LIBVIGIL_ERR("VDUMP_MAIN_FilterPathCreate() reports: %d\n", rCode);
						goto END_ERR;
						}
					}

				rCode=VDUMP_FILTER_FilterTypeAdd(
					/* I- filePath      */ filterFilePath,
					/* I- filterTypeStr */ optarg,
					/* I- filterFlags   */ VIGIL_FILTER_F__INCLUDE
					);
				switch(rCode)
					{
					case VIGIL_SUCCESS:
						break;

					default:
						LIBVIGIL_ERR("VDUMP_FILTER_FilterTypeAdd[%s] reports: %d.\n", optarg, rCode);
						goto END_ERR;
					}
				break;

			case 'l':
				{
				size_t old=outputFileSizeMax;

				outputFileSizeMax = strtoul(optarg, NULL, 0);
				if(outputFileSizeMax < (sizeof(VIGIL_AUDIT_REC_T) * 2))
					outputFileSizeMax = (sizeof(VIGIL_AUDIT_REC_T) * 2);

				LIBVIGIL_NOTE("   Vigil auditing client output file-size max changed from %zd to %zd\n", old, outputFileSizeMax);
				break;
				}

			case 'o':
				errno=0;
				VDUMP_MAIN_outStream=fopen(optarg, "w");
				if(NULL==VDUMP_MAIN_outStream)
					{
					rCode=errno;
					LIBVIGIL_ERR("fopen(\"%s\", \"w\") failed.  errno[%d]\n", optarg, rCode);
					goto END_ERR;
					}

				LIBVIGIL_NOTE("   Sending output to file[%s].\n", optarg);
				break;

			case 'r':
				{
				uint64_t old=outputRecordsLimit;

				outputRecordsLimit = strtoull(optarg, NULL, 0);
				if(outputRecordsLimit < 25LL)
					outputRecordsLimit = 25LL;

				LIBVIGIL_NOTE("   Vigil auditing client output file max-records-limit changed from %zd to %zd\n", old, outputRecordsLimit);
				break;
				}

			case 'v':
				++VDUMP_MAIN_verbose;
				break;

			case 'V':
				VDUMP_MAIN_verbose = strtoul(optarg, NULL, 0);
				break;

			case 'x':
				if(NULL == filterFilePath)
					{
					rCode=VDUMP_MAIN_FilterPathCreate(&filterFilePath);
					if(rCode)
						{
						LIBVIGIL_ERR("VDUMP_MAIN_FilterPathCreate() reports: %d\n", rCode);
						goto END_ERR;
						}
					}

				rCode=VDUMP_FILTER_FilterTypeAdd(
					/* I- filePath      */ filterFilePath,
					/* I- filterTypeStr */ optarg,
					/* I- filterFlags   */ VIGIL_FILTER_F__EXCLUDE
					);
				switch(rCode)
					{
					case VIGIL_SUCCESS:
						break;

					default:
						LIBVIGIL_ERR("VDUMP_FILTER_FilterTypeAdd[%s] reports: %d.\n", optarg, rCode);
						goto END_ERR;
					}
				break;

			default:
					LIBVIGIL_ERR("Unknown option `-%c'.\n", optopt);
					goto END_ERR;
			}
		}

	/*---------------------------------------------------------------------------
	** Initiate follow mode?
	*/
	if(follow)
		{
		/*---------------------------------------------------------------------------
		** If clientName is unspecified,
		**   set a default clientName, clientKey & outputDir.
		*/
		if(NULL == VDUMP_MAIN_clientName_ALLOC)
			{
			if(filterFilePath)
				{
				size_t clientNameSize;
	
				clientNameSize = strlen("vdump") + 1 + 10 + 1;
				rCode=LIBVIGIL_MEM_Alloc(
					/* I- ptr  */ (void **)VDUMP_MAIN_clientName_ALLOC,
					/* I- size */ clientNameSize
					);
				if(rCode)
					{
					LIBVIGIL_ERR("LIBVIGIL_MEM_Alloc(VDUMP_MAIN_clientName_ALLOC) reports: %d.\n", rCode);
					goto END_ERR;
					}

				snprintf(VDUMP_MAIN_clientName_ALLOC, clientNameSize, "%s_0x%08X", VDUMP_MAIN_clientName_ALLOC, getpid());
				}
			else
				{
				rCode=LIBVIGIL_MEM_StrDup(
					/* I- src */ "vdump",
					/* I- dup */ &VDUMP_MAIN_clientName_ALLOC
					);
				if(rCode)
					{
					LIBVIGIL_ERR("LIBVIGIL_MEM_StrDup(VDUMP_MAIN_clientName_ALLOC) reports: %d.\n", rCode);
					goto END_ERR;
					}
				}
	
			clientKey  = "MonkeyLizzard";
			outputDir  = "/var/log/audit/vdump";
			}

		rCode=VDUMP_MAIN_Follow(
			/* I- clientKey         */ clientKey,
			/* I- outputDir         */ outputDir,
			/* I- filterPath        */ filterFilePath,
			/* I- outputFileSizeMax */ outputFileSizeMax,
			/* I- outputRecsLimit   */ outputRecordsLimit
			);
		switch(rCode)
			{
			case VIGIL_SUCCESS:
				break;

			case ENOENT:
				LIBVIGIL_ERR(
					"Cannot open client[%s].\n"
					"     Verify that kernel module vigil.ko is loaded.\n"
					"",
					VDUMP_MAIN_clientName_ALLOC);
				if(!strcmp(VDUMP_MAIN_clientName_ALLOC, VIGIL_ARCHIVE_CLIENT_NAME))
					LIBVIGIL_ERR("Verify that kernel module vigil.ko is loaded with option: archive=\"yes\".\n", VDUMP_MAIN_clientName_ALLOC);

				if(strcmp(VDUMP_MAIN_clientName_ALLOC, "vdump"))
					LIBVIGIL_ERR("Verify that client[%s] exists.\n", VDUMP_MAIN_clientName_ALLOC);

				goto END_ERR;

			default:
				LIBVIGIL_ERR("VDUMP_MAIN_Follow() reports [%d]\n", rCode);
				goto END_ERR;
			}
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Initiate "Dump from the beginning of time" mode for the specified client.
	*/
	if(NULL == VDUMP_MAIN_clientName_ALLOC)
		{
		LIBVIGIL_ERR("No clientName specified.\n");
		goto END_ERR;
		}

	rCode=LIBVIGIL_SYS_InitialFile(
		/* I- clientName       */ VDUMP_MAIN_clientName_ALLOC,
		/* -O outputFile_ALLOC */ &VDUMP_MAIN_auditLogPath_ALLOC
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_SYS_InitialFile(\"%s\", ...) reports: %d\n", VDUMP_MAIN_clientName_ALLOC, rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Dump records from audit file
	*/
	rCode=VDUMP_MAIN_ProcessAuditLog(
		/* I- outStream */ VDUMP_MAIN_outStream
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_MAIN_ProcessAuditLog(path[%s]) reports: %d\n",
			VIGIL_CLIENT_TAG_FILEPATHINITIAL, rCode);
		goto END_ERR;
		}

END_ERR:

	if(stdout != VDUMP_MAIN_outStream)
		fclose(VDUMP_MAIN_outStream);

	if(filterFilePath)
		{
		errno=0;
		if((-1) == unlink(filterFilePath))
			LIBVIGIL_ERR("unlink(\"%s\") failed.  errno[%d]\n", filterFilePath, errno);

		LIBVIGIL_MEM_Free((void **)&filterFilePath);
		}

	if(VDUMP_MAIN_clientName_ALLOC)
		LIBVIGIL_MEM_Free((void **)&VDUMP_MAIN_clientName_ALLOC);

	if(VDUMP_MAIN_auditLogPath_ALLOC)
		LIBVIGIL_MEM_Free((void **)&VDUMP_MAIN_auditLogPath_ALLOC);

	CloseNCPServLib();

	/*---------------------------------------------------------------------------
	** Sign-off.
	*/
	LIBVIGIL_NOTE("TERMINATED: %s\n", PROGRAM_NAME);

	return(rCode);
	}






