/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES Auditing Sample Client (vdump).
 |
 | File: vdump_out_csv.c
 |
 | Desc: Parses records and displays them in a "comma separated values" (csv)
 |       format.
 +-------------------------------------------------------------------------*/

/*******************************************************************************
** Compiler setup.
*/

	/*---------------------------------------------------------------------------
	** Auto Tools
	*/
#ifdef HAVE_CONFIG_H
 #ifndef __config_h__
	#include <config.h>  /* PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT */
	#define __config_h__
 #endif 
#endif

	/*---------------------------------------------------------------------------
	** ANSI/POSIX
	*/
	#include <stdlib.h>	/* exit() */
	#include <stdio.h>	/* stderr, fprintf() */
	#include <errno.h>
	#include <time.h>
	#include <string.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <dirent.h>

	#include <sys/stat.h>
	#include <fcntl.h>

	#include <signal.h>

	/*---------------------------------------------------------------------------
	** Project
	*/
	#include <vigil.h>
	#include <libvigil.h>
	#include "vdump_out_csv.h"

/*******************************************************************************
** Global storage.
*/

/*******************************************************************************
** Print a unicode as an element.
*/
int VDUMP_OUT_CSV_Print_UnicodeStr(
		/* -O outStream  */ FILE     *outStream,
		/* I- label      */ char     *label,
		/* I- unicodeStr */ uint16_t *unicodeStr
		)
	{
	int rCode=VIGIL_SUCCESS;

	fprintf(outStream, ", %s=\"", label ? label : "Path");

	while(*unicodeStr)
		{
		fprintf(outStream, "%c", (char)(0x00FF & *unicodeStr));
		++unicodeStr;
		}

	fprintf(outStream, "\"");

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_PrintDate_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- date      */ uint16_t date
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, ", %s=\"%2d %02d %04d MDY\"",
		label ? label : "",
		/* Month */ (0x01E0 & date) >> 5,
		/* Day   */ (0x001F & date),
		/* Year  */ ((0xFE00 & date) >> 9) + 1980
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_PrintTime_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- time      */ uint16_t time
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, ", %s=\"%2d:%02d:%02d HMS\"",
		label ? label : "???",
		/* Hours   */ (0xF800 & time) >> 11,
		/* Minutes */ (0x07E0 & time) >> 5,
		/* Seconds */ (0x001F & time) * 2
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintTime_64()
**                   VDUMP_OUT_CSV_PrintTimes()
*/
int VDUMP_OUT_CSV_PrintTime(
		/* -O outStream */ FILE   *outStream,
		/* I- label     */ char   *label,
		/* I- timep     */ time_t *timep
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm tm;
	char tmStr[63+1];
	
	localtime_r(timep, &tm);
	strftime(tmStr, sizeof(tmStr), "%02m/%02d/%Y %02H:%02M:%02S MDY HMS", &tm);
	fprintf(outStream, ", %s=\"%s\"",
		label ? label : "???",
		tmStr
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_PrintTime_64(
		/* -O outStream */ FILE     *outStream,
		/* I- label     */ char     *label,
		/* I- timep     */ uint64_t *timep_64
		)
	{
	int  rCode=VIGIL_SUCCESS;

	rCode=VDUMP_OUT_CSV_PrintTime(
		/* -O outStream */ outStream,
		/* I- label     */ label,
		/* I- timep     */ (time_t *)timep_64
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_PrintTimes(
		/* -O outStream */ FILE              *outStream,
		/* I- times     */ VIGIL_TIME_INFO_T *times
		)
	{
	int  rCode=VIGIL_SUCCESS;
	
	rCode=VDUMP_OUT_CSV_PrintTime_64(outStream, "Accessed", &times->accessedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintTime(accessedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_CSV_PrintTime_64(outStream, "Created", &times->createdTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintTime(createdTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_CSV_PrintTime_64(outStream, "Modified", &times->modifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintTime(times->modifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_CSV_PrintTime_64(outStream, "MetaDataModified", &times->metaDataModifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintTime(metaDataModifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_ElementPrint_Guid(
		/* -O outStream */ FILE         *o_outStream,
		/* I- label     */ char         *i_label,
		/* I- guid      */ VIGIL_GUID_T *i_guid
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char *cp = (char *)i_guid;
	char *guidStr=NULL;
	uint8_t *guidStr_ALLOC = NULL;

	/*-------------------------------------------------------------------------
	** Validate caller arg(s).
	*/
	if((NULL==o_outStream) || (NULL == i_guid))
		{
		rCode=EINVAL;
		LIBVIGIL_ERR("Bad arg[%s%s]\n",
			o_outStream ? "" : " NULL=o_outStream",
			i_guid      ? "" : " NULL=i_guid"
			);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Discover if the i_guid is a magic (nds/nss) constant.
	*/

	/** First, resolve NSS guid constants. (Root/supervisor, etc.) **/
	rCode=LIBVIGIL_GUID_ConstStr(
		/* I- guid    */ i_guid,
		/* -O guidStr */ &guidStr
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_GUID_ConstChar() reports %d\n", rCode);
		goto END_ERR;
		}

	/** Second, resolve NDS guids.  (NDS users, groups, etc.) **/
	if(NULL == guidStr)
		{
		uint16_t strDn[((255 + 1) * 2)];

		rCode=NCPMapGUIDToDN(
			/* I- guid   */ (char *)i_guid,
			/* I- dnSize */ sizeof(strDn),
			/* I- dn     */ strDn
			);
		switch(rCode)
			{
			case VIGIL_SUCCESS:
				VDUMP_OUT_STD_StrUnicode2Ascii(
					/* I- unicodeStr */ strDn,
					/* -O asciiStr   */ (uint8_t *)strDn
					);
	
				guidStr=(uint8_t*)strDn;
				break;

			case (-765): /* eDirectory error code: FFFFFD03 ERR EOF HIT */
				rCode=VIGIL_SUCCESS;
				break;

			default:
				LIBVIGIL_ERR("LIBVIGIL_EDIR_MapGuidToDn_ALLOC() reports %d\n", rCode);
				goto END_ERR;
			}
		}

	/** Third, Resolve NSS guids. (VolumeIDs, PoolIDs, etc.)**/
	if(NULL == guidStr)
		{
		rCode=LIBVIGIL_NSS_MapNssGUIDToNssObjectName(
			/* I- objectGuid       */ i_guid,
			/* -O objectName_ALLOC */ &guidStr_ALLOC
			);
		if(rCode)
			rCode=VIGIL_SUCCESS;
		else
			guidStr=guidStr_ALLOC;
		}

	/*-------------------------------------------------------------------------
	** Output the guid string.
	*/
	fprintf(o_outStream, ", %s=\"%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%s%s\"",
		i_label?i_label:"Guid",
		0xFF & cp[0],
		0xFF & cp[1],
		0xFF & cp[2],
		0xFF & cp[3],
		0xFF & cp[4],
		0xFF & cp[5],
		0xFF & cp[6],
		0xFF & cp[7],
		0xFF & cp[8],
		0xFF & cp[9],
		0xFF & cp[10],
		0xFF & cp[11],
		0xFF & cp[12],
		0xFF & cp[13],
		0xFF & cp[14],
		0xFF & cp[15],
		guidStr ? " " : "",
		guidStr ? guidStr : ""
		);

END_ERR:

	if(guidStr_ALLOC)
		LIBVIGIL_MEM_Free((void **)&guidStr_ALLOC);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecordElement()
*/
int VDUMP_OUT_CSV_PrintHex(
		/* -O outStream */ FILE  *outStream,
		/* I- buf       */ void  *buf,
		/* I- bufLen    */ size_t bufLen,
		/* I- label     */ char  *label
		)
	{
	int  rCode=VIGIL_SUCCESS;
	size_t nCnt;
	char *b=(char *)buf;

	fprintf(outStream, ", %s=\"[%zd]", label?label:"???", bufLen);

	for(nCnt=0; nCnt < bufLen; ++nCnt)
		fprintf(outStream, " %02X", 0xFF & b[nCnt]);

	fprintf(outStream, "\"");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintNcpRecord
**                   VDUMP_OUT_CSV_PrintNebRecord
**                   VDUMP_OUT_CSV_PrintVigilRecord
*/
int VDUMP_OUT_CSV_PrintRecordElement(
		/* -O outStream     */ FILE  *outStream,
		/* I- elementHeader */ char **element
		)
	{
	int  rCode=VIGIL_SUCCESS;
	VIGIL_ELEMENT_HDR_T *elementHeader = (VIGIL_ELEMENT_HDR_T *)*element;

	switch(elementHeader->type)
		{
		/*-----------------------------------------------------------------------
		** VIGIL_ELEMENT_PATH_T
		*/
		case VIGIL_ET_PATH:
			{
			VIGIL_ELEMENT_PATH_T *e=(void *)elementHeader;
			char *pathTypeStr;

			switch(e->pathType)
				{
				default:
				case VIGIL_ELEMENT_PATH_TYPE_ANONYMOUS:   pathTypeStr="path";   break;
				case VIGIL_ELEMENT_PATH_TYPE_TARGET:      pathTypeStr="target"; break;
				case VIGIL_ELEMENT_PATH_TYPE_SOURCE:      pathTypeStr="source"; break;
				case VIGIL_ELEMENT_PATH_TYPE_DESTINATION: pathTypeStr="destination"; break;
				}

			switch(e->nameSpace)
				{
				default:
				case VIGIL_ELEMENT_PATH_NS_ANONYMOUS:
					fprintf(outStream, ", %s=\"???\"", pathTypeStr);
					break;

				case VIGIL_ELEMENT_PATH_NS_FAMILIAR:
					fprintf(outStream, ", %s=\"%s\"", pathTypeStr, e->data.familiar);
					break;

				case VIGIL_ELEMENT_PATH_NS_UNICODE:
					VDUMP_OUT_CSV_Print_UnicodeStr(outStream, pathTypeStr, e->data.unicode);
					break;
				}

			break;
			}

		case VIGIL_ET_NEB_EXIT:
			{
			VIGIL_ELEMENT_NEB_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, ", EnterRetStatus=\"%d\"", e->enterRetStatus);
			fprintf(outStream, ", OpRetCode=\"%d\"", e->opRetCode);
			break;
			}

		case VIGIL_ET_NEB_CREATE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CREATE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, ", FileType=\"%d\"", e->fileType);
			fprintf(outStream, ", FileAttributes=\"0x%08X\"", e->fileAttributes);
			fprintf(outStream, ", CreateFlags=\"0x%08X\"", e->createFlags);
			fprintf(outStream, ", RequestedRights=\"0x%08X\"", e->requestedRights);
			fprintf(outStream, ", CreateAndOpen=\"0x%08X\"", e->createAndOpen);
			break;
			}

		case VIGIL_ET_NEB_CREATE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CREATE_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, ", RetOpenCreateAction=\"0x%08X\"", e->retOpenCreateAction);
			fprintf(outStream, ", RetKey=\"%llX\"", e->retKey);
			fprintf(outStream, ", RetZid=\"%llX\"", e->retZid);
			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "RetVolID", &e->retVolID);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_OPEN_ENTER:
			{
			VIGIL_ELEMENT_NEB_OPEN_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, ", RequestedRights=\"0x%08X\"", e->requestedRights);
			break;
			}

		case VIGIL_ET_NEB_OPEN_EXIT:
			{
			VIGIL_ELEMENT_NEB_OPEN_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, ", RetKey=\"%llX\"", e->retKey);
			fprintf(outStream, ", RetZid=\"%llX\"", e->retZid);
			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "RetVolID", &e->retVolID);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_CLOSE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CLOSE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, ", Key=\"%llX\"", e->key);
			fprintf(outStream, ", FhState=\"0x%08X\"", e->fhState);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NEB_CLOSE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CLOSE_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, ", FileDeleted=\"0x%08X\"", e->fileDeleted);
			break;
			}

		case VIGIL_ET_NEB_RENAME_ENTER:
			{
			VIGIL_ELEMENT_NEB_RENAME_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, ", RenameFlags=\"0x%08X\"", e->renameFlags);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER:
			{
			VIGIL_ELEMENT_MODIFY_T *e=(void *)elementHeader;

			fprintf(outStream, ", ModifyInfoMask=\"0x%08X\"", e->modifyInfoMask);
			fprintf(outStream, ", ModifyTypeInfoMask=\"0x%08X\"", e->modifyTypeInfoMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_FILE_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream, ", FileAttributes=\"0x%08X\"", e->fileAttributes);
			fprintf(outStream, ", FileAttributesModMask=\"0x%08X\"", e->fileAttributesModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_CREATED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "Created", &e->created);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "Archived", &e->archived);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "Modified", &e->modified);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ACCESSED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "Accessed", &e->accessed);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "MetaDataModified", &e->metaDataModified);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_OWNER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "Owner", &e->owner);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "Archiver", &e->archiver);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "Modifier", &e->modifier);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "MetaDataModifier", &e->metaDataModifier);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_PRIMARY_NAMESPACE:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T *e=(void *)elementHeader;

			fprintf(outStream, ", PrimaryNameSpaceID=\"%d\"", e->primaryNameSpaceID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DELETED_INFO:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintTime_64(outStream, "Time", &e->time);
			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "Id", &e->id);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MAC_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.finderInfo,
				/* I- bufLen    */ sizeof(e->info.finderInfo),
				/* I- label     */ "FinderInfo"
				);

			VDUMP_OUT_CSV_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.proDOSInfo,
				/* I- bufLen    */ sizeof(e->info.proDOSInfo),
				/* I- label     */ "ProDOSInfo"
				);

			VDUMP_OUT_CSV_PrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.filler,
				/* I- bufLen    */ sizeof(e->info.filler),
				/* I- label     */ "Filler"
				);

			fprintf(outStream, ", DirRightsMask=\"0x%08X\"", e->info.dirRightsMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_UNIX_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T *e=(void *)elementHeader;

			fprintf(outStream, ", FMode=\"0x%08X\"",            e->info.fMode);
			fprintf(outStream, ", RDev=\"0x%08X\"",             e->info.rDev);
			fprintf(outStream, ", MyFlags=\"0x%08X\"",          e->info.myFlags);
			fprintf(outStream, ", NfsUID=\"0x%08X\"",           e->info.nfsUID);
			fprintf(outStream, ", NfsGID=\"0x%08X\"",           e->info.nfsGID);
			fprintf(outStream, ", NwUID=\"0x%08X\"",            e->info.nwUID);
			fprintf(outStream, ", NwGID=\"0x%08X\"",            e->info.nwGID);
			fprintf(outStream, ", NwEveryone=\"0x%08X\"",       e->info.nwEveryone);
			fprintf(outStream, ", NwUIDRights=\"0x%08X\"",      e->info.nwUIDRights);
			fprintf(outStream, ", NwGIDRights=\"0x%08X\"",      e->info.nwGIDRights);
			fprintf(outStream, ", NwEveryoneRights=\"0x%08X\"", e->info.nwEveryoneRights);
			fprintf(outStream, ", AcsFlags=\"%08X\"",           e->info.acsFlags);
			fprintf(outStream, ", FirstCreated=\"0x%08X\"",     e->info.firstCreated);
			fprintf(outStream, ", VariableSize=\"0x%08X\"",     e->info.variableSize);
			if(e->info.variableSize)
				{
				VDUMP_OUT_CSV_PrintHex(
					/* -O outStream */ outStream,
					/* I- buf       */ e->info.variableData,
					/* I- bufLen    */ e->info.variableSize,
					/* I- label     */ "VariableData"
					);
				}
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_EXTATTR_FLAGS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T *e=(void *)elementHeader;

			fprintf(outStream, ", ExtAttrUserFlags=\"0x%08X\"", e->extAttrUserFlags);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolFeaturesEnabled=\"0x%08X\"", e->enabled);
			fprintf(outStream, ", VolFeaturesEnableModMask=\"0x%08X\"", e->enableModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "VolNdsObjectID", &e->ndsObjectID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MIN_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolSalvageMinKeepSeconds=\"0x%08X\"", e->minKeepSeconds);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MAX_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolSalvageMaxKeepSeconds=\"0x%08X\"", e->maxKeepSeconds);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_LOW_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolSalvageLowWaterMark=\"0x%08X\"", e->lowWaterMark);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_HIGH_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolSalvageHighWaterMark\"0x%08X\"", e->highWaterMark);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream, ", PoolFeaturesEnabled=\"0x%08X\"", e->enabled);
			fprintf(outStream, ", PoolFeaturesEnableModMask=\"0x%08X\"", e->enableModMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "PoolNdsObjectID", &e->ndsObjectID);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_DATA_SHREDDING_COUNT:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolDataShreddingCount=\"0x%08X\"", e->dataShreddingCount);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream, ", VolTotalSpaceQuota=\"%llu\"", e->totalSpaceQuota);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DIR_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream, ", DirQuotQuota=\"%lld\"", e->quota);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_READ_AHEAD_BLOCKS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T *e=(void *)elementHeader;

			fprintf(outStream, ", ReadAheadBlocks=\"\"");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_INH_RIGHTS_MASK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T *e=(void *)elementHeader;

			fprintf(outStream, ", InheritedRightsMask=\"0x%08X\"", e->inheritedRightsMask);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ALL_TRUSTEES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T *e=(void *)elementHeader;
			uint32_t entry;

			fprintf(outStream, ", NumEntries=\"%d\"", e->numEntries);
			for(entry=0; entry < e->numEntries; ++entry)
				{
				fprintf(outStream, ", {#%d", entry);
				VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "Trustee", (VIGIL_GUID_T *)(&e->trusteeTable[entry].trustee));
				fprintf(outStream, " Rights[0x%08X]", e->trusteeTable[entry].rights);
				fprintf(outStream, "}");
				}
			break;
			}

		case VIGIL_ET_WHO__ANONYMOUS:
			{
			VIGIL_ELEMENT_WHO_ANONYMOUS_T *e=(void *)elementHeader;
			break;
			}

		case VIGIL_ET_WHO__NCP_POSIX:
			{
			VIGIL_ELEMENT_WHO_NCP_POSIX_T *e=(void *)elementHeader;

			fprintf(outStream, ", ConnID=\"%d\"", e->connID);
			fprintf(outStream, ", TaskID=\"%d\"", e->taskID);
			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "guid", &e->guid);
			break;
			}

		case VIGIL_ET_WHO__LINUX:
			{
			VIGIL_ELEMENT_WHO_LINUX_T *e=(void *)elementHeader;

			fprintf(outStream, ", Uid=\"%d\"", e->uid);
			fprintf(outStream, ", Euid=\"%d\"", e->euid);
			fprintf(outStream, ", Suid=\"%d\"", e->suid);
			fprintf(outStream, ", Fsuid=\"%d\"", e->fsuid);
			fprintf(outStream, ", Gid=\"%d\"", e->gid);
			fprintf(outStream, ", Egid=\"%d\"", e->egid);
			fprintf(outStream, ", Sgid=\"%d\"", e->sgid);
			fprintf(outStream, ", Fsgid=\"%d\"", e->fsgid);
			fprintf(outStream, ", Comm=\"%s\"", e->comm);
			break;
			}

		case VIGIL_ET_NEB_ADDTRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_ADDTRUSTEE_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "TrusteeID", &e->trusteeID);
			fprintf(outStream, ", Rights=\"0x%08X\"", e->rights);
			fprintf(outStream, ", Attributes=\"0x%08X\"", e->attributes);
			break;
			}

		case VIGIL_ET_NEB_REMOVETRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_REMOVETRUSTEE_ENTER_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "TrusteeID", &e->trusteeID);
			fprintf(outStream, ", PurgedFileFlag=\"0x%08X\"", e->purgedFileFlag);
			break;
			}

		case VIGIL_ET_NEB_SETINHERITEDRIGHTS_ENTER:
			{
			VIGIL_ELEMENT_NEB_SETINHERITEDRIGHTS_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, ", InheritedRights=\"0x%08X\"", e->inheritedRights);
			fprintf(outStream, ", AuthorizeFlag=\"0x%08X\"",   e->authorizeFlag);
			break;
			}

		case VIGIL_ET_NAME:
			{
			VIGIL_ELEMENT_NAME_T *e=(void *)elementHeader;
			char *nameTypeStr;

			switch(e->nameType)
				{
				case VIGIL_ELEMENT_NAMETYPE_CLIENT:     nameTypeStr="Client"; break;
				case VIGIL_ELEMENT_NAMETYPE_CLIENTUSER: nameTypeStr="ClientUser"; break;
				default:                                nameTypeStr="(unknown)"; break;
				}

			fprintf(outStream, ", %s=\"%s\"", nameTypeStr, e->name);
			break;
			}

		case VIGIL_ET_NCP_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_NCP_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream, ", SearchAttributes=\"%d\"", 0xFF & e->searchAttributes);
			fprintf(outStream, ", DesiredAccessRights=\"%d\"", 0xFF & e->desiredAccessRights);
			fprintf(outStream, ", LinuxPath=\"%s\"", e->linuxPath);
			fprintf(outStream, ", Status\"%d\"", e->status);
			/** Check e->hdr.length to verify that the values below were passed. **/
			if(0==e->status)
				{
				fprintf(outStream, ", FileHandle=\"%02X%02X%02X%02X%02X%02X\"",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				fprintf(outStream, ", FileAttributes=\"%02X\"", 0xFF & e->fileAttributes);
				fprintf(outStream, ", FileExecuteType=\"%02X\"", 0xFF & e->fileExecuteType);
				fprintf(outStream, ", FileLen=\"%lu\"", e->fileLen);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "CreationDate", e->creationDate);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "LastAccessDate", e->lastAccessDate);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "LastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_CSV_PrintTime_MSDOS(outStream, "LastUpdateTime", e->lastUpdateTime);
				}
			break;
			}

		case VIGIL_ET_NCP_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_NCP_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream, ", FileHandle=\"%02X%02X%02X%02X%02X%02X\"",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5]
				);
			fprintf(outStream, ", Status=\"%d\"", e->status);
			break;
			}

#if 0   // TODO:
		case VIGIL_ET_WHO__EDIR_GUID:
			break;
#endif

		case VIGIL_ET_NSS_CREATE:
			{
			VIGIL_ELEMENT_NSS_Create_T *e=(void *)elementHeader;

			fprintf(outStream, ", key=\"0x%llX\"", e->key);
			fprintf(outStream, ", requestedRights=\"0x%08X\"", e->requestedRights);
			fprintf(outStream, ", createFlags=\"0x%08X\"", e->createFlags);
			fprintf(outStream, ", createAndOpen=\"0x%08X\"", e->createAndOpen);
			fprintf(outStream, ", retOpenCreateAction=\"0x%08X\"", e->retOpenCreateAction);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_OPEN:
			{
			VIGIL_ELEMENT_NSS_Open_T *e=(void *)elementHeader;

			fprintf(outStream, ", requestedRights=\"0x%08X\"", e->requestedRights);
			fprintf(outStream, ", key=\"0x%llX\"", e->key);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_CLOSE:
			{
			VIGIL_ELEMENT_NSS_Close_T *e=(void *)elementHeader;

			fprintf(outStream, ", key=\"0x%llX\"", e->key);
			fprintf(outStream, ", fhState=\"0x%08X\"", e->fhState);
			fprintf(outStream, ", fileDeleted=\"0x%08X\"", e->fileDeleted);
			VDUMP_OUT_CSV_PrintTimes(outStream, &e->times);
			break;
			}

		case VIGIL_ET_NSS_RENAME:
			{
			VIGIL_ELEMENT_NSS_Rename_T *e=(void *)elementHeader;

			fprintf(outStream, ", renameFlags=\"0x%08X\"", e->renameFlags);
			break;
			}

		case VIGIL_ET_NSS_LINK:
			{
			VIGIL_ELEMENT_NSS_Link_T *e=(void *)elementHeader;

			fprintf(outStream, ", linkFlags=\"0x%08X\"", e->linkFlags);
			break;
			}

		case VIGIL_ET_NSS_MODIFYINFO:
			{
			VIGIL_ELEMENT_NSS_ModifyInfo_T *e=(void *)elementHeader;

			fprintf(outStream, ", modifyInfoMask=\"0x%llX\"", e->modifyInfoMask);
			fprintf(outStream, ", modifyTypeInfoMask=\"0x%llX\"", e->modifyTypeInfoMask);
			break;
			}

		case VIGIL_ET_NSS_ADDTRUSTEE:
			{
			VIGIL_ELEMENT_NSS_AddTrustee_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "trusteeId", &e->trusteeID);
			fprintf(outStream, ", rights=\"0x%08X\"", e->rights);
			fprintf(outStream, ", previousRights=\"0x%08X\"", e->previousRights);
			fprintf(outStream, ", attributes=\"0x%08X\"", e->attributes);
			break;
			}

		case VIGIL_ET_NSS_REMOVETRUSTEE:
			{
			VIGIL_ELEMENT_NSS_RemoveTrustee_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "trusteeId", &e->trusteeID);
			fprintf(outStream, ", rights=\"0x%08X=\"", e->rights);
			fprintf(outStream, ", purgedFileFlag=\"0x%08X\"", e->purgedFileFlag);
			break;
			}

		case VIGIL_ET_NSS_SETINHERITEDRIGHTS:
			{
			VIGIL_ELEMENT_NSS_SetInheritedRights_T *e=(void *)elementHeader;

			fprintf(outStream, ", inheritedRights=\"0x%08X\"", e->inheritedRights);
			fprintf(outStream, ", previousInheritedRights=\"0x%08X\"", e->previousInheritedRights);
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_CIFS_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream, ", linuxPath=\"%s\"", e->linuxPath);
			fprintf(outStream, ", desiredAccessRights=\"0x%02X\"", 0xFF & e->desiredAccessRights);
			fprintf(outStream, ", status=\"%u\"", 0xFF & e->status);
			if(0 == e->status)
				{
				fprintf(outStream, ", fileHandle=\"%02X%02X%02X%02X%02X%02X\"",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				fprintf(outStream, ", fileAttributes=\"%02X\"", 0xFF & e->fileAttributes);
				fprintf(outStream, ", fileExectueType=\"%02X\"", 0xFF & e->fileExecuteType);
				fprintf(outStream, ", fileLen=\"%u\"", e->fileLen);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "creationDate", e->creationDate);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "lastAccessDate", e->lastAccessDate);
				VDUMP_OUT_CSV_PrintDate_MSDOS(outStream, "lastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_CSV_PrintTime_MSDOS(outStream, "lastUpdateTime", e->lastUpdateTime);
				}
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_CIFS_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream, ", fileHandle=\"%02X%02X%02X%02X%02X%02X\"",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5]
				);

			fprintf(outStream, " status=\"%u\"", 0xFF & e->status);
			break;
			}

		case VIGIL_ET_PMD_NCP:
			{
			VIGIL_ELEMENT_PMD_NCP_T *e=(void *)elementHeader;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "guid", &e->guid);
			fprintf(outStream, ", connID=\"%u\"", e->connID);
			fprintf(outStream, ", taskID=\"%u\"", e->taskID);

			break;
			}

		case VIGIL_ET_PMD_CIFS:
			{
			VIGIL_ELEMENT_PMD_CIFS_T *e=(void *)elementHeader;
			char *elementTmp = (char *)&e->netAddr;

			VDUMP_OUT_CSV_ElementPrint_Guid(outStream, "guid", &e->guid);
			fprintf(outStream, ", connID=\"%u\"", e->connID);

			rCode=VDUMP_OUT_CSV_PrintRecordElement(
				/* -O outStream     */ outStream,
				/* I- elementHeader */ &elementTmp
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VIGIL_ET_PMD_CIFS:VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
				rCode=VIGIL_SUCCESS;
//				goto END_ERR;
				}

			break;
			}

		case VIGIL_ET_NET_ADDR_IPv4:
			{
			VIGIL_ELEMENT_NET_ADDR_T *e=(void *)elementHeader;

			fprintf(outStream, " netAddr_IPv4[%02X.%02X.%02X.%02X]",
				0xFF & e->addr.octet[0],
				0xFF & e->addr.octet[1],
				0xFF & e->addr.octet[2],
				0xFF & e->addr.octet[3]
				);

			break;
			}

		case VIGIL_ET_NET_ADDR_IPv6:
			{
			VIGIL_ELEMENT_NET_ADDR_T *e=(void *)elementHeader;

			fprintf(outStream, " netAddr_IPv6[%4X:%4X:%4X:%4X:%4X:%4X:%4X:%4X]",
				0xFFFF & e->addr.hextet[0],
				0xFFFF & e->addr.hextet[1],
				0xFFFF & e->addr.hextet[2],
				0xFFFF & e->addr.hextet[3],
				0xFFFF & e->addr.hextet[4],
				0xFFFF & e->addr.hextet[5],
				0xFFFF & e->addr.hextet[6],
				0xFFFF & e->addr.hextet[7]
				);

			break;
			}

		default:
			fprintf(outStream, " ElementType=\"%04X [unknown]\"", elementHeader->type);
			break;
		}

	(*element) += elementHeader->length;

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecord()
*/
int VDUMP_OUT_CSV_PrintNcpRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NCP_T *ncp
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = ncp->data;

	switch(ncp->event)
		{
		case VIGIL_E_NCP_OPEN:
			fprintf(outStream, ", Event=\"%d OPEN\"", ncp->event);
			break;

		case VIGIL_E_NCP_CLOSE:
			fprintf(outStream, ", Event=\"%d CLOSE\"", ncp->event);
			break;

		default:
			fprintf(outStream, ", Event=\"%d [unknown]\"", ncp->event);
			break;
		}

	for(curElement=0; curElement < ncp->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_CSV_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecord()
*/
int VDUMP_OUT_CSV_PrintCifsRecord(
		/* -O outStream */ FILE                   *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_CIFS_T *cifs
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = cifs->data;

	switch(cifs->event)
		{
		case VIGIL_E_CIFS_OPEN:
			fprintf(outStream, ", Event=\"%d OPEN\"", cifs->event);
			break;

		case VIGIL_E_CIFS_CLOSE:
			fprintf(outStream, ", Event=\"%d CLOSE\"", cifs->event);
			break;

		default:
			fprintf(outStream, ", Event=\"%d [unknown]\"", cifs->event);
			break;
		}

	for(curElement=0; curElement < cifs->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_CSV_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecord()
*/
int VDUMP_OUT_CSV_PrintNebRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NEB_T *neb
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = neb->data;

	fprintf(outStream, ", EnterExitID=\"%u\"", neb->enterExitID);

	switch(neb->event)
		{
		case VIGIL_E_NEB_DELETE_ENTER:
			fprintf(outStream, ", Event=\"%d DELETE_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_DELETE_EXIT:
			fprintf(outStream, ", Event=\"%d DELETE_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_CREATE_ENTER:
			fprintf(outStream, ", Event=\"%d CREATE_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_CREATE_EXIT:
			fprintf(outStream, ", Event=\"%d CREATE_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_OPEN_ENTER:
			fprintf(outStream, ", Event=\"%d OPEN_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_OPEN_EXIT:
			fprintf(outStream, ", Event=\"%d OPEN_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_CLOSE_ENTER:
			fprintf(outStream, ", Event=\"%d CLOSE_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_CLOSE_EXIT:
			fprintf(outStream, ", Event=\"%d CLOSE_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_RENAME_ENTER:
			fprintf(outStream, ", Event=\"%d RENAME_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_RENAME_EXIT:
			fprintf(outStream, ", Event=\"%d RENAME_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_MODIFY_ENTER:
			fprintf(outStream, ", Event=\"%d MODIFY_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_MODIFY_EXIT:
			fprintf(outStream, ", Event=\"%d MODIFY_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_ADDTRUSTEE_ENTER:
			fprintf(outStream, ", Event=\"%d TRUSTEE_ADD_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_ADDTRUSTEE_EXIT:
			fprintf(outStream, ", Event=\"%d TRUSTEE_ADD_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_REMOVETRUSTEE_ENTER:
			fprintf(outStream, ", Event=\"%d TRUSTEE_REMOVE_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_REMOVETRUSTEE_EXIT:
			fprintf(outStream, ", Event=\"%d TRUSTEE_REMOVE_EXIT\"", neb->event);
			break;

		case VIGIL_E_NEB_SETINHERITEDRIGHTS_ENTER:
			fprintf(outStream, ", Event=\"%d INHERITED_RIGHTS_SET_ENTER\"", neb->event);
			break;

		case VIGIL_E_NEB_SETINHERITEDRIGHTS_EXIT:
			fprintf(outStream, ", Event=\"%d INHERITED_RIGHTS_SET_EXIT\"", neb->event);
			break;

		default:
			fprintf(outStream, ", Event=\"%d [unknown]\"", neb->event);
			break;
		}

	for(curElement=0; curElement < neb->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_CSV_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecord()
*/
int VDUMP_OUT_CSV_PrintNssRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NSS_T *nss
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = nss->data;

	/*---------------------------------------------------------------------------
	** Event
	*/
	switch(nss->event)
		{
		case VIGIL_E_NSS_DELETE:
			fprintf(outStream, ", Event=\"%d DELETE\"", nss->event);
			break;

		case VIGIL_E_NSS_CREATE:
			fprintf(outStream, ", Event=\"%d CREATE\"", nss->event);
			break;

		case VIGIL_E_NSS_OPEN:
			fprintf(outStream, ", Event=\"%d OPEN\"", nss->event);
			break;

		case VIGIL_E_NSS_CLOSE:
			fprintf(outStream, ", Event=\"%d CLOSE\"", nss->event);
			break;

		case VIGIL_E_NSS_RENAME:
			fprintf(outStream, ", Event=\"%d RENAME\"", nss->event);
			break;

		case VIGIL_E_NSS_LINK:
			fprintf(outStream, ", Event=\"%d LINK\"", nss->event);
			break;

		case VIGIL_E_NSS_MODIFY:
			fprintf(outStream, ", Event=\"%d MODIFY\"", nss->event);
			break;

		case VIGIL_E_NSS_ADDTRUSTEE:
			fprintf(outStream, ", Event=\"%d TRUSTEE_ADD\"", nss->event);
			break;

		case VIGIL_E_NSS_REMOVETRUSTEE:
			fprintf(outStream, ", Event=\"%d TRUSTEE_REMOVE\"", nss->event);
			break;

		case VIGIL_E_NSS_SETINHERITEDRIGHTS:
			fprintf(outStream, ", Event=\"%d SET_INHERITED_RIGHTS\"", nss->event);
			break;

		default:
			fprintf(outStream, ", Event=\"%d [unknown]\"", nss->event);
			break;
		}

	/*---------------------------------------------------------------------------
	** Event metadata
	*/
	fprintf(outStream,
		", TaskID=\"%lu\""
		", Zid=\"%llX\""
		", ParentZid=\"%llX\""
		", FileType=\"%lu\""
		", FileAttributes=\"%llX\""
		", OpRetCode=\"%d\""
		"",
		nss->taskID,
		nss->zid,
		nss->parentZid,
		nss->fileType,
		nss->fileAttributes,
		nss->opRetCode
		);

	rCode=VDUMP_OUT_CSV_ElementPrint_Guid(
		/* -O outStream */ outStream,
		/* I- label     */ "VolID",
		/* I- ndsGuid   */ &nss->volID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_ElementPrint_Guid(VolID) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_CSV_ElementPrint_Guid(
		/* -O outStream */ outStream,
		/* I- label     */ "UserID",
		/* I- ndsGuid   */ &nss->userID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_CSV_ElementPrint_Guid(UserID) reports: %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Linux metadata
	*/
	fprintf(outStream, ", uid=\"%d\"",   nss->uid);
	fprintf(outStream, ", euid=\"%d\"",  nss->euid);
	fprintf(outStream, ", suid=\"%d\"",  nss->suid);
	fprintf(outStream, ", fsuid=\"%d\"", nss->fsuid);
	fprintf(outStream, ", gid=\"%d\"",   nss->gid);
	fprintf(outStream, ", egid=\"%d\"",  nss->egid);
	fprintf(outStream, ", sgid=\"%d\"",  nss->sgid);
	fprintf(outStream, ", fsgid=\"%d\"", nss->fsgid);
	fprintf(outStream, ", comm=\"%s\"",  nss->comm);

	/*---------------------------------------------------------------------------
	** Elements
	*/
	for(curElement=0; curElement < nss->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_CSV_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_CSV_PrintRecord()
*/
int VDUMP_OUT_CSV_PrintVigilRecord(
		/* -O outStream */ FILE                    *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_VIGIL_T *rec
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = rec->data;

	switch(rec->event)
		{
		case VIGIL_E_VIGIL_ROLL:
			{
			fprintf(outStream, ", Event=\"%d Roll\"", rec->event);
			break;
			}

		case VIGIL_E_VIGIL_START:
			fprintf(outStream, ", Event=\"%d Start: vigil.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_STOP:
			fprintf(outStream, ", Event=\"%d Stop: vigil.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_NEB_START:
			fprintf(outStream, ", Event=\"%d Start: vigil_neb.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_NEB_STOP:
			fprintf(outStream, ", Event=\"%d Stop: vigil_neb.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_NCP_START:
			fprintf(outStream, ", Event=\"%d Start: vigil_ncp.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_NCP_STOP:
			fprintf(outStream, ", Event=\"%d Stop: vigil_ncp.ko\"", rec->event);
			break;

		case VIGIL_E_VIGIL_CLIENT_START:
			fprintf(outStream, ", Event=\"%d Start: AuditingClient\"", rec->event);
			break;

		case VIGIL_E_VIGIL_CLIENT_STOP:
			fprintf(outStream, ", Event=\"%d Stop: AuditingClient\"", rec->event);
			break;

		case VIGIL_E_VIGIL_USER_START:
			fprintf(outStream, ", Event=\"%d Start: AuditingClientUser\"", rec->event);
			break;

		case VIGIL_E_VIGIL_USER_STOP:
			fprintf(outStream, ", Event=\"%d Stop: AuditingClientUser\"", rec->event);
			break;

		default:
			fprintf(outStream, ", Event=\"%d [unknown]\"", rec->event);
			break;
		}

	for(curElement=0; curElement < rec->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_CSV_PrintRecordElement(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintRecordElement() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_MAIN_ProcessAuditLog()
*/
int VDUMP_OUT_CSV_PrintRecord(
		/* -O outStream */ FILE              *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_T *rec,
		/* I- recLen    */ size_t             recLen
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm timeDate;
	char timeDateStr[63+1];

	rCode=LIBVIGIL_PARSE_RecValidate(rec, recLen);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_RecValidate() reports: %d\n", rCode);
		goto END_ERR;
		}

	fprintf(outStream,
		" recNo=\"%llu\""
		",pid=\"%u\"",
		rec->hdr.recNo,
		rec->hdr.pid
		);

	/** Print record's time stamp. **/
	localtime_r(&rec->hdr.time.tv_sec, &timeDate);
	strftime(timeDateStr, sizeof(timeDateStr), "%02m/%02d/%Y %02H:%02M:%02S", &timeDate);
	fprintf(outStream, ", TimeStamp=\"%s %06lu MDY HMS USEC\"", timeDateStr, rec->hdr.time.tv_usec);

	/** Parse record by type. **/
	switch(rec->hdr.type)
		{
		case VIGIL_T_VIGIL:
			fprintf(outStream, ", Type=\"%d VIGIL\"", rec->hdr.type);
			rCode=VDUMP_OUT_CSV_PrintVigilRecord(
				/* -O outStream */ outStream,
				/* I- vigilRec  */ &rec->vigil
				);
			switch(rCode)
				{
				case VIGIL_SUCCESS:
					break;

				case ESTALE:
					break;

				default:
					LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintVigilRecord() reports: %d\n", rCode);
					goto END_ERR;
				}
			break;

		case VIGIL_T_NEB:
			fprintf(outStream, ", Type=\"%d NEB\"", rec->hdr.type);
			rCode=VDUMP_OUT_CSV_PrintNebRecord(
				/* -O outStream */ outStream,
				/* I- nebRec    */ &rec->neb
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintNebRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NCP:
			fprintf(outStream, ", Type=\"%d NCP\"", rec->hdr.type);
			rCode=VDUMP_OUT_CSV_PrintNcpRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->ncp
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintNcpRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_CIFS:
			fprintf(outStream, ", Type=\"%d CIFS\"", rec->hdr.type);
			rCode=VDUMP_OUT_CSV_PrintCifsRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->cifs
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintCifsRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NSS:
			fprintf(outStream, ", Type=\"%d NSS\"", rec->hdr.type);
			rCode=VDUMP_OUT_CSV_PrintNssRecord(
				/* -O outStream */ outStream,
				/* I- nebRec    */ &rec->nss
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_CSV_PrintNssRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		default:
			fprintf(outStream, ", Type=\"%d [unknown]\"", rec->hdr.type);
			break;
		}

	/** Print END-OF-RECORD newline **/
	fprintf(outStream, "\n");

END_ERR:

	return(rCode);
	}






