/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES Auditing Sample Client (vdump).
 |
 | File: vdump_out_xml.c
 |
 | Desc: Parses records and displays them in an XML format.
 +-------------------------------------------------------------------------*/

/*******************************************************************************
** Compiler setup.
*/

	/*---------------------------------------------------------------------------
	** Auto Tools
	*/
#ifdef HAVE_CONFIG_H
 #ifndef __config_h__
	#include <config.h>  /* PROGRAM_NAME, PROGRAM_VERSION, PROGRAM_COPYRIGHT */
	#define __config_h__
 #endif 
#endif

	/*---------------------------------------------------------------------------
	** ANSI/POSIX
	*/
	#include <stdlib.h>	/* exit() */
	#include <stdio.h>	/* stderr, fprintf() */
	#include <errno.h>
	#include <time.h>
	#include <string.h>
	#include <unistd.h>
	#include <sys/types.h>
	#include <dirent.h>

	#include <sys/stat.h>
	#include <fcntl.h>

	#include <signal.h>

	/*---------------------------------------------------------------------------
	** Project
	*/
	#include <vigil.h>
	#include <libvigil.h>
	#include "vdump_out_xml.h"

/*******************************************************************************
** Global storage.
*/

/*******************************************************************************
** Print a unicode as an element.
*/
int VDUMP_OUT_XML_Print_UnicodeStr(
		/* -O outStream  */ FILE     *outStream,
		/* I- unicodeStr */ uint16_t *unicodeStr
		)
	{
	int rCode=VIGIL_SUCCESS;

	while(*unicodeStr)
		{
		fprintf(outStream, "%c", (char)(0x00FF & *unicodeStr));
		++unicodeStr;
		}

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintDate_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- date      */ uint16_t date
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s=\"%2d/%02d/%04d MDY\"",
		label ? label : "",
		/* Day   */ (0x001F & date),
		/* Month */ (0x01E0 & date) >> 5,
		/* Year  */ ((0xFE00 & date) >> 9) + 1980
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintTime_MSDOS(
		/* -O outStream */ FILE    *outStream,
		/* I- label     */ char    *label,
		/* I- time      */ uint16_t time
		)
	{
	int  rCode=VIGIL_SUCCESS;

	fprintf(outStream, " %s=\"%2d:%02d:%02d HMS\"",
		label ? label : "???",
		/* Hours   */ (0xF800 & time) >> 11,
		/* Minutes */ (0x07E0 & time) >> 5,
		/* Seconds */ (0x001F & time) * 2
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_AttrPrintTime_64()
**                   VDUMP_OUT_XML_AttrPrintTimes()
*/
int VDUMP_OUT_XML_AttrPrintTime(
		/* -O outStream */ FILE   *outStream,
		/* I- label     */ char   *label,
		/* I- timep     */ time_t *timep
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm tm;
	char tmStr[63+1];
	
	localtime_r(timep, &tm);
	strftime(tmStr, sizeof(tmStr), "%02m/%02d/%Y %02H:%02M:%02S", &tm);
	fprintf(outStream, " %s=\"%s MDY HMS\"",
		label ? label : "???",
		tmStr
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintTime_64(
		/* -O outStream */ FILE     *outStream,
		/* I- label     */ char     *label,
		/* I- timep     */ uint64_t *timep_64
		)
	{
	int  rCode=VIGIL_SUCCESS;

	rCode=VDUMP_OUT_XML_AttrPrintTime(
		/* -O outStream */ outStream,
		/* I- label     */ label,
		/* I- timep     */ (time_t *)timep_64
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintTimes(
		/* -O outStream */ FILE              *outStream,
		/* I- times     */ VIGIL_TIME_INFO_T *times
		)
	{
	int  rCode=VIGIL_SUCCESS;
	
	rCode=VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Accessed", &times->accessedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintTime(accessedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Created", &times->createdTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintTime(createdTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Modified", &times->modifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintTime(times->modifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_XML_AttrPrintTime_64(outStream, "MetaDataModified", &times->metaDataModifiedTime);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintTime(metaDataModifiedTime) reports: %d\n", rCode);
		goto END_ERR;
		}

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintGuid(
		/* -O outStream */ FILE         *o_outStream,
		/* I- label     */ char         *i_label,
		/* I- guid      */ VIGIL_GUID_T *i_guid
		)
	{
	int  rCode=VIGIL_SUCCESS;
	char *cp = (char *)i_guid;
	char *guidStr=NULL;
	uint8_t *guidStr_ALLOC = NULL;

	/*-------------------------------------------------------------------------
	** Validate caller arg(s).
	*/
	if((NULL==o_outStream) || (NULL==i_guid))
		{
		rCode=EINVAL;
		LIBVIGIL_ERR("Bad arg[%s%s]\n",
			o_outStream ? "" : " NULL=o_outStream",
			i_guid      ? "" : " NULL=i_guid"
			);
		goto END_ERR;
		}

	/*-------------------------------------------------------------------------
	** Discover if the i_guid is a magic (nds/nss) constant.
	*/

	/** First, resolve NSS guid constants. (Root/supervisor, etc.) **/
	rCode=LIBVIGIL_GUID_ConstStr(
		/* I- guid    */ i_guid,
		/* -O guidStr */ &guidStr
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_GUID_ConstChar() reports %d\n", rCode);
		goto END_ERR;
		}

	/** Second, resolve NDS guids.  (NDS users, groups, etc.) **/
	if(NULL == guidStr)
		{
		uint16_t strDn[((255 + 1) * 2)];

		rCode=NCPMapGUIDToDN(
			/* I- guid   */ (char *)i_guid,
			/* I- dnSize */ sizeof(strDn),
			/* I- dn     */ strDn
			);
		switch(rCode)
			{
			case VIGIL_SUCCESS:
				VDUMP_OUT_STD_StrUnicode2Ascii(
					/* I- unicodeStr */ strDn,
					/* -O asciiStr   */ (uint8_t *)strDn
					);
	
				guidStr=(uint8_t*)strDn;
				break;

			case (-765): /* eDirectory error code: FFFFFD03 ERR EOF HIT */
				rCode=VIGIL_SUCCESS;
				break;

			default:
				LIBVIGIL_ERR("LIBVIGIL_EDIR_MapGuidToDn_ALLOC() reports %d\n", rCode);
				goto END_ERR;
			}
		}

	/** Third, Resolve NSS guids. (VolumeIDs, PoolIDs, etc.)**/
	if(NULL == guidStr)
		{
		rCode=LIBVIGIL_NSS_MapNssGUIDToNssObjectName(
			/* I- objectGuid       */ i_guid,
			/* -O objectName_ALLOC */ &guidStr_ALLOC
			);
		if(rCode)
			rCode=VIGIL_SUCCESS;
		else
			guidStr=guidStr_ALLOC;
		}

	/*-------------------------------------------------------------------------
	** Output the guid string.
	*/
	fprintf(o_outStream, " %s=\"%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%02X%s%s\"",
		i_label?i_label:"Guid",
		0xFF & cp[0],
		0xFF & cp[1],
		0xFF & cp[2],
		0xFF & cp[3],
		0xFF & cp[4],
		0xFF & cp[5],
		0xFF & cp[6],
		0xFF & cp[7],
		0xFF & cp[8],
		0xFF & cp[9],
		0xFF & cp[10],
		0xFF & cp[11],
		0xFF & cp[12],
		0xFF & cp[13],
		0xFF & cp[14],
		0xFF & cp[15],
		guidStr ? " " : "",
		guidStr ? guidStr : ""
		);

END_ERR:

	if(guidStr_ALLOC)
		LIBVIGIL_MEM_Free((void **)&guidStr_ALLOC);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecordElement()
*/
int VDUMP_OUT_XML_AttrPrintHex(
		/* -O outStream */ FILE  *outStream,
		/* I- buf       */ void  *buf,
		/* I- bufLen    */ size_t bufLen,
		/* I- label     */ char  *label
		)
	{
	int  rCode=VIGIL_SUCCESS;
	size_t nCnt;
	char *b=(char *)buf;

	fprintf(outStream, " %s=\"[%zd]", label?label:"???", bufLen);

	for(nCnt=0; nCnt < bufLen; ++nCnt)
		fprintf(outStream, " %02X", 0xFF & b[nCnt]);

	fprintf(outStream, "\"");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
**
*/
int VDUMP_OUT_XML_BodyPrintHex(
		/* -O outStream  */ FILE    *outStream,
		/* I- buf        */ void    *inBuf,
		/* I- bufLen     */ size_t   inBufLen,
		/* I- tag        */ char    *tag,
		/* I- margin     */ int      margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	size_t a, b;
	char *inBufCh=(char *)inBuf;
	char  hexBuf[(3 * 16) + 1];
	char  asciiBuf[16+1];

	fprintf(outStream, "%*s<%s Size=\"%zu\">\n",
		margin, "",
		tag,
		inBufLen
		);

	for(a=0; a < inBufLen; a+=16)
		{
		memset(asciiBuf, 0, sizeof(asciiBuf));
		memset(hexBuf, 0, sizeof(hexBuf));
		for(b=0; b<16; ++b)
			{
			if((a+b) < inBufLen)
				{
				asciiBuf[b] = isprint(inBufCh[a+b]) ? inBufCh[a+b] : '.';
				sprintf(&hexBuf[3*b], " %02X", 0xFF & inBufCh[a+b]);
				}
			else
				{
				asciiBuf[b] = '\0';
				sprintf(&hexBuf[3*b], "   ");
				}
			}

		fprintf(outStream,
			"%*s%s // \"%s\"\n",
			1+margin,"",
			hexBuf,
			asciiBuf
			);
		}

	fprintf(outStream,
		"%*s</%s>\n",
		margin, "",
		tag
		);

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintNcpRecord
**                   VDUMP_OUT_XML_PrintNebRecord
**                   VDUMP_OUT_XML_PrintVigilRecord
*/
int VDUMP_OUT_XML_ElementPrint(
		/* -O outStream     */ FILE  *outStream,
		/* I- elementHeader */ char **element,
		/* I- margin        */ int    margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	VIGIL_ELEMENT_HDR_T *elementHeader = (VIGIL_ELEMENT_HDR_T *)*element;

	switch(elementHeader->type)
		{
		/*-----------------------------------------------------------------------
		** VIGIL_ELEMENT_PATH_T
		*/
		case VIGIL_ET_PATH:
			{
			VIGIL_ELEMENT_PATH_T *e=(void *)elementHeader;
			char *pathTypeStr = "";
			char *nameSpaceStr = "";

			switch(e->pathType)
				{
				default:
				case VIGIL_ELEMENT_PATH_TYPE_ANONYMOUS:   pathTypeStr="Path";        break;
				case VIGIL_ELEMENT_PATH_TYPE_TARGET:      pathTypeStr="Target";      break;
				case VIGIL_ELEMENT_PATH_TYPE_SOURCE:      pathTypeStr="Source";      break;
				case VIGIL_ELEMENT_PATH_TYPE_DESTINATION: pathTypeStr="Destination"; break;
				}

			switch(e->nameSpace)
				{
				case VIGIL_ELEMENT_PATH_NS_FAMILIAR:  nameSpaceStr="familiar";  break;
				case VIGIL_ELEMENT_PATH_NS_ANONYMOUS: nameSpaceStr="anonymous"; break;
				case VIGIL_ELEMENT_PATH_NS_UNICODE:   nameSpaceStr="unicode";   break;
				default:                              nameSpaceStr="[unknown]"; break;
				}

			fprintf(outStream,
				"%*s<PATH Type=\"%d %s\" NameSpace=\"%d %s\">",
				margin, "",
				e->pathType, pathTypeStr,
				e->nameSpace, nameSpaceStr
				);

			switch(e->nameSpace)
				{
				default:
				case VIGIL_ELEMENT_PATH_NS_ANONYMOUS:
					fprintf(outStream, "%*s[??? namespace=%u]",
						margin+1, "",
						e->nameSpace
						);
					break;

				case VIGIL_ELEMENT_PATH_NS_FAMILIAR:
					fprintf(outStream, "%*s%s",
						margin+1, "",
						e->data.familiar
						);
					break;

				case VIGIL_ELEMENT_PATH_NS_UNICODE:
					VDUMP_OUT_XML_Print_UnicodeStr(outStream, e->data.unicode);
					break;
				}

			fprintf(outStream, "%*s</PATH>\n", margin, "");

			break;
			}

		case VIGIL_ET_NEB_EXIT:
			{
			VIGIL_ELEMENT_NEB_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<EXIT Status=\"%d\" RetCode=\"%d\" />\n",
				margin, "",
				e->enterRetStatus,
				e->opRetCode
				);
			break;
			}

		case VIGIL_ET_NEB_CREATE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CREATE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CREATE_ENTER FileType=\"%d\" FileAttributes=\"0x%08X\" createFlags=\"0x%08X\" requestedRights=\"0x%08X\" createAndOpen=\"0x%08X\" />\n",
				margin, "",
				e->fileType,
				e->fileAttributes,
				e->createFlags,
				e->requestedRights,
				e->createAndOpen
				);
			break;
			}

		case VIGIL_ET_NEB_CREATE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CREATE_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CREATE_EXIT RetOpenCreateAction=\"0x%08X\" RetKey=\"0x%llX\" RetZid=\"0x%llX\"",
				margin, "",
				e->retOpenCreateAction,
				e->retKey,
				e->retZid
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "RetVolID", &e->retVolID);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_OPEN_ENTER:
			{
			VIGIL_ELEMENT_NEB_OPEN_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream, 
				"%*s<OPEN_ENTER RequestedRights=\"0x%08X\" />\n",
				margin, "",
				e->requestedRights
				);
			break;
			}

		case VIGIL_ET_NEB_OPEN_EXIT:
			{
			VIGIL_ELEMENT_NEB_OPEN_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream, 
				"%*s<OPEN_EXIT RetKey=\"0x%llX\" retZid=\"0x%llX\"",
				margin, "",
				e->retKey,
				e->retZid
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "RetVolID", &e->retVolID);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_CLOSE_ENTER:
			{
			VIGIL_ELEMENT_NEB_CLOSE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CLOSE_ENTER Key=\"0x%llX\" FhState=\"0x%08X\"",
				margin, "",
				e->key,
				e->fhState
				);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_CLOSE_EXIT:
			{
			VIGIL_ELEMENT_NEB_CLOSE_EXIT_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CLOSE_EXIT FileDeleted=\"0x%08X\" />\n",
				margin, "",
				e->fileDeleted
				);
			break;
			}

		case VIGIL_ET_NEB_RENAME_ENTER:
			{
			VIGIL_ELEMENT_NEB_RENAME_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<RENAME_ENTER RenameFlags=\"0x%08X\" />\n",
				margin, "",
				e->renameFlags
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER:
			{
			VIGIL_ELEMENT_MODIFY_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<MODIFY_ENTER ModifyInfoMask=\"0x%08X\" ModifyTypeInfoMask=\"0x%08X\" />\n",
				margin, "",
				e->modifyInfoMask,
				e->modifyTypeInfoMask
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_FILE_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_FILE_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<FILE_ATTRIBUTES fileAttributes=\"0x%08X\" fileAttributesModMask=\"0x%08X\" />\n",
				margin, "",
				e->fileAttributes,
				e->fileAttributesModMask
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_CREATED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_CREATED_TIME_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CREATED_TIME",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->created);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVED_TIME_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<ARCHIVED_TIME",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->archived);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIED_TIME_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<MODIFIED_TIME",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->modified);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ACCESSED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ACCESSED_TIME_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<ACCESSED_TIME",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->accessed);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIED_TIME:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIED_TIME_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<METADATA_MODIFIED_TIME",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->metaDataModified);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_OWNER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_OWNER_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<OWNER_ID",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->owner);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ARCHIVER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ARCHIVER_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<ARCHIVER_ID",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->archiver);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MODIFIER_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<MODIFIER_ID",
				margin, ""
					);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->modifier);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_METADATA_MODIFIER_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_METADATA_MODIFIER_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<METADATA_MODIFIER_ID",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->metaDataModifier);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_PRIMARY_NAMESPACE:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_PRIMARY_NAMESPACE_T *e=(void *)elementHeader;

			fprintf(outStream, 
				"%*s<PRIMARY_NAMESPACE NameSpace=\"%d\" />\n",
				margin, "",
				e->primaryNameSpaceID
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DELETED_INFO:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DELETED_INFO_T *e=(void *)elementHeader;

			fprintf(outStream, 
				"%*s<DELETED_INFO",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintTime_64(outStream, "Time", &e->time);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->id);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_MAC_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_MAC_METADATA_T *e=(void *)elementHeader;

			fprintf(outStream, 
				"%*s<MAC_METADATA DirRightsMask=\"0x%08X\">\n", 
				margin, "",
				e->info.dirRightsMask
				);
			VDUMP_OUT_XML_BodyPrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.finderInfo,
				/* I- bufLen    */ sizeof(e->info.finderInfo),
				/* I- label     */ "FinderInfo",
				/* I- margin    */ margin+1
				);
			VDUMP_OUT_XML_BodyPrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.proDOSInfo,
				/* I- bufLen    */ sizeof(e->info.proDOSInfo),
				/* I- label     */ "ProDOSInfo",
				/* I- margin    */ margin+1
				);
			VDUMP_OUT_XML_BodyPrintHex(
				/* -O outStream */ outStream,
				/* I- buf       */ e->info.filler,
				/* I- bufLen    */ sizeof(e->info.filler),
				/* I- label     */ "Filler",
				/* I- margin    */ margin+1
				);
			fprintf(outStream,
				"%*s</MAC_METADATA>\n",
				margin, ""
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_UNIX_METADATA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_UNIX_METADATA_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<UNIX_METADATA"
				" FMode=\"0x%08X\""
				" RDev=\"0x%08X\""
				" MyFlags=\"0x%08X\""
				" NfsUID=\"0x%08X\""
				" NfsGID=\"0x%08X\""
				" NwUID=\"0x%08X\""
				" NwGID=\"0x%08X\""
				" NwEveryone=\"0x%08X\""
				" NwUIDRights=\"0x%08X\""
				" NwGIDRights=\"0x%08X\""
				" NwEveryoneRights=\"0x%08X\""
				" AcsFlags=\"%08X\""
				" FirstCreated\"0x%08X\""
				" VariableSize\"0x%08X\"",
				margin, "",
				e->info.fMode,
				e->info.rDev,
				e->info.myFlags,
				e->info.nfsUID,
				e->info.nfsGID,
				e->info.nwUID,
				e->info.nwGID,
				e->info.nwEveryone,
				e->info.nwUIDRights,
				e->info.nwGIDRights,
				e->info.nwEveryoneRights,
				e->info.acsFlags,
				e->info.firstCreated,
				e->info.variableSize
				);

			if(e->info.variableSize)
				{
				VDUMP_OUT_XML_BodyPrintHex(
					/* -O outStream  */ outStream,
					/* I- buf        */ e->info.variableData,
					/* I- bufLen     */ e->info.variableSize,
					/* I- tag        */ "VARIABLE_DATA",
					/* I- margin     */ margin+1
					);
				}

			fprintf(outStream,
				"%*s</UNIX_METADATA>\n",
				margin, ""
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_EXTATTR_FLAGS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_EXTATTR_FLAGS_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<EXTATTR_USERFLAGS Flags=\"0x%08X\" />\n",
				margin, "",
				e->extAttrUserFlags
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_ATTRIBUTES VolFeaturesEenabled=\"0x%08X\" VolFeaturesEenableModMask=\"0x%08X\" />\n",
				margin, "",
				e->enabled,
				e->enableModMask
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_NDS_OBJECT_ID",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->ndsObjectID);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MIN_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MIN_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_MIN_KEEP_SECONDS Seconds=\"%u\" />\n",
				margin, "",
				e->minKeepSeconds
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_MAX_KEEP_SECONDS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_MAX_KEEP_SECONDS_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_MAX_KEEP_SECONDS Seconds=\"%u\" />\n",
				margin, "",
				e->maxKeepSeconds
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_LOW_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_LOW_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_SALVAGE_LOW_WATER_MARK Mark=\"%u\" />\n",
				margin, "",
				e->lowWaterMark
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_HIGH_WATER_MARK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_HIGH_WATER_MARK_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_SALVAGE_HIGH_WATER_MARK Mark=\"%u\" />\n",
				margin, "",
				e->highWaterMark
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_ATTRIBUTES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_ATTRIBUTES_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<POOL_ATTRIBUTES FeaturesEnabled=\"0x%08X\" FeaturesEnableModMask=\"0x%08X\" />\n",
				margin, "",
				e->enabled,
				e->enableModMask
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_POOL_NDS_OBJECT_ID:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_POOL_NDS_OBJECT_ID_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<POOL_NDS_OBJECT_ID",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->ndsObjectID);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_DATA_SHREDDING_COUNT:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_DATA_SHREDDING_COUNT_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_DATA_SHREDDING_COUNT Count=\"0x%08X\" />\n",
				margin, "",
				e->dataShreddingCount
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_VOL_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_VOL_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<VOL_QUOTA Space=\"0x%llu\" />\n",
				margin, "",
				e->totalSpaceQuota
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_DIR_QUOTA:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_DIR_QUOTA_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<DIR_QUOTA Space=\"0x%llu\" />\n",
				margin, "",
				e->quota
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_READ_AHEAD_BLOCKS:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_READ_AHEAD_BLOCKS_T *e=(void *)elementHeader;

			fprintf(outStream, " <READ_AHEAD_BLOCKS />\n");
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_INH_RIGHTS_MASK:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_INH_RIGHTS_MASK_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<INH_RIGHTS_MASK Mask=\"0x08X\" />\n",
				margin, "",
				e->inheritedRightsMask
				);
			break;
			}

		case VIGIL_ET_NEB_MODIFY_ENTER__zMOD_ALL_TRUSTEES:
			{
			VIGIL_ELEMENT_MODIFY__zMOD_ALL_TRUSTEES_T *e=(void *)elementHeader;
			uint32_t entry;

			fprintf(outStream,
				"%*s<ALL_TRUSTEES numEntries=\"%d\">\n",
				margin, "",
				e->numEntries
				);

			for(entry=0; entry < e->numEntries; ++entry)
				{
				fprintf(outStream,
					"%*s<ENTRY No=\"%u\" Rights=\"0x%08X\"",
					margin+1, "",
					entry,
					e->trusteeTable[entry].rights
					);
				VDUMP_OUT_XML_AttrPrintGuid(outStream, "Trustee", (VIGIL_GUID_T *)(&e->trusteeTable[entry].trustee));
				fprintf(outStream, " />\n");
				}

			fprintf(outStream,
				"%*s</ALL_TRUSTEES>\n",
				margin, ""
				);
			break;
			}

		case VIGIL_ET_WHO__ANONYMOUS:
			{
			VIGIL_ELEMENT_WHO_ANONYMOUS_T *e=(void *)elementHeader;

			VDUMP_OUT_XML_BodyPrintHex(
				/* -O outStream  */ outStream,
				/* I- buf        */ e->data,
				/* I- bufLen     */ e->hdr.length - sizeof(e->hdr),
				/* I- tag        */ "WHO_ANONYMOUS",
				/* I- margin     */ margin+1
				);
			break;
			}

		case VIGIL_ET_WHO__NCP_POSIX:
			{
			VIGIL_ELEMENT_WHO_NCP_POSIX_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<WHO_NCP_POSIX ConnID=\"%d\" TaskID=\"%d\"",
				margin, "",
				e->connID,
				e->taskID
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "guid", &e->guid);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_WHO__LINUX:
			{
			VIGIL_ELEMENT_WHO_LINUX_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<WHO_LINUX"
				" uid=\"%d\""
				" euid=\"%d\""
				" suid=\"%d\""
				" fsuid=\"%d\""
				" gid=\"%d\""
				" egid=\"%d\""
				" sgid=\"%d\""
				" fsgid=\"%d\""
				" comm=\"%s\""
				" />\n",
				margin, "",
				e->uid,
				e->euid,
				e->suid,
				e->fsuid,
				e->gid,
				e->egid,
				e->sgid,
				e->fsgid,
				e->comm
				);
			break;
			}

		case VIGIL_ET_NEB_ADDTRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_ADDTRUSTEE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<ADDTRUSTEE_ENTER Rights=\"0x%08X\" Attributes=\"0x%08X\"",
				margin, "",
				e->rights,
				e->attributes
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->trusteeID);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_REMOVETRUSTEE_ENTER:
			{
			VIGIL_ELEMENT_NEB_REMOVETRUSTEE_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<REMOVETRUSTEE_ENTER PurgedFileFlag=\"0x%08X\"",
				margin, "",
				e->purgedFileFlag
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "TrusteeID", &e->trusteeID);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NEB_SETINHERITEDRIGHTS_ENTER:
			{
			VIGIL_ELEMENT_NEB_SETINHERITEDRIGHTS_ENTER_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<SETINHERITEDRIGHTS_ENTER InheritedRights=\"0x%08X\" AuthorizeFlag=\"0x%08X\" />\n",
				margin, "",
				e->inheritedRights,
				e->authorizeFlag
				);
			break;
			}

		case VIGIL_ET_NAME:
			{
			VIGIL_ELEMENT_NAME_T *e=(void *)elementHeader;
			char *nameTypeStr;

			switch(e->nameType)
				{
				case VIGIL_ELEMENT_NAMETYPE_CLIENT:     nameTypeStr="Client"; break;
				case VIGIL_ELEMENT_NAMETYPE_CLIENTUSER: nameTypeStr="ClientUser"; break;
				default:                                nameTypeStr="(unknown)"; break;
				}

			fprintf(outStream, 
				"%*s<NAME Type=\"%d %s\">%s</NAME>\n",
				margin, "",
				e->nameType, nameTypeStr,
				e->name
				);
			break;
			}

		case VIGIL_ET_NCP_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_NCP_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<NCP_LOCAL__OPENFILE searchAttributes=\"0x%02X\" DesiredAccessRights=\"0x%02X\" LinuxPath=\"%s\" Status=\"%d\"",
				margin, "",
				0xFF & e->searchAttributes,
				0xFF & e->desiredAccessRights,
				e->linuxPath,
				e->status
				);

			/** Check e->hdr.length to verify that the values below were passed. **/
			if(0==e->status)
				{
				fprintf(outStream, " FileHandle=\"%02X%02X%02X%02X%02X%02X\"",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				fprintf(outStream, " FileAttributes=\"%02X\"", 0xFF & e->fileAttributes);
				fprintf(outStream, " FileExecuteType=\"%02X\"", 0xFF & e->fileExecuteType);
				fprintf(outStream, " FileLen=\"%lu\"", e->fileLen);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "CreationDate", e->creationDate);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "LastAccessDate", e->lastAccessDate);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "LastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_XML_AttrPrintTime_MSDOS(outStream, "LastUpdateTime", e->lastUpdateTime);
				fprintf(outStream, " />\n");
				}
			break;
			}

		case VIGIL_ET_NCP_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_NCP_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<NCP_LOCAL__CLOSEFILE FileHandle=\"0x%02X%02X%02X%02X%02X%02X\" status=\"%d\" />\n",
				margin, "",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5],
				e->status
				);
			break;
			}

#if 0   // TODO:
		case VIGIL_ET_WHO__EDIR_GUID:
			break;
#endif

		case VIGIL_ET_NSS_CREATE:
			{
			VIGIL_ELEMENT_NSS_Create_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<CREATE", margin, "");
			fprintf(outStream, " key=\"0x%llX\"", e->key);
			fprintf(outStream, " requestedRights=\"0x%08X\"", e->requestedRights);
			fprintf(outStream, " createFlags=\"0x%08X\"", e->createFlags);
			fprintf(outStream, " createAndOpen=\"0x%08X\"", e->createAndOpen);
			fprintf(outStream, " retOpenCreateAction=\"0x%08X\"", e->retOpenCreateAction);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_OPEN:
			{
			VIGIL_ELEMENT_NSS_Open_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<OPEN", margin, "");
			fprintf(outStream, " requestedRights=\"0x%08X\"", e->requestedRights);
			fprintf(outStream, " key=\"0x%llX\"", e->key);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_CLOSE:
			{
			VIGIL_ELEMENT_NSS_Close_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<CLOSE", margin, "");
			fprintf(outStream, " key=\"0x%llX\"", e->key);
			fprintf(outStream, " fhState=\"0x%08X\"", e->fhState);
			fprintf(outStream, " fileDeleted=\"0x%08X\"", e->fileDeleted);
			VDUMP_OUT_XML_AttrPrintTimes(outStream, &e->times);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_RENAME:
			{
			VIGIL_ELEMENT_NSS_Rename_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<RENAME", margin, "");
			fprintf(outStream, " renameFlags=\"0x%08X\"", e->renameFlags);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_LINK:
			{
			VIGIL_ELEMENT_NSS_Link_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<LINK", margin, "");
			fprintf(outStream, " linkFlags=\"0x%08X\"", e->linkFlags);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_MODIFYINFO:
			{
			VIGIL_ELEMENT_NSS_ModifyInfo_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<MODIFY_INFO", margin, "");
			fprintf(outStream, " modifyInfoMask=\"0x%llX\"", e->modifyInfoMask);
			fprintf(outStream, " modifyTypeInfoMask=\"0x%llX\"", e->modifyTypeInfoMask);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_ADDTRUSTEE:
			{
			VIGIL_ELEMENT_NSS_AddTrustee_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<ADD_TRUSTEE", margin, "");
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "trusteeId", &e->trusteeID);
			fprintf(outStream, " rights=\"0x%08X\"", e->rights);
			fprintf(outStream, " previousRights=\"0x%08X\"", e->previousRights);
			fprintf(outStream, " attributes=\"0x%08X\"", e->attributes);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_REMOVETRUSTEE:
			{
			VIGIL_ELEMENT_NSS_RemoveTrustee_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<REMOVE_TRUSTEE", margin, "");
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "trusteeId", &e->trusteeID);
			fprintf(outStream, " rights=\"0x%08X\"", e->rights);
			fprintf(outStream, " purgedFileFlag=\"0x%08X\"", e->purgedFileFlag);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_NSS_SETINHERITEDRIGHTS:
			{
			VIGIL_ELEMENT_NSS_SetInheritedRights_T *e=(void *)elementHeader;

			fprintf(outStream, "%*s<SET_INHERITED_RIGHTS", margin, "");
			fprintf(outStream, " InheritedRights=\"0x%08X\"", e->inheritedRights);
			fprintf(outStream, " PreviousInheritedRights=\"0x%08X\"", e->previousInheritedRights);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__OPENFILE:
			{
			VIGIL_ELEMENT_CIFS_OPENFILE_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CIFS_LOCAL__OPENFILE DesiredAccessRights=\"0x%02X\" LinuxPath=\"%s\" Status=\"%d\"",
				margin, "",
				0xFF & e->desiredAccessRights,
				e->linuxPath,
				e->status
				);

			/** Check e->hdr.length to verify that the values below were passed. **/
			if(0 == e->status)
				{
				fprintf(outStream, " FileHandle=\"%02X%02X%02X%02X%02X%02X\"",
					0xFF & e->fileHandle[0],
					0xFF & e->fileHandle[1],
					0xFF & e->fileHandle[2],
					0xFF & e->fileHandle[3],
					0xFF & e->fileHandle[4],
					0xFF & e->fileHandle[5]
					);
				fprintf(outStream, " FileAttributes=\"%02X\"", 0xFF & e->fileAttributes);
				fprintf(outStream, " FileExecuteType=\"%02X\"", 0xFF & e->fileExecuteType);
				fprintf(outStream, " FileLen=\"%lu\"", e->fileLen);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "CreationDate", e->creationDate);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "LastAccessDate", e->lastAccessDate);
				VDUMP_OUT_XML_AttrPrintDate_MSDOS(outStream, "LastUpdateDate", e->lastUpdateDate);
				VDUMP_OUT_XML_AttrPrintTime_MSDOS(outStream, "LastUpdateTime", e->lastUpdateTime);
				}

			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_CIFS_LOCAL__CLOSEFILE:
			{
			VIGIL_ELEMENT_CIFS_CLOSEFILE_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<CIFS_LOCAL__CLOSEFILE FileHandle=\"0x%02X%02X%02X%02X%02X%02X\" status=\"%d\" />\n",
				margin, "",
				0xFF & e->fileHandle[0],
				0xFF & e->fileHandle[1],
				0xFF & e->fileHandle[2],
				0xFF & e->fileHandle[3],
				0xFF & e->fileHandle[4],
				0xFF & e->fileHandle[5],
				e->status
				);
			break;
			}

		case VIGIL_ET_PMD_NCP:
			{
			VIGIL_ELEMENT_PMD_NCP_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<PMD_NCP ConnID=\"%d\" TaskID=\"%d\"",
				margin, "",
				e->connID,
				e->taskID
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->guid);
			fprintf(outStream, " />\n");
			break;
			}

		case VIGIL_ET_PMD_CIFS:
			{
			VIGIL_ELEMENT_PMD_CIFS_T *e=(void *)elementHeader;

			fprintf(outStream,
				"%*s<PMD_CIFS",
				margin, ""
				);
			VDUMP_OUT_XML_AttrPrintGuid(outStream, "Guid", &e->guid);
			fprintf(outStream, " ConnID=\"%d\"", e->connID);
			switch(e->netAddr.hdr.type)
				{
				case VIGIL_ET_NET_ADDR_IPv4:
					fprintf(outStream, " netAddr_IPv4=\"%02X.%02X.%02X.%02X\"",
						0xFF & e->netAddr.addr.octet[0],
						0xFF & e->netAddr.addr.octet[1],
						0xFF & e->netAddr.addr.octet[2],
						0xFF & e->netAddr.addr.octet[3]
						);
					break;

				case VIGIL_ET_NET_ADDR_IPv6:
					fprintf(outStream, " netAddr_IPv6=\"%4X:%4X:%4X:%4X:%4X:%4X:%4X:%4X\"",
						0xFFFF & e->netAddr.addr.hextet[0],
						0xFFFF & e->netAddr.addr.hextet[1],
						0xFFFF & e->netAddr.addr.hextet[2],
						0xFFFF & e->netAddr.addr.hextet[3],
						0xFFFF & e->netAddr.addr.hextet[4],
						0xFFFF & e->netAddr.addr.hextet[5],
						0xFFFF & e->netAddr.addr.hextet[6],
						0xFFFF & e->netAddr.addr.hextet[7]
						);
					break;

				default:
					fprintf(outStream, " netAddr=\"0X%04hX[Unknown netAddr Type]\"", e->netAddr.hdr.type);
					break;
				}

			fprintf(outStream, " />\n");
			break;
			}

		default:
			fprintf(outStream,
				"%*s<UNKNOWN_ELEMENT Type=\"%d\">\n",
				margin, "",
				elementHeader->type
				);
			VDUMP_OUT_XML_BodyPrintHex(
				/* -O outStream  */ outStream,
				/* I- buf        */ (void  *)elementHeader,
				/* I- bufLen     */ elementHeader->length,
				/* I- tag        */ "ELEMENT",
				/* I- margin     */ margin+1
				);
			fprintf(outStream,
				"%*s</UNKNOWN_ELEMENT>\n",
				margin, ""
				);
			break;
		}

	(*element) += elementHeader->length;

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecord()
*/
int VDUMP_OUT_XML_PrintNcpRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NCP_T *ncp,
		/* I- margin    */ int                    margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = ncp->data;
	char *auditEventStr;
	char *enterExitStr;

	/** Get the (neb) audit event string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ ncp->hdr.type,
		/* I- auditEvent   */ ncp->event,
		/* -O auditTypeStr */ NULL,
		/* -O auditTypeStr */ &auditEventStr,
		/* -O enterExitStr */ &enterExitStr
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Print NCP record's opening tag **/
	fprintf(outStream, "%*s<NCP", margin, "");

	/* Print event */
	fprintf(outStream, " Event=\"%d %s_%s\"", ncp->event, auditEventStr, enterExitStr);

	/* Terminate vigil opening tag. */
	fprintf(outStream, ">\n");

	for(curElement=0; curElement < ncp->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_XML_ElementPrint(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr,
			/* I- margin     */ margin+1
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_XML_ElementPrint() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	fprintf(outStream, "%*s</NCP>\n", margin, "");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecord()
*/
int VDUMP_OUT_XML_PrintCifsRecord(
		/* -O outStream */ FILE                   *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_CIFS_T *cifs,
		/* I- margin    */ int                     margin
		)
	{
	int   rCode=VIGIL_SUCCESS;
	int   curElement;
	char *curDataPtr = cifs->data;
	char *auditEventStr;

	/** Get the (cifs) audit event string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ cifs->hdr.type,
		/* I- auditEvent   */ cifs->event,
		/* -O auditTypeStr */ NULL,
		/* -O auditTypeStr */ &auditEventStr,
		/* -O enterExitStr */ NULL
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Print CIFS record's opening tag **/
	fprintf(outStream, "%*s<CIFS", margin, "");

	/* Print event */
	fprintf(outStream, " Event=\"%d %s\"", cifs->event, auditEventStr);

	/* Terminate vigil opening tag. */
	fprintf(outStream, ">\n");

	for(curElement=0; curElement < cifs->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_XML_ElementPrint(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr,
			/* I- margin     */ margin+1
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_XML_ElementPrint() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	fprintf(outStream, "%*s</CIFS>\n", margin, "");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecord()
*/
int VDUMP_OUT_XML_PrintNebRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NEB_T *neb,
		/* I- margin    */ int                    margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = neb->data;
	char *auditEventStr;
	char *enterExitStr;

	/** Get the (neb) audit event string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ neb->hdr.type,
		/* I- auditEvent   */ neb->event,
		/* -O auditTypeStr */ NULL,
		/* -O auditTypeStr */ &auditEventStr,
		/* -O enterExitStr */ &enterExitStr
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Print NEB record's opening tag **/
	fprintf(outStream, "%*s<NEB", margin, "");

	/* Print enterExitID */
	fprintf(outStream, " EnterExitID=\"%u\"", neb->enterExitID);

	/* Print event */
	fprintf(outStream, " Event=\"%d %s_%s\"", neb->event, auditEventStr, enterExitStr);

	/* Terminate vigil opening tag. */
	fprintf(outStream, ">\n");

	for(curElement=0; curElement < neb->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_XML_ElementPrint(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr,
			/* I- margin     */ margin+1
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_XML_ElementPrint() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	fprintf(outStream, "%*s</NEB>\n", margin, "");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecord()
*/
int VDUMP_OUT_XML_PrintVigilRecord(
		/* -O outStream */ FILE                    *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_VIGIL_T *rec,
		/* I- margin    */ int                      margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = rec->data;
	char *auditEventStr;

	/** Get the (vigil) audit event string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ rec->hdr.type,
		/* I- auditEvent   */ rec->event,
		/* -O auditTypeStr */ NULL,
		/* -O auditTypeStr */ &auditEventStr,
		/* -O enterExitStr */ NULL
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Print VIGIL record's opening tag **/
	fprintf(outStream, "%*s<VIGIL", margin, "");

	/* Print event */
	fprintf(outStream, " Event=\"%d %s\"", rec->event, auditEventStr);

	/* Terminate vigil opening tag. */
	fprintf(outStream, ">\n");

	for(curElement=0; curElement < rec->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_XML_ElementPrint(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr,
			/* I- margin     */ margin+1
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_XML_ElementPrint() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	fprintf(outStream, "%*s</VIGIL>\n", margin, "");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_OUT_XML_PrintRecord()
*/
int VDUMP_OUT_XML_PrintNssRecord(
		/* -O outStream */ FILE                  *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_NSS_T *nss,
		/* I- margin    */ int                    margin
		)
	{
	int  rCode=VIGIL_SUCCESS;
	int  curElement;
	char *curDataPtr = nss->data;
	char *auditEventStr;

	/** Get the (neb) audit event string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ nss->hdr.type,
		/* I- auditEvent   */ nss->event,
		/* -O auditTypeStr */ NULL,
		/* -O auditTypeStr */ &auditEventStr,
		/* -O enterExitStr */ NULL
		);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Print NEB record's opening tag **/
	fprintf(outStream, "%*s<NSS", margin, "");

	/* Print event */
	fprintf(outStream, " Event=\"%d %s\"", nss->event, auditEventStr);

	/*---------------------------------------------------------------------------
	** Event metadata
	*/
	fprintf(outStream,
		" TaskID=\"%lu\""
		" Zid=\"%llX\""
		" ParentZid=\"%llX\""
		" FileType=\"%lu\""
		" FileAttributes=\"%llX\""
		" OpRetCode=\"%d\""
		"",
		nss->taskID,
		nss->zid,
		nss->parentZid,
		nss->fileType,
		nss->fileAttributes,
		nss->opRetCode
		);

	rCode=VDUMP_OUT_XML_AttrPrintGuid(
		/* -O outStream */ outStream,
		/* I- label     */ "VolID",
		/* I- ndsGuid   */ &nss->volID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintGuid(VolID) reports: %d\n", rCode);
		goto END_ERR;
		}

	rCode=VDUMP_OUT_XML_AttrPrintGuid(
		/* -O outStream */ outStream,
		/* I- label     */ "UserID",
		/* I- ndsGuid   */ &nss->userID
		);
	if(rCode)
		{
		LIBVIGIL_ERR("VDUMP_OUT_XML_AttrPrintGuid(UserID) reports: %d\n", rCode);
		goto END_ERR;
		}

	/*---------------------------------------------------------------------------
	** Linux metadata
	*/
	fprintf(outStream, " uid\"%d\"",   nss->uid);
	fprintf(outStream, " euid\"%d\"",  nss->euid);
	fprintf(outStream, " suid\"%d\"",  nss->suid);
	fprintf(outStream, " fsuid\"%d\"", nss->fsuid);
	fprintf(outStream, " gid\"%d\"",   nss->gid);
	fprintf(outStream, " egid\"%d\"",  nss->egid);
	fprintf(outStream, " sgid\"%d\"",  nss->sgid);
	fprintf(outStream, " fsgid\"%d\"", nss->fsgid);
	fprintf(outStream, " comm\"%s\"",  nss->comm);

	/* Terminate opening tag. */
	fprintf(outStream, ">\n");

	/*---------------------------------------------------------------------------
	** Elements
	*/
	for(curElement=0; curElement < nss->dataElements; ++curElement)
		{
		rCode=VDUMP_OUT_XML_ElementPrint(
			/* -O outStream  */ outStream,
			/* I- elementPtr */ &curDataPtr,
			/* I- margin     */ margin+1
			);
		if(rCode)
			{
			LIBVIGIL_ERR("VDUMP_OUT_XML_ElementPrint() reports: %d\n", rCode);
			goto END_ERR;
			}
		}

	fprintf(outStream, "%*s</NSS>\n", margin, "");

END_ERR:

	return(rCode);
	}

/*******************************************************************************
** Local references: VDUMP_MAIN_ProcessAuditLog()
*/
int VDUMP_OUT_XML_PrintRecord(
		/* -O outStream */ FILE              *outStream,
		/* I- record    */ VIGIL_AUDIT_REC_T *rec,
		/* I- recLen    */ size_t             recLen
		)
	{
	int  rCode=VIGIL_SUCCESS;
	struct tm timeDate;
	char timeDateStr[63+1];
	char *auditTypeStr;

	rCode=LIBVIGIL_PARSE_RecValidate(rec, recLen);
	if(rCode)
		{
		LIBVIGIL_ERR("LIBVIGIL_PARSE_RecValidate() reports: %d\n", rCode);
		goto END_ERR;
		}

	/** Get the record's type string. **/
	rCode=LIBVIGIL_PARSE_AuditTypeStr(
		/* I- auditType    */ rec->hdr.type,
		/* -O auditTypeStr */ &auditTypeStr
		);
	switch(rCode)
		{
		case VIGIL_SUCCESS:
			break;
		case ENOENT:
			rCode=VIGIL_SUCCESS;
			break;
		default:
			LIBVIGIL_ERR("LIBVIGIL_PARSE_AuditTypeEventStateStr() reports: %d\n", rCode);
			goto END_ERR;
		}

	/** Print Audit record's opening tag **/
	fprintf(outStream, "<AUDIT");

	/* Print the record number. */
	fprintf(outStream, 
		" recNo=\"%llu\""
		" pid=\"%u\"",
		rec->hdr.recNo,
		rec->hdr.pid
		);

	/* Print record's time stamp. */
	localtime_r(&rec->hdr.time.tv_sec, &timeDate);
	strftime(timeDateStr, sizeof(timeDateStr), "%02m/%02d/%Y %02H:%02M:%02S", &timeDate);
	fprintf(outStream, " TimeStamp=\"%s %06lu MDY HMS USEC\"", timeDateStr, rec->hdr.time.tv_usec);

	/* Print the record's type. */
	fprintf(outStream, " Type=\"%d %s\"", rec->hdr.type, auditTypeStr);
	
	/* Terminate AUDIT opening tag. */
	fprintf(outStream, ">\n");

	/** Parse record by type. **/
	switch(rec->hdr.type)
		{
		case VIGIL_T_VIGIL:
			rCode=VDUMP_OUT_XML_PrintVigilRecord(
				/* -O outStream */ outStream,
				/* I- vigilRec  */ &rec->vigil,
				/* I- margin    */ 1
				);
			switch(rCode)
				{
				case VIGIL_SUCCESS:
					break;

				case ESTALE:
					break;

				default:
					LIBVIGIL_ERR("VDUMP_OUT_XML_PrintVigilRecord() reports: %d\n", rCode);
					goto END_ERR;
				}
			break;

		case VIGIL_T_NEB:
			rCode=VDUMP_OUT_XML_PrintNebRecord(
				/* -O outStream */ outStream,
				/* I- nebRec    */ &rec->neb,
				/* I- margin    */ 1
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_XML_PrintNebRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NCP:
			rCode=VDUMP_OUT_XML_PrintNcpRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->ncp,
				/* I- margin    */ 1
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_XML_PrintNcpRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_CIFS:
			rCode=VDUMP_OUT_XML_PrintCifsRecord(
				/* -O outStream */ outStream,
				/* I- ncpRec    */ &rec->cifs,
				/* I- margin    */ 1
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_XML_PrintCifsRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		case VIGIL_T_NSS:
			rCode=VDUMP_OUT_XML_PrintNssRecord(
				/* -O outStream */ outStream,
				/* I- nssRec    */ &rec->nss,
				/* I- margin    */ 1
				);
			if(rCode)
				{
				LIBVIGIL_ERR("VDUMP_OUT_NSS_PrintNssRecord() reports: %d\n", rCode);
				goto END_ERR;
				}
			break;

		default:
			break;
		}

	/** Print Record's closing tag **/
	fprintf(outStream, "</AUDIT>\n");

END_ERR:

	return(rCode);
	}












