/****************************************************************************
 |
 | Copyright (c) 2008-2009 Novell, Inc.
 |
 | Permission is hereby granted, free of charge, to any person obtaining a
 | copy of this software and associated documentation files (the "Software"),
 | to deal in the Software without restriction, including without limitation
 | the rights to use, copy, modify, merge, publish, distribute, sublicense,
 | and/or sell copies of the Software, and to permit persons to whom the
 | Software is furnished to do so, subject to the following conditions:
 |
 | The above copyright notice and this permission notice shall be included
 | in all copies or substantial portions of the Software.
 |
 | THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 | OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 | MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
 | NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 | DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 | OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
 | USE OR OTHER DEALINGS IN THE SOFTWARE.
 |
 |***************************************************************************
 |
 | OES NEB Auditing Userspace Library (libvigil).
 |
 | File: libvigil.h
 |
 | Desc: libvigil - NSS Auditing Library.
 +-------------------------------------------------------------------------*/

#ifndef __LIBVIGIL_H__
   #define __LIBVIGIL_H__

#include <dirent.h>

#include <vigil.h>

/*****************************************************************************
** TypeDefs
*/
#if defined(__cplusplus)
   #define LIBVIGIL_CDECL "C"
#else
   #define LIBVIGIL_CDECL
#endif

/*****************************************************************************
** LIBVIGIL_LOG
*/
#define __LIBVIGIL_DEBUG__     1

	/*-------------------------------------------------------------------------
	** LIBVIGIL_ERR()
	**
	** Referenced by:
	**   v2ld_laf.c
	**   v2ld_main.c
	**   v2ld_parse.c
	**   v2ld_seq.c
	**   v2ld_seq_ncp.c
	**   v2ld_seq_neb.c
	**   v2ld_xml.c
	**   vdump_main.c
	*/
	#if defined(__LIBVIGIL_DEBUG__)
		#define LIBVIGIL_ERR(fmt, ...)\
			LIBVIGIL_LOG_Err(__FILE__, __func__, __LINE__, fmt, ##__VA_ARGS__)
	#else
		#define LIBVIGIL_ERR(...)
	#endif

	/*-------------------------------------------------------------------------
	** LIBVIGIL_NOTE()
	**
	** Referenced by:
	**   v2ld_main.c
	**   vdump_main.c
	*/
	#define LIBVIGIL_NOTE(fmt, ...)\
		LIBVIGIL_LOG_Note(fmt, ##__VA_ARGS__)

extern int LIBVIGIL_LOG_Err(
		/* I- srcfile      */ const char *file,
		/* I- functionName */ const char *functionName,
		/* I- srcFileLine  */ int         srcFileLine,
		/* I- fmtStr       */ const char *fmtStr,
		/* I- ...          */ ...
		);

extern int LIBVIGIL_LOG_Note(
		/* I- fmtStr       */ char *fmtStr,
		/* I- ...          */ ...
		);

/*****************************************************************************
** LIBVIGIL_MEM
*/
extern int LIBVIGIL_MEM_Free(
		/* -O addr */ void **addr
		);

extern int LIBVIGIL_MEM_Alloc(
		/* I- ptr  */ void **addr,
		/* I- size */ size_t size
		);

extern int LIBVIGIL_MEM_StrDup(
		/* I- src */ const char  *src,
		/* I- dup */ char       **dup
		);

extern int LIBVIGIL_MEM_Dup(
		/* I- data   */ void    *i_data,
		/* I- length */ size_t   i_length,
		/* -O dup    */ void   **o_dup
		);

/*****************************************************************************
** LIBVIGIL_LIST
*/

	/*-------------------------------------------------------------------------
	** List flags
	**
	** Referenced by:
	**   v2ld_seq_ncp.c
	**   v2ld_seq_ncp.c
	*/
	#define LIBVIGIL_LIST_F_NONE          0x00000000
	#define LIBVIGIL_LIST_F_NODUPS        0x00000001

	/*-------------------------------------------------------------------------
	** LIBVIGIL_LIST_NODE_T
	**
	** Referenced by:
	**   v2ld_seq.c
	*/
	typedef struct LIBVIGIL_LIST_NODE
		{
		struct LIBVIGIL_LIST_NODE *prev;
		struct LIBVIGIL_LIST_NODE *next;
		void                      *data;
		int                      (*data_FreeFn)(void **addr);
		void                      *list;
		} LIBVIGIL_LIST_NODE_T;

	/*-------------------------------------------------------------------------
	** LIBVIGIL_LIST_T
	**
	** Referenced by:
	**   v2ld_seq.c
	**   v2ld_seq_ncp.c
	**   v2ld_seq_neb.c
	*/
	typedef struct LIBVIGIL_LIST
		{
		LIBVIGIL_LIST_NODE_T *head;
		LIBVIGIL_LIST_NODE_T *tail;
		int                 (*NodeCmpFn)(LIBVIGIL_LIST_NODE_T *, LIBVIGIL_LIST_NODE_T *, int *);
		uint32_t              flags;
		uint32_t              nodeCnt;
		char                 *name;
		} LIBVIGIL_LIST_T;

extern int LIBVIGIL_LIST_Unlock(
		/* I- list */ LIBVIGIL_LIST_T *list
		);

extern int LIBVIGIL_LIST_Lock(
		/* I- list */ LIBVIGIL_LIST_T *list
		);

extern int LIBVIGIL_LIST_GetNodeEqualTo(
		/* I- list */ LIBVIGIL_LIST_T       *list,
		/* I- data */ void                  *data,
		/* -O node */ LIBVIGIL_LIST_NODE_T **node
		);

extern int LIBVIGIL_LIST_GetNodeGreaterThan(
		/* I- list */ LIBVIGIL_LIST_T		   *list,
		/* I- node */ LIBVIGIL_LIST_NODE_T  *node,
		/* -O cur  */ LIBVIGIL_LIST_NODE_T **cur
		);

extern int LIBVIGIL_LIST_Iter(
		/* I- list           */ LIBVIGIL_LIST_T *list,
		/* I- callBack_Fn    */ int (*callBack_Fn)(LIBVIGIL_LIST_NODE_T *, void *),
		/* I- callBack_arg   */ void *callBack_arg,
		/* -O callBack_rCode */ int  *callBack_rCode
		);

extern int LIBVIGIL_LIST_NodeUnlink(
		/* I- node */ LIBVIGIL_LIST_NODE_T *node
		);

extern int LIBVIGIL_LIST_NodeLink(
		/* I- list */ LIBVIGIL_LIST_T      *list,
		/* I- node */ LIBVIGIL_LIST_NODE_T *node
		);

extern int LIBVIGIL_LIST_NodeCloseAll(
		/* I- listPtr */ LIBVIGIL_LIST_T *list
		);

extern int LIBVIGIL_LIST_NodeClose(
		/* I- node */ LIBVIGIL_LIST_NODE_T **node
		);

extern int LIBVIGIL_LIST_NodeOpen(
		/* I- list        */ LIBVIGIL_LIST_T       *list,
		/* I- data        */ void                  *data,
		/* I- data_FreeFn */ int                  (*data_FreeFn)(void **memPtr),
		/* -O node        */ LIBVIGIL_LIST_NODE_T **n
		);

extern int LIBVIGIL_LIST_Close(
		/* -O listPtr  */ LIBVIGIL_LIST_T **list
		);

extern int LIBVIGIL_LIST_Open(
		/* -O listPtr    */ LIBVIGIL_LIST_T **list,
		/* I- NodeCmpFn  */ int             (*NodeCmpFn)(LIBVIGIL_LIST_NODE_T *, LIBVIGIL_LIST_NODE_T *, int *),
		/* I- flags      */ uint32_t          flags,
		/* I- listName   */ const char       *listName
		);

/*****************************************************************************
** LIBVIGIL_PARSE.
*/
	/*-------------------------------------------------------------------------
	** VIGIL record states
	**
	** Referenced by:
	**   v2ld_parse.c
	**   v2ld_xml.c
	*/
	#define LIBVIGIL_PARSE_STATE_UNKNOWN        (-1)
	#define LIBVIGIL_PARSE_STATE_SINGULAR       0
	#define LIBVIGIL_PARSE_STATE_ENTER          1
	#define LIBVIGIL_PARSE_STATE_EXIT           2

extern int LIBVIGIL_PARSE_FileValidate(
		/* I- filePath */ const char *filePath
		);

extern int LIBVIGIL_PARSE_RecValidate(
		/* I- rec    */ void  *recVoid,
		/* I- recLen */ size_t recLen
		);

extern int LIBVIGIL_PARSE_RecElement_ByTypeInstance(
		/* I- recData         */ void      *recData,
		/* I- recDataElements */ uint16_t   recDataElements,
		/* I- elementType     */ uint16_t   elementType,
		/* I- instance        */ uint16_t   instance,
		/* -O elementPtr      */ void     **elementPtr_out
		);

extern int LIBVIGIL_PARSE_Rec2TypeEventSeq(
		/* I- record          */ VIGIL_AUDIT_REC_T *rec,
		/* -O auditType       */ uint32_t          *auditType_OUT,
		/* -O auditEvent      */ uint32_t          *auditEvent_OUT,
		/* -O auditSequenceNo */ uint64_t          *auditSequenceNo_OUT
		);

extern int LIBVIGIL_PARSE_EnterExitCheck(
		/* I- auditType    */ uint32_t auditType,
		/* I- auditEvent   */ uint32_t auditEvent,
		/* -O state        */ int     *state
		);

extern int LIBVIGIL_PARSE_AuditTypeStr(
		/* I- auditType    */ uint32_t   auditType,
		/* -O auditTypeStr */ char     **auditTypeStr_OUT
		);

extern int LIBVIGIL_PARSE_AuditTypeEventStateStr(
		/* I- auditType    */ uint32_t   auditType,
		/* I- auditEvent   */ uint32_t   auditEvent,
		/* -O auditTypeStr */ char     **auditTypeStr_OUT,
		/* -O auditTypeStr */ char     **auditEventStr_OUT,
		/* -O enterExitStr */ char     **enterExitStr_OUT
		);

#if 0
extern int LIBVIGIL_PARSE_IterRecs(
		/* I- filePath       */ const char *filePath,
		/* I- callBack_fn    */ int       (*callBack_fn)(VIGIL_AUDIT_REC_T *, size_t, void *),
		/* I- callBack_arg   */ void       *callBack_arg,
		/* I- callBack_rCode */ int        *callBack_rCode
		);
#endif

/*****************************************************************************
** LIBVIGIL_TIME
*/
extern int LIBVIGIL_TIME_TimeValCmp(
		/* I- timeRef1 */ struct timeval *timeRef1,
		/* I- timeRef2 */ struct timeval *timeRef2,
		/* -O result   */ int            *result
		);

/*****************************************************************************
** LIBVIGIL_FILE
*/
extern int LIBVIGIL_FILE_WriteString(
		/* I- filePath */ const char *filePath,
		/* I- offset   */ long        offset,
		/* I- string   */ const char *string
		);

extern int LIBVIGIL_FILE_ReadString(
		/* I- filePath */ const char  *filePath,
		/* I- offset   */ long         offset,
		/* -O buf      */ char        *buf,
		/* I- bufSize  */ size_t       bufSize,
		/* -O bufLen   */ size_t      *bufLen
		);

extern int LIBVIGIL_FILE_Size(
		/* I- filePath */ const char  *filePath,
		/* -O size     */ off_t       *size
		);

extern int LIBVIGIL_FILE_Size_2(
		/* I- filePath */ const char  *i_filePath,
		/* -O size     */ off_t       *o_size
		);

extern int LIBVIGIL_FILE_ToMem(
		/* I- filePath   */ const uint8_t  *i_filePath,
		/* -O file_ALLOC */ uint8_t       **o_file_ALLOC,
		/* -O fileSize   */ size_t         *o_fileSize
		);

/*****************************************************************************
** LIBVIGIL_DIR
*/
extern int LIBVIGIL_DIR_PathPaste(
		/* I- pathSeg1   */ const uint8_t  *i_pathSeg1,
		/* I- pathSeg2   */ const uint8_t  *i_pathSeg2,
		/* -O path_ALLOC */ uint8_t       **o_path_ALLOC
		);

extern int LIBVIGIL_DIR_Iter(
		/* I- dirPath       */ const char *i_dirPath,
		/* I- callBackFn    */ int       (*i_callBackFn)(struct dirent *dirEnt, void *argVoid),
		/* I- argVoid       */ void       *i_argVoid,
		/* -O callBackRCode */ int        *o_callBackRCode
		);


/*****************************************************************************
** LIBVIGIL_SYS
*/

/** Maximum size of sysfs data transfer buffer. **/
#define LIBVIGIL_SYS_CMDBUF_SIZE  (4 * 1024)

extern int LIBVIGIL_SYS_UserPathAssemble(
		/* I- clientName */ const char  *clientName,
		/* I- userName   */ const char  *userName,
		/* -O path_ALLOC */ char       **path_ALLOC
		);

extern int LIBVIGIL_SYS_ClientPathAssemble(
		/* I- clientName */ const char *clientName,
		/* -O path_ALLOC */ char      **path_ALLOC
		);

extern int LIBVIGIL_SYS_ClientPathVerify(
		/* I- clientName */ const char *clientName
		);

extern int LIBVIGIL_SYS_UserClose(
		/* I- clientName */ const char *clientName,
		/* I- userName   */ const char *userName,
		/* I- userKey    */ const char *userKey
		);

extern int LIBVIGIL_SYS_ClientClose(
		/* I- clientName */ const char *clientName,
		/* I- clientKey  */ const char *clientKey
		);

extern int LIBVIGIL_SYS_ClientOpen(
		/* I- clientName        */ const char *clientName,
		/* I- clientKey         */ const char *clientKey,
		/* I- clientFlags       */ uint32_t    clientFlags,
		/* I- outputDirPath     */ const char *outputDirPath,
		/* I- outputFileSizeMax */ size_t      outputSizeMax,
		/* I- recordsLimit      */ uint64_t    outputRecordsLimit,
		/* I- userName          */ const char *userName,
		/* I- userKey           */ const char *userKey,
		/* I- userFlags         */ uint32_t    userFlags,
		/* I- sigioPid          */ int         outputSigioPid
		);

extern int LIBVIGIL_SYS_ParseTagValue(
		/* I- buf            */ const char  *buf,
		/* I- tag            */ const char  *tag,
		/* -O value32        */ uint32_t    *value32,
		/* -O value64        */ uint64_t    *value64,
		/* -O valueStr_ALLOC */ char       **valueStr_ALLOC
		);

extern int LIBVIGIL_SYS_ClientInfo_Alloc(
		/* I- clientName       */ const char  *clientName,
		/* -O clientInfo_ALLOC */ char       **clientInfo_ALLOC,
		/* -O clientInfoLen    */ size_t      *clientInfoLen
		);

extern int LIBVIGIL_SYS_CurrentFileAndOffset(
		/* I- clientName       */ const char  *clientName,
		/* -O outputFile_ALLOC */ char       **outputFile_ALLOC,
		/* -O outputFileOffset */ size_t      *outputFileOffset
		);

extern int LIBVIGIL_SYS_InitialFile(
		/* I- clientName       */ const char *clientName,
		/* -O outputFile_ALLOC */ char      **outputFile_ALLOC
		);

#if 0
extern int LIBVIGIL_SYS_Element(
		/* I- clientName    */ const char  *clientName,
		/* I- tag           */ const char  *tag,
		/* -O element_ALLOC */ char       **element_ALLOC
		);
#endif

extern int LIBVIGIL_SYS_FilterFileImport(
		/* I- clientName     */ const char *clientName,
		/* I- clientKey      */ const char *clientKey,
		/* I- filterFilePath */ const char *filterFilePath
		);

extern int LIBVIGIL_SYS_GenerateUsername(
		/* -O userName_ALLOC */ char **userName_ALLOC
		);

/*****************************************************************************
** LIBVIGIL_PROC
*/
extern int LIBVIGIL_PROC_Pause(void);

/*****************************************************************************
** LIBVIGIL_GUID
*/
extern int LIBVIGIL_GUID_ConstStr(
		/* I- guid    */ const VIGIL_GUID_T  *i_guid,
		/* -O guidStr */ char               **o_guidStr
		);

extern int LIBVIGIL_GUID_GUIDStrToGUID(
		/* I- utfStr */ const uint8_t *i_utfStr,
		/* -O guid   */ VIGIL_GUID_T  *o_guid
		);

extern int LIBVIGIL_GUID_StrGUIDToGuidMatch(
		/* I- guidStr */ const uint8_t       *i_guidStr,
		/* I- guidObj */ const VIGIL_GUID_T  *i_guidObj,
		/* -O match   */ int                 *o_match
		);

/*****************************************************************************
** LIBVIGIL_NSS
*/
extern int LIBVIGIL_NSS_MapNssGUIDToNssObjectName(
		/* I- objectGuid       */ VIGIL_GUID_T  *i_objectGuid,
		/* -O objectName_ALLOC */ uint8_t      **o_objectName_ALLOC
		);

/*****************************************************************************
** LIBVIGIL_XML
*/
extern int LIBVIGIL_STR_Stat(
		/* I- str         */ const uint8_t  *i_str,
		/* -O strLen      */ size_t         *o_strLen,
		/* -O strSize     */ size_t         *o_strSize,
		/* -O lastCharPtr */ uint8_t       **o_lastCharPtr,
		/* -O termPtr     */ uint8_t       **o_termPtr
		);

/*****************************************************************************
** LIBVIGIL_XML
*/
extern int LIBVIGIL_XML_GetContentBySimpleRootLabel(
		/* I- filePath      */ uint8_t       *i_filePath,
		/* I- label         */ const uint8_t *i_label,
		/* -O content_ALLOC */ uint8_t      **o_content_ALLOC
		);

#endif /* __LIBVIGIL_H__ */

