/*
===============================================================================
Novell Software Developer Kit Sample Code License

Copyright (C) 2005, Novell, Inc.  All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
  *  Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  *  Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.
  *  Neither the name of Novell, Inc. nor the names of its contributors
     may be used to endorse or promote products derived from this
     software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT
SHALL NOVELL, INC., THE COPYRIGHT OWNER, OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

NAME OF FILE:
	smuni.c

PURPOSE/COMMENTS:
	Implements the UTF-8 conversion and UTF-8 string premitive functions for Linux.

NDK COMPONENT NAME AND VERSION:
	SMS Developer Components

LAST MODIFIED DATE: 
	13 May 2005

===============================================================================
*/

/*
 * Headers and Defines
 */

#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <locale.h>
#include <limits.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include<iconv.h>
#include<langinfo.h>
#include <smuni.h>

 
#define FALSE 0
#define TRUE 1

#pragma pack(push, 1 )
typedef struct
{
  int     cmask;
  int     cval;
  int     shift;
  long    lmask;
  long    lval;
} Tab;
#pragma pack (pop)

static unsigned  initICONV = FALSE;
static int 		 do_conv = 0;			/* default = no mapping */
static char 	 current_locale[PATH_MAX];
static iconv_t 	 LocToUTF8 = (iconv_t) -1, UTF8ToLoc = (iconv_t) -1;

Tab table[] =
{
        { 0x80, 0x00, 0*6,       0x7F,         0 },     /* 1 byte sequence */
        { 0xE0, 0xC0, 1*6,      0x7FF,      0x80 }, /* 2 byte sequence */
        { 0xF0, 0xE0, 2*6,     0xFFFF,     0x800 },     /* 3 byte sequence */
        { 0xF8, 0xF0, 3*6,   0x1FFFFF,   0x10000 },     /* 4 byte sequence */
        { 0xFC, 0xF8, 4*6,  0x3FFFFFF,  0x200000 },     /* 5 byte sequence */
        { 0xFE, 0xFC, 5*6, 0x7FFFFFFF, 0x4000000 },     /* 6 byte sequence */
        { 0,       0,   0,          0,         0 }  /* end of table    */
};

/*********************************************************************
* NAME: loc2utf
*
* WHAT: Converts one local codeset character to utf-8
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int loc2utf(char *utc, const char *lcc, int *charsused, int bytesLeft)
{
    int len = 0; 
    *utc = 0; 

    if (LocToUTF8 != (iconv_t) -1)
    {
        char *inbuf = (char*)lcc;
        char *outbuf = utc;
        size_t inbytesleft, outbytesleft;
        int inbytessave;

        inbytessave = inbytesleft = mblen(lcc, MB_CUR_MAX); 
        
        if (inbytessave <= 0)
        {
            return -1;
        }

        outbytesleft = bytesLeft;

        if (iconv(LocToUTF8, &inbuf, &inbytesleft, &outbuf, &outbytesleft) == -1)
        {
            return -1;
        }

        /* we used up this many bytes in outbuf */
        len = bytesLeft - outbytesleft;

        utc[len] = 0;           /* NULL terminator      */
        *charsused = len;
        return inbytessave - inbytesleft;
    }
    else if( !do_conv)
    {
		/* locale itself is UTF-8*/
		len = mblen(lcc, MB_CUR_MAX);
		if (!len || len == -1)
		{
			return -1;
		}
		memcpy(utc, lcc, len);
		*charsused = len;
		utc[len] = 0;	/* NULL terminator	*/
		return len;
    }
    else
    {
		return -1;
    }
    
} 

/*********************************************************************
* NAME: utf2loc
*
* WHAT: Converts one utf-8 character to local codeset 
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/

int utf2loc(char *lcc, const char *utc, int *charsused, int bytesLeft)
{
	int c0, len; 
	Tab *t;

	*lcc = 0; 

	/* The next loop determines the number of UTF-8 charcters */
	len = 0;
	c0 = *utc & 0xff;
	for (t = table; t->cmask; t++)
	{
		len++;
		if ((c0 & t->cmask) == t->cval)
			break;
	}

	if (UTF8ToLoc != (iconv_t) -1)
	{
		char *inbuf; char *outbuf;
		size_t inbytesleft, outbytesleft;

		inbuf = (char*)utc; inbytesleft = len; 
		outbuf = lcc; outbytesleft = bytesLeft;
		if (iconv(UTF8ToLoc, &inbuf, &inbytesleft, &outbuf, &outbytesleft) == -1) 
			return -1;

		if (inbytesleft != 0)
		{
			*charsused = 0;
			return -1;
		}

		*charsused = outbuf - lcc;    /* so many characters in local code set */
		if( *charsused > MB_CUR_MAX )
		{
			return -1;
		}
	} 
	else if( !do_conv)
	{
		/* locale itself is UTF-8*/
		strncpy(lcc, utc, len);
		*charsused = len;
	}
	else
	{
		return -1;
	}
        
	lcc[*charsused] = 0;
	return len; 
} 

/*********************************************************************
* NAME: utf2loc
*
* WHAT: Converts utf-8 string to local codeset
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
size_t locs2utfs(char *uts, const char *lcs, size_t n)
{
    int len, nutf;
    int utf_bytes;
    int nloc;
    int outbytesleft;

    /*
      Convert each local code character of the string one by one, 
      Adjust source and destination strings, as required
     */
	outbytesleft = n;
    for(nutf = utf_bytes = nloc = 0; *lcs && ( outbytesleft > 0 );
            lcs += len, uts += utf_bytes, nutf += utf_bytes )
    {
        if((len = loc2utf(uts, lcs, &utf_bytes, outbytesleft)) == -1)
            return -1;
        nloc += len;
        outbytesleft -= utf_bytes;
    }
    if( outbytesleft )
    	*uts = 0;
    else 
    	return -1;

    return nloc;
}

/*********************************************************************
* NAME: utfs2locs
*
* WHAT: Converts utf-8 string to local  string
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
size_t utfs2locs(char *lcs, const char *uts, size_t n)
{
    int len, nutf, nloc;
    int loc_bytes;
    int outbytesleft;

    if( !*uts ) 
    {
		*lcs = 0;
		return 0;
    }
    outbytesleft = n;
    for(nutf = loc_bytes = nloc = 0; *uts && (outbytesleft > 0); 
            uts += len, lcs += loc_bytes, nloc += loc_bytes )
    {
        if((len = utf2loc(lcs, uts, &loc_bytes, outbytesleft)) == -1)
            return -1;
        nutf += len;
        outbytesleft -= loc_bytes;
    }
    if( outbytesleft )
    	*lcs = 0;
    else 
    	return -1;
    return nutf;
}

/*********************************************************************
* NAME: SMutf8nicmp
*
* WHAT: Compares first n bytes of the UTF-8 string
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMutf8nicmp  (const utf8_t *s1, const  utf8_t  *s2, unsigned int n)
{
     int result;
     result = strncasecmp( s1, s2, n);
	return result;
}

/*********************************************************************
* NAME: SMInitUniLibrary
*
* WHAT: Initialises the Unicode Library
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMInitUniLibrary(void *handle)
{
 	char *oldlocale, *codeset, *tmp,*userCodeSet;

 	memset(current_locale, 0, sizeof(current_locale));
	oldlocale = setlocale(LC_ALL, 0);
	if ((tmp = setlocale(LC_ALL, "")))
	{
	  strcpy(current_locale, tmp);
	}
	codeset = nl_langinfo(CODESET);

	if(strcmp(tmp, "C") == 0) /* Assume UTF-8 code set for 'C' locale*/
		codeset = "UTF-8";

	do_conv = 1;

	if (strcmp(codeset, "UTF-8") == 0)
	{
		do_conv = 0;
	}
	else if(((LocToUTF8 = iconv_open("UTF-8", codeset)) == (iconv_t) -1 ) || 
		( (UTF8ToLoc = iconv_open(codeset, "UTF-8")) == (iconv_t) -1))
	{
		if (LocToUTF8 != (iconv_t) -1)
		{
			iconv_close(LocToUTF8);
			LocToUTF8 = (iconv_t) -1;
		}
		if (UTF8ToLoc != (iconv_t) -1)
		{
			iconv_close(UTF8ToLoc);
			UTF8ToLoc = (iconv_t) -1;
		}
		return -1;		/* no unicode tables! */
	}
	initICONV = TRUE;
	return 0;
}

/*********************************************************************
* NAME: SMShutDownUniLibrary
*
* WHAT: Destroys the Unicode Library
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMShutDownUniLibrary(void *handle)
{
	if (do_conv)
	{
		if (LocToUTF8 != (iconv_t) -1)
		{
			iconv_close(LocToUTF8);
			LocToUTF8 = (iconv_t) -1;
		}
		if (UTF8ToLoc != (iconv_t) -1)
		{
			iconv_close(UTF8ToLoc);
			UTF8ToLoc = (iconv_t) -1;
		}
	}
	initICONV = FALSE;
	return 0;
}

/*********************************************************************
* NAME: locs2utfsize
*
* WHAT: returns the size required after converting a local string to UTF-8
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int locs2utfsize(const char *lcs, size_t *n)
{
    int len, nutf;
    int utf_bytes;
    int nloc;
	char uts[MB_LEN_MAX+1] = {0};

    /* 
      Convert each local code character of the string one by one, 
      return the totel buffer size required
     */

    for(nutf = utf_bytes = nloc = 0; *lcs;
            lcs += len, nutf += utf_bytes )
    {
        if((len = loc2utf(uts, lcs, &utf_bytes, MB_LEN_MAX)) == -1)
            return  -1;
        nloc += len;
    }

	*n =nutf;
    return 0;
}

/*********************************************************************
* NAME: utfs2locsize
*
* WHAT: returns the size required after converting a UTF-8 string to locale
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int utfs2locsize( const char *uts, size_t *n)
{
    int len, nutf, nloc;
    int loc_bytes;
	char lcs[MB_LEN_MAX+1] = {0};


    if( !*uts ) 
    {
		*lcs = 0;
		return 0;
    }
    
    for(nutf = loc_bytes = nloc = 0; *uts; uts += len, nloc += loc_bytes )
    {
        if((len = utf2loc(lcs, uts, &loc_bytes, MB_LEN_MAX)) == -1)
            return -1;
        nutf += len;
    }
       *n = nloc; 
    return 0;
}

/*********************************************************************
* NAME: SMloc2utf8
*
* WHAT: Converts local string to UTF-8
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMloc2utf8(utf8_t * dest, size_t *destLen,const char * src, int srclen, UNICODE_CONTEXT* uContext)
{
	int retval =0;

	if(src == NULL || destLen == NULL)
		return -1 ;
	
	if(initICONV == FALSE && (SMInitUniLibrary(NULL) == -1))
		return -1;

	if(dest) 
	{
		retval = locs2utfs(dest, src, *destLen);
		if(retval != -1) 
		{
			*destLen = retval ;
			retval=0;
		}
		return retval ;
	}
	else
	{
		return locs2utfsize( src, destLen);
	}
}

/*********************************************************************
* NAME: SMutf82loc
*
* WHAT: Converts UTF-8 string to local string
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMutf82loc( char * dest, size_t* destLen, const utf8_t * src,  int  srcLen, UNICODE_CONTEXT* uContext)
{
	int retval =0;

	if(src == NULL || destLen == NULL)
		return -1; 
	
	if(initICONV == FALSE && (SMInitUniLibrary(NULL) == -1))
		return -1;
	
	if(dest)
	{
		retval = utfs2locs(dest, src, *destLen);

		if(retval != -1) 
		{
			*destLen = retval ;
			retval = 0;
		}
		return retval ;
	}
	else
	{
		return utfs2locsize( src, destLen);
	}
}

