/*
===============================================================================
Novell Software Developer Kit Sample Code License

Copyright (C) 2005, Novell, Inc.  All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
  *  Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  *  Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.
  *  Neither the name of Novell, Inc. nor the names of its contributors
     may be used to endorse or promote products derived from this
     software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT
SHALL NOVELL, INC., THE COPYRIGHT OWNER, OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

NAME OF FILE:
	demain.c

PURPOSE/COMMENTS:
	Implements a demonstration engine using SMS interfaces 

NDK COMPONENT NAME AND VERSION:
	SMS Developer Components

LAST MODIFIED DATE: 
	19 October 2005
===============================================================================
*/

 /***For NLM Conversion***/
 /****************************************************************************
 *      NOTE: Comments are included with this code that describe how to
 *              convert this demonstration engine to an NLM. To make an NLM 
 *              well behaved, control should be passed to other processes on
 *              the server at least every 150 milliseconds (386 @ 25Mhz).
 *              To accomodate this, calls to the ThreadSwitch function (CLIB) 
 *              should be appropriately scattered through the code. 
 *              ThreadSwitches should be considered for any sections of
 *              code, such as loops, which could execute for more 
 *              than 150 milliseconds. CLIB routines that block should already
 *              pass control to other threads so you won't have to worry about
 *              these.
 *
 *              To find the changes need for converting this demonstration 
 *              to an NLM, search for the string "NLM" in this code.
 ****************************************************************************/

/*** FOR NLM CONVERSION ***
* remove #include <nwcalls.h>
****************************/

/* 
 * Headers and Defines
 */
 
#include <smstypes.h>
#include <smsutapi.h>
#include <smuni.h>
#include<nbkmsg.mlh>
#ifdef N_PLAT_NLM
#include <nwconio.h>
#include <unicode.h>
#include <nwthread.h>
#elif defined(N_PLAT_UNIX)
#include <strings.h>
#include <ncurses.h>
#include <signal.h>

#define puts  printw
#define gets getstr
#define printf printw
#define ConsolePrintf 	printw
#define strcmpi 		strcasecmp
#define EXIT_NLM 		0
#define TSR_THREAD 		0
#define GetNLMID()
#endif
NWSM_LIST_PTR *inFiles = NULL;

#include <sms.h>
#include <deoslib.h>
#include "de.h"

#define bdt "BuIlDdAtEaNdTiMe="

/*
 * Globals
 */

char 		*_cdt_ = bdt __DATE__ " @ "  __TIME__;
int 		 _cdt_o_ = sizeof(bdt) - 1;
int 		 NoOfActiveSessions = 0;
int 		 deScreenHandle,deNLMID;
UINT8 		 NO_MORE_SESSIONS = FALSE;
UINT8 		 TerminateNLM = FALSE;
char 		*buildDateAndTime, *moduleVersion;
#ifdef N_PLAT_UNIX
extern char *NbkMessages[];
SESSION_INFORMATION *curSession = NULL;
#elif defined N_PLAT_NLM
char **NbkMessages;
#endif

/*
 * Function prototypes
 */

void 	DisplayRenameList(SinglePair *);
void 	*ImportSymbol (int , const char *);
int 	UnimportSymbol(int ,const char *);
CCODE	 	DisplayList(NWSM_NAME_LIST *list);
char    	GetCommand(UINT8 toLogSession);
char    	GetBCommand(int backup, char **prompt, int *object);
char    	GetRCommand(UINT8 overwriteParent, UINT8 overwriteChild);
CCODE   	RenameRestore(SESSION_INFORMATION *sessionInfo);
UINT32  	GetNameSpace(SESSION_INFORMATION *sessionInfo);
UINT32  	GetSelectionType(SESSION_INFORMATION *sessionInfo);
void    	SetNameSpaceType(UINT32 nameSpaceType, char *string, SESSION_INFORMATION *sessionInfo);
void    	SetSelectionType(UINT32 selectionType, char *string, SESSION_INFORMATION *sessionInfo);
#ifdef N_PLAT_UNIX
void UnloadHandler(int sig);
#endif

/*
 * Function definitions
 */

/*********************************************************************
* NAME: _NbkMessage
*
* WHAT: String processing function for localization requirements.
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
char *_NbkMessage(int index)
{
	return NbkMessages[index];
}

/*********************************************************************
* NAME: ListTSAs
*
* WHAT: Display a list of the TSAs visible under the target SMDR.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMListTSAs (TSAPI)
*                               NWSMFreeNameList (TSAPI)
*
**********************************************************************/
int ListTSAs(char *target)
{
 	NWSM_NAME_LIST *list = NULL;
	char 			 strName[MAX_SCREEN_NAME_LENGTH];
	
	GetNameOfThread(strName);
	
	ClearScreen();
	NameScreen(strName);
	strcat(target, ".*");
	status("\nVisible Target Service Agents:", NULL);
	if(NWSMListTSAs(target, &list)==0)
	{
		DisplayList(list);
		NWSMFreeNameList(&list);
	}
	else
	{
		printf("Failed to list TSAs");
		NWSMFreeNameList(&list);
		return 1;
	}

	return 0;
}

/*********************************************************************
* NAME: ListSMDRs
*
* WHAT: Display a list of the SMDRs visible on the network.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMListSMDRs (TSAPI)
*                               NWSMFreeNameList (TSAPI)
*
**********************************************************************/
int ListSMDRs(void)
{
	NWSM_NAME_LIST *list = NULL;
	char 			 strName[MAX_SCREEN_NAME_LENGTH];
	
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	status("Visible Storage Management Data Requestors:", NULL);
	if(NWSMListSMDRs("*", &list)==0)      
	{
		DisplayList(list);
		NWSMFreeNameList(&list);
	}	
	else
	{
		printf("Failed to list SMDRs");
		NWSMFreeNameList(&list);	
		return 1;
	}
	return 0;
}

/*********************************************************************
* NAME: ListTSs
*
* WHAT: Display a list of the Target Services on the network.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSListTargetServices (TSAPI)
*                               NWSMFreeNameList (TSAPI)
*
**********************************************************************/
int ListTSs(SESSION_INFORMATION *sessionInfo)
{
	NWSM_NAME_LIST *list=NULL;
	char 			 strName[MAX_SCREEN_NAME_LENGTH];
	
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	status("Target Services:", NULL);
	if(NWSMTSListTargetServices(sessionInfo->connection, "*", &list)==0)      
	{
		DisplayList(list);
		NWSMFreeNameList(&list);
	}	
	else
	{
		printf("Failed to list SMDRs");
		NWSMFreeNameList(&list);	
		return 1;
	}
	return 0;
}


/*********************************************************************
* NAME: ListSelectionStrings
*
* WHAT: Display a list of selection strings used to include/exclude
*       data sets to be scanned on the Target Service. Selection types
*       are used in conjunction with scan types to control data set 
*       scanning.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSGetTargetSelectionTypeStr (SIDR)
*
**********************************************************************/
void ListSelectionStrings(SESSION_INFORMATION *sessionInfo)
{
	CCODE 	ccode = 0;
	char 	string[RESOURCE_NAME_LENGTH];
	UINT8 	typeNumber;
	
	puts("Selection Strings: ");
	for (typeNumber = 1; !ccode; typeNumber ++)
	{
		ccode = NWSMTSGetTargetSelectionTypeStr(sessionInfo->connection, typeNumber, string, "++");
		printf("  %s\n", string);
	}
	DEPause();

	return;
}

/*********************************************************************
* NAME: ListScanTypeStrings
*
* WHAT: Display a list of scan type strings used to include/exclude
*       data sets to be scanned on the Target Service. Scan types are
*       used in conjunction with selection types to control data set
*       scanning.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSGetTargetScanTypeString (SIDR)
*
**********************************************************************/
void ListScanTypeStrings(SESSION_INFORMATION *sessionInfo)
{
    CCODE 	ccode = 0;
    char 	string[RESOURCE_NAME_LENGTH];
    UINT8 	typeNumber;
    UINT32 	req, dis;
	
   printf("Selection Strings: ");
    for (typeNumber = 0; !ccode; typeNumber ++)
    {
        ccode = NWSMTSGetTargetScanTypeString(sessionInfo->connection, typeNumber, string, &req, &dis);
        printf("  %s\n", string);
	}
    DEPause();

	return;
}

/*********************************************************************
* NAME: EstablishConnection
*
* WHAT: Establish a connection between the engine and the Target Service
*       Agent on the target to be scanned, backed up or restored.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMConnectToTSA (TSAPI)
*								NWSMTSListTargetServices (TSAPI)
*								OpenService (demain.c)
*
**********************************************************************/
CCODE EstablishConnection(SESSION_INFORMATION *sessionInfo)
{
	CCODE 			 ccode = 0;
	NWSM_NAME_LIST *services = NULL;
	NWSM_NAME_LIST *p;
	char 			 target[MAX_TARGET_NAME_LENGTH];
	char 			 pattern[MAX_TARGET_NAME_LENGTH];

	/* List and get the SMDR to connect to */

	ListSMDRs();
	
	status("SMDR:       ",NULL); gets(target);
#ifdef N_PLAT_UNIX
	noecho();
#endif
	status("\n           Please wait...\n", NULL);
	strcpy(pattern, target);
	ListTSAs(target);
#ifdef N_PLAT_UNIX
	echo();
#endif
	
	status("\nTSA:       ", NULL); gets(target);
	status("\n           Please wait...\n", NULL);
#ifdef N_PLAT_UNIX
	noecho();
#endif

	/* Connect to the TSA */
	if ((ccode = NWSMConnectToTSA(target, &sessionInfo->connection)) != 0)
	{
		printf("Error NWSMConnectToTSA %x",ccode);
		puts("Failure to connect to Target Service Agent.");
		sessionInfo->connection= 0;
		goto Return;
	}

	/* Query and list the available target services */
	strcpy(sessionInfo->tsaName, target);
	if (strstr(sessionInfo->tsaName, "NetWare Cluster File System") || strstr(sessionInfo->tsaName, "NetWare File System"))
		ccode = NWSMTSListTargetServices(sessionInfo->connection, pattern, &services);
	else
		ccode = NWSMTSListTargetServices(sessionInfo->connection, "*", &services);
	if (ccode) 
	{
		NWSMReleaseTSA(&sessionInfo->connection);
		goto Return;
	}
	
	status("\nTarget Service(s): ", NULL);
	for (p = services; p != NULL; p = p->next) 
	{
		if (p->name) status("%s\n",(char *)p->name);
	}
#ifdef N_PLAT_UNIX
	echo();
#endif
	status("\n\rWhich target service do you want to connect to?  ", NULL); gets(sessionInfo->svcName);

	/* Prompt for the user name and password to use for connecting to the target service */
	status("User name: ", NULL); 
	getun(sessionInfo->CuserName);
	status("Password:  ", NULL); 
	getpw(sessionInfo->CpassWord);
	status("\n", NULL);

	/* Attempt to connect to the target service */
	ccode = OpenService(sessionInfo);
	if (ccode)
	{
		NWSMReleaseTSA(&sessionInfo->connection);
		goto Return;
	}
	
Return:
	if(services) 
		NWSMFreeNameList(&services);
	
	return (ccode);
}


/*********************************************************************
* NAME: BuildSelectionList
*
* WHAT: Build a list for including/excluding data sets in/from a scan
*       of a Target Service.
*       The user is prompted for name space type, selection type, and
*       selection item for each element in the list.
*
* SMS-SPECIFIC ROUTINES CALLED: GetNameSpace (Deutil.c)
*                               GetSelectionType (Deutil.c)
*                               NWSMCloseName (SMS UTILITY API)
*								InsertToSelectionList(Deutil.c)
**********************************************************************/
void BuildSelectionList(SESSION_INFORMATION *sessionInfo)
{
	BUFFER 			 buffer[MAX_SELECTION_ITEM_LENGTH];
    CCODE 			 ccode;
    STRING_BUFFER 	*firstSep = NULL, *nextSep = NULL;
    NWBOOLEAN 		 firstTime = TRUE;
    NWBOOLEAN 		 reverseOrder = FALSE;
    UINT32 			 nameHandle = 0, nameSpaceType, selectionType;
    char 			 strName[MAX_SCREEN_NAME_LENGTH];
	char 			*choice;

	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);

	if (sessionInfo->selectionList)
	{
		free(sessionInfo->selectionList);
		sessionInfo->selectionList = NULL;
	}
	printf("\n");

	/* Try to get the user specified selection list.
	 * This is done in a loop as there may be multiple sesection entries */
	for(;;)
	{
		
		/* Get the selection type for the following entry */
		if ((selectionType = GetSelectionType(sessionInfo)) == 100)
			goto Return;
		
		/* Get the name space in which this selection list is being entered */
		nameSpaceType = GetNameSpace(sessionInfo);
		if (nameSpaceType == 100)
			goto Return;

		printf("\n\nEnter selection item: ");
		gets(buffer);

		ccode = InsertToSelectionList(sessionInfo, buffer, nameSpaceType, selectionType, &nameHandle);
		if(ccode)
			goto Return;
		
		ClearScreen();
		NameScreen(strName);
		printf("\n\nDo you wish to enter another selection item?(y/n):");
		while ((choice = strchr("yn", getch())) == NULL) ;
		if(*choice == 'n')
		{
			goto Return;
		}
	}
Return:
	if (nameHandle)
		NWSMCloseName(&nameHandle);

	return;
}

/*********************************************************************
* NAME: DisplaySelectionList
*
* WHAT: Display the list used for selecting data sets from the target
*       service. This list is created by the BuildSelectionList function.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMGetFirstName (SMS UTILITY API)
*                               SetNameSpaceType (Deutil.c)
*                               SetSelectionType (Deutil.c)
*                               NWSMGetNextName (SMS UTILITY API)
*                               NWSMCloseName (SMS UTILITY API)
*
**********************************************************************/
void DisplaySelectionList(SESSION_INFORMATION *sessionInfo)
{
    CCODE 		 ccode;
    UINT32 		 handle = 0;
    char 		 string1[NAME_SPACE_TYPE_STRING_LENGTH];
    char 		 string2[SELECTION_TYPE_STRING_LENGTH];
    char 		 strName[MAX_SCREEN_NAME_LENGTH];
	size_t		 destLen;
	char 		*dest = NULL;
    NWSM_DATA_SET_NAME		name;
	
    GetNameOfThread(strName);

	if(sessionInfo->selectionList == NULL)
		goto Return;

	/* Get the first name in the selection list */
	if ((ccode = NWSMGetFirstName((void *)sessionInfo->selectionList, &name, &handle)) != 0)
    {
        printf("\nNWSMGetFirstName = 0x%X\n", ccode);
        goto Return;
    }
	ClearScreen();
	NameScreen(strName);
	printf("\n");
    puts("SelectionList:\n");

	/* Loop till all elements in the list are displayed */
    do
    {
		if(sessionInfo->IsUTF8Supp)
		{
			/* If the lsit is maintained in UTF-8 then convert to MBCS for displayint it to the user */
			name.nameSpaceType = MapToMBCSNameSpace(name.nameSpaceType);
			ccode = ConvertUTF8ToLoc( name.name, &destLen, &dest );
			if(ccode)
			{
				goto Return;
			}
			strncpy(name.name, dest, strlen(dest));
			if(dest)
			{
				free(dest);
				dest = NULL;
			}		
		}
		
		SetNameSpaceType(name.nameSpaceType, string1, sessionInfo);
        SetSelectionType(name.selectionType, string2, sessionInfo);
        printf("%s, %s:\n  \"%s\"\n\n", string1, string2, name.name);

		/* Get the next name in the list */
        if (((ccode = NWSMGetNextName(&handle, &name)) != 0) && (ccode != NWSMUT_NO_MORE_NAMES))
			printf("\nNWSMGetNextName = 0x%X\n", ccode);
    } while (!ccode);
	
    puts("");
    DEPause();
	
Return: 
    if (handle && ((ccode = NWSMCloseName(&handle)) != 0))
		printf("\nNWSMCloseName = 0x%X\n", ccode);

	return;
}

/*********************************************************************
* NAME: FreeSelectionList
*
* WHAT: Free the memory allocated for the data set selection list.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void FreeSelectionList(SESSION_INFORMATION *sessionInfo)
{
    if (sessionInfo->selectionList)
    {
		free(sessionInfo->selectionList);
		sessionInfo->selectionList = NULL;
    }

	return;
}

/*********************************************************************
* NAME: Backup
*
* WHAT: This function controls the overall process of backing up the
*       Target Service. A complimentary function (Restore) exists for
*       restoring data to a Target Service.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMPutOneName (SMS UTILITY API)
*                               NWSMTSScanDataSetBegin (TSAPI)
*                               NWSMGetDataSetName (SMS UTILITY API)
*                               NWSMTSGetNameSpaceTypeInfo (TSAPI)
*                               NWSMTSScanDataSetEnd (TSAPI)
*                               NWSMTSOpenDataSetForBackup (TSAPI)
*                               NWSMTSCloseDataSet (TSAPI)
*                               NWSMTSReadDataSet (TSAPI)
*                               NWSMTSScanNextDataSet (TSAPI)
*                               NWSMTSScanDataSetContinue (TSAPI)
*                               NWSMCatStrings (SMS UTILITY API)
**********************************************************************/
void ProcessBackupParams(SESSION_INFORMATION *sessionInfo)
{
	CCODE 			 ccode;
	char 			 myName[SERVICE_RESOURCE_NAME_LENGTH];
	char 			*prompt;
	char 			 option[5];
	STRING_BUFFER 	*firstSep = NULL, *nextSep = NULL;
	NWBOOLEAN 		 reverseOrder = FALSE; 
	int 			 backup = TRUE, command, resource, restart = TRUE;
	int 			 resourceListCount=1;
	int 			 x, y;
	int 			 index;
	char 			 strName[MAX_SCREEN_NAME_LENGTH];
	size_t			 destLen;
	utf8_t 			*dest = NULL;
	int				 rcode;	
	NWSM_DATA_SET_NAME_LIST *resourceName;
	NWSM_NAME_LIST 				*serviceResourceList;
	NWSM_NAME_LIST 				*saveFirstResourceList;
	BACKUP_CONTEXT *bctx = sessionInfo->bctx ;
	char  dataFileNo[2] = {0};

	
	GetNameOfThread(strName);
	serviceResourceList = saveFirstResourceList = NULL;

	if(bctx == NULL)
	{
		return;
	}

    /* prepare the data file name for the session  */
	IntegerToString( sessionInfo->sessionNo, dataFileNo, 10,"%d");
	strcpy(sessionInfo->destFile, "De-");
	strcat(sessionInfo->destFile, dataFileNo);
	strcat(sessionInfo->destFile, ".sdf");

Loop:
	resourceName = NULL;
	sessionInfo->scanInfo = NULL;
	bctx->scanOnly = FALSE;
	/*Init statistics */
	ResetStats(sessionInfo);

	bctx->dsetList= NULL;
	/*********************************************************************
	*       Prompt the user for a backup or scan command: Backup Resource,
	*               Backup Subdirectories, Scan Resource, Scan Subdirectories,
	*               Save Backup Data (toggle on/off).
	**********************************************************************/
	switch (command = GetBCommand(bctx->saveBackup, &prompt, &resource))
	{
		case 'T':
		bctx->saveBackup = !bctx->saveBackup;
		goto Loop;

		/*********************************************************************
		*       Prompt the user for the name of the resource or subdirectory to
		*       scan. This resource/subdirectory is added to a list for backup
		*       or scan.
		**********************************************************************/
		case 'S':
			bctx->scanOnly = TRUE;
		case 'B':
		if (resource)
		{
			fprintf( stderr, "\nEnter %s to %s: ", prompt, command == 'B' ? "backup" : "scan");

			/* Query and display the list of resources */
			resourceListCount = 1;
			NWSMTSListTSResources(sessionInfo->connection, &serviceResourceList);
			saveFirstResourceList = serviceResourceList;
			while (serviceResourceList != NULL)
			{
				status("\n\r\t%d: ", resourceListCount);
				status("%s\n\r", serviceResourceList->name);
				serviceResourceList = serviceResourceList->next;
				resourceListCount ++;
			}
			status("\t%d: Return to previous menu\n\r", resourceListCount);
			status("\n\rPlease enter selection: ", NULL);
			CursorPosition(&x, &y);
			gets(option);
			index = atoi(option);
			if(index==resourceListCount)
				goto Loop;
			while ((index < 1) || (index > resourceListCount))
			{
				//printf("\a\a");
				SetCursor(x, y);
				gets(option);
				index = atoi(option);
			}

			/* Process the selected resource and insert it into the resourceName parameter */
			serviceResourceList = saveFirstResourceList;
			for (resourceListCount=1; resourceListCount < index; resourceListCount++)
			{
				serviceResourceList = serviceResourceList->next;
			}
			strcpy(myName, (char *)serviceResourceList->name);
			ccode = NWSMPutOneName((void **)&bctx->dsetList, NWSM_TSA_DEFINED_RESOURCE_TYPE, 
							0L, 0, NULL, NULL, myName);

		}
		else
		{
			/* Prompt for the name space in which the resource to scan is  to be entered */
			sessionInfo->nameSpaceType = GetNameSpace(sessionInfo);		
			if (bctx->nameSpaceType == 100)
			{
				bctx->nameSpaceType = DOSNameSpace;
				goto Loop;
			}

			/* Get the namespaces information */
			if (sessionInfo->IsUTF8Supp)
			{
				sessionInfo->nameSpaceType = MapToUTF8NameSpace( sessionInfo->nameSpaceType);
				ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, sessionInfo->nameSpaceType, &reverseOrder,
								&firstSep, &nextSep);
			}
			else
				ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, sessionInfo->nameSpaceType, &reverseOrder, &firstSep, &nextSep);
			if (ccode != 0)
			{
				PrintError(ccode, "NWSMTSGetNameSpaceTypeInfo", __LINE__, sessionInfo);
				restart = FALSE;
				goto Terminate;
			}

			/* Prompt for the resource name */
			printf("\nEnter %s to %s: ", prompt, command == 'B' ? "backup" : "scan");
			gets(myName);

			/* IF UTF-8 is supported then pass in the UTF-8 encoding of the resourceName */
			if (sessionInfo->IsUTF8Supp)
			{
				ccode = ConvertLocToUTF8( myName, &destLen, &dest);
				if (ccode)
				{
					if (dest)
					{
						free(dest);
						dest = NULL;
					}
					goto Terminate;
				}
				
				ccode = NWSMPutOneName((void **)&bctx->dsetList, sessionInfo->nameSpaceType, 0L, reverseOrder, firstSep->string, nextSep->string, dest);
				if (dest)
				{
					free(dest);
					dest = NULL;
				}
			}
			else
				ccode = NWSMPutOneName((void **)&bctx->dsetList, sessionInfo->nameSpaceType, 0L, reverseOrder, firstSep->string, nextSep->string, myName);
		}
		break;
		
		default:
		restart = FALSE;
		goto Terminate;
	}

	if (ccode)
	{
		printf("\nNWSMPutOneName:   ccode = %lu\n", ccode);
		restart = FALSE;
		goto Terminate;
	}
	ClearScreen();
	NameScreen(strName);

	ccode = InitBackupSession( sessionInfo);
	if(ccode)
	{
		printf("\nEither memory allocation failed or error in opending file\n");
		restart = FALSE;
		goto Terminate;	
	}
	/* Save session information - session and backup file names */
	if (bctx->saveBackup )
	{
		if ((SaveSessionInfo(sessionInfo)) != 0)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_SAVE_SESSION_ERR));
			goto Terminate;
		}
	}
	Backup(sessionInfo, &restart);
	ccode = CloseBackupSession( sessionInfo);
	if(ccode)
	{
		restart = FALSE;
		goto Terminate;	
	}

	if (restart)
	{
		status("\n\rNumber of resources scanned: %u\n", sessionInfo->stats.resCount);
		status("\n\rNumber of Directories scanned: %u\n", sessionInfo->stats.dirCount);
		status("\n\rNumber of Files scanned: %u\n", sessionInfo->stats.fileCount );
		DEPause();

		if (resourceName) 
			free(resourceName);
		if(bctx->dsetList)
		{
			free(bctx->dsetList);
			bctx->dsetList= NULL;
		}
		goto Loop;
	}
	Terminate:
	if (resourceName) 
		free(resourceName);

	NWSMFreeString(&firstSep);
	firstSep=NULL;
	
	NWSMFreeString(&nextSep);
	nextSep=NULL;	
	return;
}

/*********************************************************************
* NAME: Restore
*
* WHAT: This function controls the overall process of restoring data
*       to a Target Service. A complimentary routine (Backup) exists
*       for backing up Target Service data.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSSetRestoreOptions (TSAPI)
*                               NWSMTSCloseDataSet (TSAPI)
*                               NWSMTSOpenDataSetForRestore (TSAPI)
*                               NWSMTSWriteDataSet (TSAPI)
*
**********************************************************************/
void ProcessRestoreParams(SESSION_INFORMATION *sessionInfo)
{
    CCODE 			 ccode;
    int 			 command, restart = TRUE;
	int 			 x, y;
    char 			 strName[MAX_SCREEN_NAME_LENGTH];
    RESTORE_CONTEXT *rctx = sessionInfo->rctx;
	int listOnly = FALSE;
	char  dataFileNo[2] = {0};
	

	GetNameOfThread(strName);
	if(rctx == NULL)
	{
		return;
		
	}
    /* Obtain the list of sessions previously backed up and dispay them */
    /* Restore will be performed from the file indicated by the restoreSessionIndex parameter */
    if ((ObtainSessionInfo(&rctx->restoreSessionIndex)) != 0)
    {
        if (rctx->restoreSessionIndex != -1)
        {
            printf("\nError in Reading Session Information!\n");		
            DEPause();				
        }
        goto Return;
    }  
    /* prepare the data file name for the session to be restored */
    IntegerToString(rctx->restoreSessionIndex, dataFileNo,10,"%d");
	strcpy(sessionInfo->destFile, "De-");
	strcat(sessionInfo->destFile, dataFileNo);
	strcat(sessionInfo->destFile, ".sdf");

Loop:
    ccode = 0;

	/*********************************************************************
	*       Prompt the user to select the option(s) of overwriting/not overwriting
	*       the Parent/Child data set.
	**********************************************************************/
    switch (command = GetRCommand(rctx->overwriteParent, rctx->overwriteChild))
    {
    case 'R':
		break;

	case 'N':
		ccode = RenameRestore(sessionInfo);
		if(ccode != 0)
			goto Terminate;
		DisplayRenameList(rctx->renameList);
		goto Loop;

	case 'D':
		DisplayRenameList(rctx->renameList);
		goto Loop;

	case 'F':
		FreeRenameList(rctx->renameList);
		rctx->renameList = NULL;
		goto Loop;

	case 'P':
        rctx->overwriteParent =!rctx->overwriteParent;
        goto Loop;

    case 'C':
        rctx->overwriteChild =!rctx->overwriteChild;
        goto Loop;

    default:
        restart = FALSE;
        goto Terminate;
    }
	ClearScreen();
	NameScreen(strName);

	/*Init statistics */
	ResetStats(sessionInfo);
	ccode=InitRestoreSession( sessionInfo);
	if(ccode)
	{
		goto Terminate;
	}

	Restore(sessionInfo, &restart);
	/* Display Statistics */
	if (restart)
    {
		float kb = sessionInfo->stats.totalBytesRead / ONE_KB;
		float	min = sessionInfo->stats.totalTime / SIXTY_SECONDS;
		/* The index file begins with a END_OF_RECORD_MARKER so we need to decrement resourceCount */
		status("\n\rNumber of resources restored: %u\n\r", --sessionInfo->stats.resCount );
		status("Total written: %.2f KB\n\r", kb);
		status("Elapsed time:  %.2f minutes\n\r", min);
		if(min) status("Throughput:    %.2f MB/min\n\r", kb / min / ONE_KB);
    }
	DEPause();
	if(restart)
		goto Loop;
	Terminate:
	FreeRenameList(rctx->renameList);
	rctx->renameList = NULL;


	Return :
    	return;
}

/*********************************************************************
* NAME: EraseFile
*
* WHAT: Delete a selected Data Set
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMPutOneName (SMS UTILITY API)
*                               NWSMTSScanDataSetBegin (TSAPI)
*                               NWSMTSScanNextDataSet (TSAPI)
*                               NWSMGetDataSetName (SMS UTILITY API)
*                               NWSMTSDeleteDataSet (TSAPI)
*                               NWSMTSScanDataSetEnd (TSAPI)
*
**********************************************************************/
void EraseFile(SESSION_INFORMATION *sessionInfo)
{
	CCODE 			 ccode;
	UINT32 			 sequence = 0L;
	char 			 myName[FILE_OR_DIRECTORY_NAME_LENGTH], mode;	
	UINT32 			 nameSpaceType = DOSNameSpace;  
	STRING_BUFFER 	*firstSep = NULL, *nextSep = NULL;
	NWBOOLEAN 		 reverseOrder = FALSE; 
	NWSM_DATA_SET_NAME 			 dataSetName;
	NWSM_DATA_SET_NAME_LIST 	*resourceName = NULL, *dataSetNameList = NULL;
	STRING_BUFFER 				*resourceParentDataSetName = NULL, *resourceChildDataSetName = NULL;
	NWSM_DATA_SET_NAME 			 toDeleteDataSetName;

	sessionInfo->scanInfo = NULL;
	nameSpaceType = GetNameSpace(sessionInfo);		
	if (nameSpaceType == 100)
	{
		goto Terminate;
	}
	ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, nameSpaceType, &reverseOrder, &firstSep, &nextSep);
	if (ccode != 0)
	{
		printf("\nNWSMTSGetNameSpaceTypeInfo:   ccode = %lu\n", ccode);
		goto Terminate;
	}

	/*********************************************************************
	*       Prompt the user to select whether to delete a file or a
	*       directory. Then get the name of the file/directory to delete.
	**********************************************************************/
	printf("\nDelete File or Directory? (f/d): ");
	GetCharacter(&mode);
	mode = toupper(mode);
	printf("\nEnter %s to %s: ", mode == 'F' ? "file" : "directory", "delete: ");
	gets(myName); 

	/*********************************************************************
	*       Create a data set name list with an entry for the above selected
	*       file/directory.
	**********************************************************************/
	ccode = NWSMPutOneName((void **)&resourceName, nameSpaceType, 0L, 0, firstSep->string, nextSep->string, myName);
	if (ccode)
	{
		PrintError(ccode, "NWSMPutOneName", 0, sessionInfo);
		goto Terminate;
	}

	/*********************************************************************
	*       Scan for the selected file/directory. If a directory was selected,
	*       then the NWSMTSScanDataSetBegin will be sufficient to return the
	*       correct data set. If a file was selected (mode = F), another
	*       scan is needed since the first scan gets the directory (parent)
	*       and the second scan returns the desired file (child).
	**********************************************************************/
	if ((ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, resourceName, &sessionInfo->scanControl, sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0)
	{
		PrintError(ccode, "NWSMTSScanDataSetBegin", 0, sessionInfo);
		goto Terminate;
	}

	if (mode == 'F')
	{
		if ((ccode = NWSMTSScanNextDataSet(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0)
		{
			PrintError(ccode, "NWSMTSScanNextDataSet", __LINE__, sessionInfo);
			goto Terminate;
		}
	}

	/*********************************************************************
	*       This call to NWSMGetDataSetName provides the name that results
	*       from the above scan. This is only used to provide feedback to 
	*       the user.
	**********************************************************************/
	NWSMGetDataSetName(dataSetNameList, nameSpaceType, &dataSetName);

	/*********************************************************************
	*               If the item returned by NWSMTSScanNextDataSet does not match with
	*               the last item in full path of resource name, then that item is not
	*               to be deleted.
	**********************************************************************/                
	NWSMGetDataSetName(resourceName, nameSpaceType, &toDeleteDataSetName);
	NWSMTSSeparateDataSetName(sessionInfo->connection, nameSpaceType, toDeleteDataSetName.name, &resourceParentDataSetName, &resourceChildDataSetName);          
	/**********************************************************************
	*               Check if the current Data Set Item matches with the last item in
	*               the specified path. If it is a file then NWSMTSScanNextDataSet
	*               returns only the terminal node name in dataSetName.
	**********************************************************************/
	if(mode == 'F' && strcmpi(resourceChildDataSetName->string,dataSetName.name))
	{
		printf("\nCannot Delete This Data Set!\n");
		goto Terminate;
	}

	/**********************************************************************
	*               If the item is a directory then the dataSetName will contain a
	*               terminating '/' to indicate that it is a parent. Also, for a parent,
	*               full path is returned by NWSMTSScanDataSetBegin.
	**********************************************************************/
	if(mode != 'F' && strcmpi(dataSetName.name, strcat(toDeleteDataSetName.name, "/")))
	{
		printf("\nCannot Delete This Data Set!\n");
		goto Terminate;
	}

	/*********************************************************************
	*       Display the name of the file/directory returned by the scan.
	**********************************************************************/
	status("\n\rDelete: %s\n\r", dataSetName.name);

	/*********************************************************************
	* Delete the selected file/directory 
	**********************************************************************/
	if ((ccode = NWSMTSDeleteDataSet(sessionInfo->connection, sequence)) != 0)
		PrintError(ccode, "NWSMTSDeleteDataSet", 0, sessionInfo);

	/*********************************************************************
	* Close the scan and clean up 
	**********************************************************************/
Terminate:
	NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);

	if (resourceName)
		free(resourceName);
	
	if (resourceParentDataSetName) NWSMFreeString(&resourceParentDataSetName);
	if (resourceChildDataSetName) NWSMFreeString(&resourceChildDataSetName);           
	
	DEPause();
	
	return;
}

/*********************************************************************
* NAME: RenameFile
*
* WHAT: Rename a selected data set.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMPutOneName (SMS UTILITY API)
*                               NWSMTSScanDataSetBegin (TSAPI)
*                               NWSMTSScanNextDataSet (TSAPI)
*                               NWSMGetDataSetName (SMS UTILITY API)
*                               NWSMTSRenameDataSet (TSAPI)
*                               NWSMTSScanDataSetEnd (TSAPI)
*
**********************************************************************/
void RenameFile(SESSION_INFORMATION *sessionInfo)
{
	CCODE 			 ccode;
	UINT32 			 sequence = 0L;
	char 			 myName[FILE_OR_DIRECTORY_NAME_LENGTH], newName[FILE_OR_DIRECTORY_NAME_LENGTH], mode;	
	UINT32 			 nameSpaceType = DOSNameSpace;  
	STRING_BUFFER 	*firstSep=NULL, *nextSep=NULL;
	NWBOOLEAN 		 reverseOrder = FALSE; 
	NWSM_DATA_SET_NAME 			 dataSetName;
	NWSM_DATA_SET_NAME_LIST 	*resourceName = NULL, *dataSetNameList = NULL;
	STRING_BUFFER 				*resourceParentDataSetName = NULL, *resourceChildDataSetName = NULL;
	NWSM_DATA_SET_NAME 			 toDeleteDataSetName;

	sessionInfo->scanInfo = NULL;
	nameSpaceType = GetNameSpace(sessionInfo);		
	if (nameSpaceType == 100)
	{
		goto Terminate;
	}
	ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, nameSpaceType, &reverseOrder, &firstSep, &nextSep);
	if (ccode != 0)
	{
		printf("\nNWSMTSGetNameSpaceTypeInfo:   ccode = %lu\n", ccode);
		goto Terminate;
	}
	/*********************************************************************
	*       Prompt the user to select whether to rename a file or a directory.
	*       Then get the name of the file/directory to rename.
	**********************************************************************/
	printf("\nRename File or Directory? (f/d): "); 
	GetCharacter(&mode);
	mode=toupper(mode);
	printf("\nEnter %s to %s: ", mode == 'F' ? "file" : "directory", "rename"); gets(myName);
	/*********************************************************************
	*       Create a data set name list with an entry for the above
	*       selected file/directory.
	**********************************************************************/
	ccode = NWSMPutOneName((void **)&resourceName, nameSpaceType, 0L, 0,firstSep->string, nextSep->string, myName);
	if (ccode)
	{
		PrintError(ccode, "NWSMPutOneName", 0, sessionInfo);
		goto Terminate;
	}
	/*********************************************************************
	*       Scan for the directory that was selected above or for the
	*       parent directory if a file was selected for renaming.
	**********************************************************************/
	if ((ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, resourceName, &sessionInfo->scanControl, sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0)
	{
		PrintError(ccode, "NWSMTSScanDataSetBegin", 0,sessionInfo);
		goto Terminate;
	}

	/*********************************************************************
	*       Prompt for the new name of the file/directory.
	**********************************************************************/
	printf("Enter new %s %s: ", mode == 'F' ? "file" : "directory", "name");
	gets(newName);

	/*********************************************************************
	*       If a file was selected above (mode = F) do another scan since
	*       the first scan only returned the parent of this file.
	**********************************************************************/
	if (mode == 'F')
	{
		if ((ccode = NWSMTSScanNextDataSet(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0)
		{
			PrintError(ccode, "NWSMTSScanNextDataSet", __LINE__, sessionInfo);
			goto Terminate;
		}
	}

	/*********************************************************************
	*       This call to NWSMGetDataSetName provides the name that resulted
	*       from the above scan. This is used to provide feedback to
	*       the user and also to make a final compare before renaming.
	**********************************************************************/
	NWSMGetDataSetName(dataSetNameList, nameSpaceType, &dataSetName);

	/*********************************************************************
	*               If the item returned by NWSMTSScanNextDataSet does not match with
	*               the last item in full path of resource name, then that item is not
	*               to be renamed.
	**********************************************************************/                
	NWSMGetDataSetName(resourceName, nameSpaceType, &toDeleteDataSetName);
	NWSMTSSeparateDataSetName(sessionInfo->connection,nameSpaceType,toDeleteDataSetName.name,&resourceParentDataSetName,&resourceChildDataSetName);          

	/**********************************************************************
	*               Check if the current Data Set Item matches with the last item in
	*               the specified path. If it is a file then NWSMTSScanNextDataSet
	*               returns only the terminal node name in dataSetName.
	**********************************************************************/
	if (mode == 'F' && strcmpi(resourceChildDataSetName->string,dataSetName.name))
	{
		printf("\nCannot Rename This Data Set!\n");
		goto Terminate;
	}


	/**********************************************************************
	*               If the item is a directory then the dataSetName will contain a
	*               terminating '/' to indicate that it is a parent. Also, for a parent,
	*               full path is returned by NWSMTSScanDataSetBegin.
	**********************************************************************/
	if (mode != 'F' && strcmpi(dataSetName.name,strcat(toDeleteDataSetName.name,"/")))
	{
		printf("\nCannot Rename This Data Set!\n");
		goto Terminate;
	}
	/*********************************************************************
	*       Display the name of the file/directory returned by the scan.
	**********************************************************************/
	fprintf(stderr, "\nRename %s to %s\n", dataSetName.name, newName);
	
	/*********************************************************************
	* Rename the selected file 
	**********************************************************************/
	if ((ccode = NWSMTSRenameDataSet(sessionInfo->connection, sequence, nameSpaceType, newName)) != 0)
		PrintError(ccode, "NWSMTSRenameDataSet", 0, sessionInfo);

	/*********************************************************************
	* Close the scan and clean up 
	**********************************************************************/
Terminate:
	NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
	
	if (resourceName)
		free(resourceName);
	
	if (resourceParentDataSetName) NWSMFreeString(&resourceParentDataSetName);
	if (resourceChildDataSetName) NWSMFreeString(&resourceChildDataSetName);           
	
	DEPause();
	
	return;
}



/*********************************************************************
* NAME: Init_Cluster_Params
*
* WHAT: Initializes the Cluster Parameters if the selected Resource is a Cluster file system
*
* SMS-SPECIFIC ROUTINES CALLED: 
**********************************************************************/
CCODE GetClusterParams(SESSION_INFORMATION *sessionInfo)
{
	CCODE 	 ccode = 0;
	int 	 retries = 0,waitperiod = 0,retryP = 0;
	char 	*choice;
	char 	 strName[MAX_SCREEN_NAME_LENGTH];
	char 	tempTsaName[512];

	strcpy(tempTsaName, sessionInfo->tsaName);
	strupr(tempTsaName);
	if(strstr(tempTsaName,"NETWARE CLUSTER FILE SYSTEM"))
	{
		ccode =InitClusterParams(sessionInfo ) ;
		if(ccode == 0)
		{
				GetNameOfThread(strName);
				ClearScreen();
				NameScreen(strName);
				printf("\r\n Cluster Parameters\n");
				printf("\r\n  No of Retries     : %d", DEFAULT_RE_TRIES);
				printf("\r\n  Retry Period      : %d secs", DEFAULT_RE_TRY_PERIOD);
				printf("\r\n  Initial Wait Time : %d minute\n", DEFAULT_WAIT_TIME);
				printf("\r\n  Use Default Cluster values(y/n)?");
				while ((choice = strchr("ynYN", getch())) == NULL) ConsoleSpeakerToBeep();
					if (*choice == 'n' || *choice =='N')
					{
						do {
							printf("\r\n Enter No of Retries default(10)      :");
							scanf("%d",&retries);
						} while(retries < 0);
						
						do {
							printf("\r\n Enter Retry Period(secs) (default 30secs) :");
							scanf("%d",&retryP);
						} while(retryP < 0);
						
						do {
							printf("\r\n Enter Initial Wait period(mins) default(1 minimum):");
							scanf("%d",&waitperiod);
						} while(waitperiod < 1);
						
						sessionInfo->clusterparams->reTries = retries;
						sessionInfo->clusterparams->reTryPeriod = retryP;
						sessionInfo->clusterparams->waitTime = waitperiod;
					}
			}
    	}

	return ccode;
}

/*********************************************************************
* NAME: EndSession
*
* WHAT: Exits the thread when the particular session has ended, and destroys
*               the screen associated with it. Also releases the target Service and 
*               the Connection to TSA for that session.
* 
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSReleaseTargetService (TSAPI)
*                               NWSMReleaseTSA (TSAPI)
**********************************************************************/
void EndSession(int screenHandle, SESSION_INFORMATION *sessionInfo)
{
	static int 	count = 0;
	int 		sessionsRemaining;
	
	sessionsRemaining = NoOfActiveSessions - (++count);
	CloseTarget(sessionInfo);
	ReleaseSessionInfoBuffer(sessionInfo);
	DestroyTheScreen(screenHandle);
	
	if (sessionsRemaining == 0)
		TerminateNLM = TRUE;
	else 
		TerminateNLM = FALSE;
	
	if (NO_MORE_SESSIONS && (sessionsRemaining == 0))
	{
		ClearModuleDontUnloadFlag(GetNLMID());
		ExitTheThread(EXIT_NLM, 0);
	}
	else
		ExitTheThread(TSR_THREAD, 0);

	return;
}

/*********************************************************************
* NAME: StartSession
*
* WHAT: Execution for each thread begins from this function. It
*               creates a screen for each session and names the current thread.
*               It then calls the appropriate function for the user's choice.
*               Before quitting, it frees resources and calls EndSession.
*
* SMS-SPECIFIC ROUTINES CALLED: GetCommand (Deutil.C)
*										EstablishConnection (Demain.C)
*										Init_Cluster_Params (Demain.C)
*                                       BuildSelectionList (Demain.C)
*                                       DisplaySelectionList (Demain.C)
*                                       FreeSelectionList (Demain.C)
*                                       EraseFile (Demain.C)
*                                       RenameFile (Demain.C)
*                                       Backup (Demain.C)
*                                       Restore (Demain.C)
*                                                               
**********************************************************************/
StartSession(char *sessionName)
{
	int 		screenHandle;
	UINT8 		toLogSession=FALSE;
	CCODE 		ccode;
	UINT32 		tsaMajorVersion=0;
	UINT32 		tsaMinorVersion=0;
    UINT32 		supportedTypes=0;
	SESSION_INFORMATION *sessionInfo;
#ifdef N_PLAT_NLM
	UINT32 (*GetSupportedNameTypes)(UINT32, UINT32 *) = 0;
	UINT32 (*GetTargetServiceApiVersion)(UINT32, UINT32 *, UINT32*) = 0;
#endif
	
    screenHandle = CreateNewScreen(sessionName,0);
    SetTheCurrentScreen(screenHandle);
    DisplayTheScreen(screenHandle);
    RenameTheThread(sessionName);
    NameScreen(sessionName);
    
    if ((sessionInfo = GetSessionInfoBuffer()) == NULL)
    {
		puts("Unable to Allocate Memory");
		DEPause();
		EndSession(screenHandle, sessionInfo);
    }
	
#ifdef N_PLAT_UNIX
	curSession=sessionInfo;
	strcpy(sessionInfo->sName, sessionName);
#endif
    sessionInfo->sessionNo = NoOfActiveSessions;
    NoOfActiveSessions++;
	sessionInfo->verbose = TRUE;
	SetLogLevel(NBK_ALL_MESSAGES);

    if (EstablishConnection(sessionInfo))
    {
		puts("Connection denied.");
		DEPause();
		EndSession(screenHandle, sessionInfo);
#ifdef N_PLAT_UNIX
		return;
#endif
    }

	/* Check support for UTF-8 Name Space*/
	CheckUTF8Support(sessionInfo);

    if (( GetClusterParams(sessionInfo)) != 0 )
   	{
		puts("Unable to Allocate Memory");
		DEPause();
		EndSession(screenHandle, sessionInfo);
   	}

    for(;;)
    {
		switch (GetCommand(toLogSession))
		{
		case 'L':
			LogSessionInfo(&toLogSession, sessionInfo);
			break;
		case 'C':
			BuildSelectionList(sessionInfo);
			DisplaySelectionList(sessionInfo);
			break;

		case 'D':
			DisplaySelectionList(sessionInfo);
			break;

		case 'F':
			FreeSelectionList(sessionInfo);
			break;

		case 'E':
			EraseFile(sessionInfo);
			break;

		case 'N':
			RenameFile(sessionInfo);
			break;

		case 'B':
			ProcessBackupParams(sessionInfo);
			break;

		case 'R':
			ProcessRestoreParams(sessionInfo);
			break;

		case 'Q': 
			/***FOR NLM CONVERSION***/
			/* Be sure to clean up and release all resources before calling exit
			 * if you have added any code of your own to this file--otherwise a nasty
			 * message is displayed on the server console. */ 
			EndSession(screenHandle, sessionInfo);
			return(0);
		}
    }
}

/*********************************************************************
* NAME: main
*
* WHAT: This is the main function for the demonstration engine.
*       List the TSAs that are visible on the network. Prompt the
*       user to select a TSA then attempt to establish a connection
*       with that TSA.
*       After connecting to a TSA enter an infinite loop in which user
*       commands are received and processed.
*
* SMS-SPECIFIC ROUTINES CALLED: 
*                               EstablishConnection (Demain.C)
*                               ListTSs (Demain.C)
*                               ListScanTypeStrings (Demain.C)
*                               ListSelectionStrings (Demain.C)
*                               BuildSelectionList (Demain.C)
*                               DisplaySelectionList (Demain.C)
*                               FreeSelectionList (Demain.C)
*                               EraseFile (Demain.C)
*                               RenameFile (Demain.C)
*                               Backup (Demain.C)
*                               Restore (Demain.C)
*                               NWSMTSReleaseTargetService (TSAPI)
*                               NWSMReleaseTSA (TSAPI)
*
**********************************************************************/
int main(int argc, char *argv)
{
    char 	 sessionName[MAX_SCREEN_NAME_LENGTH];
    char 	*choice;
    char 	 strName[MAX_SCREEN_NAME_LENGTH];
#ifdef N_PLAT_NLM
	UINT32 messageCount, languageID;
#endif	
/*    moduleName = "DE"; */
    buildDateAndTime = __DATE__ " @ " __TIME__;
    moduleVersion = "v4.10";
	GetNameOfThread(strName);
	NameScreen(strName);
	deScreenHandle=GetTheCurrentScreen(); 
	
#ifdef N_PLAT_NLM

	SMImportUniAPIs(GetNLMHandle());
	if (NWCallsInit(NULL, NULL) != 0)
	{
		/* Display some unable to initialise message */
		goto Return;
	}

	ReturnMessageInformation((LONG)GetNLMHandle(), (BYTE ***)&NbkMessages,
                &messageCount, &languageID, NULL);
	if (NbkMessages == NULL)
    {
		  goto Return;
    }
#endif

#ifdef N_PLAT_UNIX
 signal(SIGINT, UnloadHandler);
#endif

	/********************************************************************
	* Enter a loop to start the required number of sessions.
	********************************************************************/
#ifdef N_PLAT_UNIX
	initscr();	
#endif
	while(TRUE)
    {
#if defined(N_PLAT_UNIX)
		endwin();
		initscr();
		ClearScreen();
		setvbuf(stdout, (char *)NULL, _IONBF , 0); 
		nocbreak();
#endif
		printf("\n Enter a Name for this Session :");
		gets(sessionName);
#ifdef N_PLAT_NLM
		BeginThreadGroup(StartSession,NULL,16384,sessionName);
		printf("  \t Started Session \"%s\" ...\n",sessionName);
#elif defined(N_PLAT_UNIX)
		printf("  \t Started Session \"%s\" ...\n",sessionName);
		cbreak();
		StartSession( sessionName);
		nocbreak();
		ClearScreen();
#endif
		printf("\n\n Do You Wish to start Another Session?(y/n):");
#ifdef N_PLAT_UNIX
		noecho();
		cbreak();
#endif
		while ((choice = strchr("yn", getch())) == NULL) ConsoleSpeakerToBeep();	
#ifdef N_PLAT_UNIX
		echo();
#endif

		if(*choice == 'n')
		{
			printf("\n");
#ifdef N_PLAT_UNIX
			endwin();
#endif
			break;
		}

    }
Return:
    DestroyTheScreen(deScreenHandle);
    NO_MORE_SESSIONS = TRUE;
	if(TerminateNLM)
	{
		ClearModuleDontUnloadFlag(GetNLMID());
		ExitTheThread(EXIT_NLM, 0);
	}
	else
		ExitTheThread(TSR_THREAD, 0);

#ifdef N_PLAT_NLM
	SMUnImportUniAPIs(GetNLMHandle());
	NWCallsTerm(NULL );
#endif

	return(0);
}


/*****************************************************************************
* NAME: RenameRestore
*
* WHAT: Provides for supplying source and destination paths to rename restore a backup session.
*
* SMS-SPECIFIC ROUTINES CALLED:  GetNameSpace(Deutil.c)
*							    NWSMTSFixDataSetName(TSAPI)
*
*****************************************************************************/
CCODE RenameRestore(SESSION_INFORMATION *sessionInfo)
{
	BUFFER 	 source[MAX_SELECTION_ITEM_LENGTH];
	BUFFER 	 dest[MAX_SELECTION_ITEM_LENGTH];
    CCODE 	 ccode = 0;
    UINT32   nameSpaceType;
    char 	 strName[MAX_SCREEN_NAME_LENGTH];	
	int 	 term = 0;
	char 	*choice;
	RESTORE_CONTEXT *rctx = sessionInfo->rctx;

	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	FreeRenameList(rctx->renameList);

	/* Loop till the user has no more name pairs for rename */
    for(;;)
    {
		if(term != 0)
		{
			printf("\n\nDo you wish to enter another rename restore list?(y/n):");
			while ((choice = strchr("yn", getch())) == NULL) ;
			if(*choice == 'n')
			{
				goto Return;
			}
		}
		term ++;

		/* Get the name space in which the names are going to be entered */
		nameSpaceType = GetNameSpace(sessionInfo);
        if (nameSpaceType == 100 )
			goto Return;

		/* Prompt for the source and destination names */
        printf("\n\nEnter Source Path: "),
		gets(source);
		printf("\nEnter Destination Path: "), 
		gets(dest);
		ccode = AddToRenameList(sessionInfo, source, dest, nameSpaceType);
		if(ccode)
			goto Return;
		ClearScreen();
		NameScreen(strName);
    }
	
Return:
   	return(ccode);
}


/*****************************************************************************
* NAME: DisplayRenameList
*
* WHAT: Display the list used for Rename Restore, this list is created by the RenameRestore function.
*
* SMS-SPECIFIC ROUTINES CALLED:  
*
*****************************************************************************/
void DisplayRenameList(SinglePair *head)
{  
	char 		 strName[MAX_SCREEN_NAME_LENGTH]; 
	SinglePair 	*traverselist = NULL;

	GetNameOfThread(strName);
	if(head == NULL)
		goto Return;
	
	ClearScreen();
	NameScreen(strName);
	
	status("\n\rRename Restore List:\n", NULL);

	/* Loop till we have displayed all name pairs */
	for(traverselist = head; traverselist  != NULL; traverselist = traverselist->next)
	{
		status("\n\rSource Path : \"%s\"\n\r", traverselist->source);
		status("Destination Path : \"%s\"\n\r", traverselist->destination);
		status("Name Space Type: %d\n\r", traverselist->nameSpace);
		status("\n\r", NULL);
	}
	
	puts("");
	DEPause();
	
Return: 
	return;
}

/*********************************************************************
* NAME: DisplayList
*
* WHAT: Display the names (strings) in a linked list that is passed
*	as a parameter to this function. Specifically a list structure
*	of type NWSM_NAME_LIST is displayed by this function.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE DisplayList (NWSM_NAME_LIST *list)
{
	CCODE ccode = 0;
	NWSM_NAME_LIST *ptr, *next;
	int count = 0;
	
	if (!list)
		goto Return;
    ptr = list;
	while (ptr)
	{
		next = ptr->next;
		status("\n\r%s", ptr->name);
		//printf ("\n");
		ptr = next;
		count++;
		if ( !(count % 20))
		{
			DEPause ();
		}
	}
	status("\n\r", NULL);
	Return:
	return (ccode);
}


/*********************************************************************
* NAME: GetCommand
*
* WHAT: Prompt the user to enter a command, then get the command
*	and return it to the caller. The valid commands are: 
*		Create Selection List
*		Display Selection List
*		Free Selection List
*		Erase File
*		Rename File
*		List TSA strings
*		Backup
*		Restore
*		Quit
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
char GetCommand (UINT8 toLog)
{
	char command;
	int x, y;
	char strName[MAX_SCREEN_NAME_LENGTH];

#ifdef N_PLAT_UNIX
Repaint:
#endif
	/* Work around for Linux, clear the screen and repaint to avoid cluttering of the screen with invalid characters*/
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
    printf("\n");  
	printf(	"L: %sog Session Output\n"
			"C: Create SelectionList\n"
			"D: Display SelectionList\n"
			"F: Free SelectionList\n"
			"E: Delete File\n"
			"N: Rename File\n"
			"B: Backup\n"
			"R: Restore\n\n"
			"Q: Quit\n" 
			"\nSelect option: ",toLog?"L":"Do not l");
	CursorPosition(&x,&y);
	for(;;)
	{
		SetCursor(x,y);
		printf(" ");
		SetCursor(x,y);
		GetCharacter(&command);
		command = toupper(command);
		if (strchr("CDFENLVBRQ", command))
			goto Return;
		PutCharacter('\a');	
#ifdef N_PLAT_UNIX
		goto Repaint;
#endif
	}
Return:
	return (command);
}

/*********************************************************************
* NAME: GetBCommand
*
* WHAT: Prompt the user to select one of the following Backup/Scan options:
*
*	Backup Resource
*	Backup Subdirectory
*	Scan Resource
*	Scan Subdirectory
*	Save Backup Data to File (Toggle on/off)
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
char GetBCommand(int backup, char **prompt, int *resource)
{
	char command,test,option;
	int x, y;
	char strName[MAX_SCREEN_NAME_LENGTH];

#ifdef N_PLAT_UNIX
Repaint:
#endif
	/* Work around for Linux, clear the screen and repaint to avoid cluttering of the screen with invalid character*/
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	*prompt = "resource";
	*resource = TRUE;
	printf("\n");
	printf(	"T:  %save backup data\n"
			"BR: Backup Resource\n"
			"BS: Backup Subdirectories\n"
			"SR: Scan Resource\n"
			"SS: Scan Subdirectories\n\n"
			"Q:  Return to previous menu\n"
			"\nSelect option: ", backup ? "Do not s" : "S");
	CursorPosition(&x, &y);
	for(;;)
	{
		SetCursor(x, y);
		printf("  ");
		SetCursor(x, y);
		GetCharacter(&command);
		switch(command = toupper(command))
		{
		case 'Q':
		case 'T':
			goto Return;

		case 'S':
		case 'B':
#ifdef N_PLAT_NLM
			PutCharacter(command);
#endif
			GetCharacter(&test);
			switch(test = toupper(test))
			{
			case 'S':
				*prompt = "subdirectory";
				*resource = FALSE;

			case 'R':
				goto Return;
			}

		default:
			PutCharacter('\a');
			break;
		}
#ifdef N_PLAT_UNIX
		goto Repaint;
#endif
	}

Return:
	while(IsKeyStrokeAvailable()) 
		GetCharacter(&option);
	ClearScreen();
	NameScreen(strName);
	return (command);
}

/*********************************************************************
* NAME: GetRCommand
*
* WHAT: Prompt the user to select the following "Restore" options:
*	Restore Session
*	Rename Restore
*	Display Rename Restore Path
*	Free Rename Restore Path
*	Overwrite/Don't Overwrite Parent
*	Overwrite/Don't Overwrite Child
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
char GetRCommand(UINT8 overwriteParent,  UINT8 overwriteChild)
{
	char command,option;
	int x, y;
	char strName[MAX_SCREEN_NAME_LENGTH];

#ifdef N_PLAT_UNIX
Repaint:	
#endif
	/* Work around for Linux, clear the screen and repaint to avoid cluttering of the screen with invalid characters*/
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
    printf("\n");   
	printf( "R: Restore Session\n"
			"N: Rename Restore\n"
			"D: Display Rename Restore Path\n"
			"F: Free Rename Restore Path\n"
			"P: %sverwrite Parent\n"
			"C: %sverwrite Child\n\n"
			"Q: Return to previous menu\n"
			"\nSelect option: ", overwriteParent ? "O" : "Do not o", overwriteChild ? "O" : "Do not o");
	CursorPosition(&x,&y);
	for(;;)
	{
		SetCursor(x,y);
		printf("  ");
		SetCursor(x,y);
		GetCharacter(&command);
		command=toupper(command);
		if (strchr("RNDFPCQ", command))
			goto Return;
		PutCharacter('\a');
#ifdef N_PLAT_UNIX
		goto Repaint;
#endif
	}

Return:
	while(IsKeyStrokeAvailable()) 
		GetCharacter(&option);
	ClearScreen();
	NameScreen(strName);
	return (command);
}


/*********************************************************************
* NAME: GetNameSpace
*
* WHAT: Prompt the user to select the Data Set Name Space Type. Verify
*	that the selected type is valid. The name space type is used
*	in scanning data sets on a Target Service.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
UINT32 GetNameSpace(SESSION_INFORMATION *sessionInfo)
{
	char option;
	int nameSpaceCount;
	int realIndex, subIndex;
	int x, y;
	UINT32 nameSpaceType = NOT_ASSIGNED;
	UINT32 sequence = 0;
	NWSM_NAME_LIST *supportedNameSpaces = NULL;
	NWSM_NAME_LIST *saveFirstNameSpace = NULL;
	CCODE ccode = 0;
	BUFFER resourceName[RESOURCE_NAME_LENGTH];
	char strName[MAX_SCREEN_NAME_LENGTH];
	
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	ccode = NWSMTSScanTargetServiceResource(sessionInfo->connection, &sequence, resourceName);

	/* we will get all resources here (sequence = 0) so that supported name spaces      
	 * will list all supported name spaces available on the system. */
	if (ccode != 0)
	{
		PrintError(ccode, "NWSMTSScanTargetServiceResource", __LINE__,sessionInfo);
		LogSession(sessionInfo->logFile,"NWSMTSScanTargetServiceResource error : 0x%x\n", ccode);
		nameSpaceType = 100;
		goto NameSpaceEnd;
	}
	ccode = NWSMTSListSupportedNameSpaces(sessionInfo->connection, resourceName, &supportedNameSpaces);
	if (ccode == NWSMTS_OUT_OF_MEMORY)
	{
		PrintError(ccode, "NWSMTSListSupportedNameSpaces", __LINE__,sessionInfo);
		LogSession(sessionInfo->logFile,"\t\aOut of Memory\n");
		DEPause();
		nameSpaceType = 100;
		goto NameSpaceEnd;
	}
	saveFirstNameSpace = supportedNameSpaces;

#ifdef N_PLAT_UNIX
Repaint:
#endif
	/* Work around for Linux, clear the screen and repaint to avoid cluttering of the screen with invalid characters*/
#ifdef N_PLAT_UNIX
	ClearScreen();
	supportedNameSpaces = saveFirstNameSpace;
#endif
	/* This routine is based on the idea that at least one name space will be returned from the TSA. */
	nameSpaceCount = 1;
	printf("\n\nName Space Type:\n");
	LogSession(sessionInfo->logFile,"\n\nName Space Type:\n");
	while (supportedNameSpaces != NULL)
	{
		printf("\t%d: %s\n",nameSpaceCount, supportedNameSpaces->name+4);
		LogSession(sessionInfo->logFile,"\t%d: %s\n",nameSpaceCount, supportedNameSpaces->name+4);
		supportedNameSpaces = supportedNameSpaces->next;
		nameSpaceCount++;
	}
	printf("\t%d: Quit\n", nameSpaceCount);
	printf("\n\nEnter option: ");
	CursorPosition(&x,&y);
	//cbreak();
	GetCharacter(&option);
	realIndex = option - '0';
	if (realIndex == nameSpaceCount)
	{
		nameSpaceType = 100;
		goto NameSpaceEnd;
	}
	while ((realIndex < 1) || (realIndex > nameSpaceCount))
	{
		printf("\a\a");
		SetCursor(x,y);
		GetCharacter(&option);
		realIndex = option - '0';
		if (realIndex == nameSpaceCount)
		{
			nameSpaceType = 100;
			goto NameSpaceEnd;
		}
#ifdef N_PLAT_UNIX
		goto Repaint;
#endif

	}
	supportedNameSpaces = saveFirstNameSpace;
	for (subIndex=1; subIndex < realIndex; subIndex++)
	{
		supportedNameSpaces = supportedNameSpaces->next;
	}
	nameSpaceType = *(UINT32 *)supportedNameSpaces->name;

/*** nameSpaceType is what we're passing back  ***/

NameSpaceEnd:
	NWSMFreeNameList(&saveFirstNameSpace);
	saveFirstNameSpace = NULL;
	return (nameSpaceType);

}

/*********************************************************************
* NAME: GetSelectionType
*
* WHAT: Prompt the user to enter the type of selection desired for the
*	data set scan. 
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
UINT32 GetSelectionType(SESSION_INFORMATION *sessionInfo)
{
	char *strings[62]; /*This builds an array of pointers to string1 and string2 */
	char string1[SELECTION_TYPE_STRING_LENGTH];
	char string2[SELECTION_TYPE_STRING_LENGTH];
	char option;
	int x, y;
	int index = 1, endindex = 1;
	UINT8 typenum = 1;
	UINT32 selectionType = NOT_ASSIGNED;
	UINT32 value = 2;
	UINT32 retValue[62];
	CCODE ccode = 0;
	char strName[MAX_SCREEN_NAME_LENGTH];
	
	GetNameOfThread(strName);
	ClearScreen();
	NameScreen(strName);
	while (ccode == 0)
	{
		ccode = NWSMTSGetTargetSelectionTypeStr(sessionInfo->connection, typenum, string1, string2);
		if (ccode == NWSMTS_INVALID_CONNECTION_HANDL)
		{
			PrintError(ccode, "NWSMTSGetTargetSelectionTypeStr", __LINE__,sessionInfo);
			LogSession(sessionInfo->logFile,"\nINVALID CONNECTION HANDLE");
			break;
		}
		if (ccode == NWSMTS_INVALID_SELECTION_TYPE)
		{
			PrintError(ccode, "NWSMTSGetTargetSelectionTypeStr", __LINE__,sessionInfo);
			LogSession(sessionInfo->logFile,"\nINVALID SELECTION TYPE");
			break;
		}
		if (!ccode)
		{
			if (*string1 != 0)
			{
				strings[index] = malloc(strlen(string1)+1);
				strcpy(strings[index], string1);
				retValue[index] = value;
				index++;
			}

			if (*string2 != 0)
			{
				strings[index] = malloc(strlen(string2)+1);
				strcpy(strings[index], string2);
				retValue[index] = (value|1);
				index++;
			}
			typenum++;		
			value <<= 1;
		}
	}
	endindex = index;
	
#ifdef N_PLAT_UNIX
Repaint:
	ClearScreen();
#endif

	printf("\n\nSelection Type:\n");
	LogSession(sessionInfo->logFile,"\n\nSelection Type:\n");
	for (index = 1; index < endindex; index++)
	{
		if (strings[index] != NULL)
		{
			printf("\t%d: %s\n",index, strings[index]);
			LogSession(sessionInfo->logFile,"\t%d: %s\n",index, strings[index]);
		}
	} 
	printf("\t%d: Return to previous menu\n", endindex);
	printf("\n\nEnter Selection Type: ");
	CursorPosition(&x,&y);
	while (selectionType == NOT_ASSIGNED)
	{
		SetCursor(x,y);
		printf("  ");
		SetCursor(x,y);
		GetCharacter(&option);
		index = option - '0';
		if ((index < 1) || (index > endindex))
		{
			printf("\a\a");
			selectionType = NOT_ASSIGNED;
#ifdef N_PLAT_UNIX
			goto Repaint;
#elif defined (N_PLAT_NLM)
			continue;
#endif
		}
		if (index == endindex)
		{
#ifdef N_PLAT_NLM
			PutCharacter(option);
#endif
			selectionType = 100;
			break;
		}
#ifdef N_PLAT_NLM
		PutCharacter(option);
#endif
		selectionType=retValue[index];
	}

	for (index=1; index < endindex; index++)
	{
		if (strings[index] != NULL)
			free(strings[index]);
	}
 	return (selectionType);
}

/*********************************************************************
* NAME: SetNameSpaceType
*
* WHAT: Convert a name space type code (UINT32) into a string describing
*	the name space type passed as a parameter to this function.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void SetNameSpaceType(UINT32 selectedNameSpaceType, char *string, SESSION_INFORMATION *sessionInfo)
{
	UINT32 sequence = 0;
	NWSM_NAME_LIST *supportedNameSpaces = NULL; 
	NWSM_NAME_LIST *saveFirstNameSpace = NULL;
	CCODE ccode = 0;
	BUFFER resourceName[RESOURCE_NAME_LENGTH];
	
	ccode = NWSMTSScanTargetServiceResource(sessionInfo->connection, &sequence, resourceName);
	if (ccode != 0)
	{
		PrintError(ccode, "NWSMTSScanTargetServiceResource", __LINE__,sessionInfo);
		LogSession(sessionInfo->logFile,"NWSMTSScanTargetServiceResource error : 0x%x\n", ccode);
		strcpy(string, "Bad nameSpaceType");
		goto GetStringEnd;
	}
	ccode = NWSMTSListSupportedNameSpaces(sessionInfo->connection, resourceName, &supportedNameSpaces);
	if (ccode )
	{
		PrintError(ccode, "NWSMTSListSupportedNameSpaces", __LINE__,sessionInfo);
		LogSession(sessionInfo->logFile,"\t\aOut of Memory\n");
		DEPause();
		strcpy(string, "Bad nameSpaceType");
		goto GetStringEnd;
	}
	saveFirstNameSpace = supportedNameSpaces;

	/*This routine is based on the idea that at least one name space will be returned from the TSA.*/
	while (*supportedNameSpaces->name != selectedNameSpaceType)
	{
		supportedNameSpaces = supportedNameSpaces->next;
		if (supportedNameSpaces == NULL)
			break;
	}
	if (*supportedNameSpaces->name != selectedNameSpaceType)
	{
		strcpy(string, "Bad nameSpaceType");
		goto GetStringEnd;
	}
	strcpy(string, (char *)supportedNameSpaces->name+4);

	/* string is what we're passing back */
	GetStringEnd:
	NWSMFreeNameList(&saveFirstNameSpace);
	saveFirstNameSpace = NULL;
	return;
}

/*********************************************************************
* NAME: SetSelectionType
*
* WHAT: Convert a selection type code (UINT32) into a string describing
*	the selection type passed as a parameter to this function.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void SetSelectionType(UINT32 selectionType, char *string, SESSION_INFORMATION *sessionInfo)
{
	int index=1;
	UINT8 typenum = 1;
	char string1[SELECTION_TYPE_STRING_LENGTH], string2[SELECTION_TYPE_STRING_LENGTH];
	UINT32 value = 2;
	CCODE ccode = 0;
	
	for (index = 1, value=2; index <= 32; index++, value <<= 1)
	{
		ccode = NWSMTSGetTargetSelectionTypeStr(sessionInfo->connection, typenum, string1, string2);
		if (ccode == NWSMTS_INVALID_CONNECTION_HANDL)
		{
			PrintError(ccode, "NWSMTSGetTargetSelectionTypeStr", __LINE__,sessionInfo);
			LogSession(sessionInfo->logFile,"\nINVALID CONNECTION HANDLE");
			break;
		}
		if (ccode == NWSMTS_INVALID_SELECTION_TYPE)
		{
			PrintError(ccode, "NWSMTSGetTargetSelectionTypeStr", __LINE__,sessionInfo);
			LogSession(sessionInfo->logFile,"\nINVALID SELECTION TYPE");
			continue;
		}
		if (value == selectionType)
		{
			strcpy(string, (char *)string1);
			break;
		}
		if ((value|1) == selectionType)
		{
			strcpy(string, (char *)string2);
			break;
		}
		typenum++;
	}
	return;
}
#ifdef N_PLAT_UNIX
void UnloadHandler(int sig)
{
     if(sig != SIGINT && sig != SIGTERM)
        return ;
	 if(curSession)
	  CloseBackupSession(curSession);
	endwin();
     exit(0);
}
#endif

