/*
===============================================================================
Novell Software Developer Kit Sample Code License

Copyright (C) 2005, Novell, Inc.  All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
  *  Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  *  Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.
  *  Neither the name of Novell, Inc. nor the names of its contributors
     may be used to endorse or promote products derived from this
     software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT
SHALL NOVELL, INC., THE COPYRIGHT OWNER, OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

NAME OF FILE:
	deutil.c

PURPOSE/COMMENTS:
	Demo engine utility functions

NDK COMPONENT NAME AND VERSION:
	SMS Developer Components

LAST MODIFIED DATE: 
	18 October 2005
===============================================================================
*/


 /***For NLM Conversion***/
 /****************************************************************************
 *	NOTE: Comments are included with this code that describe how to
 *		convert this demonstration engine to an NLM. To make an NLM 
 *		well behaved, control should be passed to other processes on
 *		the server at least every 150 milliseconds (386 @ 25Mhz).
 *		To accomodate this, calls to the ThreadSwitch function (CLIB) 
 *		should be appropriately scattered through the code. 
 *		ThreadSwitches should be considered for any sections of
 *		code, such as loops, which could execute for more 
 *		than 150 milliseconds. CLIB routines that block should already
 *		pass control to other threads so you won't have to worry about
 *		these.
 *
 *		To find the changes need for converting this demonstration 
 *		to an NLM, search for the string "NLM" in this code.
 ****************************************************************************/
/* 
 * Headers and Defines
 */
 
#include <smstypes.h>
#include <smsutapi.h>
#include <smuni.h>
#include <sidffids.inc>
#include <errno.h>

#ifdef N_PLAT_NLM
#include <nwconio.h>
#include <unicode.h>
#include <nwthread.h>
#include <nwadv.h>
#elif defined(N_PLAT_UNIX)
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <strings.h>
#include <ncurses.h>
#include<sys/time.h>
#include<unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#define puts  printw
#define gets getstr
#define printf printw
#define ConsolePrintf 	printf
#define strcmpi 		strcasecmp
#define EXIT_NLM 		0
#define TSR_THREAD 		0
#define GetNLMID()
#endif

#include "deoslib.h"
#include <sms.h>
#include "de.h"
#include<nbkmsg.mlh>
 UINT32  	MoveDataSet;
UINT32 SetExFileData= 0;
UINT32 noIDs=0;
UINT32 TotalDataWritten= 0;

#ifdef N_PLAT_GNU
/* The passwd structure.  */
struct passwd
{
  char *pw_name;        /* Username.  */
  char *pw_passwd;      /* Password.  */
  uid_t pw_uid;       /* User ID.  */
  gid_t pw_gid;       /* Group ID.  */
  char *pw_gecos;       /* Real name.  */
  char *pw_dir;         /* Home directory.  */
  char *pw_shell;       /* Shell program.  */
};
#endif

int 		 safeUnloadFlag;
static int ungetEOFFlag = FALSE;
FILE *sessionInfoFile = NULL;
clock_t begin_time;
int logLevel;
char moduleName[256] = {0};
BUFFER sdiSidfIdString[]       = {'S','I','D','F'};
BUFFER sdiSidfVersion101[]     = {1,0,1,0};
#ifdef N_PLAT_UNIX
extern char *NbkMessages[];
#elif defined N_PLAT_NLM
extern char **NbkMessages;
#endif
extern NWSM_LIST_PTR *inFiles;
time_t timep, scanTimeConvert;
#ifdef N_PLAT_UNIX
struct timeval tv;
struct timezone tz;
#endif
int	safeFlag=0;

UINT32 MapToUtf8NameSpace(UINT32 nameSpaceType);

/*********************************************************************
* NAME: PrintError
*
* WHAT: Convert Error Return (CCODE) to a string and display it.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMConvertError (SIDR)
*
**********************************************************************/
void PrintError(CCODE ccode, char *routine, UINT16 line, SESSION_INFORMATION *sessionInfo)
{
	BUFFER buffer[ERROR_BUFFER_LENGTH] = {0};
	NWSMConvertError(sessionInfo->connection, ccode, buffer);

	if( logLevel >= NBK_DEBUG_MSG)
	{
		if (line)
			fprintf(stderr, "\n\n%s:  (line = %u)\n   \"%s\"\n\n", routine, line, buffer);
		else 
			fprintf(stderr, "\n\n%s:\n   \"%s\"\n\n", routine, buffer);
	}
	else
	{
	/* If the log level is less than debug_msg, then do not print function name, line no etc*/
		if( buffer[0] != 0)
			fprintf(stderr, "\n%s: %s\n", moduleName, buffer);
	}

	return;
}

/*********************************************************************
* NAME: PrintMsg
*
* WHAT: Format message and display based on the Loglevel
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
void PrintMsg(int msgType, char *format, ...)
{
    va_list argList;
    unsigned char buffer[2048] = {0};

    
	if( logLevel && (msgType >  logLevel) && (msgType != NBK_GENERAL_MESSAGES))
		return;
	
	/*  Print error message */
    va_start(argList, format);
#ifdef N_PLAT_NLM
	/* Using Message enabled vsprintf */
if(msgType != NBK_ERR_MSG)
    NWvsprintf (buffer, format, argList) ;
#elif N_PLAT_UNIX
	vsnprintf (buffer,2047, format, argList) ;
#endif
    va_end(argList);

	 if(moduleName && ((msgType == NBK_ERR_MSG) || (msgType == NBK_GENERAL_MESSAGES)))
	 {
		if(msgType == NBK_GENERAL_MESSAGES)
			fprintf(stderr, "%s: %s\n", moduleName, buffer);
		else
			fprintf(stderr, "\n%s: %s\n", moduleName, buffer);
	 }
	else 
		fprintf(stderr,"%s", buffer);

}

/*********************************************************************
* NAME: SetLogLevel
*
* WHAT: set loglevel
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void SetLogLevel(int mode)
{
	logLevel = mode;
}

/*********************************************************************
* NAME: SetModuleName
*
* WHAT: set Module name
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
void SetModuleName(char *name)
{
	if(name)
		strcpy(moduleName, name);
}

/*********************************************************************
* NAME: ResetStats
*
* WHAT: Resets statistics to zeros
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
void ResetStats(SESSION_INFORMATION *sessionInfo)
{
	sessionInfo->stats.fileCount = 0;
	sessionInfo->stats.dirCount = 0;
	sessionInfo->stats.resCount = 0;
	sessionInfo->stats.totalBytesRead = 0;
	sessionInfo->stats.totalTime = 0;

}

/*********************************************************************
* NAME: ConvertLocToUTF8
*
* WHAT:  Converts MBCS string to UTF8 string
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
CCODE ConvertLocToUTF8( unsigned char *path, size_t *destLen, utf8_t **utf8Path)
{
	CCODE ccode;

	/* If the target service connected to supporte UTF-8 then convert user input to UTF-8 */
	ccode = SMloc2utf8(NULL, destLen, path, strlen(path), NULL) ;
	if(ccode)
	{
		PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_UTF8_CONV_ERR), ccode);
		goto Return;
	}
	(*destLen) ++;
	*utf8Path = (utf8_t *)malloc(*destLen);
	if(*utf8Path == NULL)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		ccode = NBK_OUT_OF_MEMROY;
		goto Return;
	}
	
	ccode = SMloc2utf8(*utf8Path, destLen, path, strlen(path), NULL) ;
	if(ccode)
	{
		PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_UTF8_CONV_ERR), ccode);
		goto Return;
	}

	Return:
	if(ccode && *utf8Path)
	{
		free(*utf8Path);
		*utf8Path = NULL;
	}
	return ccode;
}

/*********************************************************************
* NAME: ConvertUTF8ToLoc
*
* WHAT:  Converts UTF8 strings to MBCS string
*
* SMS-SPECIFIC ROUTINES CALLED: 
*
**********************************************************************/
CCODE ConvertUTF8ToLoc( utf8_t *utf8path, size_t *destLen,char **locpath)
{
	CCODE ccode;

	ccode = SMutf82loc(NULL, destLen, (utf8_t*)utf8path, strlen(utf8path), NULL) ;
	if(ccode)
		goto Return;
	
	(*destLen) ++;
	*locpath = (char *)malloc(*destLen);
	if(*locpath == NULL)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		ccode = NBK_OUT_OF_MEMROY;
		goto Return;
	}
	
	ccode = SMutf82loc(*locpath, destLen, (utf8_t *)utf8path, strlen(utf8path), NULL) ;
	if(ccode)
		goto Return;

	Return:
	if(ccode && *locpath)
	{
		free(*locpath);
		*locpath = NULL;
	}
	return ccode;
}

/*********************************************************************************************
* NAME: InsertToSelectionList
*
* WHAT: Insert a name into SelectionList
*
* SMS-SPECIFIC ROUTINES CALLED: 	NWSMPutFirstName (SMS UTILITY API)
*                               			NWSMPutNextName (SMS UTILITY API)
*                               			NWSMCloseName (SMS UTILITY API)
*								NWSMTSGetNameSpaceTypeInfo
*								NWSMFreeString
*
**********************************************************************************************/
CCODE InsertToSelectionList(SESSION_INFORMATION * sessionInfo, char *buffer, UINT32 nameSpaceType, UINT32 selectionType, SMS_HANDLE *nameHandle)
{
	CCODE ccode;
	STRING_BUFFER 	*firstSep = NULL, *nextSep = NULL;
    size_t			 destLen;
	utf8_t 			*dest = NULL;
	NWBOOLEAN 		 reverseOrder = FALSE;
	

	/* Get the selected namespaces information using NWSMTSGetNameSpaceTypeInfo */
	if(sessionInfo->IsUTF8Supp)
	{
		nameSpaceType = MapToUtf8NameSpace(nameSpaceType);	
		ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, nameSpaceType, &reverseOrder, &firstSep, &nextSep);
	}
	else
		ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, nameSpaceType, &reverseOrder, &firstSep, &nextSep);
	
	if (ccode)
	{
		PrintError(ccode, "NWSMTSGetNameSpaceTypeInfo", __LINE__,sessionInfo);
		goto Return;
	}
	if(sessionInfo->IsUTF8Supp)
	{
		/* If the target service connected to supporte UTF-8 then convert user input to UTF-8 */
		ccode = ConvertLocToUTF8(buffer, &destLen, &dest);
		if(ccode)
		{
			goto Return;
		}
	}
	else
		/* Type cast to avoid warnings, as this is used as a void * later */
		dest = (utf8_t *)buffer;

	/* Add the selsection item into the selection list */
	if (*nameHandle==0)
	{
		ccode = NWSMPutFirstName((void *)&(sessionInfo->selectionList), nameSpaceType,
				selectionType, reverseOrder, firstSep->string, nextSep->string, dest, nameHandle);
	}
	else
	{
		ccode = NWSMPutNextName((void *)&(sessionInfo->selectionList), nameHandle, nameSpaceType,
		selectionType, reverseOrder, firstSep->string, nextSep->string, dest);
	}
	
	if (ccode)
	{
		PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_PUTNAME_ERR));
		goto Return;
	}
	Return:
	if (dest && ((unsigned long)dest != (unsigned long)buffer))
	{
		free(dest);
		dest = NULL;
	} 
	NWSMFreeString(&firstSep);
	NWSMFreeString(&nextSep);
	return ccode;
}

/*********************************************************************************************
* NAME: AddToRenameList
*
* WHAT: Insert a name into rename List
*
* SMS-SPECIFIC ROUTINES CALLED: 	NWSMTSFixDataSetName (TS API)
*								NWSMFreeString
*                               
**********************************************************************************************/
CCODE AddToRenameList(SESSION_INFORMATION *sessionInfo, char *source, char *dest, UINT32 nameSpaceType)
{
	CCODE ccode;
	SinglePair 		*newpair = NULL;
	STRING_BUFFER 	*newsource = NULL, *newdest = NULL;
	RESTORE_CONTEXT *rctx = sessionInfo->rctx;

	/* Fix the user entered names for correct placement of separators and names */
	ccode = NWSMTSFixDataSetName(sessionInfo->connection, (STRING)source, nameSpaceType, TRUE, FALSE, &newsource);
	if (ccode)
    {
		PrintError(ccode, "NWSMTSFixDataSetName", __LINE__, sessionInfo);
		goto Return;
    }
	ccode = NWSMTSFixDataSetName(sessionInfo->connection, (STRING)dest, nameSpaceType, TRUE, FALSE, &newdest);
	if (ccode)
    {
		PrintError(ccode, "NWSMTSFixDataSetName", __LINE__, sessionInfo);
		goto Return;
	}

	/* Allocate space and copy the names into the rename list */
	if( (newpair = (SinglePair *)malloc(sizeof(SinglePair))) == NULL) 
    {
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		goto Return;
    }
	if( (newpair->source = (char *)malloc(strlen(newsource->string) + 1)) == NULL) 
    {
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
        goto Return;
    }
	if( (newpair->destination = (char *)malloc(strlen(newdest->string) + 1)) == NULL) 
    {
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
        goto Return;
    }
	newpair->nameSpace = nameSpaceType;
	newpair->next = NULL;
	memcpy(newpair->source, newsource->string, strlen(newsource->string) + 1);
	memcpy(newpair->destination, newdest->string, strlen(newdest->string) + 1);
	if( rctx->renameList == NULL)
	{
		rctx->renameList = newpair;
	}
	else
	{
		Insert(rctx->renameList , newpair);
	}
	Return:
	NWSMFreeString(&newsource);
	NWSMFreeString(&newdest);  
		return ccode;
}

/*********************************************************************************************
* NAME: GetSupportedNameSpace
*
* WHAT: Check supported namespaces. Select LONG namespace, if supported. Select NFSNameSpace, otherwise.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMTSScanTargetServiceResource
*								NWSMTSListSupportedNameSpaces	
*								NWSMFreeNameList
*
**********************************************************************************************/
UINT32 GetSupportedNameSpace(SESSION_INFORMATION *sessionInfo)
{
	UINT32 nameSpaceType = NOT_ASSIGNED;
	UINT32 sequence = 0;
	NWSM_NAME_LIST *supportedNameSpaces = NULL;
	NWSM_NAME_LIST *saveFirstNameSpace = NULL;
	CCODE ccode = 0;
	BUFFER resourceName[RESOURCE_NAME_LENGTH];
	
	ccode = NWSMTSScanTargetServiceResource(sessionInfo->connection, &sequence, resourceName);

	/* we will get all resources here (sequence = 0) so that supported name spaces      
	 * will list all supported name spaces available on the system. */
	if (ccode != 0)
	{
		PrintError(ccode, "NWSMTSScanTargetServiceResource",__LINE__,sessionInfo);
		goto NameSpaceEnd;
	}
	ccode = NWSMTSListSupportedNameSpaces(sessionInfo->connection, resourceName, &supportedNameSpaces);
	if (ccode != 0)
	{
		PrintError(ccode, "NWSMTSListSupportedNameSpaces",__LINE__,sessionInfo);
		goto NameSpaceEnd;
	}
	saveFirstNameSpace = supportedNameSpaces;
	/* This routine is based on the idea that at least one name space will be returned from the TSA. */
	while (supportedNameSpaces != NULL)
	{
		nameSpaceType = *(UINT32 *)supportedNameSpaces->name;
		if( nameSpaceType == OS2NameSpace )
		{	
			goto NameSpaceEnd;				
		}
		else if(nameSpaceType == NFSNameSpace)
		{	
			goto NameSpaceEnd;			
		}
		supportedNameSpaces = supportedNameSpaces->next;
	}
	
/*** nameSpaceType is what we're passing back  ***/

NameSpaceEnd:
	NWSMFreeNameList(&saveFirstNameSpace);
	saveFirstNameSpace = NULL;
	return (nameSpaceType);

}

/*********************************************************************
* NAME: CloseService
*
* WHAT: Releases the Target Service
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSReleaseTargetService (TSAPI)
*
**********************************************************************/
void CloseService(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode;
	
	if (sessionInfo->connection)
		ccode = NWSMTSReleaseTargetService(&sessionInfo->connection);

	return;
}

/*********************************************************************
* NAME: CloseTarget
*
* WHAT: Releases the Target service and also the TSA connection.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMReleaseTSA (TSAPI)
*
**********************************************************************/
void CloseTarget(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode;
	
	CloseService(sessionInfo);
	if (sessionInfo->connection)
	   ccode = NWSMReleaseTSA(&sessionInfo->connection);
	sessionInfo->connection = 0;

	return;
}

/*********************************************************************
* NAME: OpenTarget
*
* WHAT: Opens the TSA connection with the target name.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMConnectToTSA (TSAPI)
*
**********************************************************************/
CCODE OpenTarget(SESSION_INFORMATION *sessionInfo, UINT16 numRetry)
{
	CCODE ccode;
	
	ccode = NWSMConnectToTSA(sessionInfo->tsaName, &sessionInfo->connection);
	if (ccode)
	{
		if(numRetry == 1)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_TSA_CONNECT_FAIL), sessionInfo->tsaName,ccode);
	        	sessionInfo->connection= 0;
		} 
		else
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OPEN_TARGET_ERR));
	}
	return ((ccode)? -1 : sessionInfo->connection);
}

/*********************************************************************
* NAME: OpenService
*
* WHAT: Opens the Target service conneciton using Username and Password
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSConnectToTargetService (TSAPI)
*								NWSMTSConnectToTargetServiceEx (TSAPI)
*
**********************************************************************/
CCODE OpenService(SESSION_INFORMATION *sessionInfo)
{
	CCODE		ccode;
	size_t		actLen;
	unsigned char	utf8pwd[MAX_BUF_LEN * MAX_UTF8_CHAR_SIZE];

	/* Functions that will be imported */
	CCODE  (*mPtrConnectToTargetServiceEx)(UINT32 *, STRING , STRING , void *, UINT32 ) = NULL;

	/* Connect to the chosen target service, with the entered user name and password */
#ifdef N_PLAT_NLM
	/* Try the NWSMTSConnectToTargetServiceEx function first */
	mPtrConnectToTargetServiceEx = ImportSymbol(GetNLMHandle(), "NWSMTSConnectToTargetServiceEx");
#elif N_PLAT_UNIX
	/* This symbol is not dynamically resolved as it is supported on all SMS versions on Linux */
	mPtrConnectToTargetServiceEx = NWSMTSConnectToTargetServiceEx;
#endif
	if(mPtrConnectToTargetServiceEx)
	{
		if ( strcmp(sessionInfo->CuserName, "CasaPrincipal") == 0)
		{
			/* NWSMTSConnectToTargetServiceEx exists. Convert the input to UTF-8 and attempt to connect */
			sessionInfo->CpassWord[*(UINT16 *)sessionInfo->CpassWord + sizeof(UINT16)] = 0;
			ccode = (*mPtrConnectToTargetServiceEx)(&sessionInfo->connection, sessionInfo->svcName, sessionInfo->CuserName, sessionInfo->CpassWord, NWSM_AUTH_CASA_TOKEN );
			if (ccode)
			{
				PrintError(ccode, "NWSMTSConnectToTargetServiceEx", __LINE__,sessionInfo);
				goto Return;
			}
		}
		else
		{
			/* NWSMTSConnectToTargetServiceEx exists. Convert the input to UTF-8 and attempt to connect */
			sessionInfo->CpassWord[*(UINT16 *)sessionInfo->CpassWord + sizeof(UINT16)] = 0;
	
			actLen = sizeof(utf8pwd);
			ccode = SMloc2utf8(utf8pwd, &actLen, &(sessionInfo->CpassWord[sizeof(UINT16)]), 
						*(UINT16 *)sessionInfo->CpassWord + sizeof(char), NULL);
			if (ccode)
			{
				/* Failed conversion, signal error and exit */
				PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_UTF8_CONV_ERR));
				PrintError(ccode, "SMloc2utf8", __LINE__,sessionInfo);
				goto Return;
			}
	
			*((UINT16 *)(sessionInfo->UPassWord)) = strlen(utf8pwd); 
			memcpy(&(sessionInfo->UPassWord[2]), utf8pwd, strlen(utf8pwd));

			
			/* Call NWSMTSConvertToTargetServiceEx with the UTF-8 encoded password */		
			ccode = (*mPtrConnectToTargetServiceEx)(&sessionInfo->connection, sessionInfo->svcName, 
						sessionInfo->CuserName, sessionInfo->UPassWord, NWSM_AUTH_UTF8_DATA );
	
		/* The function is not supported by the remote SDMR or target service, use the older API to authenticate */
			if(ccode == NWSMDR_UNSUPPORTED_FUNCTION || ccode == NWSMTS_UNSUPPORTED_FUNCTION)
			{
	#ifdef N_PLAT_NLM
				UnimportSymbol(GetNLMHandle(), "NWSMTSConnectToTargetServiceEx");
	#endif
				mPtrConnectToTargetServiceEx = NULL;
			}
			/* Function is supported but the UTF-8 password failed authentication */
			else if (ccode)
			{
				/* Possibly the user password may still be valid in RAW format.
				* Try with NWSM_AUTH_RAW_DATA */
				ccode = (*mPtrConnectToTargetServiceEx)(&sessionInfo->connection, sessionInfo->svcName, 
							sessionInfo->CuserName, sessionInfo->CpassWord, NWSM_AUTH_RAW_DATA );
				if (ccode)
				{
					PrintError(ccode, "NWSMTSConnectToTargetServiceEx", __LINE__,sessionInfo);
					goto Return;
				}
			}	
		}	

	}
	
	if (!mPtrConnectToTargetServiceEx)
	{
		/* Target service does not support NWSMTSConnectToTargetServiceEx, 
		 * use the older NWSMTSConnectToTargetService API */
		ccode = NWSMTSConnectToTargetService(&sessionInfo->connection, sessionInfo->svcName, 
					sessionInfo->CuserName, sessionInfo->CpassWord);
		if(ccode)
		{
			PrintError(ccode, "NWSMTSConnectToTargetService", __LINE__,sessionInfo);
			goto Return;
		}
	}

Return:
	
#ifdef N_PLAT_NLM
	if (mPtrConnectToTargetServiceEx)
		UnimportSymbol(GetNLMHandle(), "NWSMTSConnectToTargetServiceEx");
#endif
	
	return ccode;
}

/*********************************************************************
* NAME: IsRecoveryError
*
* WHAT: Checks for cluster failure
*
* SMS-SPECIFIC ROUTINES CALLED:

**********************************************************************/
int IsRecoveryError(CCODE ccode)
{
	if ((ccode == NWSMDR_WRITE_FAILURE ) || (ccode == NWSMDR_READ_FAILURE ) ||
		(ccode == NWSMDR_OPEN_FAILURE ) || (ccode == NWSMDR_TRANSPORT_FAILURE) ||
		(ccode == NWSMDR_DISCONNECTED ) ||   (ccode == NWSMTS_CLUSTER_TARGET_DOES_NOT_EXIST ))
		return TRUE;
	else
		return FALSE;
}

/*********************************************************************
* NAME: CheckUTFSupport
*
* WHAT: Check TSAFS for UTF8 support
*
* SMS-SPECIFIC ROUTINES CALLED: 
*								NWSMTSGetTargetServiceAPIVersion
*								NWSMTSGetSupportedNameTypes
*                                                               
**********************************************************************/
int CheckUTF8Support(SESSION_INFORMATION *sessionInfo)
{
	CCODE 		ccode;
	UINT32 		tsaMajorVersion=0;
	UINT32 		tsaMinorVersion=0;
    UINT32 		supportedTypes=0;
#ifdef N_PLAT_NLM
	UINT32 (*GetSupportedNameTypes)(UINT32, UINT32 *) = 0;
	UINT32 (*GetTargetServiceApiVersion)(UINT32, UINT32 *, UINT32*) = 0;
#endif

#ifdef N_PLAT_NLM
	GetTargetServiceApiVersion= ImportSymbol(GetNLMHandle(), "NWSMTSGetTargetServiceAPIVersion");
	if (GetTargetServiceApiVersion)
	{
		if ((ccode = GetTargetServiceApiVersion(sessionInfo->connection, &tsaMajorVersion, &tsaMinorVersion)) == 0)
		{
			if (tsaMajorVersion > 0x2)
			{
				GetSupportedNameTypes = ImportSymbol(GetNLMHandle(), "NWSMTSGetSupportedNameTypes");
				if (GetSupportedNameTypes)
				{
					if ((ccode = GetSupportedNameTypes(sessionInfo->connection, &supportedTypes)) == 0)
					{
						if (supportedTypes & NWSM_NAME_TYPE_UTF8)
						{
							sessionInfo->scanControl.returnNameSpaceType = NWSM_ALL_NAME_SPACES_UTF8;
							sessionInfo->IsUTF8Supp = TRUE;
						}
					}
					UnimportSymbol(GetNLMHandle(), "NWSMTSGetSupportedNameTypes");
				}
			}
		}
		UnimportSymbol(GetNLMHandle(), "NWSMTSGetTargetServiceAPIVersion");
	}
#elif defined(N_PLAT_UNIX)
	if ((ccode = NWSMTSGetTargetServiceAPIVersion(sessionInfo->connection, &tsaMajorVersion, &tsaMinorVersion)) == 0)
	{
		if(tsaMajorVersion > 0x2)
		{
			if ((ccode = NWSMTSGetSupportedNameTypes(sessionInfo->connection, &supportedTypes)) == 0)
			{
				if(supportedTypes & NWSM_NAME_TYPE_UTF8)
				{
					sessionInfo->scanControl.returnNameSpaceType = NWSM_ALL_NAME_SPACES_UTF8;
					sessionInfo->IsUTF8Supp = TRUE;
				}
			}
		}
	}
#endif

return 0;
}

/*********************************************************************
* NAME: Init_Cluster_Params
*
* WHAT: Initializes the Cluster Parameters if the selected Resource is a Cluster file system
*
* SMS-SPECIFIC ROUTINES CALLED: 
**********************************************************************/
CCODE InitClusterParams(SESSION_INFORMATION *sessionInfo )
{
	CCODE 	 ccode = 0;
	
	if((sessionInfo->clusterparams = (ClusterParams*)malloc(sizeof(ClusterParams))))
	{
			sessionInfo->clusterparams->reTries = DEFAULT_RE_TRIES;
			sessionInfo->clusterparams->reTryPeriod = DEFAULT_RE_TRY_PERIOD;
			sessionInfo->clusterparams->waitTime = DEFAULT_WAIT_TIME;
	}
	else
		ccode = -1;

	return ccode;
}

/*********************************************************************
* NAME: Reconnect
*
* WHAT: Closes the previous connection and Reconnects to the same target 
*       after a Failure of the Target(failover / failback)
*
*
* SMS-SPECIFIC ROUTINES CALLED: CloseTarget (Demain.c)
*                               OpenTarget  (Demain.c)
*                               OpenService (Demain.c)
*
**********************************************************************/
int Reconnect(SESSION_INFORMATION *sessionInfo)
{
	CCODE 	ccode;
	UINT32 	connection = 0;
	int 	numRetry = 1;
	
	if (sessionInfo->clusterparams->reTries > 0)
		numRetry = sessionInfo->clusterparams->reTries;
	
	PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_FAIL_OVER));
	CloseTarget(sessionInfo);
	PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_RECONNECTING));

	/* Wait for a period of time for services to initialize on other cluster nodes */
	Delay_milliSec(sessionInfo->clusterparams->waitTime, 60000);

	/* Attempt connection to the target service again. 
	 * As services may not yet be up, try upto retry count and then fail */
	while(numRetry--)
	{
		/* Open the target service that we originally connected to */
		connection = OpenTarget(sessionInfo, numRetry);
		if(connection != -1)
		{
			sessionInfo->connection = connection;
			
			/* Try and get a connection to the target service */
			ccode = OpenService(sessionInfo);
			if(ccode)
				CloseTarget(sessionInfo);
			else
				return 0;
		}
		Delay_milliSec(sessionInfo->clusterparams->reTryPeriod,1000);
	}
	
	return RECONNECT_FAIL; //return reconnection time out
}

/*********************************************************************
* NAME: Backup
*
* WHAT: This function controls the overall process of backing up the
*       Target Service. A complimentary function (Restore) exists for
*       restoring data to a Target Service.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMPutOneName (SMS UTILITY API)
*                               NWSMTSScanDataSetBegin (TSAPI)
*                               NWSMGetDataSetName (SMS UTILITY API)
*                               NWSMTSGetNameSpaceTypeInfo (TSAPI)
*                               NWSMTSScanDataSetEnd (TSAPI)
*                               NWSMTSOpenDataSetForBackup (TSAPI)
*                               NWSMTSCloseDataSet (TSAPI)
*                               NWSMTSReadDataSet (TSAPI)
*                               NWSMTSScanNextDataSet (TSAPI)
*                               NWSMTSScanDataSetContinue (TSAPI)
*                               NWSMCatStrings (SMS UTILITY API)
**********************************************************************/
void Backup(SESSION_INFORMATION *sessionInfo,  UINT32 *canrestart) 
{
	CCODE 			 ccode;
	int 			 restart = TRUE;
	int 			 x, y;
#ifdef N_PLAT_UNIX	
	unsigned long long			 total, utotal;	
#else
	unsigned __int64 			total,utotal;
#endif
	UINT16 			 resourceCount;
	UINT32 			 bytesRead, dataHandle, sequence, readSize;
	UINT32 			 noOfDirectories,noOfFiles;
	NWSM_DATA_SET_NAME 			dataSetName;
	NWSM_DATA_SET_NAME_LIST 	*dataSetNameList;
	NWSM_DATA_SET_NAME_LIST 	*lastDataSetNameList = NULL;
	char 			*Cparent = NULL;
	char 			*Cchild = NULL;
	UINT32 			 CnameSpaceType;
	NWBOOLEAN 		 CreverseOrder;
	STRING_BUFFER	*CfirstSeparater = NULL;
	STRING_BUFFER	*CsecondSeperater = NULL;
	STRING_BUFFER	*CfullPath = NULL;
	UINT32 			 mParentAllocSize = 0;
	UINT32 			 mChildAllocSize = 0;
	size_t			 destLen;
	char 			*dest = NULL;
	char 			*lastParent = NULL;
	NWSM_SCAN_INFO_EXTN_NFS_DATA_1 	nfsInfo = {0};
	CCODE 		(*mPtrScanDataSetContinue)(UINT32, NWSM_DATA_SET_NAME_LIST *,
					NWSM_SCAN_CONTROL *, NWSM_SELECTION_LIST *, NWSM_DATA_SET_NAME_LIST *, 
					UINT32 *, NWSM_SCAN_INFORMATION **, NWSM_DATA_SET_NAME_LIST **) = NULL;
	
	ECMATime		eTime;
	UINT32			unixTime;
	INT32			tzOff;
	BACKUP_CONTEXT *bctx = sessionInfo->bctx;
	BOOL			useStrictMode = FALSE;
	NWBOOLEAN		isbackup=TRUE;
	
	/* Import the NWSMTSScanDataSetContinue API */
#ifdef N_PLAT_NLM
	mPtrScanDataSetContinue = ImportSymbol(GetNLMHandle(),"NWSMTSScanDataSetContinue");
#else
	mPtrScanDataSetContinue = NWSMTSScanDataSetContinue;
#endif

#ifdef N_PLAT_GNU
	if(strstr(sessionInfo->tsaName, "Linux File System") || strstr(sessionInfo->tsaName,"Linux Cluster File System"))
	{
		/*Check for root user*/
		pw = getpwnam(sessionInfo->CuserName);
		if(pw != NULL)
		{
			if(pw->pw_uid == 0)
				useStrictMode = TRUE;
		}
	}
#endif	

	utotal = 0;
	total = 0;
	dataSetNameList = NULL;
	resourceCount = 0;
	readSize = BUFFER_SIZE;
	sequence = 0xff00ff00L;
	sessionInfo->scanInfo = NULL;
	noOfDirectories = 0;
	noOfFiles = 0;

	/* A warning message will be printed as long as this is non-zero. */
	safeUnloadFlag ++;

	/* Check, if there is another backup session */
	if (Cparent)
		free(Cparent);

	/* Predefined amount of memory is allocated for Parent */
	Cparent = (char *)calloc((size_t)PREMEMALLOC, sizeof(char));
	if (!Cparent)
	{
		PrintMsg(NBK_ERR_MSG,  _NbkMessage(NBK_OUT_OF_MEMROY));
		restart = FALSE;
		goto Terminate;
	}
	mParentAllocSize = PREMEMALLOC;

	/* Check, if there is another backup session */
	if (Cchild)
		free(Cchild);

	/* Predefined amount of memory is allocated for child */
	Cchild = (char *)calloc((size_t)PREMEMALLOC, sizeof(char));
	if (!Cchild)
	{
		PrintMsg(NBK_ERR_MSG,  _NbkMessage(NBK_OUT_OF_MEMROY));
		restart = FALSE;
		goto Terminate;
	}
	mChildAllocSize = PREMEMALLOC;

	/* Check, if there is another backup session */
	if (lastDataSetNameList)
		free(lastDataSetNameList);
	lastDataSetNameList = NULL;
	
	StartTimer();

	/*********************************************************************
	*       Begin scanning for the selected data sets on the Target Service.
	**********************************************************************/
	if ((ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl, 
	                      sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0)
	{
		/***************************************************************************
		*	      During a cluster backup, if a fail over / fail back happens during
		*       the NWSMTSScanDataSetBegin() call, the following code will be executed.
		*
		*       This will re-establish connection with the TSA and call
		*       NWSMTSScanDataSetBegin(). If the failure is in the first dataset, then continue
		*       should not happen, as there is nothing to continue from.
		****************************************************************************/
		if (IsRecoveryError(ccode) && (sessionInfo->clusterparams))
		{
			if (!Reconnect( sessionInfo))
			{
				ccode=NWSMTSScanDataSetBegin(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
							sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList); 
			}
			else
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_RECONNECTING_ERR));
				goto Terminate;
			}
		}
		else
		{
			NWSMGetDataSetName(bctx->dsetList, sessionInfo->nameSpaceType, &dataSetName);
			PrintMsg(NBK_ERR_MSG,_NbkMessage(NBK_BKP_DSET_OPENFAIL),  dataSetName.name);
			PrintError(ccode, "NWSMTSScanDataSetBegin", 0, sessionInfo);
		}

	}
	/**************************************************************************     
	* 	     Constructing the name space type info for the resource being
	*        backed up. Will need this for constructing the cursor that is to
	*        be passed to TSA after a fail over / fail back. This is done only
	*        once for a resource.
	**************************************************************************/
	if (!ccode && (sessionInfo->clusterparams))
	{
		NWSMGetDataSetName(dataSetNameList, 0L, &dataSetName);
		CnameSpaceType = dataSetName.nameSpaceType;
		NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, CnameSpaceType, &CreverseOrder, &CfirstSeparater, &CsecondSeperater);
	}
	else if (ccode)
		goto Terminate;
	
	

	/*********************************************************************
	*       This loop completes the scan of the selected data sets on the
	*       Target Service. The data set name is displayed to the user
	*       as it is scanned.
	**********************************************************************/

	while (!ccode)
	{
		CCODE error;
		resourceCount++;
		NWSMGetDataSetName(dataSetNameList, bctx->nameSpaceType, &dataSetName);
		/* If the returned names were in MBCS then convert the same to UTF-8 to display the same to the user */
		if (!sessionInfo->IsUTF8Supp)
		{
			ccode = ConvertLocToUTF8( dataSetName.name, &destLen, &dest );
			if(ccode)
			{
				dest = malloc(strlen(dataSetName.name) +1 );
				if( dest == NULL)
				{
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
					goto Terminate;
				}
				strcpy(dest, dataSetName.name);
				ccode = 0;
			}
		}
		else
		{
			dest = dataSetName.name;
		}
		LogSession(sessionInfo->logFile, "\n\r" );
		if(!sessionInfo->scanInfo->parentFlag)
		{
#ifdef N_PLAT_UNIX
			NWSMDOSTimeToECMA(sessionInfo->scanInfo->modifiedDateAndTime, &eTime);
			NWSMECMAToUnixTime(&eTime, &unixTime, &tzOff);
			unixTime -= tzOff * 60;
			if(unixTime < timep)
			{
				goto NEXT;
			}
#endif
			LogSession(sessionInfo->logFile,sessionInfo->scanInfo->parentFlag ? "\n%-60s":"\n  %-58s", dataSetName.name);
		}
		/*Prints the data set name*/
		ccode=PrintDataSetName(sessionInfo->scanInfo->parentFlag , &lastParent, dest, sessionInfo, dest, isbackup);
		if(ccode)
			goto Terminate;
		if(!sessionInfo->IsUTF8Supp && dest)
		{
    			free(dest);    
    			dest = NULL;
		}
		PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_PRIMDATA_STREAM_SIZE), sessionInfo->scanInfo->primaryDataStreamSize);
		LogSession(sessionInfo->logFile, "\n\r\t\t primaryDataStreamSize= %lu", sessionInfo->scanInfo->primaryDataStreamSize);
		PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_TOTALSTREAM_SIZE), sessionInfo->scanInfo->totalStreamsDataSize);
		LogSession(sessionInfo->logFile, "\n\r\t\t totalStreamsDataSize = %lu", sessionInfo->scanInfo->totalStreamsDataSize);
		
		if (sessionInfo->scanInfo->parentFlag) 
		{
			/* If we have additional meta data we can cross check parentFlag status */
			if (!ProcessScanInfoExtension(sessionInfo->scanInfo, &nfsInfo))
			{
				if (nfsInfo.nfs_st_mode & S_IFDIR)
				{
					noOfDirectories++;
				}
			}
			else
			{
				noOfDirectories++;
			}
		}
		else noOfFiles ++;

		/*********************************************************************
		*       If the scan option was selected and the user presses any key,
		*       terminate the scan.
		*
		*       NOTE: To clean up properly, NWSMTSScanDataSetEnd must be called
		*       if ending a scan prematurely. See NWSMTSScanDataSetEnd in the
		*       DR API document for more details.
		**********************************************************************/
		if (bctx->scanOnly)
		{
//			status("\n");
			if (IsKeyStrokeAvailable())
			{
				NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
				sessionInfo->scanInfo = NULL ;
				dataSetNameList = NULL ;
				goto Terminate;
			}
		}

		/*********************************************************************
		*       For the Backup option, each data set returned by the scan
		*       is opened for backup. If the open fails, this data set is skipped
		*       and the next one is scanned.
		*       If the open succeeds a check is made for a non-zero data handle
		*       returned by the open. If the data handle is non-zero, read the 
		*       data set until complete or until the occurrence of an error.
		*       If the option to save the backup data was selected, write the
		*       data to a file on disk.
		**********************************************************************/
		else
		{
			UINT32 mode;
			if(sessionInfo->scanInfo->attributes & 0x40000000)
				PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_SYSMOD_BITSET ));
			else
				PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_SYSMOD_BITRESET));
			total = 0;
			
			if(strstr(sessionInfo->tsaName, "Linux File System") || strstr(sessionInfo->tsaName,"Linux Cluster File System"))
			{
				/*In case of non-root user set the open mode to less stricter mode*/
				if(useStrictMode)
					mode = NWSM_OPEN_READ_DENY_WRITE;
				else
					mode = NWSM_NO_LOCK_NO_PROTECTION;
			}
			else
				mode = NWSM_OPEN_READ_DENY_WRITE;
			
			if ((error = NWSMTSOpenDataSetForBackup(sessionInfo->connection, sequence, mode, &dataHandle)) != 0)
			{
				/*************************************************************************
				*		During a cluster backup, if a fail over / fail back happens during
				*       the OpenDataSetForBackup() call, the following loop will be executed.
				*       This will re-establish connection with the TSA, build the
				*       lastDataNameList and call ScanDataSetContinue() 
				**************************************************************************/
				if (mPtrScanDataSetContinue  && IsRecoveryError(error) && (sessionInfo->clusterparams))
				{
					if (sessionInfo->scanInfo || dataSetNameList)
					{
#ifdef N_PLAT_NLM
						NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
#endif
						sessionInfo->scanInfo = NULL ;
						dataSetNameList = NULL ;
					}
					if (!Reconnect(sessionInfo))
					{
						if (*Cparent || *Cchild)
						{
							 /* Cat the parent and child */
							NWSMCatStrings(2, &CfullPath, Cparent, Cchild);
							if (lastDataSetNameList)
							{
								free(lastDataSetNameList);
								lastDataSetNameList = NULL;
							} 
							NWSMPutOneName((void **)&lastDataSetNameList, CnameSpaceType, 0L, CreverseOrder,
									(char *)&CfirstSeparater, (char *)&CsecondSeperater, (char *)&(CfullPath->string));
							ccode = (*mPtrScanDataSetContinue)(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
								sessionInfo->selectionList, lastDataSetNameList, &sequence, &sessionInfo->scanInfo, &dataSetNameList); 
						}
						else
						{
							ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
								sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList); 
						}
						if (ccode)
						{
							PrintError(ccode,(*Cparent || *Cchild)?"NWSMTSScanDataSetContinue":"NWSMTSScanDataSetBegin",0,sessionInfo);
							goto Terminate1;
						}
						
						if (sessionInfo->scanInfo->parentFlag) noOfDirectories--;
						else noOfFiles--;
						resourceCount--;

						/* Continue from NWSMTSOpenDataSetForBackup loop. */
						continue;
					}
					else
					{
						PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_RECONNECTING_ERR));
						goto Terminate;
					}
				}

				PrintError(error, "NWSMTSOpenDataSetForBackup", 0, sessionInfo);
				LogSession(sessionInfo->logFile, "\nCCODE from OpenDataSetForBackup %X\n", error);
				goto NEXT;
			}
			else
			{
				if (bctx->saveBackup)
				{
					
					error=PutDataSetHeader(dataSetNameList, sessionInfo);
					if(error)
					{
						PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_ERR_WRITE_HDR));
						restart = FALSE;
						goto Terminate1;	
					}
				}
				PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_OPEN));
				LogSession(sessionInfo->logFile, " Open ");
			}
			PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_READ));
			LogSession(sessionInfo->logFile, " Read");
			CursorPosition(&x, &y);
			while (!error)
			{
				SetCursor(x, y);
				/*********************************************************************
				*       If the user presses any key - terminate the scan/backup.
				**********************************************************************/
				if (IsKeyStrokeAvailable())
				{
					if(ccode == 0)
					{
						NWSMTSCloseDataSet(sessionInfo->connection, &dataHandle);
						NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
						sessionInfo->scanInfo = NULL ;
						dataSetNameList = NULL ;
					}
					else
						NWSMTSCloseDataSet(sessionInfo->connection, &dataHandle);
					utotal += total;
					goto Terminate;
				}
				sessionInfo->dataBufferPtr=sessionInfo->dataBuffer;

				/**************************************************************************
				*                       Read the Data Set 
				**************************************************************************/
				if ((error = NWSMTSReadDataSet(sessionInfo->connection, dataHandle, readSize, &bytesRead, (BUFFER *)sessionInfo->dataBuffer)) == 0)
				{

					total += bytesRead;

					/**************************************************************************
					*               Here's where we write the backup data to files 
					**************************************************************************/
					if (bctx->saveBackup)
					{
						if(PutData(sessionInfo, (UINT32)bytesRead))  
							break;
					}
				}
				PrintMsg(NBK_STATUS_MSG, "\n\r %-8lu ", total);
				LogSession(sessionInfo->logFile, "\n\r %-8llu ", total);
				
				/* Save bytesRead bytes from dataBuffer */
				if (bytesRead < readSize)
					break;
			}
			
			/***************************************************************************
			*       The code below is to update the Cchild / Cparent with the name
			*       of the successfully backed up file/directory 
			****************************************************************************/
			if (sessionInfo->clusterparams)
			{
				if (!error)
				{
					if (sessionInfo->scanInfo->parentFlag)
					{
						/* This is for if the name of the dir is more than the allocated size */
						while ((strlen(dataSetName.name) > mParentAllocSize))
						{
							mParentAllocSize += PREMEMALLOC;
							Cparent = (char *)realloc(Cparent, mParentAllocSize);
							if (!Cparent)
							{
								PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
								restart = FALSE;
								goto Terminate1;
							}
								
						}
						strcpy(Cparent, dataSetName.name);
						if (*Cchild)
						*Cchild = 0;
					}
					else
					{
						/* This is for if the name of the file is more than the allocated size */
						while ((strlen(dataSetName.name) > mChildAllocSize))
						{
							mChildAllocSize += PREMEMALLOC;
							Cchild = (char *)realloc(Cchild, mChildAllocSize);
							if (!Cchild)
							{
								PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
								restart = FALSE;
								goto Terminate1;
							}
						}	
						strcpy(Cchild, dataSetName.name);
					}
				}
				/***************************************************************************
				*       During a cluster backup, if a fail over / fail back happens
				*       during the ReadDataSet() call, the following loop will be executed.
				*       This will re-establish connection with the TSA, build the
				*		    lastDataNameList and call ScanDataSetContinue() 
				***************************************************************************/
				if (mPtrScanDataSetContinue && IsRecoveryError(error))
				{
					/****************************************************************************
					*			End the partially backed up file. During restore, consecutive file
					*			name can be checked and overwritten even if overwrite option is disabled
					****************************************************************************/
					if (sessionInfo->scanInfo || dataSetNameList)
					{
#ifdef N_PLAT_NLM
						NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
#endif
						sessionInfo->scanInfo = NULL ;
						dataSetNameList = NULL ;
					}
					
					if (!Reconnect(sessionInfo))
					{ 
						if (*Cparent || *Cchild)
						{
							/* Cat the parent and child */
							NWSMCatStrings(2, &CfullPath, Cparent, Cchild);
							if (lastDataSetNameList)
							{
								free(lastDataSetNameList);
								lastDataSetNameList=NULL;
							}

							NWSMPutOneName((void **)&lastDataSetNameList, CnameSpaceType, 0, CreverseOrder,
									(char *) &CfirstSeparater, (char *) &CsecondSeperater, (char *) &(CfullPath->string));
							ccode = (*mPtrScanDataSetContinue)(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
									sessionInfo->selectionList, lastDataSetNameList, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
						}
						else
						{
							ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
									sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
						}
						if (ccode)
						{
							PrintError(ccode,(*Cparent || *Cchild) ? "NWSMTSScanDataSetContinue" : "NWSMTSScanDataSetBegin", 0, sessionInfo);
							goto Terminate1;
						}

						/* Continue from NWSMTSOpenDataSetForBackup loop. */
						continue;
					} 
					else
					{
						PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_RECONNECTING_ERR));
						goto Terminate;
					}
				}
			}
			
			if (error)
			{
				PrintError(error, "NWSMTSReadDataSet", 0, sessionInfo);
				PrintMsg(NBK_ERR_MSG,_NbkMessage(NBK_BKP_DSET_READFAIL),  dataSetName.name);
			}
			if (bctx->saveBackup)
			{
				DataEnd(sessionInfo);
			}
			/***************************************************************************
			*	   Handling NWSMTSCloseDataSet during cluster failure is ignored here and is left to 
			*	   third parties to define the scheme that best suits them.
			***************************************************************************/
			if ((error = NWSMTSCloseDataSet(sessionInfo->connection, &dataHandle)) != 0)
				PrintError(error, "NWSMTSCloseDataSet", 0, sessionInfo);
			//status1("\n");
		}

		/*********************************************************************
		*       Increment the count of the total number of bytes saved in
		*       this backup session.
		**********************************************************************/
NEXT:
		utotal += total;

		/*********************************************************************
		*       Scan for the next data set - if another data set is available,
		*       the loop continues; otherwise display the number of resources
		*       scanned and terminate.
		**********************************************************************/
		if ((ccode = NWSMTSScanNextDataSet(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList)) != 0
			&& ccode != NWSMTS_NO_MORE_DATA_SETS)
		{
			/*********************************************************************     
			*		During a cluster backup, if a fail over / fail back happens
			*       during the ScanNextDataSet() call, the following loop will be
			*       executed. This will re-establish connection with the TSA, build
			*       the lastDataNameList and call ScanDataSetContinue()
			*		If the failure is in first data set(possible, as the first file to be backed
			*		up may be a Swap file or some other such as it cannot be opened.
			*********************************************************************/
			if (mPtrScanDataSetContinue && IsRecoveryError(ccode) && (sessionInfo->clusterparams))
			{
				if (sessionInfo->scanInfo || dataSetNameList)
				{
#ifdef N_PLAT_NLM
					NWSMTSScanDataSetEnd(sessionInfo->connection, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
#endif
					sessionInfo->scanInfo = NULL ;
					dataSetNameList = NULL ;
				}
				if (!Reconnect(sessionInfo))	
				{
					if (*Cparent || *Cchild)
					{
						/* Cat the parent and child */
						NWSMCatStrings(2, &CfullPath, Cparent, Cchild);
						if (lastDataSetNameList)
						{
							free(lastDataSetNameList);
							lastDataSetNameList = NULL;
						}
						NWSMPutOneName((void **)&lastDataSetNameList, CnameSpaceType, 0, CreverseOrder,
								(char *) &CfirstSeparater, (char *) &CsecondSeperater, (char *) &(CfullPath->string));
						ccode = (*mPtrScanDataSetContinue)(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl, sessionInfo->selectionList,
								lastDataSetNameList, &sequence, &sessionInfo->scanInfo, &dataSetNameList); 
					}
					else
					{
						ccode = NWSMTSScanDataSetBegin(sessionInfo->connection, bctx->dsetList, &sessionInfo->scanControl,
								sessionInfo->selectionList, &sequence, &sessionInfo->scanInfo, &dataSetNameList);
					}
					if(ccode)
					{
						PrintError(ccode,(*Cparent || *Cchild)?"NWSMTSScanDataSetContinue":"NWSMTSScanDataSetBegin", 0, sessionInfo);
						goto Terminate1;
					}
				} 	 						  							
				else
				{
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_RECONNECTING_ERR));
					goto Terminate;
				}
			}
			else
			{
				if(ccode==NWSMUT_BUFFER_OVERFLOW)
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_PROCESS_BKUP_PATH_ERR));
				else
					PrintError(ccode, "NWSMTSScanNextDataSet", 0, sessionInfo);
			}
		}
	}

	/*********************************************************************
	*       This is the common termination point for the backup function.
	*       If the restart flag has not been set to FALSE, go back to loop
	*       and start over.
	*       If we were doing a backup (not just scan) display some statistics.
	**********************************************************************/
	sessionInfo->stats.fileCount += noOfFiles;
	sessionInfo->stats.dirCount += noOfDirectories;
	sessionInfo->stats.resCount += resourceCount;
	sessionInfo->stats.totalBytesRead += utotal;
	sessionInfo->stats.totalTime += ElapsedTime();
	
Terminate:
	if (restart)
	{
		*canrestart = restart;
	}

Terminate1:
	/*********************************************************************
	*       End of Backup/Scan - do some clean up and return.
	**********************************************************************/
	safeUnloadFlag--;
	
	if (lastDataSetNameList)
		free(lastDataSetNameList);
	
	NWSMFreeString(&CfirstSeparater);
	CfirstSeparater=NULL;
	
	NWSMFreeString(&CsecondSeperater);
	CsecondSeperater=NULL;
	
	NWSMFreeString(&CfullPath);
	CfullPath=NULL;
	
	if (Cparent)
	{ 
		free(Cparent);
		Cparent=NULL;
	}  
	
	if (Cchild)
	{  
		free(Cchild);
		Cchild=NULL;
	}
	if(!sessionInfo->IsUTF8Supp && dest)
	{
		free(dest);    
		dest = NULL;
	}
	if( lastParent)
	{
		free(lastParent);
		lastParent = NULL;
	}
#ifdef N_PLAT_NLM
	if (mPtrScanDataSetContinue)
		UnimportSymbol(GetNLMHandle(), "NWSMTSScanDataSetContinue");
#endif
	
	return;
}

/*********************************************************************
* NAME: Restore
*
* WHAT: This function controls the overall process of restoring data
*       to a Target Service. A complimentary routine (Backup) exists
*       for backing up Target Service data.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMTSSetRestoreOptions (TSAPI)
*                               NWSMTSCloseDataSet (TSAPI)
*                               NWSMTSOpenDataSetForRestore (TSAPI)
*                               NWSMTSWriteDataSet (TSAPI)
*
**********************************************************************/
void Restore(SESSION_INFORMATION *sessionInfo, int *canrestart) 
{
	BUFFER 			*SMSData;
	CCODE 			 ccode = 0;
	int 			 restart = TRUE;
	int 			 x, y;
	UINT8 			 parentFlag, terminalNodeNameOnlyFlag;
	UINT16 			 resourceCount =0 ;
	UINT32 			 bytesToWrite =0 , dataSetHandle = 0, parentHandle = 0;
	UINT8 			 parentWasExcluded = FALSE;
	UINT8			nextdatablock;
#ifdef N_PLAT_UNIX	
	unsigned long long			 total, utotal;	
#else
	unsigned __int64 			total,utotal;
#endif
	int moredatasets = 	0 ;
	CCODE  herror=0;
	NWSM_DATA_SET_NAME_LIST *resourceName = NULL;
	NWSM_DATA_SET_NAME_LIST *newList = NULL;
	NWSM_DATA_SET_NAME dsetName;
	NWSM_DATA_SET_NAME dsetExtractName;
	NWSM_DATA_SET_NAME dsetSource;
	NWSM_DATA_SET_NAME NextResourceName;
	RESTORE_CONTEXT *rctx = sessionInfo->rctx;
	char *lastParent = NULL;
	NWBOOLEAN isUTFName = FALSE;
	NWBOOLEAN isExtract = TRUE;
	NWBOOLEAN isParentFound = TRUE;
	NWBOOLEAN prevParentFlag = FALSE;
	NWBOOLEAN isFirstFile = TRUE;
	NWBOOLEAN isTime = FALSE;
	NWBOOLEAN nextdSet = FALSE;
	NWBOOLEAN checkUTF8 =FALSE;
	NWBOOLEAN	isbackup=FALSE;
	size_t destLen;
	char *locpath = NULL;
	char *extractPath=NULL;
	char *listPath=NULL;
	char *tmpStr=NULL;
	NWSM_LIST *cur = NULL;
	NWSM_LIST *savecur = NULL;
	char 	fileWithOutPath[1024];
	ECMATime		eTime;
	UINT32			unixTime;
	INT32			tzOff;
 	/* A warning message will be printed as long as this is non-zero. */
	safeUnloadFlag ++;
	utotal = 0;
	total = 0;

	/*********************************************************************
	*       If a selection list exists - set up the restore options so
	*       that data sets will be checked against the list and the appropriate
	*       data sets excluded.
	**********************************************************************/
	if(sessionInfo->selectionList)
		NWSMTSSetRestoreOptions(sessionInfo->connection, sessionInfo->scanControl.generateCRC, FALSE, sessionInfo->selectionList);
	

	/*********************************************************************
	*       For this example engine, backed up data is stored in DOS files
	*       on disk instead of tape (or other backup media). Here we open
	*       the disk files for reading. Two files are maintained - an index
	*       file and a data file.
	**********************************************************************/
	sessionInfo->tUsed=0;
	sessionInfo->tLeft= 0;
	sessionInfo->atHeader=TRUE;
	sessionInfo->newDataSetFlag=1;
	sessionInfo->recInfo.scanInformation=NULL;
	sessionInfo->recInfo.dataSetName=NULL;
	StartTimer();
	/*Read the current transfer buffer from the archive*/
	if(!GetTransferBuffer(sessionInfo))
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		restart = FALSE;
		goto Terminate;		
	}
	/*Checks for new data set*/
	moredatasets = GetDataHeader(&parentFlag, &terminalNodeNameOnlyFlag, &resourceName,sessionInfo, &herror);
	while (moredatasets)
	{
		UINT16 mode;
		terminalNodeNameOnlyFlag= TRUE;
		resourceCount ++;

		if(herror)
			goto Terminate;
            if (dataSetHandle)
            {
				PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_CLOSE));
				LogSession(sessionInfo->logFile," Close");
				NWSMTSCloseDataSet(sessionInfo->connection, &dataSetHandle);
				dataSetHandle = 0;
            }
#ifdef N_PLAT_NLM
		if(parentFlag)
			sessionInfo->recInfo.pathIsFullyQualified=TRUE;
		else
			sessionInfo->recInfo.pathIsFullyQualified=FALSE;
#endif
					
		if (parentFlag) parentWasExcluded = FALSE;
		if (parentWasExcluded) 
			goto NextData;

		isTime=FALSE;

#ifdef N_PLAT_UNIX
		if(!parentFlag)
		{
			if(sessionInfo->recInfo.scanInformation)
			{
				NWSMDOSTimeToECMA(sessionInfo->recInfo.scanInformation->modifiedDateAndTime, &eTime);
				NWSMECMAToUnixTime(&eTime, &unixTime, &tzOff);
				unixTime -= tzOff * 60;
				if(unixTime < timep)
				{
					isTime=TRUE;
					goto NextData;
				}
			}
	
		}
#endif
		if(newList)
		{
			free(newList);
			newList = NULL;
		}
		
		if(nextdSet)
		{
			if(!parentFlag)
			{
				SMSData = GetData(&bytesToWrite,sessionInfo);
				while(SMSData)
				{
					bytesToWrite=0;
					SMSData = GetData(&bytesToWrite,sessionInfo);
				}
				goto NextDataHeader;	
			}
		}
		if(parentFlag)
			prevParentFlag=TRUE;

		if(rctx->multiVol)
		{
			
			if((sessionInfo->recInfo.dataSetName ==NULL) ||( !sessionInfo->recInfo.pathIsFullyQualified && (!parentFlag) &&( !prevParentFlag)))
			{
				SMSData = GetData(&bytesToWrite,sessionInfo);
				while(SMSData)
				{
					bytesToWrite=0;
					SMSData = GetData(&bytesToWrite,sessionInfo);
				}
				if(sessionInfo->ControlBlock.transferBufferSequence==END_OF_RECORD_MARKER)
					break;
				goto NextDataHeader;		
			}

		}
		
		if(rctx->renameList != NULL && sessionInfo->recInfo.pathIsFullyQualified)
		{
			if(MoveDataSet)
			{
				checkUTF8=FALSE;
				/*get the resource name in specified name space type*/
				GetResourceName(resourceName, checkUTF8, &dsetSource, &isUTFName);
				if(*(dsetSource.name) != '/')
				{
					tmpStr = strchr(dsetSource.name, ':');
					if(tmpStr)
					{
						tmpStr++;
						*tmpStr=0;
						if( rctx->renameList->source)
						{
							free(rctx->renameList->source);
							rctx->renameList->source=NULL;
						}
						rctx->renameList->source = malloc( MAX_BUF_LEN );
						if( rctx->renameList->source == NULL)
						{
							PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
							goto Terminate;
						}
						strcpy( rctx->renameList->source ,dsetSource.name);
					}
					
				}

			}
			/*Get the destination path if rename restore is specified*/
			ccode = GetNewName(sessionInfo, resourceName, &newList);
			if(ccode)
			{
				nextdSet=TRUE;
				checkUTF8=FALSE;
				GetResourceName(resourceName, checkUTF8, &NextResourceName, &isUTFName);
				ccode=PrintDataSetName(parentFlag, &lastParent, NextResourceName.name, sessionInfo, locpath, isbackup);
				if(ccode)
					goto Terminate;				
				if(ccode==NWSMUT_BUFFER_OVERFLOW)
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_PROCESS_RES_PATH_ERR));
				else
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_PROCESS_NAME_ERROR));
				SMSData = GetData(&bytesToWrite,sessionInfo);
				while(SMSData)
				{
					bytesToWrite=0;
					SMSData = GetData(&bytesToWrite,sessionInfo);
				}
				goto NextDataHeader;
				
			}
			nextdSet=FALSE;
		}	
		if(resourceName)
		{			
			isUTFName = FALSE;
			/*Get the resource name and copy it in the specified buffer.
			* Copies the full path in case of directories and terminal node name only in case of files.
			*This code is executed only if the user wants to extract some files from the archive*/
			if(sessionInfo->bctx->extractFiles)
			{	
				checkUTF8=TRUE;
				GetResourceName(resourceName, checkUTF8, &dsetExtractName, &isUTFName);
				ccode=CopyResourceName(parentFlag, dsetExtractName.name, sessionInfo,isUTFName, &extractPath);
				if(ccode)
					goto Terminate;		
			}
			if(parentFlag)
			{
				isFirstFile=TRUE;
				if(listPath)
				{
					free(listPath);
					listPath=NULL;
				}
			}
			/*Decides whether a particular file needs to be extracted or not*/
			ccode=IsFileExtracted(parentFlag, &isFirstFile, sessionInfo, extractPath, &listPath, fileWithOutPath, &cur,&savecur, &isParentFound,&isExtract);
			if(ccode)
				goto Terminate;
				
			/* If the name is UTF8 format, need to convert UTF8 to local */
			if(isExtract)
			{
				isUTFName = FALSE;
				if(sessionInfo->recInfo.pathIsFullyQualified && rctx->renameList != NULL && newList != NULL)
				{
					checkUTF8=TRUE;
					GetResourceName(newList, checkUTF8, &dsetName, &isUTFName);
				}
				else
				{
					checkUTF8=TRUE;
					GetResourceName(resourceName, checkUTF8, &dsetName, &isUTFName);
				}	
			}
			if(!isUTFName && sessionInfo->bctx->extractFiles && extractPath)
			{
	    			free(extractPath);    
	    			extractPath = NULL;
			}
			
			
		}
				
		if (parentFlag && terminalNodeNameOnlyFlag && isExtract)
		{
			if (parentHandle)
			{
				PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_CLOSE_PARENT));
				LogSession(sessionInfo->logFile, "\nClose parent\n");
				NWSMTSCloseDataSet(sessionInfo->connection, &parentHandle);
				parentHandle = 0;
			}
			
			mode = rctx->overwriteParent ? NWSM_OVERWRITE_DATA_SET : NWSM_CREATE_PARENT_HANDLE;
			if (SetExFileData)
			{
				mode |= NWSM_NO_DATA_STREAMS; 
			}
			if(noIDs)
			{
				mode |= NWSM_NO_CHILD_TRUSTEES;
				mode |= NWSM_NO_PARENT_TRUSTEES;
			}
			PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_CREATE_PARENT));
			LogSession(sessionInfo->logFile, "\nCreate parent");
			PrintMsg(NBK_STATUS_MSG, _NbkMessage(NBK_MODE), mode);
			if ((ccode = NWSMTSOpenDataSetForRestore(sessionInfo->connection, 0L, rctx->renameList ? newList : NULL,
			                mode, &parentHandle)) != 0)
			PrintError(ccode, "NWSMTSOpenDataSetForRestore", __LINE__,sessionInfo);
		}

		else if (terminalNodeNameOnlyFlag && isExtract)
		{
			mode = rctx->overwriteChild ? NWSM_OVERWRITE_DATA_SET : NWSM_DO_NOT_OVERWRITE_DATA_SET;
			if (SetExFileData)
			{
				mode |= NWSM_NO_DATA_STREAMS; 
			}
			if(noIDs)
			{
				mode |= NWSM_NO_CHILD_TRUSTEES;
				mode |= NWSM_NO_PARENT_TRUSTEES;
			}
			PrintMsg(NBK_STATUS_MSG,  _NbkMessage(NBK_OPEN_CHILD_WPARENT ));
			LogSession(sessionInfo->logFile,"\nOpen child /w parent handle");
			if ((ccode = NWSMTSOpenDataSetForRestore(sessionInfo->connection, parentHandle,  rctx->renameList ? newList : NULL, mode, &dataSetHandle)) != 0)
				PrintError(ccode, "NWSMTSOpenDataSetForRestore", __LINE__, sessionInfo);
		}

		else if (isExtract)
		{
			mode = rctx->overwriteChild ? NWSM_OVERWRITE_DATA_SET : NWSM_DO_NOT_OVERWRITE_DATA_SET;
			if (SetExFileData)
			{
				mode |= NWSM_NO_DATA_STREAMS; 
			}
			if(noIDs)
			{
				mode |= NWSM_NO_CHILD_TRUSTEES;
				mode |= NWSM_NO_PARENT_TRUSTEES;
			}
			if(parentFlag)
				PrintMsg(NBK_STATUS_MSG,  _NbkMessage(NBK_PARENT));
			else
				PrintMsg(NBK_STATUS_MSG, _NbkMessage( NBK_OPEN_CHILD_WOPARENT ));
			LogSession(sessionInfo->logFile,parentFlag ? "\nOpen parent, parent handle not used" : "\nOpen child w/o parent handle");
			if ((ccode = NWSMTSOpenDataSetForRestore(sessionInfo->connection, 0L, NULL, mode, &dataSetHandle)) != 0)
				PrintError(ccode, "NWSMTSOpenDataSetForRestore", __LINE__, sessionInfo);
		}
            
        

		/***************************************************************************
		*       Increment the byte count and display the count to the user.
		***************************************************************************/
        /* We don't want to write any terminal node children of
         * a parent that has been excluded */
		//if(parentWasExcluded) continue;
		CursorPosition(&x,&y);
		SetCursor(60, y);
		PrintMsg(NBK_STATUS_MSG,  " %-8lu\n\r " , total);
		LogSession(sessionInfo->logFile," %-8llu ", total);

		/*********************************************************************
		*       Attempt to write the data set to the Target Service. If 
		*       successful - continue reading more backup data.
		*       If the write fails - display the reason, skip this data set
		*       and look for another data set to restore.
		**********************************************************************/
		NextData:
		SMSData = GetData(&bytesToWrite,sessionInfo);
		nextdatablock=0;
		while(SMSData)
		{
			if(bytesToWrite)
			{
				utotal += bytesToWrite;
				if( !parentWasExcluded && !ccode && isExtract && !isTime)
				{
					if ((ccode = NWSMTSWriteDataSet(sessionInfo->connection, parentFlag &&
			                        terminalNodeNameOnlyFlag ? parentHandle : dataSetHandle, bytesToWrite, SMSData)) != 0)
					{
						if (ccode == NWSMTS_VALID_PARENT_HANDLE)
						{
							PrintMsg(NBK_STATUS_MSG, _NbkMessage( NBK_VALID ));
							LogSession(sessionInfo->logFile, " Valid");
							total = 0;
							ccode = 0;
						}
						else if (ccode == NWSMTS_DATA_SET_EXCLUDED)
						{
							CursorPosition(&x, &y);
							SetCursor(60, y);
							PrintMsg(NBK_STATUS_MSG,  _NbkMessage(NBK_EXCLUDED ));
							LogSession(sessionInfo->logFile, " Excluded ");
							total = 0;
							ccode = 0;
							if(parentFlag) parentWasExcluded = TRUE;
						}
						else if (ccode == NWSMTS_DATA_SET_ALREADY_EXISTS)
						{
							CursorPosition(&x, &y);
							SetCursor(60, y);
							PrintMsg(NBK_STATUS_MSG, _NbkMessage( NBK_EXISTS) );
							LogSession(sessionInfo->logFile, " Exists   ");
							ccode = 0;
							total = 0;
						}
						else 
						{
							PrintError(ccode, "NWSMTSWriteDataSet", __LINE__, sessionInfo);
							continue;
						}

						/*********************************************************************
						*       Flush this data set -- Data Sets are separated by 
						*       END_OF_RECORD_MARKERs
						**********************************************************************/

						/*********************************************************************
						*       The loop immediately above will read up to and including
						*       an END_OF_RECORD_MARKER -- This call to UngetEOF prevents
						*       the next ReadDEFiles call from reading any more data until
						*       the END_OF_RECORD_MARKER has been processed.
						**********************************************************************/
						UngetEOF();
					}
					if(!parentWasExcluded && (nextdatablock==0))
					{
						nextdatablock++;
						if(!isUTFName)
						{
							ccode = ConvertLocToUTF8(dsetName.name, &destLen, &locpath);
							if( ccode)
							{
								locpath = malloc(strlen(dsetName.name) +1 );
								if( locpath == NULL)
								{
									PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
									goto Terminate;
								}
								strcpy(locpath, dsetName.name);
								ccode = 0;
							}
						}
						else
							locpath = dsetName.name;
							ccode=PrintDataSetName(parentFlag, &lastParent, locpath, sessionInfo, locpath, isbackup);
							if(ccode)
								goto Terminate;
								
							if(!isUTFName && locpath)
							{
					    			free(locpath);    
					    			locpath = NULL;
							}
						}
				}
			}
			bytesToWrite=0;
			SMSData = GetData(&bytesToWrite, sessionInfo) ;
		}
		if(sessionInfo->ControlBlock.transferBufferSequence==END_OF_RECORD_MARKER)
			break;
		NextDataHeader:
		moredatasets = GetDataHeader(&parentFlag, &terminalNodeNameOnlyFlag, &resourceName,sessionInfo, &herror);
    }

	/*********************************************************************
	*       This is the common termination point for the Restore function.
	*       Do some clean up then check the restart flag. If restart hasn't
	*       been set to false - display some statistics (if this was a restore)
	*       then go back to loop and start over.
	**********************************************************************/
Terminate:
	PrintMsg(NBK_VERBOSE_MSG, "\n" );
    if (dataSetHandle)
    {
		PrintMsg(NBK_STATUS_MSG,  _NbkMessage(NBK_CLOSE) );
		LogSession(sessionInfo->logFile, " Close\n");
		NWSMTSCloseDataSet(sessionInfo->connection, &dataSetHandle);
		dataSetHandle = 0;
    }
    if (parentHandle)
    {
		PrintMsg(NBK_STATUS_MSG,  _NbkMessage(NBK_CLOSE_PARENT) );
		LogSession(sessionInfo->logFile, "\nClose parent\n");
		NWSMTSCloseDataSet(sessionInfo->connection, &parentHandle);
		parentHandle = 0;
    }
    CloseDEFiles(sessionInfo);
    if (restart)
    {
			sessionInfo->stats.resCount = resourceCount;
			sessionInfo->stats.totalBytesRead = utotal;
			sessionInfo->stats.totalTime = ElapsedTime();
    }
	*canrestart = restart;
    if (sessionInfo->selectionList)
    {
	    free(sessionInfo->selectionList);
	    sessionInfo->selectionList = NULL;
    }
    if(lastParent)
    {
		free(lastParent);
		lastParent=NULL;
    }
    if(sessionInfo->transferBuffer)
	{
		free(sessionInfo->transferBuffer);
		sessionInfo->transferBuffer=NULL;
	}

	if(sessionInfo->sessionHBuffer)
	{
		free(sessionInfo->sessionHBuffer);
		sessionInfo->sessionHBuffer=NULL;
		
	}
	if(sessionInfo->mediaBuffer)
	{
		free(sessionInfo->mediaBuffer);
		sessionInfo->mediaBuffer=NULL;
	}
	if(sessionInfo->recInfo.scanInformation)
	{
		free(sessionInfo->recInfo.scanInformation);
		sessionInfo->recInfo.scanInformation=NULL;
	}
	if(sessionInfo->recInfo.dataSetName)
	{
		free(sessionInfo->recInfo.dataSetName);
		sessionInfo->recInfo.dataSetName=NULL;
	}

	if (newList)
	{
		free(newList);
		newList = NULL;
	}
    safeUnloadFlag--;
    return;
}

/*********************************************************************
* NAME: ListFilesFromArchive
*
* WHAT: List files in the archive.
*
* SMS-SPECIFIC ROUTINES CALLED: 	NWSMGetDataSetName
								NWSMTSSeparateDataSetName
								NWSMFreeString
**********************************************************************/
void ListFilesFromArchive(SESSION_INFORMATION *sessionInfo) 
{
	BUFFER 			*SMSData;
	CCODE 			 ccode = 0;
	int 			 restart = TRUE;
	UINT8 			 parentFlag, terminalNodeNameOnlyFlag;
	UINT32 			 bytesToWrite = 0;
	NWBOOLEAN		isbackup=FALSE;
	NWBOOLEAN 		checkUTF8 =FALSE;
	int moredatasets = 	0 ;
	NWSM_DATA_SET_NAME_LIST *resourceName = NULL;
	NWSM_DATA_SET_NAME dsetName;
	CCODE  herror=0;
	char *lastParent = NULL;
	NWBOOLEAN isUTFName = FALSE;
	size_t destLen;
	char *locpath;
	BUFFERPTR bufferPtr=NULL;
	UINT32 _bufferSize;
	UINT16	mediaNumber;
	SMDF_FIELD_DATA		field;
 	/* A warning message will be printed as long as this is non-zero. */
	safeUnloadFlag ++;
	/*********************************************************************
	*       For this example engine, backed up data is stored in DOS files
	*       on disk instead of tape (or other backup media). Here we open
	*       the disk files for reading. Two files are maintained - an index
	*       file and a data file.
	**********************************************************************/
	if ((OpenDEFiles(O_RDONLY,sessionInfo)) != 0) 
	{
		return;
	}
	
	if(sessionInfo->sessionHBuffer== NULL)
	{
		sessionInfo->sessionHBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->sessionHBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=1;
			goto Terminate;
		}
	}
	if(sessionInfo->mediaBuffer== NULL)
	{
		sessionInfo->mediaBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->mediaBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=1;
			goto Terminate;
		}
	}
	/*Read the media header information from the sidf archive*/
	
	ccode=GetMediaHeader( sessionInfo);
	if(ccode)
		goto Terminate;
	

	bufferPtr=sessionInfo->mediaBuffer;
	_bufferSize=SECTOR_SIZE;
	/*Decide whether the archive is valid or not*/
	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       !(field.fid == SIDF_PARTITION_HEADER) )
	 {
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_INVALID_SIDF_ARCHIEVE));
		ccode=1;
		goto Terminate;	

	 }
	   while( field.fid != SIDF_PARTITION_SET_SEQUENCE)
	 {
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
			goto Terminate;
		}
		if(  field.fid == SIDF_PARTITION_SET_SEQUENCE)
		{
			memcpy(&mediaNumber, field.data, sizeof(UINT16));
			break;
		}
	 }
	  sessionInfo->rctx->volno=mediaNumber;
	 /*Get the session header information*/
	ccode=ReadOpenSession(sessionInfo);
	if(ccode)
		goto Terminate;
	if(sessionInfo->transferBuffer == NULL)
	{
		sessionInfo->transferBuffer= (BUFFER *)malloc(sessionInfo->transferBufferInfo.maxTransferBufferSize*sizeof(BUFFER));
		if (sessionInfo->transferBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=1;
			goto Terminate;
		}
		sessionInfo->currentPtr =sessionInfo->transferBuffer;
		
	}
	/*Prints the session name */
	if(sessionInfo->sessionName[0] ==0)
	{
		PrintMsg(NBK_SUMMARY_MSG,  _NbkMessage(NBK_UNKNOWN_SESSION));
	}
	else
	{
		PrintMsg(NBK_SUMMARY_MSG, _NbkMessage(NBK_SESSION_NAME), sessionInfo->sessionName);

	}


	sessionInfo->currentPtr=sessionInfo->transferBuffer;
	sessionInfo->tUsed=0;
	sessionInfo->tLeft= 0;
	sessionInfo->atHeader=TRUE;
	sessionInfo->newDataSetFlag=1;
	sessionInfo->recInfo.scanInformation=NULL;
	sessionInfo->recInfo.dataSetName=NULL;
	StartTimer();
	if(!GetTransferBuffer(sessionInfo))
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		restart = FALSE;
		goto Terminate;		
	}
	
	moredatasets = GetDataHeader(&parentFlag, &terminalNodeNameOnlyFlag, &resourceName,sessionInfo, &herror);
	while (moredatasets)
	{
		if (resourceName)
		{
			checkUTF8=FALSE;
			GetResourceName(resourceName, checkUTF8, &dsetName, &isUTFName);
			if(!isUTFName)
			{
				ccode = ConvertLocToUTF8(dsetName.name, &destLen, &locpath);
				if( ccode)
				{
					locpath = malloc(strlen(dsetName.name) +1 );
					if( locpath == NULL)
					{
						PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
						goto Terminate;
					}
					strcpy(locpath, dsetName.name);
					ccode = 0;
				}
			}
			else
				locpath = dsetName.name;
			
			ccode=PrintDataSetName(parentFlag, &lastParent, locpath, sessionInfo, locpath, isbackup);
			if(ccode)
				goto Terminate;

			if(!isUTFName && locpath)
			{
	    			free(locpath);    
	    			locpath = NULL;
			}
		}
		/*********************************************************************
		*       Attempt to write the data set to the Target Service. If 
		*       successful - continue reading more backup data.
		*       If the write fails - display the reason, skip this data set
		*       and look for another data set to restore.
		**********************************************************************/
		SMSData = GetData(&bytesToWrite,sessionInfo);
		while(SMSData)
		{
			SMSData = GetData(&bytesToWrite, sessionInfo) ;
		}
		if(sessionInfo->ControlBlock.transferBufferSequence==END_OF_RECORD_MARKER)
			break;
		moredatasets = GetDataHeader(&parentFlag, &terminalNodeNameOnlyFlag, &resourceName,sessionInfo, &herror);
		

    }

	/*********************************************************************
	*       This is the common termination point for the Restore function.
	*       Do some clean up then check the restart flag. If restart hasn't
	*       been set to false - display some statistics (if this was a restore)
	*       then go back to loop and start over.
	**********************************************************************/
Terminate:
	PrintMsg(NBK_VERBOSE_MSG, "\n" );
    CloseDEFiles(sessionInfo);
    if(!isUTFName && locpath)
    	free(locpath);    	

    if(sessionInfo->transferBuffer)
	{
		free(sessionInfo->transferBuffer);
		sessionInfo->transferBuffer=NULL;
	}
	if(sessionInfo->sessionHBuffer)
	{
		free(sessionInfo->sessionHBuffer);
		sessionInfo->sessionHBuffer=NULL;
	}
	if(sessionInfo->mediaBuffer)
	{
		free(sessionInfo->mediaBuffer);
		sessionInfo->mediaBuffer=NULL;
	}
    safeUnloadFlag--;
    return;
}

/*********************************************************************
* NAME: Init_Scan_Control
*
* WHAT: Initializes the scanControl structure for each session.
*
* SMS-SPECIFIC ROUTINES CALLED: 
**********************************************************************/
void Init_Scan_Control(NWSM_SCAN_CONTROL *scanControl)
{
	int i;
	
	scanControl->bufferSize = sizeof(NWSM_SCAN_CONTROL);
	scanControl->scanControlSize = sizeof(NWSM_SCAN_CONTROL) - sizeof(UINT16) - sizeof(UINT8);
	scanControl->scanType = NWSM_EXCLUDE_MIGRATED_CHILD;
	scanControl->firstAccessDateAndTime = 0;
	scanControl->lastAccessDateAndTime = 0;
	scanControl->firstCreateDateAndTime = 0;
	scanControl->lastCreateDateAndTime = 0;
	scanControl->firstModifiedDateAndTime = 0;
	scanControl->lastModifiedDateAndTime = 0;
	scanControl->firstArchivedDateAndTime = 0;
	scanControl->lastArchivedDateAndTime = 0;
	scanControl->returnChildTerminalNodeNameOnly = TRUE;
	scanControl->parentsOnly = FALSE;
	scanControl->childrenOnly = FALSE;
	scanControl->createSkippedDataSetsFile = TRUE;
	scanControl->generateCRC = FALSE;
	scanControl->returnNFSHardLinksInDataSetName = FALSE;
	for (i = 0; i < 6; i ++) scanControl->reserved[i] = 0;
	scanControl->scanChildNameSpaceType = NWSM_ALL_NAME_SPACES;
	scanControl->returnNameSpaceType = NWSM_ALL_NAME_SPACES;
	scanControl->callScanFilter = FALSE;
	scanControl->otherInformationSize = 0;
}

/*********************************************************************
* NAME: GetSessionInfoBuffer
*
* WHAT: Allocates memory for the session information buffer.
*
* SMS-SPECIFIC ROUTINES CALLED: 
**********************************************************************/
SESSION_INFORMATION *GetSessionInfoBuffer(void)
{
	CCODE ccode = 0;
	
	/**********************************************************************
	* Allocate memory for the session information buffer and initialize 
	* those fields to NULL for which memory will be allocated during calls
	* to the SMDR.
	**********************************************************************/
	SESSION_INFORMATION *sessionInfo;
	
	if (!(sessionInfo = (SESSION_INFORMATION *)malloc(sizeof(SESSION_INFORMATION))) )
	{
		memset(sessionInfo, 0 ,sizeof(SESSION_INFORMATION));
		ccode = 1;
	}
	else
	{
		memset(sessionInfo, 0 ,sizeof(SESSION_INFORMATION));
		if (!(sessionInfo->dataBuffer = (BUFFER *)malloc(BUFFER_SIZE*sizeof(BUFFER))) )
			ccode = 1;
		else
		{
			sessionInfo->connection = 0;
			sessionInfo->selectionList = NULL;
			sessionInfo->dataFile = 0;
			sessionInfo->logFile = NULL;
			sessionInfo->clusterparams = NULL;
			sessionInfo->verbose = FALSE;
			sessionInfo->IsUTF8Supp = FALSE;
			sessionInfo->transferBuffer=NULL;
			sessionInfo->sessionHBuffer=NULL;
			sessionInfo->sessionHTrailer=NULL;
			sessionInfo->mediaBuffer=NULL;
			sessionInfo->currentPtr=NULL;
			sessionInfo->tUsed=0;
			sessionInfo->tLeft=0;
			sessionInfo->dataread=0;
			sessionInfo->atHeader=FALSE;
			sessionInfo->newDataSetFlag=0;
			sessionInfo->dataBufferPtr=NULL;
			sessionInfo->bctx = (BACKUP_CONTEXT *) malloc( sizeof(BACKUP_CONTEXT));
			memset( sessionInfo->bctx, 0, sizeof(BACKUP_CONTEXT));
			sessionInfo->bctx->appenBackup= FALSE;
			sessionInfo->bctx->multiVol= FALSE;
			sessionInfo->bctx->settapeLength= FALSE;
			sessionInfo->bctx->volno = 1;
			sessionInfo->bctx->extractFiles= FALSE;
			sessionInfo->bctx->nameSpaceType = DOSNameSpace;  
			sessionInfo->bctx->scanOnly = FALSE; 
			sessionInfo->bctx->saveBackup = TRUE;
			sessionInfo->bctx->dsetList = NULL;
			sessionInfo->rctx =  (RESTORE_CONTEXT*) malloc( sizeof(RESTORE_CONTEXT));
			memset( sessionInfo->rctx, 0, sizeof(RESTORE_CONTEXT));
			sessionInfo->rctx->overwriteParent = sessionInfo->rctx->overwriteChild = TRUE;
			sessionInfo->rctx->extractsession=FALSE;
			sessionInfo->rctx->multiVol= FALSE;
			sessionInfo->rctx->volno = 1;
			sessionInfo->stats.fileCount = 0;
			sessionInfo->stats.dirCount = 0;
			sessionInfo->stats.resCount = 0;
			sessionInfo->stats.totalBytesRead = 0;
			sessionInfo->stats.totalTime = 0;

			Init_Scan_Control(&(sessionInfo->scanControl));
		}
	}
	return(ccode == 0 ? sessionInfo : NULL);
}

/*********************************************************************
* NAME: ReleaseSessionInfoBuffer
*
* WHAT: Releases the memory allocated for the session information buffer.
*
* SMS-SPECIFIC ROUTINES CALLED: 
**********************************************************************/
void ReleaseSessionInfoBuffer(SESSION_INFORMATION *sessionInfo)
{
	if(sessionInfo)
	{
		if(sessionInfo->selectionList) 
			free(sessionInfo->selectionList);

		if(sessionInfo->dataBuffer) 
			free(sessionInfo->dataBuffer);

		if(sessionInfo->logFile) 
			fclose(sessionInfo->logFile);
		
		sessionInfo->logFile = NULL;

		if(sessionInfo->clusterparams) 
			free(sessionInfo->clusterparams);
		if(sessionInfo->bctx)
		{
			if(sessionInfo->bctx->dsetList)
				free(sessionInfo->bctx->dsetList);
			free(sessionInfo->bctx);
		}
		if(sessionInfo->rctx)
		{
			if(sessionInfo->rctx->renameList)
			{
				FreeRenameList(sessionInfo->rctx->renameList);
				sessionInfo->rctx->renameList = NULL;
			}
			free(sessionInfo->rctx);
		}
		free(sessionInfo);
		sessionInfo=NULL;
	}

	return;
}

/*****************************************************************************
* NAME: Insert
*
* WHAT: Insert a node into the link list.
*
* SMS-SPECIFIC ROUTINES CALLED:  
*
*****************************************************************************/
void Insert(SinglePair *head, SinglePair *pair)
{
	SinglePair *travel = NULL;
	
	if(head == NULL)
	{
		head = pair;
		pair = NULL;
	}
	else
	{
		travel = head;
		while(travel->next != NULL)
			travel = travel->next;
		travel->next = pair;
		pair = NULL;
	}
}

/*****************************************************************************
* NAME: FreeRenameList
*
* WHAT: Free the memory allocated for the data set rename list..
*
* SMS-SPECIFIC ROUTINES CALLED:  
*
*****************************************************************************/
void FreeRenameList(SinglePair *head)
{
	SinglePair *traverselist=NULL;

	while(head != NULL)
	{
		traverselist = head;
		head = head->next;

		if(traverselist->source)
		{
			free(traverselist->source);
			traverselist->source = NULL;
		}
		
		if(traverselist->destination)
		{
			free(traverselist->destination);
			traverselist->destination = NULL;
		}
		
		free(traverselist);
		traverselist = NULL;
	}

	return;
}

/*****************************************************************************
* NAME: GetNewName
*
* WHAT: Builds the final destination path for rename restore.
*
* SMS-SPECIFIC ROUTINES CALLED:  NWSMTSGetNameSpaceTypeInfo
*								NWSMGetDataSetName
*								NWSMCatStrings
*								NWSMFreeString
*								NWSMCloseName
*
*****************************************************************************/
CCODE GetNewName(SESSION_INFORMATION *sessionInfo, NWSM_DATA_SET_NAME_LIST *origNameList, 
					   NWSM_DATA_SET_NAME_LIST **newNameList  )
{
	SinglePair 				*pair, *found = NULL;
	NWSM_DATA_SET_NAME 		 origName;
	STRING_BUFFER 			*newName = NULL;
	int  					 notSame = 1;
	UINT32					 nameSpaceType;
	size_t					 sourceLen, destLen;
	utf8_t					*sourceUTF8 = NULL;
	utf8_t					*destUTF8 = NULL;
	SMS_HANDLE				 nameHandle = 0;
	NWBOOLEAN				 utf8DataFound = FALSE;
	CCODE 					 ccode = 0;
	NWBOOLEAN 				 rev;			
	STRING_BUFFER 			*sep1 = NULL, *sep2 = NULL;
	RESTORE_CONTEXT *rctx = sessionInfo->rctx;

	/* Loop through all rename pairs to find a match */
	for (pair = rctx->renameList; !found && pair; pair = pair->next)
	{
		/* Try the UTF-8 data set name first */
		nameSpaceType = MapToUTF8NameSpace(pair->nameSpace);
		if(sourceUTF8)
		{
			free(sourceUTF8);
			sourceUTF8 = NULL;
		}
		/* Get the data set name from the backed up list of names */
		if (!NWSMGetDataSetName((BUFFERPTR)origNameList, nameSpaceType, &origName))
		{
			utf8DataFound = TRUE;

			/* Convert the the names to UTF-8 from MBCS */
			ccode = ConvertLocToUTF8( pair->source, &sourceLen, &sourceUTF8);
			if (ccode)
			{
				continue;
			}
			/* Compare for a match */
			notSame = SMutf8nicmp((utf8_t *)sourceUTF8, (utf8_t *)origName.name, strlen(sourceUTF8));
			if (!notSame)
				found = pair;
		}
		else if(!NWSMGetDataSetName((BUFFERPTR)origNameList, pair->nameSpace, &origName))
		{
			/* Process the MBCS name as the UTF-8 name is not available with the backup
			 * Compare for a match */
			notSame = strnicmp(pair->source, (char *)origName.name, strlen(pair->source));
			if (!notSame)
				found = pair;
		}
		else 
			PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_GETNAME_ERR), nameSpaceType);
	}
	
	pair = found;
	if (pair)
	{
		/* If we have a UTF-8 name then cat the string and use the UTF-8 name space info to,
		 * populate the renamed name. Else use the MBCS names and info, to do the same */
		if(utf8DataFound)
		{
			/* Convert the the names to UTF-8 from MBCS */
			ccode = ConvertLocToUTF8( pair->destination, &destLen, &destUTF8);
			if (ccode)
			{
				return ccode;
			}
			NWSMCatStrings(2, &newName, destUTF8, origName.name + strlen(sourceUTF8));
			ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, nameSpaceType, &rev, &sep1, &sep2);
			if (ccode)		 
			{
				PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_NSTYPEINFO_ERR), ccode);
				return(ccode);
			}
			else					   
				ccode = NWSMPutOneName((void **)newNameList, nameSpaceType, 0, rev, (char *)sep1->string, (char *)sep2->string, (char *)newName->string);

			if(ccode)
			{
				PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_PUTNAME_ERR), ccode);
				return(ccode);
			}	
		}
		else
		{
			NWSMCatStrings(2, &newName, pair->destination, origName.name + strlen(pair->source));
			ccode = NWSMTSGetNameSpaceTypeInfo(sessionInfo->connection, pair->nameSpace, &rev, &sep1, &sep2);
			if (ccode)		 
			{
				PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_NSTYPEINFO_ERR), ccode);
				return(ccode);
			}
			else					   
				ccode = NWSMPutOneName( (void **)newNameList, pair->nameSpace, 0, rev, (char *)sep1->string, (char *)sep2->string, (char *)newName->string);

			if(ccode)
			{
				PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_PUTNAME_ERR), ccode);
				return(ccode);
			}	
		}
		
		NWSMFreeString(&sep1);
		NWSMFreeString(&sep2);
		NWSMFreeString(&newName);	
	}

	/* Cleanup local allocations and return */
	if(sourceUTF8)
		free(sourceUTF8);
	
	if(destUTF8)
		free(destUTF8);
	
	if(*newNameList)
		NWSMCloseName(&nameHandle);
	
	return(ccode);
}
/*****************************************************************************
* NAME: CopyResourceName
*
* WHAT: Copies the Resource(directory/file) name.
*
* SMS-SPECIFIC ROUTINES CALLED:  NWSMStripPathChild
*
*****************************************************************************/

CCODE CopyResourceName( UINT8 parentFlag,char *oNameList, SESSION_INFORMATION *sessionInfo,NWBOOLEAN isUTFName, char **extractPath)
{

	CCODE ccode=0;
	size_t destLen;
	char *copyPath=NULL;
	char 	filePath[1024];
	
	if(!isUTFName)
	{
		/* Convert the the names to UTF-8 from MBCS */
		ccode = ConvertLocToUTF8(oNameList, &destLen, extractPath);
		if( ccode)
		{
			*extractPath = malloc(strlen(oNameList) +1 );
			if( *extractPath == NULL)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
				ccode=1;
				goto Return;
			}
			strcpy(*extractPath, oNameList);
			ccode = 0;
		}
	}
	else
		*extractPath = oNameList;

	/*Copy only the file name in case the path is fully qualified*/
	if(!parentFlag && (sessionInfo->recInfo.pathIsFullyQualified))
	{
		copyPath = malloc(strlen(oNameList) +1 );
		if( copyPath == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
		strcpy(copyPath,oNameList);
		NWSMStripPathChild(sessionInfo->nameSpaceType,copyPath,filePath,1024);
		strcpy(*extractPath, filePath);
		if(copyPath)
		{
			free(copyPath);
			copyPath=NULL;
		}
	}

	Return:
	return (ccode);
}
/*****************************************************************************
* NAME: IsFileExtracted
*
* WHAT: Decides whether a particular dir/file needs to be extracted or not.
*
* SMS-SPECIFIC ROUTINES CALLED:  NWSMStripPathChild
*								NWSMGetListHead
*
*****************************************************************************/

CCODE IsFileExtracted( UINT8 parentFlag, NWBOOLEAN *isFirstFile, SESSION_INFORMATION *sessionInfo, char *extractPath, char **listPath, char *fileWithOutPath, NWSM_LIST **cur,NWSM_LIST **savecur, NWBOOLEAN *isParentFound,  NWBOOLEAN *isExtract)
{

	CCODE ccode=0;
	char *listcurPath=NULL;
	
	if(sessionInfo->bctx->extractFiles)
	{
	
		/*No need to extract a file under the path if it doesn't match with any of the path
		* mentioned by the user*/
		if(!parentFlag && !(*isParentFound))
			goto Return;
		if(!parentFlag && !*isFirstFile)
		{
			*cur=*savecur;
			goto NextLink;
		}
		
		if(!parentFlag)
		{
			*isFirstFile=FALSE;
			/*Compares the file name(in the sidf archive) with the file name which is specified by the user.File is extracted if
			* the comparison is successful*/
			if (!strcmp(fileWithOutPath, extractPath))
    			{
				*isExtract=TRUE;
				goto Return;
    			}
			else
			{
				/*Go to next path specified by the user to extract a file from archive*/
				*cur = (*cur)->next;
				NextLink:
				while(*cur)
				{
					listcurPath= malloc(strlen((*cur)->text) +1 );
					if( listcurPath == NULL)
					{
						PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
						ccode=NBK_OUT_OF_MEMROY;
						goto Return;
					}
					strcpy(listcurPath, (*cur)->text);
					NWSMStripPathChild(sessionInfo->nameSpaceType,listcurPath,fileWithOutPath,1024);
					if (!strcmp(*listPath, listcurPath))
		        		{
						/*See if the file name match with the name of the file which user has mentioned*/
						if (!strcmp(fileWithOutPath, extractPath))
		        			{
							*isExtract=TRUE;
							if(listcurPath)
							{
								free(listcurPath);
								listcurPath=NULL;
							}
							goto Return;
		        			}
						else
						{
							*cur = (*cur)->next;
							if(listcurPath)
							{
								free(listcurPath);
								listcurPath=NULL;
							}
						}
					}
					else
					{
						*cur = (*cur)->next;	
						if(listcurPath)
						{
							free(listcurPath);
							listcurPath=NULL;
						}
					}
				}
				*isExtract=FALSE;
				goto Return;
			}
		}
		/*Gives the first file path which needs to be extracted*/
		*cur = NWSMGetListHead(inFiles);
		*listPath= malloc(strlen((*cur)->text) +1 );
		if( *listPath == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
		strcpy(*listPath, (*cur)->text);
		/*Strips the child name from the path*/
		NWSMStripPathChild(sessionInfo->nameSpaceType,*listPath,fileWithOutPath,1024);
		/*Compares the stripped path with the sidf path*/
		if (!strcmp(*listPath, extractPath))
    		{
			*isExtract=TRUE;
			*isParentFound=TRUE;
			*savecur=*cur;
			goto Return;
    		}
		if(*listPath)
		{
			free(*listPath);
			*listPath=NULL;
		}
		/*Comparison is done untill and unless there is no new path(specified by the user) which needs to be extracted */
		while(*cur)
		{
			*cur = (*cur)->next;
			if(*cur)
			{
				*listPath= malloc(strlen((*cur)->text) +1 );
				if( *listPath == NULL)
				{
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
					ccode=NBK_OUT_OF_MEMROY;
					goto Return;
				}
				strcpy(*listPath, (*cur)->text);
				NWSMStripPathChild(sessionInfo->nameSpaceType,*listPath,fileWithOutPath,1024);
				if (!strcmp(*listPath, extractPath))
        			{
					*isExtract=TRUE;
					*isParentFound=TRUE;
					*savecur=*cur;
					goto Return;
        			}
				if(*listPath)
				{
					free(*listPath);
					*listPath=NULL;
				}
			}
		}
		*isExtract=FALSE;
		*isParentFound=FALSE;
	}

	Return:
	return (ccode);

}

/*****************************************************************************
* NAME: GetResourceName
*
* WHAT: Returns the resource name in specified name space type
*
* SMS-SPECIFIC ROUTINES CALLED:  NWSMGetDataSetName
*
*****************************************************************************/
void GetResourceName(NWSM_DATA_SET_NAME_LIST *resourceName, NWBOOLEAN checkUTF8 , NWSM_DATA_SET_NAME *nNameList,  NWBOOLEAN *isUTFName)
{

	if (!NWSMGetDataSetName((BUFFERPTR)resourceName, LONGNameSpaceUtf8Type, nNameList))
		{
			if(checkUTF8)
				*isUTFName=TRUE;	
		}
	else if (!NWSMGetDataSetName((BUFFERPTR)resourceName, NFSNameSpaceUtf8Type, nNameList))
		{
			if(checkUTF8)
				*isUTFName=TRUE;	
		}
	else if (!NWSMGetDataSetName((BUFFERPTR)resourceName, OS2NameSpace, nNameList)) 
		{
		}
	else if (!NWSMGetDataSetName((BUFFERPTR)resourceName, NFSNameSpace, nNameList))
		{
		}
}

/*****************************************************************************
* NAME: PrintDataSetName
*
* WHAT: Prints the data set name.
*
* SMS-SPECIFIC ROUTINES CALLED:  NWSMTSSeparateDataSetName
*
*****************************************************************************/
CCODE  PrintDataSetName( UINT8 pFlag, char **lParent, char *NextRName, SESSION_INFORMATION *sessionInfo, char *lPath, NWBOOLEAN isBackup)
{

	CCODE ccode=0;
	STRING_BUFFER 	*resourceParentDataSetName = NULL, *resourceChildDataSetName = NULL;
	
	if(pFlag)
	{
		if( *lParent)
		{
			free(*lParent);
			*lParent = NULL;
		}
		*lParent = malloc(strlen(NextRName) +1 );
		if( *lParent == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;;
		}
		strcpy(*lParent, NextRName);
		/*Prints the directory name (full path) */
		if( strlen(*lParent) > 1024)
			PrintMsg(NBK_VERBOSE_MSG, "\n%.1024s ... ", *lParent);
		else
			PrintMsg(NBK_VERBOSE_MSG, "\n%s", *lParent );

	}
	else
	{
		/*if the path is fully qualified*/
		if(((!sessionInfo->scanControl.returnChildTerminalNodeNameOnly) && isBackup) || (sessionInfo->recInfo.pathIsFullyQualified && (!isBackup)))
		{
			if( strlen(NextRName) > 1024)
			{
				NWSMTSSeparateDataSetName(sessionInfo->connection, sessionInfo->nameSpaceType,lPath, &resourceParentDataSetName, &resourceChildDataSetName);  
				/*Prints the file name (full path) */
				PrintMsg(NBK_VERBOSE_MSG, "\n%.1024s ... %.256s", resourceParentDataSetName->string,resourceChildDataSetName->string);
				if (resourceParentDataSetName) 
					NWSMFreeString(&resourceParentDataSetName);
				if (resourceChildDataSetName) 
					NWSMFreeString(&resourceChildDataSetName); 
			}
			else
				PrintMsg(NBK_VERBOSE_MSG, "\n%s", NextRName);
		}
		/*if the path is not fully qualified*/
		else
		{
			/* if the target is a directory, the resource will
			 * always be fully qualified DN, no need to 
			 * put the parent's name.
			 */
			if(strstr(sessionInfo->srcName,"Directory")||(strstr(sessionInfo->tsaName,"Directory")))
				PrintMsg(NBK_VERBOSE_MSG, "\n%s%s", NextRName,*lParent);
			 if(*lParent)
			{
				if( strlen(*lParent) > 1024)
					PrintMsg(NBK_VERBOSE_MSG, "\n%.1024s ... %.256s", *lParent, NextRName);
				else
					PrintMsg(NBK_VERBOSE_MSG, "\n%s%.256s", *lParent, NextRName );
			}
		}
	}	

	Return:
	return (ccode);
	
}

/*********************************************************************
* NAME: UngetEOF
*
* WHAT: Set ungetEOF Flag 
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void UngetEOF(void)
{
	ungetEOFFlag = TRUE;
}

/*********************************************************************
* NAME: OpenDEFiles
*
* WHAT: Open an index file and a data file for storing backup data from
*	a Target Service.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE OpenDEFiles(int   mode, SESSION_INFORMATION *sessionInfo )
{
	CCODE  ccode = 0;
	char  dataFileName[MAX_FILE_NAME_LENGTH];

	if (sessionInfo->dataFile)
	{
		ccode = 1;
		goto Return;
	}

	/* check if file name is present in the user options. 
	 * Generate a name for the file, Since GUI will not give the file name*/
	if( sessionInfo->destFile[0] == 0)
	{
#ifdef N_PLAT_NLM
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_NO_FILENAME));
		ccode = 1;
		goto Return;
#elif defined(N_PLAT_UNIX)
		if(sessionInfo->rctx->extractsession)
			sessionInfo->dataFile=STDIN_FILENO;
		else	
			sessionInfo->dataFile=STDOUT_FILENO;
#endif
	}
	else
	{
#ifdef N_PLAT_UNIX
		mode |= O_LARGEFILE;
#endif
		strcpy(dataFileName, sessionInfo->destFile);
		if ((sessionInfo->dataFile = open(dataFileName, mode,S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH)) == -1)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_FILE_OPEN_FAILED), dataFileName);
			ccode = 1;
		}
	}
	if (ccode)
		CloseDEFiles(sessionInfo);

Return:
	return (ccode);
}

/*********************************************************************
* NAME: CloseDEFiles
*
* WHAT: Close the index and data files opened with OpenDEFiles.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void CloseDEFiles(SESSION_INFORMATION *sessionInfo)
{
	if ((sessionInfo->dataFile != -1) && close(sessionInfo->dataFile))
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_CLOSE_FAIL));
	sessionInfo->dataFile = 0;
}

/*********************************************************************
* NAME: SaveSessionInfo
*
* WHAT: Save the information for the current backup session.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE SaveSessionInfo(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;
	FILE *tempFilePtr;
	char sessionName[MAX_SCREEN_NAME_LENGTH]={' '};

	/* If de-sessn.inf already exists, then open it in read-append mode, else
 	 * it must be created . */
	if ( sessionInfoFile || ((sessionInfoFile = (tempFilePtr = fopen("de-sessn.inf","r+"))==NULL ? fopen("de-sessn.inf","w+"):tempFilePtr) == NULL) )
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_FILE_OPEN_FAILED), "de-sessn.inf");
		ccode=1;
		goto Return;
	}
	
	/*Store the session name at offset based on the session number of this
	 * session, from the start of file de-sessn.inf .*/
	GetNameOfThread(sessionName);
	fseek(sessionInfoFile, (sessionInfo->sessionNo) * MAX_SCREEN_NAME_LENGTH,SEEK_SET);
	
#ifdef N_PLAT_NLM
	if (fwrite(sessionName, MAX_SCREEN_NAME_LENGTH * sizeof(char), 1, sessionInfoFile) == 0)
	{
		perror("Session File Write Error\n");
		ccode=1;
	}
#elif defined(N_PLAT_UNIX)
	if (fwrite(sessionInfo->sName, MAX_SCREEN_NAME_LENGTH * sizeof(char), 1, sessionInfoFile) == 0)
	{
		perror("Session File Write Error\n");
		ccode=1;
	}
#endif /* N_PLAT_NLM */

	if(sessionInfoFile)
		fclose(sessionInfoFile);
	sessionInfoFile = NULL;
Return:
	return(ccode);
}


/*********************************************************************
* NAME: ObtainSessionInfo
*
* WHAT: Obtain the list of available backup sessions that can be restored.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE ObtainSessionInfo(int *sessionToRestore)
{
	CCODE ccode = 0, count = 0;
	int sessionNo;
	char sessionName[MAX_SCREEN_NAME_LENGTH];
	char choice[5] = {0};
	int x,y;

	if (sessionInfoFile || ((sessionInfoFile = fopen("de-sessn.inf","r")) == NULL) )
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_FILE_OPEN_FAILED), "de-sessn.inf");
		ccode = 1;
		goto Return;
	}

	/*Get the list of all available sessions.*/
	printf("\n\nList of Available Sessions :\n");
	while ((fread(&sessionName, MAX_SCREEN_NAME_LENGTH * sizeof(char), 1, sessionInfoFile) != 0))
	{
		if (ferror(sessionInfoFile))
		{
			ccode = 1;
			goto Return;
		}
		else
			printf("%d. %s\n",++count,sessionName);
	}
	printf("\nEnter the Choice of Session to be Restored ('q' to Escape):");
	{
		CursorPosition(&x, &y);
		while (TRUE)
		{
			gets(choice);
			if ((strcmp(choice,"q")) == 0)
			{ 
				*sessionToRestore = -1; 
				ccode = 1; 
				goto Return; 
			}
			sessionNo = atoi(choice);
			if (!((sessionNo < 1) || (sessionNo > count)))
			{
				*sessionToRestore = sessionNo - 1;
				break;
			}
			ConsoleSpeakerToBeep();
			SetCursor(x,y);
		}
	}
Return:
	if (sessionInfoFile) 
		fclose(sessionInfoFile);
	sessionInfoFile = NULL;
	return(ccode);
}

/*********************************************************************
* NAME: LogSessionInfo
*
* WHAT: Log the output of this session to the log file if the user has
*		requested.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void LogSessionInfo(UINT8 *toLog, SESSION_INFORMATION *sessionInfo)
{
	char  logFileNo[5];
	char logFileName[MAX_FILE_NAME_LENGTH];

	*toLog = !(*toLog);
	if (toLog && !(sessionInfo->logFile))
	{
		IntegerToString( sessionInfo->sessionNo, logFileNo, 10,"%d");
		strcpy(logFileName, "De-");
		strcat(logFileName, logFileNo);
		strcat(logFileName, ".log");
		if (!(sessionInfo->logFile = fopen(logFileName,"w")))
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_FILE_OPEN_FAILED), logFileName);
			*toLog = !(*toLog);
		}
	}
	else if (!toLog)
	{
		if (sessionInfo->logFile) 
			fclose(sessionInfo->logFile);
		sessionInfo->logFile=NULL;
	}
	return;
}

/*********************************************************************
* NAME: MapToUTF8NameSpace
*
* WHAT: Converts MBCS nameSpaceType to UTF8 nameSpaceType
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
UINT32 MapToUTF8NameSpace(UINT32 nameSpaceType)
{
	if (nameSpaceType == DOSNameSpace)
		return DOSNameSpaceUtf8Type;
	if (nameSpaceType == MACNameSpace)
        return MACNameSpaceUtf8Type;
	if (nameSpaceType == NFSNameSpace)
        return NFSNameSpaceUtf8Type;
	if (nameSpaceType == OS2NameSpace)
        return LONGNameSpaceUtf8Type;

    return nameSpaceType;
}

/*********************************************************************
* NAME: MapToMBCSNameSpace
*
* WHAT: Converts UTF8 nameSpaceType to MBCS nameSpaceType.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
UINT32 MapToMBCSNameSpace(UINT32 nameSpaceType) 
{
    if (nameSpaceType == DOSNameSpaceUtf8Type)
        return DOSNameSpace;
	
    if (nameSpaceType == MACNameSpaceUtf8Type)
        return MACNameSpace;
	
    if (nameSpaceType == NFSNameSpaceUtf8Type)
        return NFSNameSpace;
	
    if (nameSpaceType == LONGNameSpaceUtf8Type)
        return OS2NameSpace;
	
	return nameSpaceType;
} 

/*********************************************************************
* NAME: StartTimer
*
* WHAT: Start the timer used by the ElapsedTime function.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void StartTimer(void)
{	
#ifdef N_PLAT_NLM
	begin_time = clock();
#else
	gettimeofday(&tv, &tz);
	begin_time = tv.tv_sec;
#endif	
}

/*********************************************************************
* NAME: ElapsedTime
*
* WHAT: Return the elapsed time (in milliseconds) since the StartTimer
*	function was last called.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
UINT32 ElapsedTime(void)
{
	UINT32 time_now;
#ifdef N_PLAT_NLM
	time_now = clock();
#else
	gettimeofday(&tv, &tz);
	time_now = tv.tv_sec;
#endif	
	
#ifdef N_PLAT_NLM
	return((UINT32)((time_now - begin_time) / CLOCKS_PER_SEC) );
#elif defined N_PLAT_UNIX
	return((UINT32)((time_now - begin_time)));
#endif
}

/*
 *	NOTE: This is just an example of how to replace ftime(), the timer
 *	granularity is not as good as ftime() -- hundredths of seconds instead
 *	of milliseconds.
 */
/*****************************************************************************
* NAME: CheckDEUnload
*
* WHAT: Checks if there are active session on unload and prompts a warning message.
*
* SMS-SPECIFIC ROUTINES CALLED:  
*
*****************************************************************************/
#ifdef N_PLAT_NLM
int CheckDEUnload(void)
{
	if (safeUnloadFlag) PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_UNLOADING_WARN));
	return safeUnloadFlag;
}
#endif

/*********************************************************************
* NAME: PutDataSetHeader
*
* WHAT: Puts the RecordHeader into the current transfer buffer
* 		as well as initialises the recInfo structure.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMSetNewRecordHeader
*								NWSMSetNewRecordHeader
*
**********************************************************************/
CCODE PutDataSetHeader(NWSM_DATA_SET_NAME_LIST *dataSet, SESSION_INFORMATION *sessionInfo)
{

	CCODE ccode=0;
	
	sessionInfo->recInfo.dataSetName = dataSet;
	sessionInfo->recInfo.scanInformation = sessionInfo->scanInfo;
	sessionInfo->recInfo.dataSetIsInvalid = FALSE;
	ccode=PutDataHeader(FALSE,sessionInfo);
	if(ccode)
		return (ccode);
	sessionInfo->recInfo.scanInformation = NULL;
	sessionInfo->recInfo.dataSetName	   = NULL;

	return(ccode);	
}

/*********************************************************************
* NAME: PutDataHeader
*
* WHAT: Puts the RecordHeader into the current transfer buffer
* 		If the buffer is not sufficient, it calls FlushTransferBuffer()
*
* SMS-SPECIFIC ROUTINES CALLED: 	NWSMSetNewRecordHeader
*								NWSMSetNewRecordHeader
*								NWSMPadBlankSpace
*
**********************************************************************/
CCODE PutDataHeader(NWBOOLEAN subRecord, SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;
	
	sessionInfo->recInfo.isSubRecord = subRecord;
	if (sessionInfo->recInfo.scanInformation && (sessionInfo->recInfo.scanInformation->parentFlag || (!sessionInfo->scanControl.returnChildTerminalNodeNameOnly) ))
		sessionInfo->recInfo.pathIsFullyQualified = TRUE;
	else
		sessionInfo->recInfo.pathIsFullyQualified = FALSE;
	
	/*Creates a record or a sub record header*/
	ccode = NWSMSetNewRecordHeader(
							&(sessionInfo->currentPtr), &(sessionInfo->tLeft), &(sessionInfo->tUsed), FALSE, &(sessionInfo->recInfo));
	// Not enough space to write the header.
	if(ccode)
	{
		if (ccode == NWSMUT_BUFFER_OVERFLOW)
		{

			/*Write the current transfer buffer to the archive*/
			ccode = FlushTransferBuffer(sessionInfo);
			if(ccode)
			{
				return(ccode);
			}
			/*Creates a record or a sub record header*/
			ccode = NWSMSetNewRecordHeader(
							&(sessionInfo->currentPtr), &(sessionInfo->tLeft), &(sessionInfo->tUsed), FALSE, &(sessionInfo->recInfo));
			if (ccode)
			{
				PrintError(ccode, "NWSMSetNewRecordHeader", __LINE__,sessionInfo);
				return(ccode);
			}
		}
		else
		{
			PrintError(ccode, "NWSMSetNewRecordHeader", __LINE__,sessionInfo);
			return(ccode);	
		}
	}
	return (ccode);

}

/*********************************************************************
* NAME: FlushTransferBuffer
*
* WHAT: Write the current transfer buffer to the media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMPadBlankSpace
*
**********************************************************************/
CCODE FlushTransferBuffer(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode = 0;
	
	if ( sessionInfo->dataFile == -1)
	{
		ccode = 1;
		return (ccode);
	}
	/* Pad unused buffer area*/
	NWSMPadBlankSpace( sessionInfo->currentPtr, sessionInfo->tLeft);
	sessionInfo->ControlBlock.transferBufferSizeData =sessionInfo->tUsed;
	sessionInfo->ControlBlock.sessionDataType = NWSMSD_SDT_TSA_DATA;
	sessionInfo->ControlBlock.transferBufferState = NWSMSD_TBS_READY_TO_TRANSFER;
	sessionInfo->ControlBlock.transferBufferSequence=0;
	/*Writes data from the transfer buffer to archive*/
	ccode= SessionWriteData(sessionInfo);
	if(ccode)
	{
		PrintError(ccode, "SessionWriteData", __LINE__,sessionInfo);
		return(ccode);
	}

	sessionInfo->currentPtr =sessionInfo->transferBuffer;
	sessionInfo->tUsed =sessionInfo->transferBufferInfo.transferBufferDataOffset;;
	sessionInfo->tLeft = sessionInfo->transferBufferInfo.maxTransferBufferSize -
		sessionInfo->transferBufferInfo.transferBufferDataOffset;
	sessionInfo->currentPtr += sessionInfo->tUsed;

	return(ccode);

}

/*********************************************************************
* NAME: FlushSessionHeader
*
* WHAT: Write the session header information to the media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:	
*
**********************************************************************/
CCODE FlushSessionHeader(SESSION_INFORMATION *sessionInfo)
{

	ssize_t datawriten=0;
	CCODE ccode = 0;

	if ( sessionInfo->dataFile == -1)
	{
		ccode = 1;
		return (ccode);
	}
	/*write it to archive*/
	if(sessionInfo->bctx->settapeLength)
	{
		if( sessionInfo->tape_length - TotalDataWritten < 4)
		{
			TotalDataWritten=0;
			ccode = ContinueMediaHeaderToNextTape(sessionInfo,  &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}	
		}
	}
	writedata:
	datawriten=write(sessionInfo->dataFile, sessionInfo->sessionHBuffer, SECTOR_SIZE*sizeof(BUFFER));
	if(sessionInfo->bctx->settapeLength && (datawriten == (SECTOR_SIZE*sizeof(BUFFER))))
	{
		TotalDataWritten += 4;
	}
	if(datawriten != (SECTOR_SIZE*sizeof(BUFFER)))
	{

		if(sessionInfo->bctx->multiVol )
		{
			TotalDataWritten=0;
			ccode=ContinueMediaHeaderToNextTape(sessionInfo , &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}
			goto writedata;
			
		}
		PrintError(ccode, "write", __LINE__,sessionInfo);
		ccode=1;
		return(ccode);
	}
	Return:
		
	return(ccode);

}

/*********************************************************************
* NAME: FlushMediaHeader
*
* WHAT: Write the media header information to the media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:	
*
**********************************************************************/
CCODE FlushMediaHeader(SESSION_INFORMATION *sessionInfo)
{

	ssize_t datawriten=0;
	CCODE ccode = 0;
	
	if (  sessionInfo->dataFile == -1)
	{
		ccode = 1;
		return (ccode);
	}
	/*write it to archive*/
	if(sessionInfo->bctx->settapeLength)
	{
		if( sessionInfo->tape_length - TotalDataWritten < 4)
		{
			TotalDataWritten=0;
			ccode = ContinueToNextTape(sessionInfo,  &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}	
		}
	}
	writedata:
	datawriten=write(sessionInfo->dataFile, sessionInfo->mediaBuffer, SECTOR_SIZE*sizeof(BUFFER));
	if(sessionInfo->bctx->settapeLength && (datawriten == (SECTOR_SIZE*sizeof(BUFFER))))
	{
		TotalDataWritten += 4;
	}
	if(datawriten != (SECTOR_SIZE*sizeof(BUFFER)))
	{
		if(sessionInfo->bctx->multiVol  )
		{
			sessionInfo->bctx->volno ++;
			CloseDEFiles(sessionInfo);
			PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE), sessionInfo->bctx->volno, sessionInfo->destFile);
			getchar();
			if(sessionInfo->bctx->saveBackup )
			{
				ccode= OpenDEFiles(O_CREAT|O_WRONLY|O_TRUNC,sessionInfo);
				if(ccode)
					goto Return;
			}
			goto writedata;
			
		}
		PrintError(ccode, "write", __LINE__,sessionInfo);
		ccode=1;
		return(ccode);
	}
	Return:
	return(ccode);

}

/*********************************************************************
* NAME: FlushSessionTrailer
*
* WHAT: Write the session trailer information to the media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMPadBlankSpace
*
**********************************************************************/
CCODE FlushSessionTrailer(SESSION_INFORMATION *sessionInfo)
{

	ssize_t	datawriten=0;
	CCODE	ccode = 0;
   	UINT32	bufferSize;
  
	BUFFERPTR                         addressOfData, bufferPtr;
	UINT64                            save16 = UINT64_ZERO;
	UINT16                            offsetToEnd, sync;

	sync = NWSM_SYNC_DATA;
	bufferPtr = sessionInfo->sessionHTrailer;
	bufferSize = sessionInfo->transferBufferInfo.sectorSize;
	/*Puts the data across the sidf tag*/
	 if ((ccode = PutField(  SIDF_SESSION_TRAILER,
                              &sync,
                               sizeof(sync),
                               sizeof(sync),
                               NULL,
                              &bufferPtr,
                              &bufferSize,
                               NULL)) != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	SetUINT64Value(&save16, sizeof(UINT16));

       if ((ccode = PutField(
                           SIDF_OFFSET_TO_END,
                           &save16,
                           sizeof(UINT16),
                           sizeof(UINT16),
                           &addressOfData,
                           &bufferPtr,
                           &bufferSize,
                           NULL)) != 0)
       {
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
       }

       memcpy(
           bufferPtr,
           sessionInfo->sHandle.tbHeaderPtr +
                sessionInfo->sHandle.sessionTrailerInfoOffset,
            sessionInfo->sHandle.sessionTrailerInfoSize);

	bufferPtr +=   sessionInfo->sHandle.sessionTrailerInfoSize;
	bufferSize -=  sessionInfo->sHandle.sessionTrailerInfoSize;
	offsetToEnd = bufferPtr - (addressOfData + sizeof(UINT16));
       memcpy(addressOfData, &offsetToEnd, sizeof(UINT16));
       if ((ccode = PutField(
                            SIDF_SESSION_TRAILER,
                            NULL,
                            0,
                            0,
                            NULL,
                           &bufferPtr,
                           &bufferSize,
                            NULL)) != 0)
       {
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
       }
	/* Pad unused buffer area*/
       NWSMPadBlankSpace(bufferPtr, bufferSize);

	if (  sessionInfo->dataFile == -1)
	{
		ccode = 1;
		return (ccode);
	}
	/*write it to archive*/
	if(sessionInfo->bctx->settapeLength)
	{
		if( sessionInfo->tape_length - TotalDataWritten < 4)
		{
			TotalDataWritten=0;
			ccode = ContinueToNextTape(sessionInfo,  &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}	
		}
	}
	writedata:
	datawriten=write(sessionInfo->dataFile, sessionInfo->sessionHTrailer, SECTOR_SIZE*sizeof(BUFFER));
	if(sessionInfo->bctx->settapeLength && (datawriten == (SECTOR_SIZE*sizeof(BUFFER))))
	{
		TotalDataWritten += 4;
	}
	if(datawriten != (SECTOR_SIZE*sizeof(BUFFER)))
	{
		
		if(sessionInfo->bctx->multiVol  )
		{
			TotalDataWritten=0;
			ccode = ContinueToNextTape(sessionInfo , &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}
			goto writedata;	
		}
		PrintError(ccode, "write", __LINE__,sessionInfo);;
		ccode=1;
		return(ccode);
	}

	Return:
	return(ccode);

}

/*********************************************************************
* NAME: PutData
*
* WHAT:Writes size- bytes into the transfer buffer
* 		If no space in TB, it calls FlushTransferBuffer()
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMUpdateRecordHeader
*
**********************************************************************/
CCODE PutData(SESSION_INFORMATION *sessionInfo, UINT32 size)
{
	CCODE ccode = 0;
	UINT32 dataWritten , dataToWrite;
	
	if ( sessionInfo->dataFile == -1)
		ccode = 1;
	else
	{
		sessionInfo->recInfo.dataSetIsInvalid=FALSE;
		dataWritten = (size <sessionInfo->tLeft)?size:sessionInfo->tLeft;
		dataToWrite = (size > sessionInfo->tLeft)?(size-sessionInfo->tLeft):0;
		memcpy(sessionInfo->currentPtr, sessionInfo->dataBufferPtr,dataWritten);
		sessionInfo->tLeft -= dataWritten;
		sessionInfo->tUsed += dataWritten;
		sessionInfo->currentPtr += dataWritten;
		sessionInfo->recInfo.recordSize += dataWritten;
		if (!sessionInfo->tLeft)
		{
			/*Update record/subrecord size and CRC information*/
			NWSMUpdateRecordHeader(&(sessionInfo->recInfo));
			/*Flush the current transfer buffer to the archive.*/
			ccode=FlushTransferBuffer(sessionInfo);
			if(ccode)
				goto Return;
			ccode=PutDataHeader(TRUE,sessionInfo);
			if(ccode)
				goto Return;
			if (dataToWrite)
			{
				sessionInfo->dataBufferPtr += dataWritten;
				PutData(sessionInfo, dataToWrite);
			}
		}				
	}

	Return:
		return (ccode);
}

/*********************************************************************
* NAME: DataEnd
*
* WHAT:Update the record header information when the backup of a dataset is complete.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMUpdateRecordHeader
*
**********************************************************************/
void DataEnd(SESSION_INFORMATION *sessionInfo)
{
	/*Update record/subrecord size and CRC information*/
	NWSMUpdateRecordHeader(&(sessionInfo->recInfo));
}

/*********************************************************************
* NAME: GetDataHeader
*
* WHAT: Checks whether the current data set is the last data set.
*
* SMS-SPECIFIC ROUTINES CALLED:NWSMGetRecordHeaderOnly
*
**********************************************************************/
int GetDataHeader(UINT8 *pflag, UINT8 *tflag, NWSM_DATA_SET_NAME_LIST **newList,SESSION_INFORMATION *sessionInfo ,CCODE *error)
{
	int rcode=1;
	CCODE ccode=0;

	if (sessionInfo->atHeader)
	{
		/*Returns record/subrecord along with dataset information.*/
		ccode=GetRecordInfo(sessionInfo);
		if(ccode)
		{
			/*signifies no more data set in the archive*/
			if(sessionInfo->ControlBlock.transferBufferSequence==END_OF_RECORD_MARKER)
				return 0;
			else
				*error=1;
		}
	}
	
	if( sessionInfo->recInfo.scanInformation)
	*pflag = sessionInfo->recInfo.scanInformation->parentFlag;
	//Here we need to get the correct tflag
	if( sessionInfo->recInfo.dataSetName)
	*newList= sessionInfo->recInfo.dataSetName;
	
	
	return(rcode);

}

/*********************************************************************
* NAME: GetRecordInfo
*
* WHAT:Returns record/subrecord along with dataset information.
*
* SMS-SPECIFIC ROUTINES CALLED:	NWSMGetRecordHeaderOnly
*								NWSMGetDataSetInfo
*
**********************************************************************/
CCODE GetRecordInfo(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;

	if ((int)(sessionInfo->tLeft) <= 0)
	{
		/*Read the transfer buffer from the archive*/
		if(!GetTransferBuffer( sessionInfo))
		{
			if(sessionInfo->dataread != SECTOR_SIZE)
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
			ccode=1;
			return(ccode);		
		}
	}
	/*Get the record/subrecord header information */
	ccode = NWSMGetRecordHeaderOnly(&(sessionInfo->currentPtr), &(sessionInfo->tLeft), &(sessionInfo->recInfo));
	if(ccode)
	{
		if (ccode == NWSMUT_BUFFER_UNDERFLOW)
		{
			/*Read the transfer buffer from the archive*/
			if(!GetTransferBuffer( sessionInfo))
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
				ccode=1;
				return(ccode);		
			}
			/*Get the record/subrecord header information */
			ccode = NWSMGetRecordHeaderOnly(&(sessionInfo->currentPtr), &(sessionInfo->tLeft), &(sessionInfo->recInfo));
			if (ccode)
			{
				PrintError(ccode, "NWSMGetRecordHeaderOnly", __LINE__, sessionInfo);
				return(ccode);
			}	
		}
		else
		{
			PrintError(ccode, "NWSMGetRecordHeaderOnly", __LINE__, sessionInfo);
			return(ccode);	
		}
	}

	if(sessionInfo->ControlBlock.transferBufferSequence == END_OF_RECORD_MARKER)
		return 1;

	if ((sessionInfo->recInfo.dataSetInfoRetrieved == DATA_SET_INFO_NOT_STARTED)
	|| (sessionInfo->recInfo.dataSetInfoRetrieved == DATA_SET_INFO_SPANNED))
	{
		/*Get the data set information*/
		ccode  = NWSMGetDataSetInfo(&sessionInfo->currentPtr, &sessionInfo->tLeft, &sessionInfo->recInfo);
		if (ccode)
		{
			PrintError(ccode, "NWSMGetDataSetInfo", __LINE__,sessionInfo);
			return(ccode);
		}
		
	}
	sessionInfo->tUsed = BUFFER_SIZE*sizeof(BUFFER) -sessionInfo->tLeft;
	sessionInfo->atHeader = FALSE;

	return(ccode);
}

/*********************************************************************
* NAME: GetTransferBuffer
*
* WHAT:Read the current transfer buffer from the media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
NWBOOLEAN GetTransferBuffer(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;
	NWBOOLEAN more = TRUE;

	if(sessionInfo->ControlBlock.transferBufferSequence==0xffffffff)
	{
		ccode=1;
		goto Return;	
	}
	sessionInfo->ControlBlock.transferBufferSequence = 0xffffffff;
	/*Read data from archive and parse transfer buffer header information.*/
	ccode=SessionReadData( sessionInfo);
	if(ccode)
	{
		more = FALSE;
		goto Return;	
	}
		
	if (sessionInfo->ControlBlock.transferBufferSequence == 0xffffffff)
	{
		more = FALSE;
	}
	
	Return:
	return(more);
	
}

/*********************************************************************
* NAME: GetData
*
* WHAT:Get the size as well as buffer containing the data.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
BUFFERPTR GetData (UINT32 *size, SESSION_INFORMATION *sessionInfo)
{
	BUFFERPTR buffer = NULL;

	if(!NewDataSet(sessionInfo))
			buffer = RestoreRead(sessionInfo, size);

	return buffer;
}

/*********************************************************************
* NAME: NewDataSet
*
* WHAT:Checks whether the data set is new or it is the continuation of old data set.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
NWBOOLEAN NewDataSet(SESSION_INFORMATION *sessionInfo)
{
	CCODE rcode =0;
	
	if (sessionInfo->newDataSetFlag)
	{
		sessionInfo->newDataSetFlag = 0;
		return FALSE;
	}

	if (sessionInfo->atHeader)
	{
		/*Get record/subrecord along with dataset information.*/
		rcode = GetRecordInfo(sessionInfo);
		if(rcode)
		{
			if(sessionInfo->ControlBlock.transferBufferSequence == END_OF_RECORD_MARKER)
				return TRUE;
		}
	}

	sessionInfo->newDataSetFlag = (sessionInfo->recInfo.dataSetInfoRetrieved == DATA_SET_INFO_COMPLETE);		
	return sessionInfo->newDataSetFlag?TRUE:FALSE;
}

/*********************************************************************
* NAME: RestoreRead
*
* WHAT: Get record/subrecord along with data set information and updates the field of the
*		sessionInfo struct appropriately.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
BUFFERPTR RestoreRead(SESSION_INFORMATION *sessionInfo, UINT32 *size)
{
	BUFFERPTR buffer=NULL;
	CCODE rcode=0;


	if (sessionInfo->atHeader)
	{			
		/*Get record/subrecord along with dataset information.*/
		rcode=GetRecordInfo(sessionInfo);
		if(rcode)
		{
			if(sessionInfo->ControlBlock.transferBufferSequence==END_OF_RECORD_MARKER)
				return buffer;
		}
	}
	/*Update the field of the sessionInfo struct appropriately*/
	*size = sessionInfo->recInfo.recordSize;		
	buffer = sessionInfo->currentPtr;
	sessionInfo->currentPtr += *size;
	sessionInfo->tUsed += *size;
	sessionInfo->tLeft -= *size;
	sessionInfo->atHeader = TRUE;
	
	 return buffer;
}

/*********************************************************************
* NAME: InitBackupSession
*
* WHAT:Initializes the backup session
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE InitBackupSession(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode = 0;
	/*********************************************************************
	*       For this demonstration engine, the backup data is placed in
	*       DOS files instead of tape(or other backup media). Here we open
	*       the files.
	**********************************************************************/
	if(sessionInfo->bctx->saveBackup && (sessionInfo->bctx->appenBackup==FALSE))
	{
		ccode= OpenDEFiles(O_CREAT|O_WRONLY|O_TRUNC,sessionInfo);
		if(ccode)
			goto Return;
	}
	/*Allocates memory for session header buffer*/
	if(sessionInfo->sessionHBuffer== NULL)
	{
		sessionInfo->sessionHBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->sessionHBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
	}
	/*Allocates memory for media header buffer*/
	if(sessionInfo->mediaBuffer== NULL)
	{
		sessionInfo->mediaBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->mediaBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
	}
	/*Build media header , session header and transfer buffer header informations.
	*Flush session header and media header information*/
	ccode=WriteOpenSession(sessionInfo);
	if(ccode)
	{
		PrintError(ccode, "WriteOpenSession", __LINE__, sessionInfo);
		goto Return;
	}
	/*Allocates memory for transfer buffer*/
	if(sessionInfo->transferBuffer == NULL)
	{
		sessionInfo->transferBuffer= (BUFFER *)malloc(65536*sizeof(BUFFER));
		if (sessionInfo->transferBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
		/*Update the fields of sessionInfo struct*/
		sessionInfo->currentPtr =sessionInfo->transferBuffer;
		sessionInfo->tUsed =sessionInfo->transferBufferInfo.transferBufferDataOffset;
		sessionInfo->tLeft = sessionInfo->transferBufferInfo.maxTransferBufferSize -
					sessionInfo->transferBufferInfo.transferBufferDataOffset;
		sessionInfo->currentPtr += sessionInfo->tUsed;
	}

	Return:
		return(ccode);
	
}

/*********************************************************************
* NAME: InitRestoreSession
*
* WHAT:Initializes the restore session
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE InitRestoreSession(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode = 0;
	BUFFERPTR bufferPtr=NULL;
	UINT32 _bufferSize;
	SMDF_FIELD_DATA		field;
	UINT16	mediaNumber;
	
	ccode= OpenDEFiles(O_RDONLY,sessionInfo);
	if(ccode)
		goto Return;

	/*Allocates memory for session header buffer*/
	if(sessionInfo->sessionHBuffer== NULL)
	{
		sessionInfo->sessionHBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->sessionHBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
	}
	/*Allocates memory for media header buffer*/
	if(sessionInfo->mediaBuffer== NULL)
	{
		sessionInfo->mediaBuffer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
		if (sessionInfo->mediaBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=NBK_OUT_OF_MEMROY;
			goto Return;
		}
	}
	/*Reads media header information from  archive.*/
	ccode=GetMediaHeader( sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		ccode=NBK_ERR_READING_ARCH;
		goto Return;
	}

	bufferPtr=sessionInfo->mediaBuffer;
	_bufferSize=SECTOR_SIZE;
	/*Checks whether the archive is a valid sidf archive*/
	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       !(field.fid == SIDF_PARTITION_HEADER) )
	 {
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_INVALID_SIDF_ARCHIEVE));
		ccode=1;
		goto Return;	

	 }

	  while( field.fid != SIDF_PARTITION_SET_SEQUENCE)
	 {
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			ccode=1;
			goto Return;
		}
		if(  field.fid == SIDF_PARTITION_SET_SEQUENCE)
		{
			memcpy(&mediaNumber, field.data, sizeof(UINT16));
			break;
		}
	 }
	  sessionInfo->rctx->volno=mediaNumber;
	  
	/*Allocates buffer for session header info and reads the session header information from archive.*/
	ccode=ReadOpenSession(sessionInfo);
	if(ccode)
		goto Return;
	/*Allocates memory for transfer buffer*/
	if(sessionInfo->transferBuffer == NULL)
	{
		sessionInfo->transferBuffer= (BUFFER *)malloc(sessionInfo->transferBufferInfo.maxTransferBufferSize*sizeof(BUFFER));
		if (sessionInfo->transferBuffer == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode=1;
			goto Return;
		}
		sessionInfo->currentPtr =sessionInfo->transferBuffer;	
	}

	Return:
		return(ccode);
	
}

/*********************************************************************
* NAME: CloseBackupSession
*
* WHAT:Flushes the last transfer buffer.
*
* SMS-SPECIFIC ROUTINES CALLED:NWSMUpdateRecordHeader
*
**********************************************************************/
CCODE CloseBackupSession(SESSION_INFORMATION *sessionInfo)
{

	CCODE ccode=0;
	if(sessionInfo->bctx->saveBackup)
	{
		/*Update record/subrecord size and CRC information*/
		NWSMUpdateRecordHeader(&(sessionInfo->recInfo));
		/*Flush the current transfer buffer to sidf archive*/
		ccode = FlushTransferBuffer( sessionInfo);
		if(ccode)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
			goto Return;	
		}
		/*Allocates memory for session trailer information*/
		if(sessionInfo->sessionHTrailer== NULL)
		{
			sessionInfo->sessionHTrailer= (BUFFER *)malloc(SECTOR_SIZE*sizeof(BUFFER));
			if (sessionInfo->sessionHTrailer == NULL)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
				ccode=NBK_OUT_OF_MEMROY;
				goto Return;
			}
		}
		if(safeFlag)
		{
			goto NextLevel;
		}
		/*Write the session trailer information to the archive.*/
		ccode=FlushSessionTrailer( sessionInfo);
		if(ccode)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
			goto Return;
		}
		CloseDEFiles( sessionInfo);

	}
	NextLevel:

	/*Free all the allocated buffers*/	
	if(sessionInfo->transferBuffer)
	{
		free(sessionInfo->transferBuffer);
		sessionInfo->transferBuffer=NULL;		
	}
	if(sessionInfo->sessionHBuffer)
	{
		free(sessionInfo->sessionHBuffer);
		sessionInfo->sessionHBuffer=NULL;
	}
	if(sessionInfo->sessionHTrailer)
	{
		free(sessionInfo->sessionHTrailer);
		sessionInfo->sessionHTrailer=NULL;	
	}
	if(sessionInfo->mediaBuffer)
	{
		free(sessionInfo->mediaBuffer);
		sessionInfo->mediaBuffer=NULL;	
	}
	if(sessionInfo->sHandle.tbHeaderPtr)
	{
		free(sessionInfo->sHandle.tbHeaderPtr);
		sessionInfo->sHandle.tbHeaderPtr=NULL;
	}

	Return:
	return(ccode);

}

/*********************************************************************
* NAME: ProcessScanInfoExtension
*
* WHAT: Process extra information in the scan Info struct.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int	ProcessScanInfoExtension(NWSM_SCAN_INFORMATION *scanInformation, NWSM_SCAN_INFO_EXTN_NFS_DATA_1 *nfsInfo)
{
	CCODE		 cCode = 0;
	int			 retval = 0;
	UINT32		 size;
	void			*scanExtensionBuffer;
	SMS_HANDLE	 handle;
	NWSM_EXTENSION_INFORMATION		*extension = NULL;
	NWSM_SCAN_INFO_EXTN_NFS_DATA_1	*nfsExtension;

	/* Functions that will be imported */
	static CCODE (*mPtrNWSMGetExtension)(void *, UINT32, UINT32, NWSM_EXTENSION_INFORMATION **, SMS_HANDLE *) = NULL;
	static CCODE (*mPtrNWSMCloseExtension)(SMS_HANDLE *) = NULL;

#ifdef N_PLAT_NLM
	if (!mPtrNWSMGetExtension)
		mPtrNWSMGetExtension = ImportSymbol(GetNLMHandle(), "NWSMGetExtension");

	if (!mPtrNWSMCloseExtension)
		mPtrNWSMCloseExtension = ImportSymbol(GetNLMHandle(), "NWSMCloseExtension");
#elif N_PLAT_UNIX
	/* This symbol is not dynamically resolved as it is supported on all SMS versions on Linux */
	mPtrNWSMGetExtension = NWSMGetExtension;
	mPtrNWSMCloseExtension = NWSMCloseExtension;
#endif

	/* If the functions are not present then return error */
	if (!mPtrNWSMCloseExtension || !mPtrNWSMGetExtension)
		return -1;

	/* Check if we have scan information */
	if (!scanInformation)
		return -1;

	
	/* Check if we have any extensions */
	if (!scanInformation->otherInformationSize)
		return -1;

	/* Get the size of the extensions in the SCAN_INFORMATION structure. */
	/* NOTE: As the other Information field of scan information is already extended by 4 bytes to hold primary stream
	 * sizes that are greater than UINT32 size (NSS QUAD sizes), we need to move the buffer by UINT32 size to actually 
	 * get to the extension buffer */
	size = scanInformation->otherInformationSize - sizeof(UINT32);
	
	/* Check if we have any extensions */
	if (!size)
		return -1;

	/* Set the extension buffer pointer */
	/* NOTE: As the other Information field of scan information is already extended by 4 bytes to hold primary stream
	 * sizes that are greater than UINT32 size (NSS QUAD sizes), we need to move the buffer by UINT32 size to actually 
	 * get to the extension buffer */
	scanExtensionBuffer = scanInformation->otherInformation + sizeof(UINT32);

	/* Get the desired extension, or loop through all extensions using NWSMGet(First/Next)Extension APIs */
	cCode = mPtrNWSMGetExtension(scanExtensionBuffer, size, NWSM_SCAN_INFO_EXTN_NFS_TAG, &extension, &handle);
	if (cCode)
	{
		return -1;
	}
	else
	{
		/* Check if any information is available */
		if (extension->info)
		{
			/* Type cast it to the extension of choice, if type casting to higher extension versions check extension->tagVersion,
			 * before type casting the pointer */
			nfsExtension = (NWSM_SCAN_INFO_EXTN_NFS_DATA_1 *)(extension->info);
			
			/* Copy the information to the output pointer.
			 * What we are pointing to in nfsExtension will become invalid once the extension handle is closed */
			nfsInfo->nfs_st_mode = nfsExtension->nfs_st_mode;
			nfsInfo->nfs_st_nlinks = nfsExtension->nfs_st_nlinks;
			nfsInfo->nfs_st_uid = nfsExtension->nfs_st_uid;
			nfsInfo->nfs_st_gid = nfsExtension->nfs_st_gid;
			nfsInfo->nfs_st_ctime = nfsExtension->nfs_st_ctime;
		}
		else
			retval = -1;

		/* Close the extension handle to free the handle and the extension pointers, this is only if there are no errors */
		mPtrNWSMCloseExtension(&handle);
	}

	return retval;
}

/*********************************************************************
* NAME: BuildSessionAndTBHeader
*
* WHAT:Initializes session and transfer buffer header informations.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE BuildSessionAndTBHeader(SESSION_INFORMATION *sessionInfo,NWSMSD_HEADER_BUFFER  *sessionHeaderInfo,
   										NWSMSD_TRANSFER_BUF_INFO *transferBufferInfo,BUFFERPTR  *endingFidPtr,
   											UINT32   *endingBufferSize)
  
{
	CCODE		ccode;
	UINT32		bufferSize,
	                     dataOverflow,
	                     tbHeaderData,
	                     tbHeaderSpace;
	BUFFERPTR	addressOfData,
	                     bufferPtr,
	                     tbHeaderPtr;
	UINT64		save16,
	                     save32;
	UINT16		sync,
	                     offsetToEnd,
	                     tbHeaderOffsetToEndIndex;
	UINT8		transferBufferType;
	SMDF_FIELD_DATA               field;
	
	ccode = 0;
	tbHeaderData = 0;
	sync = NWSM_SYNC_DATA;
	/*
	In this version, only type Data Set are written
	Since the headers are already checked in, hard-code
	here for now
	*/
#define TB_TYPE_DATA_SET 0x01
	transferBufferType = TB_TYPE_DATA_SET;
	memset(&save16, 0, sizeof(UINT64));
	memset(&save32, 0, sizeof(UINT64));
	memset(&field,  0, sizeof(SMDF_FIELD_DATA));

	SetUINT64Value(&save16, sizeof(UINT16));
	SetUINT64Value(&save32, sizeof(UINT32));

	/* Build transferbuffer header info not found in the session header */
	sessionInfo->sHandle.tbHeaderAllocSize=transferBufferInfo->sectorSize;
	/*Allocate space for transfer buffer header*/
	if ((sessionInfo->sHandle.tbHeaderPtr = malloc(sessionInfo->sHandle.tbHeaderAllocSize)) == NULL)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		ccode = NBK_OUT_OF_MEMROY;
		goto Return;
	}

	tbHeaderPtr = sessionInfo->sHandle.tbHeaderPtr;
	tbHeaderSpace = sessionInfo->sHandle.tbHeaderAllocSize;
	
	/*Build tranfer buffer header by putting different sidf fields
	*All the relevant informations will be copied to transfer buffer header before flusing the transfer buffer to archive*/
	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_TRANSFER_BUFFER_HEADER,
	                  &sync,
	                   sizeof(sync),
	                   sizeof(sync),
	                   NULL,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_OFFSET_TO_END,
	                  &save16,
	                   sizeof(UINT16),
	                   sizeof(UINT16),
	                  &addressOfData,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	tbHeaderOffsetToEndIndex = addressOfData - sessionInfo->sHandle.tbHeaderPtr;

	ccode = PutTBHeaderField( &(sessionInfo->sHandle),
	                   SIDF_TRANSFER_BUFFER_TYPE,
	                  &transferBufferType,
	                   sizeof(UINT8),
	                   sizeof(UINT8),
	                   NULL,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_TRANSFER_BUFFER_SIZE,
	                  &save32,
	                   sizeof(UINT32),
	                   sizeof(UINT32),
	                  &addressOfData,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);

	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	sessionInfo->sHandle.tbHeaderSizeOffset =
	                       addressOfData - sessionInfo->sHandle.tbHeaderPtr;

	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_UNUSED_IN_THIS_BUFFER,
	                  &save32,
	                   sizeof(UINT32),
	                   sizeof(UINT32),
	                  &addressOfData,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	sessionInfo->sHandle.tbHeaderUnusedOffset =
	                       addressOfData -sessionInfo->sHandle.tbHeaderPtr;

	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_TRANSFER_BUFFER_SEQUENCE,
	                  &save32,
	                   sizeof(UINT32),
	                   sizeof(UINT32),
	                  &addressOfData,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	sessionInfo->sHandle.tbHeaderSequenceOffset =
	                       addressOfData - sessionInfo->sHandle.tbHeaderPtr;

	ccode = PutTBHeaderField(&(sessionInfo->sHandle),
	                   SIDF_TRANSFER_BUFFER_ADDRESS,
	                  &save32,
	                   sizeof(UINT32),
	                   sizeof(UINT32),
	                  &addressOfData,
	                  &tbHeaderPtr,
	                  &tbHeaderSpace,
	                  &tbHeaderData);
	if (ccode != 0)
	{
		PrintError(ccode, "PutTBHeaderField", __LINE__,sessionInfo);
		goto Return;
	}
	
	sessionInfo->sHandle.tbHeaderSectorAddressOffset =
	                       addressOfData - sessionInfo->sHandle.tbHeaderPtr;

	bufferPtr = sessionHeaderInfo->headerBuffer;
	bufferSize = sessionHeaderInfo->headerSize;

	sessionInfo->sHandle.sessionTrailerInfoOffset =
	                       tbHeaderPtr - sessionInfo->sHandle.tbHeaderPtr;
	
	/*Move some more data to transfer buffer header
	*This is a sidf complaint session header information */

	while (bufferSize)
	{
		ccode = SMDFGetNextField(bufferPtr, bufferSize, &field);
		if (ccode != 0)
		{
			goto Return;
		}

		if (SMDFGetUINT64(&field.dataOverflow, &dataOverflow) ||  dataOverflow)
		{
			goto Return;
		}

		switch (field.fid)
		{
		case SIDF_SESSION_DATE_TIME:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);	
			goto Return;
		}
		break;

		case SIDF_SOURCE_NAME_TYPE:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);
			goto Return;
		}
		break;

		case SIDF_SOURCE_NAME:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);
			goto Return;
		}
		break;

		case SIDF_SOURCE_OS:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);
			goto Return;
		}
		break;

		case SIDF_SOURCE_OS_VERSION:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);
			goto Return;
		}
		break;

		case SIDF_SOFTWARE_NAME:
		case SIDF_SOFTWARE_TYPE:
		case SIDF_SOFTWARE_VERSION:
		case SIDF_SESSION_LABEL:
		case SIDF_SESSION_ID:
		ccode = MoveDataToTBHeader(&(sessionInfo->sHandle),
		                           bufferPtr,
		                           field.bytesTransfered,
		                          &tbHeaderPtr,
		                          &tbHeaderSpace,
		                          &tbHeaderData);
		if (ccode != 0)
		{
			PrintError(ccode, "MoveDataToTBHeader", __LINE__,sessionInfo);
			goto Return;
		}
		break;

		case SIDF_SESSION_HEADER:
		case SIDF_OFFSET_TO_END:
		case SIDF_CRC_TYPE:
		case SIDF_TRANSFER_BUFFER_SIZE:
		ccode = NWSMSD_BUFFER_INCORRECT;
		goto Return;
		}

		bufferPtr += field.bytesTransfered;
		bufferSize -= field.bytesTransfered;
	}

	sessionInfo->sHandle.sessionTrailerInfoSize =
	                       tbHeaderPtr -
	                       sessionInfo->sHandle.tbHeaderPtr -
	                       sessionInfo->sHandle.sessionTrailerInfoOffset;

	/* Caculate transferBufferInfo data */

	sessionInfo->sHandle.applicationAreaOffset =
	                   transferBufferInfo->applicationAreaOffset =
	                   tbHeaderData;
	sessionInfo->sHandle.applicationAreaSize =
	                   transferBufferInfo->applicationAreaSize;
	sessionInfo->sHandle.tbHeaderEndFidOffset =
	                   tbHeaderData +
	                   sessionInfo->sHandle.applicationAreaSize;
	sessionInfo->sHandle.tbHeaderSize = tbHeaderData;

	/* Update the offsetToEnd field in the TB header, will always be the same */
	offsetToEnd =
	sessionInfo->sHandle.tbHeaderEndFidOffset -
	(tbHeaderOffsetToEndIndex + sizeof(UINT16));

	memcpy( sessionInfo->sHandle.tbHeaderPtr + tbHeaderOffsetToEndIndex,
	&offsetToEnd,
	sizeof(UINT16));

	/* ?? CRC - RBC, if CRC + 6 instead of 2 */

	transferBufferInfo->transferBufferDataOffset =
	                           sessionInfo->sHandle.tbHeaderEndFidOffset + 2;

	/* Only support Transfer Buffers <= sector size for now */
	if (transferBufferInfo->transferBufferDataOffset >
	                                 transferBufferInfo->sectorSize)
	{
		ccode = NWSMSD_BUFFER_INCORRECT;
		goto Return;
	}

	bufferSize = transferBufferInfo->sectorSize;
	sessionInfo->sHandle.spannedSessionHeader =sessionInfo->sessionHBuffer;
	bufferPtr = sessionInfo->sessionHBuffer;

	/*Build session header info*/

	ccode = PutField(   SIDF_SESSION_HEADER,
	          &sync,
	           sizeof(sync),
	           sizeof(sync),
	           NULL,
	          &bufferPtr,
	          &bufferSize,
	           NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutField(   SIDF_OFFSET_TO_END,
	          &save16,
	           sizeof(UINT16),
	           sizeof(UINT16),
	          &addressOfData,
	          &bufferPtr,
	          &bufferSize,
	           NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutField(   SIDF_TRANSFER_BUFFER_SIZE,
	          &transferBufferInfo->maxTransferBufferSize,
	           sizeof(UINT32),
	           sizeof(UINT32),
	           NULL,
	          &bufferPtr,
	          &bufferSize,
	           NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	field.fid = SIDF_SESSION_HEADER;
	if ((sessionHeaderInfo->headerSize + SMDFSizeOfFID(field.fid) + 1) >
	                                                       bufferSize)
	{
		ccode = NWSMSD_HEADER_TOO_LARGE;
		goto Return;
	}
	/*Copies session header information to session header buffer*/
	memcpy( bufferPtr,
	sessionHeaderInfo->headerBuffer,
	sessionHeaderInfo->headerSize);

	bufferPtr += sessionHeaderInfo->headerSize;
	bufferSize -= sessionHeaderInfo->headerSize;

	/* Fix offsetToEnd */
	offsetToEnd = bufferPtr - (addressOfData + sizeof(UINT16));
	memcpy(addressOfData, &offsetToEnd, sizeof(UINT16));

	ccode = PutField(   SIDF_SESSION_HEADER,
	           NULL,
	           0,
	           0,
	           NULL,
	          &bufferPtr,
	          &bufferSize,
	           NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	  *endingFidPtr = bufferPtr;
   	*endingBufferSize = bufferSize;
	
	/*Pad blank space of the session header buffer*/
	NWSMPadBlankSpace(bufferPtr, bufferSize);


	Return:
	return (ccode);

} /* BuildSessionAndTBHeader() */

/*********************************************************************
* NAME: PutTBHeaderField
*
* WHAT: Put sidf tags into the transfer buffer header  and check for error.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE PutTBHeaderField(
   WRITE_SESSION_HANDLE         *sessionHandle,
   UINT32                        fid,
   void                         *data,
   UINT8                         dataSizeMap,
   UINT32                        sizeOfData,
   BUFFERPTR                    *addressOfData,
   BUFFERPTR                    *bufferPtr,
   UINT32                       *bufferSize,
   UINT32                       *bufferData)
{
	CCODE                         ccode;
	UINT32                        allocIncrement;

	ccode = 0;

	/*Put sidf tags into the transfer buffer header and incremets the
	transfer buffer header pointer*/
	if ((ccode = PutField(fid, data, dataSizeMap, sizeOfData, addressOfData,
	           bufferPtr, bufferSize, bufferData)) != 0)
	{
		
		/*In case of error*/
		allocIncrement = sizeOfData + HEADER_ALLOC_INCREMENT;
		sessionHandle->tbHeaderPtr = realloc(
		           sessionHandle->tbHeaderPtr,
		           sessionHandle->tbHeaderAllocSize + allocIncrement);

		if (sessionHandle->tbHeaderPtr == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode = NBK_OUT_OF_MEMROY;
			goto Return;
		}
		*bufferPtr = sessionHandle->tbHeaderPtr + *bufferData;
		*bufferSize += allocIncrement;
		sessionHandle->tbHeaderAllocSize += allocIncrement;
		ccode = PutField(fid, data, dataSizeMap, sizeOfData, addressOfData,
		bufferPtr, bufferSize, bufferData);
	}

	Return:
	return (ccode);

}

/*********************************************************************
* NAME: MoveDataToTBHeader
*
* WHAT:Moves data information to transfer buffer header.Basically used for copying 
*		session header info to transfer buffer header.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE MoveDataToTBHeader(
           WRITE_SESSION_HANDLE     *sessionHandle,
           BUFFERPTR                 sourceBuffer,
           UINT32                    sourceSize,
           BUFFERPTR                *bufferPtr,
           UINT32                   *bufferSize,
           UINT32                   *bufferData)
{
	CCODE                     ccode;
	UINT32                    allocIncrement;

	ccode = 0;

	/*Reallocation needs to be done if the space left in the destination buffer
		is less then the source size*/
	if (*bufferSize < sourceSize)
	{
		allocIncrement = (sourceSize - *bufferSize) + HEADER_ALLOC_INCREMENT;

		sessionHandle->tbHeaderPtr = realloc(
		           sessionHandle->tbHeaderPtr,
		           sessionHandle->tbHeaderAllocSize + allocIncrement);

		if (sessionHandle->tbHeaderPtr == NULL)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
			ccode = NBK_OUT_OF_MEMROY;
			goto Return;
		}

		*bufferPtr = sessionHandle->tbHeaderPtr + *bufferData;
		*bufferSize += allocIncrement;
		sessionHandle->tbHeaderAllocSize += allocIncrement;
	}
	/*Copies the data from source to destination*/
	memcpy(*bufferPtr, sourceBuffer, sourceSize);
	/*Update destination buffer pointer*/
	*bufferPtr += sourceSize;
	*bufferSize -= sourceSize;
	*bufferData += sourceSize;

	Return:
	return (ccode);

} /* MoveDataToTBHeader() */

/*********************************************************************
* NAME: WriteOpenSession
*
* WHAT:Build media header , session header and transfer buffer header informations.
*
* SMS-SPECIFIC ROUTINES CALLED: 	NWSMSetMediaHeaderInfo
*								NWSMGetCurrentDateAndTime
*								NWSMDOSTimeToECMA
*								NWSMSetSessionHeaderInfo
*
**********************************************************************/
CCODE WriteOpenSession(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;
	NWSMSD_HEADER_BUFFER *sessionHeaderInfo=NULL;
	NWSM_SESSION_INFO SessionInfo;
	NWSM_MEDIA_INFO smMediaInfo;
	UINT32 dosTime;
	ECMATime	ECMADateAndTime;
	UINT32		dosDateAndTime;

	/*allocate space for session header info*/
	sessionHeaderInfo = (NWSMSD_HEADER_BUFFER *) malloc( sizeof(NWSMSD_HEADER_BUFFER)+HBUFSIZE);
	if(sessionHeaderInfo == NULL)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		ccode = NBK_OUT_OF_MEMROY;
		goto Return;
	}
	sessionHeaderInfo->bufferSize = HBUFSIZE+sizeof(NWSMSD_HEADER_BUFFER);
	dosDateAndTime = NWSMGetCurrentDateAndTime();
	NWSMDOSTimeToECMA( dosDateAndTime, & ECMADateAndTime);
	memcpy( (void *) &smMediaInfo.timeStamp, (void *)&ECMADateAndTime , sizeof(ECMATime) );
	memcpy( (void *)&smMediaInfo.setTimeStamp,(void *)&ECMADateAndTime , sizeof(ECMATime) );
	smMediaInfo.number = 1;

	/*Formats media header information into a sidf complaint media header*/
	ccode = NWSMSetMediaHeaderInfo( &smMediaInfo,
									sessionHeaderInfo->headerBuffer,
									HBUFSIZE,
									&sessionHeaderInfo->headerSize); 
	if(ccode)
	{
		if(sessionHeaderInfo)
		{
			free(sessionHeaderInfo);
			sessionHeaderInfo=NULL;
		}
		PrintError(ccode, "NWSMSetMediaHeaderInfo", __LINE__,sessionInfo);
		goto Return;
	}

	/*Copies essential info to media header buffer and flushes the media header info
		to sidf archive*/
	if(sessionInfo->bctx->saveBackup)
	{
		ccode=MediaLabel( sessionInfo, sessionHeaderInfo , &(sessionInfo->bctx->volno));
		if(ccode)
		{
			if(sessionHeaderInfo)
			{
				free(sessionHeaderInfo);
				sessionHeaderInfo=NULL;
			}
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
			goto Return;
		}
		
	}
	
	dosTime = NWSMGetCurrentDateAndTime();
	NWSMDOSTimeToECMA(dosTime, &SessionInfo.timeStamp);
	SessionInfo.sessionID = dosTime;
	strncpy((char *)&SessionInfo.description,sessionInfo->sessionName,NWSM_MAX_DESCRIPTION_LEN);
	strncpy((char *)&SessionInfo.sidfSourceName,sessionInfo->tsaName,NWSM_MAX_SIDF_SRC_NAME_LEN);
	/*Formats session header information into a sidf complaint session header*/
	ccode = NWSMSetSessionHeaderInfo(&SessionInfo,
									sessionHeaderInfo->headerBuffer,
									HBUFSIZE,
									&sessionHeaderInfo->headerSize);

	if(ccode)
	{
		if(sessionHeaderInfo)
		{
			free(sessionHeaderInfo);
			sessionHeaderInfo=NULL;
		}
		PrintError(ccode, "NWSMSetSessionHeaderInfo", __LINE__,sessionInfo);
		goto Return;
	}
	memset(&(sessionInfo->transferBufferInfo), 0, sizeof(NWSMSD_TRANSFER_BUF_INFO));
	sessionInfo->transferBufferInfo.maxTransferBufferSize = 65536;
	sessionInfo->transferBufferInfo.sectorSize = SECTOR_SIZE;
	sessionInfo->transferBufferInfo.applicationAreaSize = 0;

	/*Build session and  transfer buffer header info and flushes the session header info
	to a sidf archive*/
	ccode = SessionOpenForWriting(sessionInfo,sessionHeaderInfo,&(sessionInfo->transferBufferInfo));
	if(ccode)
	{
		if(sessionHeaderInfo)
		{
			free(sessionHeaderInfo);
			sessionHeaderInfo=NULL;
		}
		PrintError(ccode, "SessionOpenForWriting", __LINE__,sessionInfo);
		goto Return;
	}

	memset(&(sessionInfo->ControlBlock), 0, sizeof(NWSMSD_CONTROL_BLOCK));
	sessionInfo->ControlBlock.transferBufferSizeAllocated = sessionInfo->transferBufferInfo.maxTransferBufferSize;
	sessionInfo->ControlBlock.transferBufferDataOffset = sessionInfo->transferBufferInfo.transferBufferDataOffset;
	sessionInfo->ControlBlock.transferBufferSequence = 0;
	sessionInfo->ControlBlock.sessionDataType = NWSMSD_SDT_TSA_DATA;
	if(sessionHeaderInfo)
	{
		free(sessionHeaderInfo);
		sessionHeaderInfo=NULL;
	}
	

	Return:
	return (ccode);

	
}

/*********************************************************************
* NAME: SessionOpenForWriting
*
* WHAT:Build session and  transfer buffer header info and flushes the session header info
*	to a sidf archive
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE SessionOpenForWriting(SESSION_INFORMATION *sessionInfo,NWSMSD_HEADER_BUFFER  *sessionHeaderInfo,
   										NWSMSD_TRANSFER_BUF_INFO *transferBufferInfo)
{
	CCODE ccode=0;
	  BUFFERPTR                         endingFidPtr;
        UINT32                            endingBufferSize;
         BUFFERPTR                         spannedSessionHeaderPtr;
        UINT32                            bufferSize;
        UINT16                             sync;
     

	 sessionInfo->sHandle.maxTransferBufferSize =
                        transferBufferInfo->maxTransferBufferSize;

	/*Build session and transfer buffer header*/
	ccode = BuildSessionAndTBHeader(sessionInfo,sessionHeaderInfo,transferBufferInfo,&endingFidPtr, &endingBufferSize);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
		goto Return;
	}

	/*Flush session header info if the data is not to be appended into already existing session*/
	if(sessionInfo->bctx->saveBackup && (sessionInfo->bctx->appenBackup==FALSE))
	{
		ccode = FlushSessionHeader(sessionInfo);
		if(ccode)
		{
			PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
			goto Return;
		}
		bufferSize = transferBufferInfo->sectorSize;
		spannedSessionHeaderPtr = sessionInfo->sessionHBuffer;

		 /* Replace SESSION_HEADERs with SESSION_SUB_HEADERs */
		   ccode = PutField(   SIDF_SESSION_SUB_HEADER,
		                      &sync,
		                       sizeof(sync),
		                       sizeof(sync),
		                       NULL,
		                      &spannedSessionHeaderPtr,
		                      &bufferSize,
		                       NULL);
		  if (ccode != 0)
		{
			PrintError(ccode, "PutField", __LINE__,sessionInfo);
			goto Return;
		}

		   ccode = PutField(   SIDF_SESSION_SUB_HEADER,
		                       NULL,
		                       0,
		                       0,
		                       NULL,
		                      &endingFidPtr,
		                      &endingBufferSize,
		                       NULL);
		 if (ccode != 0)
		{
			PrintError(ccode, "PutField", __LINE__,sessionInfo);
			goto Return;
		}
	}

	Return:
	return (ccode);
	
} 

/*********************************************************************
* NAME: SessionWriteData
*
* WHAT:Writes data from the transfer buffer to media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE SessionWriteData(SESSION_INFORMATION *sessionInfo)
{
	CCODE		ccode=0;
	UINT32		sectorSize,
	         	size,
	         	tbSize,
	         	unused;
	BUFFERPTR	tbPtr,
                        ptr;
	ssize_t 	datawriten=0;
	
	sectorSize=SECTOR_SIZE;

	tbPtr = sessionInfo->transferBuffer;
	/*Copies transfer buffer header information before flushing the transfer buffer to archive*/
	memcpy(tbPtr, sessionInfo->sHandle.tbHeaderPtr, sessionInfo->sHandle.tbHeaderSize);
	
	tbSize = sessionInfo->sHandle.maxTransferBufferSize;
	unused = tbSize - sessionInfo->ControlBlock.transferBufferSizeData;

	if(tbPtr + sessionInfo->sHandle.tbHeaderSizeOffset==NULL)
	{
		safeFlag=1;
		goto Return;
	}
	/*Copies relevant information to different sidf fields of the transfer buffer header*/
	memcpy( tbPtr + sessionInfo->sHandle.tbHeaderSizeOffset,
	&tbSize,
	sizeof(UINT32));
	memcpy( tbPtr + sessionInfo->sHandle.tbHeaderUnusedOffset,
	&unused,
	sizeof(UINT32));
	
	memcpy( tbPtr + sessionInfo->sHandle.tbHeaderSequenceOffset,
	&sessionInfo->ControlBlock.transferBufferSequence,
	sizeof(UINT32));
	
	ptr = tbPtr + sessionInfo->sHandle.tbHeaderEndFidOffset;
	size = sessionInfo->ControlBlock.transferBufferSizeData -
	           sessionInfo->sHandle.tbHeaderEndFidOffset;
	ccode=PutField(   SIDF_TRANSFER_BUFFER_HEADER,
				NULL,
				0,
				0,
				NULL,
				&ptr,
				&size,
				NULL);

	if(ccode)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	
	
	/*
	Pad unused portion, if any, of Transfer Buffer with
	BlankSpace section.
	*/
	if (unused != 0)
	{
		NWSMPadBlankSpace(tbPtr + sessionInfo->ControlBlock.transferBufferSizeData,
		unused);
	}

	/*flush the data*/
	if(sessionInfo->bctx->settapeLength)
	{
		
		if( sessionInfo->tape_length - TotalDataWritten < 64)
		{
			TotalDataWritten=0;
			ccode = ContinueToNextTape(sessionInfo,  &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}	
		}
	}
	writedata:
	datawriten=write(sessionInfo->dataFile, sessionInfo->transferBuffer, 65536*sizeof(BUFFER));
	if(sessionInfo->bctx->settapeLength && (datawriten == (65536*sizeof(BUFFER))))
	{
		TotalDataWritten += 64;
	}
	if(datawriten != (65536*sizeof(BUFFER)))
	{
		if(sessionInfo->bctx->multiVol )
		{
			TotalDataWritten=0;
			ccode = ContinueToNextTape(sessionInfo,  &(sessionInfo->bctx->volno));
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
				goto Return;
			}
			goto writedata;	
		}
		PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_TBUF_FLUSH_ERR) );
		ccode=1;
		return(ccode);
	}

	Return:
	return (ccode);

}

/*********************************************************************
* NAME: ReadOpenSession
*
* WHAT:Allocates buffer for session header info and reads the session header information from media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE ReadOpenSession(SESSION_INFORMATION *sessionInfo)
{
	CCODE ccode=0;
	NWSMSD_HEADER_BUFFER *sessionHeaderInfo=NULL;
	
	/*allocate space for session header info*/
	sessionHeaderInfo = (NWSMSD_HEADER_BUFFER *) malloc( sizeof(NWSMSD_HEADER_BUFFER)+HBUFSIZE);
	if(sessionHeaderInfo == NULL)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_OUT_OF_MEMROY));
		ccode = NBK_OUT_OF_MEMROY;
		goto Return;
	}
	sessionHeaderInfo->bufferSize = HBUFSIZE+sizeof(NWSMSD_HEADER_BUFFER);
	
	memset(&(sessionInfo->transferBufferInfo), 0, sizeof(NWSMSD_TRANSFER_BUF_INFO));   

	/*Reads session header information from  archive*/
	ccode= SessionOpenForReading(sessionInfo,sessionHeaderInfo);
	if(ccode)
	{
		if(sessionHeaderInfo)
		{
			free(sessionHeaderInfo);
			sessionHeaderInfo=NULL;
		}
		PrintError(ccode, "SessionOpenForReading", __LINE__,sessionInfo);
		goto Return;
	}

	memset(&(sessionInfo->ControlBlock), 0, sizeof(NWSMSD_CONTROL_BLOCK));
	sessionInfo->ControlBlock.transferBufferSizeAllocated = sessionInfo->transferBufferInfo.maxTransferBufferSize;
	sessionInfo->ControlBlock.transferBufferSequence = 0;
	if(sessionHeaderInfo)
	{
		free(sessionHeaderInfo);
		sessionHeaderInfo=NULL;
	}
		
	Return:
	return (ccode);
	
}

/*********************************************************************
* NAME: SessionOpenForReading
*
* WHAT:Reads session header information from  media/file and copies the required info
*	to session info struct
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/

CCODE SessionOpenForReading(SESSION_INFORMATION *sessionInfo,NWSMSD_HEADER_BUFFER  *sessionHeaderInfo)
   										
{
	CCODE ccode=0;
	BUFFERPTR bufferPtr=NULL;
	UINT32 _bufferSize,sectionHead,dataSize,size,bufferSize;
	SMDF_FIELD_DATA		field;

	/*Read session header info from archive*/
	ccode=GetSessionHeader(sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		ccode=NBK_ERR_READING_ARCH;
		goto Return;
	}

	bufferPtr=sessionInfo->sessionHBuffer;
	bufferSize=_bufferSize=SECTOR_SIZE;

	
	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       ((field.fid != SIDF_SESSION_HEADER)  &&   (field.fid != SIDF_SESSION_SUB_HEADER)) )
	 {
       	goto Return;
	 }
	 
	sectionHead = field.fid;
	field.fid = 0;

	/*session header info is used to get the session name and transfer buffer size*/
	 while (field.fid != sectionHead)
	{
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			goto Return;
		}

		switch (field.fid)
		{
			case SIDF_SESSION_DATE_TIME:
			break;

			case SIDF_SESSION_DATE_TIME_OLD:    /* To read old format tapes */
			break;

			case SIDF_SESSION_LABEL:
			SMDFGetUINT64(&field.dataSize, &dataSize);
			size = min(dataSize, NWSM_MAX_DESCRIPTION_LEN - 1);
			memcpy(sessionInfo->sessionName, field.data, size);
			sessionInfo->sessionName[size] = 0;
			break;

			case SIDF_SOURCE_NAME:
			SMDFGetUINT64(&field.dataSize, &dataSize);
			size = min(dataSize, NWSM_MAX_SIDF_SRC_NAME_LEN - 1);
			memcpy(sessionInfo->srcName, field.data, size);
			sessionInfo->srcName[size] = 0;
			break;

			case SIDF_SOURCE_OS:
			break;

			case SIDF_SOURCE_OS_VERSION:
			break;

			case SIDF_SESSION_ID:
			break;

			case SIDF_TRANSFER_BUFFER_SIZE:
			SMDFGetUINT64(&field.dataSize, &dataSize);
			size = min(dataSize, sizeof(UINT32));
			memcpy(&(sessionInfo->transferBufferInfo.maxTransferBufferSize), field.data, size);
			break;

			case SIDF_AUTHENTICATION:
			break;
		}
	}

	if (sessionHeaderInfo != NULL)
	{
		sessionHeaderInfo->headerSize = bufferSize - _bufferSize;
		if (sessionHeaderInfo->headerSize > sessionHeaderInfo->bufferSize)
		{
			sessionHeaderInfo->overflowSize = sessionHeaderInfo->headerSize -
			           sessionHeaderInfo->bufferSize;
			sessionHeaderInfo->headerSize -= sessionHeaderInfo->overflowSize;
		}
		else
		{
			sessionHeaderInfo->overflowSize = 0;
		}

		memcpy(sessionHeaderInfo->headerBuffer, sessionInfo->sessionHBuffer,
		sessionHeaderInfo->headerSize);
	}


	Return:
	return (ccode);
	
} 

/*********************************************************************
* NAME: GetSessionHeader
*
* WHAT:Reads session header information from  media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE GetSessionHeader(SESSION_INFORMATION *sessionInfo)
{

	CCODE ccode=0;
	ssize_t dataread=0;
	
	/*read session header info from archive*/
	readdata:
	dataread=read(sessionInfo->dataFile, sessionInfo->sessionHBuffer,SECTOR_SIZE*sizeof(BUFFER) ); 
	if(dataread != (SECTOR_SIZE*sizeof(BUFFER)))
	{	
		if(sessionInfo->rctx->multiVol)
		{
			sessionInfo->rctx->volno ++;
			ccode= ContinueMediaHeaderFromNextTape(sessionInfo);
			if(ccode)
			{
				PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
				goto Return;
			}
			
			goto readdata;
		}
		PrintError(ccode, "read", __LINE__,sessionInfo);
		ccode=1;
		goto Return;
	}
	
	Return:
	return(ccode);

}

/*********************************************************************
* NAME: GetMediaHeader
*
* WHAT:Reads media header information from  media/file.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE GetMediaHeader(SESSION_INFORMATION *sessionInfo)
{

	CCODE ccode=0;
	ssize_t dataread=0;
	
	/*read media header info from archive*/
	readdata:
	dataread=read(sessionInfo->dataFile, sessionInfo->mediaBuffer,SECTOR_SIZE*sizeof(BUFFER) ); 
	if(dataread != (SECTOR_SIZE*sizeof(BUFFER)))
	{	
		if(sessionInfo->rctx->multiVol)
		{
			sessionInfo->rctx->volno ++;
			CloseDEFiles(sessionInfo);
			PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE), sessionInfo->rctx->volno, sessionInfo->destFile);
			getchar();
			ccode= OpenDEFiles(O_RDONLY,sessionInfo);
			if(ccode)
				goto Return;
			goto readdata;	
		}
		PrintError(ccode, "read", __LINE__,sessionInfo);
		ccode=1;
		goto Return;
	}
	
	Return:
	return(ccode);

}

/*********************************************************************
* NAME: GetField
*
* WHAT: Gets the data for the particular sidf tag.
*
* SMS-SPECIFIC ROUTINES CALLED: SMDFGetNextField
*
**********************************************************************/
CCODE  GetField(
   SMDF_FIELD_DATA              *field,
   BUFFERPTR                    *bufferPtr,
   UINT32                       *bufferSize)
{
	CCODE                         ccode;
	UINT32                        dataOverflow;


	ccode = 0;
	/*Seperated SIDF complaint field into its various components*/
	if ((ccode = SMDFGetNextField(*bufferPtr, *bufferSize, field)) != 0)
	{
		goto Return;
	}
	/*returns the lower four bytes from a UINT64 value*/
	if (SMDFGetUINT64(&field->dataOverflow, &dataOverflow) || dataOverflow)
	{
		goto Return;
	}

	*bufferPtr += field->bytesTransfered;
	*bufferSize -= field->bytesTransfered;

	Return:
	return (ccode);

} // GetField()

/*********************************************************************
* NAME: SessionReadData
*
* WHAT: Read data from media/file and parse transfer buffer header information.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
CCODE SessionReadData(SESSION_INFORMATION *sessionInfo)
{
	CCODE	ccode=0;
	ssize_t dataread=0;
	ssize_t readdata=0;
	BUFFER 	*tempBuffer=NULL;
	BUFFERPTR bufferPtr=NULL;
	UINT32 _bufferSize;
	SMDF_FIELD_DATA		field; 
	
	/*read one block of transfer buffer from archive*/
	ReadAgain:
	dataread=read(sessionInfo->dataFile, sessionInfo->transferBuffer,sessionInfo->ControlBlock.transferBufferSizeAllocated*sizeof(BUFFER) );
	if(sessionInfo->rctx->multiVol)
	{
		if(dataread != (sessionInfo->ControlBlock.transferBufferSizeAllocated*sizeof(BUFFER)))
		{
			if(dataread==SECTOR_SIZE)
			{
				bufferPtr=sessionInfo->transferBuffer;
				_bufferSize=SECTOR_SIZE;
				if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
				!(field.fid == SIDF_SESSION_TRAILER) )
				{
					ccode=RetrieveFromNextTape(sessionInfo);
					if(ccode)
					{
						PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
						ccode=NBK_ERR_READING_ARCH;
						goto Return;
					}
					goto ReadAgain;	
				}
				ccode=0;
				sessionInfo->dataread=dataread;
				goto Return;
			}
			else
			{
				ccode=RetrieveFromNextTape(sessionInfo);
				if(ccode)
				{
					PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
					ccode=NBK_ERR_READING_ARCH;
					goto Return;
				}
				goto ReadAgain;		
			}
		
		}
		goto ParseTB;
	}
	while(dataread != (sessionInfo->ControlBlock.transferBufferSizeAllocated*sizeof(BUFFER)))
	{
			
		if(dataread == -1)
		{
			PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_TBGET_FAILED));
			ccode=1;
			goto Return;
		}
		/*reading 4096 bytes means reading session trailer information.Go to NextRead if it is not the session trailer*/
		
		if(dataread==SECTOR_SIZE)
		{
			bufferPtr=sessionInfo->transferBuffer;
			_bufferSize=SECTOR_SIZE;
			if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
			!(field.fid == SIDF_SESSION_TRAILER) )
			{
				goto NextRead;	
			}
			ccode=0;
			sessionInfo->dataread=dataread;
			goto Return;
		}
		NextRead:
		tempBuffer=sessionInfo->transferBuffer+dataread;
		readdata=read(sessionInfo->dataFile, tempBuffer,sessionInfo->ControlBlock.transferBufferSizeAllocated*sizeof(BUFFER)-dataread );
		if((readdata == -1) || (readdata == 0))
		{
			PrintMsg(NBK_DEBUG_MSG, _NbkMessage(NBK_TBGET_FAILED));
			ccode=1;
			goto Return;
		}
		dataread=dataread+readdata;
	}
	/* Parse transfer buffer header information*/
	ParseTB:
	ccode=FMTSIDFParseTBHeader(sessionInfo);
	if(ccode)
	{
		PrintError(ccode, "FMTSIDFParseTBHeader", __LINE__,sessionInfo);
		goto Return;
	}
	
	sessionInfo->currentPtr = sessionInfo->transferBuffer;
	sessionInfo->tUsed =sessionInfo->ControlBlock.transferBufferDataOffset;
	sessionInfo->tLeft =  sessionInfo->ControlBlock.transferBufferSizeData-sessionInfo->tUsed;
	sessionInfo->currentPtr += sessionInfo->tUsed;
	
	Return:
	return (ccode);

}

/*********************************************************************
* NAME: FMTSIDFParseTBHeader
*
* WHAT:  Parse transfer buffer header information.
*
* SMS-SPECIFIC ROUTINES CALLED: SMDFGetUINT64
*
**********************************************************************/
CCODE FMTSIDFParseTBHeader(SESSION_INFORMATION *sessionInfo)
 
{
	CCODE                         ccode;
	BUFFERPTR                     bufferPtr;
	UINT32                        bufferSize,_bufferSize, dataSize ,tBSize, unused;
	SMDF_FIELD_DATA               field;
	

	ccode = 0;
	tBSize = 0;
	unused = 0;
	memset(&field, 0, sizeof(SMDF_FIELD_DATA));


	bufferSize = _bufferSize =sessionInfo->ControlBlock.transferBufferSizeAllocated;
	bufferPtr = sessionInfo->transferBuffer;
	
	/*Gets the data for the particular sidf tag.*/
	if ((ccode = GetField(&field, &bufferPtr, &bufferSize)))
		goto Return;

	/*Parse only if the starting fid is SIDF_TRANSFER_BUFFER_HEADER*/
	switch (field.fid)
	{
		case SIDF_TRANSFER_BUFFER_HEADER:
		break;

		default:
		goto Return;
	}
	
	field.fid = 0;
	/*Go on parsing untill and unless the next fid is SIDF_TRANSFER_BUFFER_HEADER*/
	while (field.fid != SIDF_TRANSFER_BUFFER_HEADER)
	{
		if ((ccode = GetField(&field, &bufferPtr, &bufferSize)))
		{
			goto Return;
		}
		
		switch (field.fid)
		{
			case SIDF_TRANSFER_BUFFER_SIZE:
			SMDFGetUINT64(&field.dataSize, &dataSize);
			memcpy(&tBSize, field.data, dataSize);
			break;

			case SIDF_UNUSED_IN_THIS_BUFFER:
			SMDFGetUINT64(&field.dataSize, &dataSize);
			memcpy(&unused, field.data, dataSize);
			break;

			case SIDF_TRANSFER_BUFFER_SEQUENCE:
			SMDFGetUINT64(&field.dataSize, &dataSize);
           		sessionInfo->ControlBlock.transferBufferSequence = 0;
           		memcpy(&sessionInfo->ControlBlock.transferBufferSequence, field.data, dataSize);
			break;

			case SIDF_SOURCE_NAME:
			break;
			
			case SIDF_SOURCE_OS:
			break;
			
			case SIDF_SOURCE_OS_VERSION:
			break;
			
			case SIDF_SESSION_DATE_TIME:
			break;
			
			case SIDF_SESSION_DATE_TIME_OLD:                /* obsolete FID */
			break;

			case SIDF_SESSION_ID:
			break;
		}
	}

	
	sessionInfo->ControlBlock.transferBufferSizeData = tBSize - unused;
	sessionInfo->ControlBlock.transferBufferDataOffset = _bufferSize - bufferSize;

	Return:
	return (ccode);

} 

/*********************************************************************
* NAME: PutField
*
* WHAT: Puts the data across a particular sidf tag.
*
* SMS-SPECIFIC ROUTINES CALLED:SMDFPutNextField
*								SMDFGetUINT64
*
**********************************************************************/
CCODE  PutField(
   UINT32                        fid,
   void                         *data,
   UINT8                         dataSizeMap,
   UINT32                        sizeOfData,
   BUFFERPTR                    *addressOfData,
   BUFFERPTR                    *bufferPtr,
   UINT32                       *bufferSize,
   UINT32                       *bufferData)
{
	CCODE                         ccode;
	UINT32                        dataOverflow;
	SMDF_FIELD_DATA               field;

	// This is called too often for TRACE_BEGIN or TRACE_END

	ccode = 0;
	memset(&field, 0, sizeof(SMDF_FIELD_DATA));


	field.fid = fid;
	field.data = data;

	/*Formats data into a SIDF complaint field*/
	if (SMDFPutNextField(*bufferPtr, *bufferSize, &field, dataSizeMap, sizeOfData) != 0)
	{
		ccode = NWSMUT_BUFFER_OVERFLOW;
		goto Return;
	}
	/*Returns the lower four bytes from a UINT64 value*/
	if (SMDFGetUINT64(&field.dataOverflow, &dataOverflow) || dataOverflow)
	{
		ccode = NWSMUT_BUFFER_OVERFLOW;
		goto Return;
	}
	if (addressOfData)
	*addressOfData = *bufferPtr + (field.bytesTransfered - sizeOfData);

	*bufferPtr += field.bytesTransfered;
	*bufferSize -= field.bytesTransfered;
	if (bufferData)
	*bufferData += field.bytesTransfered;

	Return:
	return (ccode);

} // PutField()

/*********************************************************************
* NAME: MediaLabel
*
* WHAT: Label and flushes the media header information into media/file.
*
* SMS-SPECIFIC ROUTINES CALLED: NWSMPadBlankSpace
*
**********************************************************************/
CCODE MediaLabel(SESSION_INFORMATION *sessionInfo,NWSMSD_HEADER_BUFFER  *sessionHeaderInfo,   UINT16  *MediaNum)
{

	CCODE	ccode;
	UINT32	bufferSize;
	UINT64	save16 = UINT64_ZERO;
	SMDF_FIELD_DATA		field = { 0 };
	UINT16	sync = NWSM_SYNC_DATA;
	UINT32	unitSize;
	UINT8	mediaMarkUsageDataSizeMap;
	BUFFERPTR	addressOfData,bufferPtr;
	UINT16	offsetToEnd; 
	UINT16	mediaNumber;
                                    

	bufferSize =unitSize= SECTOR_SIZE;
	bufferPtr = sessionInfo->mediaBuffer;

	/* Build Media Header */
	ccode = PutField(   SIDF_PARTITION_HEADER,
					&sync,
					sizeof(sync),
					sizeof(sync),
					NULL,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	mediaNumber = *MediaNum;
	ccode= PutField(
                            SIDF_PARTITION_SET_SEQUENCE,
                           &mediaNumber,
                            sizeof(mediaNumber),
                            sizeof(mediaNumber),
                            NULL,
                           &bufferPtr,
                           &bufferSize,
                            NULL);
         if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
        (*MediaNum )++;
	SetUINT64Value(&save16, sizeof(UINT16));
	ccode = PutField(   SIDF_OFFSET_TO_END,
					&save16,
					sizeof(UINT16),
					sizeof(UINT16),
					&addressOfData,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutField(   SIDF_NAME_ID_STRING,
					sdiSidfIdString,
					SDI_SIZEOF_ID_VERSION,
					SDI_SIZEOF_ID_VERSION,
					NULL,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutField(   SIDF_VERSION_NUMBER,
					sdiSidfVersion101,
					SDI_SIZEOF_ID_VERSION,
					SDI_SIZEOF_ID_VERSION,
					NULL,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	ccode = PutField(   SIDF_SECTOR_SIZE,
					&unitSize,
					sizeof(UINT32),
					sizeof(UINT32),
					NULL,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	/* Put media marks on session headers and trailers */
	mediaMarkUsageDataSizeMap = SDI_MEDIA_MARK_USAGE_SH | SDI_MEDIA_MARK_USAGE_ST;
	ccode = PutField(   SIDF_MEDIA_MARK_USAGE,
					NULL,
					mediaMarkUsageDataSizeMap,
					0,
					NULL,
					&bufferPtr,
					&bufferSize,
					NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}

	field.fid = SIDF_PARTITION_HEADER;
	/* CRC, ?  1 or 5 RBC */
	if ((sessionHeaderInfo->headerSize+SMDFSizeOfFID(field.fid) + 1) > bufferSize)
	{
		ccode = NWSMSD_HEADER_TOO_LARGE;
		goto Return;
	}

	memcpy( bufferPtr,
	sessionHeaderInfo->headerBuffer,
	sessionHeaderInfo->headerSize);
	bufferPtr += sessionHeaderInfo->headerSize;
	bufferSize -= sessionHeaderInfo->headerSize;

	/* Fix offsetToEnd and PutEndField */
	offsetToEnd = bufferPtr - (addressOfData + sizeof(UINT16));
	memcpy(addressOfData, &offsetToEnd, sizeof(UINT16));

	ccode = PutField(   SIDF_PARTITION_HEADER,
	NULL,
	0,
	0,
	NULL,
	&bufferPtr,
	&bufferSize,
	NULL);
	if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	/*Pad the unused area of the buffer*/
	NWSMPadBlankSpace(bufferPtr, bufferSize);
	/*Flush media header info in an archive only if backup data is not going to append in a already existing session*/
	if(sessionInfo->bctx->appenBackup== FALSE)
	{
		ccode=FlushMediaHeader( sessionInfo);
		if(ccode)
		{
			PrintError(ccode, "FlushMediaHeader", __LINE__,sessionInfo);
			goto Return;
		}
	}

	Return:
		return(ccode);

}


CCODE ContinueToNextTape(SESSION_INFORMATION *sessionInfo,UINT16  *MediaNum)
{
	CCODE	ccode;
	UINT32	bufferSize;
	BUFFERPTR bufferPtr;
	SMDF_FIELD_DATA		field;
	UINT16	mediaNumber;

	
	CloseDEFiles(sessionInfo);
	PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE), sessionInfo->bctx->volno, sessionInfo->destFile);
	getchar();
	if(sessionInfo->bctx->saveBackup)
	{
		ccode= OpenDEFiles(O_CREAT|O_WRONLY|O_TRUNC,sessionInfo);
		if(ccode)
			goto Return;
	}
	
	bufferSize = SECTOR_SIZE;
	bufferPtr = sessionInfo->mediaBuffer;

	 if ((GetField(&field, &bufferPtr, &bufferSize) != 0)   ||
		(field.fid != SIDF_PARTITION_HEADER)  )
		{
			ccode=1;
			goto Return;
		}

	mediaNumber = *MediaNum;
	ccode= PutField(
                            SIDF_PARTITION_SET_SEQUENCE,
                           &mediaNumber,
                            sizeof(mediaNumber),
                            sizeof(mediaNumber),
                            NULL,
                           &bufferPtr,
                           &bufferSize,
                            NULL);
         if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	(*MediaNum )++;
	ccode=FlushMediaHeader( sessionInfo);
	if(ccode)
	{
		PrintError(ccode, "FlushMediaHeader", __LINE__,sessionInfo);
		goto Return;
	}
	ccode = FlushSessionHeader(sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_WRITE_ERROR));
		goto Return;
	}

	Return:
		return(ccode);
}


CCODE RetrieveFromNextTape(SESSION_INFORMATION *sessionInfo)
{
	CCODE	ccode;
	BUFFERPTR bufferPtr;
	UINT32 _bufferSize;
	SMDF_FIELD_DATA		field;
	UINT16	mediaNumber;
	UINT32 dataSize,size;

	sessionInfo->rctx->volno ++;
	CloseDEFiles(sessionInfo);
	NextTape:
	PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE), sessionInfo->rctx->volno, sessionInfo->destFile);
	getchar();
	ccode= OpenDEFiles(O_RDONLY,sessionInfo);
	if(ccode)
		goto Return;
	ccode=GetMediaHeader( sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		ccode=NBK_ERR_READING_ARCH;
		goto Return;
	}
	ccode=GetSessionHeader(sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		ccode=NBK_ERR_READING_ARCH;
		goto Return;
	}

	bufferPtr=sessionInfo->mediaBuffer;
	_bufferSize=SECTOR_SIZE;

	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       (field.fid != SIDF_PARTITION_HEADER))
	 {
		ccode=1;
       	goto Return;
	 }

	 while( field.fid != SIDF_PARTITION_SET_SEQUENCE)
	 {
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			goto Return;
		}
		if(  field.fid == SIDF_PARTITION_SET_SEQUENCE)
		{
			memcpy(&mediaNumber, field.data, sizeof(UINT16));
			break;
		}
	 }
	 if(mediaNumber != sessionInfo->rctx->volno)
	 {
		CloseDEFiles(sessionInfo);
		PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_CORRECT_TAPE), sessionInfo->rctx->volno, sessionInfo->sessionName);
		goto NextTape;
		
	 }

	 bufferPtr=sessionInfo->sessionHBuffer;
	_bufferSize=SECTOR_SIZE;

	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       (field.fid != SIDF_SESSION_SUB_HEADER))
	 {
		ccode=1;
       	goto Return;
	 }

	  while( field.fid != SIDF_SESSION_LABEL)
	 {
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			goto Return;
		}
		if(  field.fid == SIDF_SESSION_LABEL)
		{
			SMDFGetUINT64(&field.dataSize, &dataSize);
			size = min(dataSize, NWSM_MAX_DESCRIPTION_LEN - 1);
			memcpy(sessionInfo->rctx->session_Name, field.data, size);
			sessionInfo->rctx->session_Name[size] = 0;
			break;
		}
	 }

	  if(strncmp(sessionInfo->sessionName, sessionInfo->rctx->session_Name, size) )
	  {
		CloseDEFiles(sessionInfo);
		PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_CORRECT_TAPE), sessionInfo->rctx->volno, sessionInfo->sessionName);
		goto NextTape;
	  }
	  
	Return:
		return(ccode);
}


CCODE ContinueMediaHeaderToNextTape(SESSION_INFORMATION *sessionInfo , UINT16  *MediaNum)
{

	CCODE	ccode;
	UINT32	bufferSize;
	BUFFERPTR bufferPtr;
	UINT32 _bufferSize;
	SMDF_FIELD_DATA		field;
	UINT16	mediaNumber;

	
	CloseDEFiles(sessionInfo);
	PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE),  sessionInfo->bctx->volno, sessionInfo->destFile);
	getchar();
	if(sessionInfo->bctx->saveBackup)
	{
		ccode= OpenDEFiles(O_CREAT|O_WRONLY|O_TRUNC,sessionInfo);
		if(ccode)
			goto Return;
	}

	bufferSize = SECTOR_SIZE;
	bufferPtr = sessionInfo->mediaBuffer;

	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
		(field.fid != SIDF_PARTITION_HEADER)  )
		{
			ccode=1;
			goto Return;
		}

	mediaNumber = *MediaNum;
	ccode= PutField(
                            SIDF_PARTITION_SET_SEQUENCE,
                           &mediaNumber,
                            sizeof(mediaNumber),
                            sizeof(mediaNumber),
                            NULL,
                           &bufferPtr,
                           &bufferSize,
                            NULL);
         if (ccode != 0)
	{
		PrintError(ccode, "PutField", __LINE__,sessionInfo);
		goto Return;
	}
	(*MediaNum) ++;
	ccode=FlushMediaHeader( sessionInfo);
	if(ccode)
	{
		PrintError(ccode, "FlushMediaHeader", __LINE__,sessionInfo);
		goto Return;
	}

	Return:
		return(ccode);
}



CCODE ContinueMediaHeaderFromNextTape(SESSION_INFORMATION *sessionInfo)
{

	CCODE	ccode;
	BUFFERPTR bufferPtr;
	UINT32 _bufferSize;
	SMDF_FIELD_DATA		field;
	UINT16	mediaNumber;

	CloseDEFiles(sessionInfo);
	NextTape:
	PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_TAPE), sessionInfo->rctx->volno, sessionInfo->destFile);
	getchar();
	ccode= OpenDEFiles(O_RDONLY,sessionInfo);
	if(ccode)
		goto Return;
	ccode=GetMediaHeader( sessionInfo);
	if(ccode)
	{
		PrintMsg(NBK_ERR_MSG, _NbkMessage(NBK_ERR_READING_ARCH));
		ccode=NBK_ERR_READING_ARCH;
		goto Return;
	}
	bufferPtr=sessionInfo->mediaBuffer;
	_bufferSize=SECTOR_SIZE;

	 if ((GetField(&field, &bufferPtr, &_bufferSize) != 0)   ||
       (field.fid != SIDF_PARTITION_HEADER))
	 {
		ccode=1;
       	goto Return;
	 }

	 while( field.fid != SIDF_PARTITION_SET_SEQUENCE)
	 {
		if ((ccode = GetField(&field, &bufferPtr, &_bufferSize)) != 0)
		{
			goto Return;
		}
		if(  field.fid == SIDF_PARTITION_SET_SEQUENCE)
		{
			memcpy(&mediaNumber, field.data, sizeof(UINT16));
			break;
		}
	 }
	 if(mediaNumber != sessionInfo->rctx->volno)
	 {
		CloseDEFiles(sessionInfo);
		PrintMsg(NBK_VERBOSE_MSG, _NbkMessage(NBK_INSERT_CORRECT_TAPE), sessionInfo->rctx->volno, sessionInfo->sessionName);
		goto NextTape;
		
	 }

	 Return:
		return(ccode);

}
			


