/*
===============================================================================
Novell Software Developer Kit Sample Code License

Copyright (C) 2005, Novell, Inc.  All rights reserved.
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
  *  Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  *  Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.
  *  Neither the name of Novell, Inc. nor the names of its contributors
     may be used to endorse or promote products derived from this
     software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT
SHALL NOVELL, INC., THE COPYRIGHT OWNER, OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

NAME OF FILE:
	smuni.c

PURPOSE/COMMENTS:
	Implements UTF-8 Conversion and UTF-8 string premitives for NetWare

NDK COMPONENT NAME AND VERSION:
	SMS Developer Components

LAST MODIFIED DATE: 
	12 May 2005

===============================================================================
*/


/*
 * Headers and Defines
 */
 
#include <smsutapi.h>
#include<nunicode.h>
#include <smuni.h>

#define UNICODE_CHAR_SIZE		(sizeof(unicode_t))
#define UNI_LOCAL_DEFAULT       (-1)  
#define UNI_MAP_NO_CHAR         0


/*
 * Global variable declaration
 */

NWBOOLEAN importedAPI = FALSE; 
int (*imp_loc2utf8)(UniRuleTable_t handle, char * dest, const char * src, char noMapCh, int noMapFlag)   = NULL;
int (*imp_LocToUtf8Size) (UniRuleTable_t table, const char *str, size_t unmappedCharSize, int noMapFlag, size_t *utf8BufSize) = NULL;
int (*imp_utf82loc)(UniRuleTable_t handle, char * dest, const char * src, char noMapCh, int noMapFlag) = NULL;
int (*imp_Utf8ToLocSize) (UniRuleTable_t table,const char *str,size_t unmappedCharSize,int noMapFlag, size_t *locBufSize);

/*
 * Function prototypes
 */
void 	*ImportSymbol (int , const char *);
int 	UnimportSymbol(int ,const char *);

/*********************************************************************
* NAME: SMloc2utf8
*
* WHAT: Converts strings from local code page string to UTF-8, using LIBC APIs whereever 
*		available and CLib otherwise. This function is a wrapper to provide transparency.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMloc2utf8(utf8_t * dest, size_t *destLen,const char * src, int srclen, UNICODE_CONTEXT *uContext) 
{
    unicode_t *tempstr= NULL;
    UniRuleTable_t  table = UNI_LOCAL_DEFAULT;
    size_t   tempBufferLen; 
    int retval=0; 

    if(src == NULL || destLen == NULL)
        return -1; 
        
    if(dest == NULL) /* Do not convert , return len of the buffer required*/
    {
        if (imp_LocToUtf8Size ) 
        {
           return imp_LocToUtf8Size( table , src, 0, UNI_MAP_NO_CHAR, destLen) ;
        }
        else
        {
        	retval =NWUSByteToUnicode(NULL, NULL, (const unsigned char *)src, &tempBufferLen) ;
        	if( retval)
        		return retval;
        	tempBufferLen += sizeof(unicode_t);
        	tempstr = (unicode_t *) calloc(1, tempBufferLen*sizeof(unicode_t) );
        	if(tempstr == NULL)
            	return -1;
            retval = NWUSByteToUnicode((unsigned short *) tempstr, tempBufferLen, (const unsigned char *)src, &tempBufferLen);
            if( retval)
            {
            	free(tempstr);
        		return retval;
            }
            *destLen = NWLUnicodeToUTF8Size(tempstr);
            free (tempstr);
            return 0;
        }
    }
    else
    {
        if (imp_loc2utf8 )
        {
            return imp_loc2utf8 ( table , dest, src, 0, UNI_MAP_NO_CHAR) ;
        }
        else 
        {
        	retval =NWUSByteToUnicode(NULL, NULL,(const unsigned char *) src, &tempBufferLen);
        	if( retval)
        		return retval;
        	
        	tempBufferLen += sizeof(unicode_t);
        	tempstr = (unicode_t *) calloc(1, tempBufferLen*sizeof(unicode_t));
        	if(tempstr == NULL)
            	return -1;
        	
            retval = NWUSByteToUnicode((unsigned short *) tempstr, tempBufferLen, (const unsigned char *)src, &tempBufferLen);
            if( retval)
            {
            	free(tempstr);
        		return retval;
            }
            retval = NWLUnicodeToUTF8((const unsigned short *)tempstr, *destLen,(unsigned char *) dest, destLen);
            free(tempstr);
            return retval;
        }
    }
}

/*********************************************************************
* NAME: SMutf82loc
*
* WHAT: Converts UTF-8 string to local code page, using LIBC APIs whereever available 
*		and CLib otherwise. This function is a wrapper to provide transparency.
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int SMutf82loc( char * dest, size_t* destLen, const utf8_t * src,  int  srcLen,UNICODE_CONTEXT *uContext)
{
    unicode_t 		*tempstr =NULL;
	UniRuleTable_t  table = UNI_LOCAL_DEFAULT;
    size_t    		tempBufferLen; 
    int 			retval = 0;

    if(src == NULL || destLen == NULL)
        return -1; 
        
        
    if(dest == NULL) /* Do not convert , return len of the buffer required*/
    {
        if (imp_Utf8ToLocSize )
            return imp_Utf8ToLocSize ( table , src, 0, UNI_MAP_NO_CHAR, destLen) ;
        else
        {
        	retval = NWLUTF8ToUnicodeSize((const unsigned char *)src, &tempBufferLen);
           	if(retval)
           		return retval;
            tempstr = (unicode_t *) calloc (1, tempBufferLen );
            if(tempstr == NULL)
            	return -1;
                        
            retval = NWLUTF8ToUnicode((const unsigned char *)src, tempBufferLen, tempstr, &tempBufferLen, NULL);
            if(retval)
            {
            	free(tempstr);
            	return retval;
            }
            retval = NWUSUnicodeToByte(NULL, NULL, tempstr, destLen);
            free(tempstr);
            return retval;
        }
    }
    else
    {
         if (imp_utf82loc)
             return imp_utf82loc(table , dest, src, 0, UNI_MAP_NO_CHAR) ;
        else 
        {
        	retval = NWLUTF8ToUnicodeSize((const unsigned char *)src, &tempBufferLen);
           	if(retval)
           		return retval;
            tempstr = (unicode_t *) calloc (1, tempBufferLen );
            if(tempstr == NULL)
            	return -1;
                        
            retval = NWLUTF8ToUnicode((const unsigned char *)src, tempBufferLen, tempstr, &tempBufferLen, NULL);
            if(retval)
            {
            	free(tempstr);
            	return retval;
            }
            retval = NWUSUnicodeToByte((unsigned char *)dest, *destLen, tempstr, destLen);
            free(tempstr);
            return retval;
        }
     }
}

/*********************************************************************
* NAME: SMutf82loc
*
* WHAT: Imports the required LIBC APIs
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/

void SMImportUniAPIs(UINT32 nlmHandle)
{
	if( importedAPI == FALSE)
	{
	    imp_LocToUtf8Size    = ImportSymbol (nlmHandle, "LIBC@LocToUtf8Size"); 
	    imp_loc2utf8         = ImportSymbol (nlmHandle, "LIBC@loc2utf8");
	    imp_Utf8ToLocSize    = ImportSymbol (nlmHandle, "LIBC@Utf8ToLocSize");
	    imp_utf82loc         = ImportSymbol (nlmHandle, "LIBC@utf82loc");
	   
	    if(imp_LocToUtf8Size &&  imp_loc2utf8 && imp_Utf8ToLocSize && imp_utf82loc)
		{
	   		 importedAPI = TRUE;
		}
	}
}

/*********************************************************************
* NAME: SMUnImportUniAPIs
*
* WHAT: UnImports the Imported LIBC APIs
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
void SMUnImportUniAPIs(UINT32 nlmHandle)
{ 
    if(importedAPI)
    {
	    if( imp_LocToUtf8Size)
	   	{
	        UnimportSymbol(nlmHandle, "LIBC@LocToUtf8Size"); 
	        imp_LocToUtf8Size = NULL;
	    }
	    if(imp_loc2utf8 )
    	{
	        UnimportSymbol(nlmHandle, "LIBC@loc2utf8");
	         imp_loc2utf8= NULL;
    	}
	    if(imp_Utf8ToLocSize )
    	{
			UnimportSymbol(nlmHandle, "LIBC@Utf8ToLocSize");
			imp_Utf8ToLocSize= NULL;
    	}
        if(imp_utf82loc )
    	{
			UnimportSymbol(nlmHandle, "LIBC@utf82loc");
			imp_utf82loc= NULL;
    	}
    }
    importedAPI = FALSE;
}

/*********************************************************************
* NAME: SMutf8nicmp
*
* WHAT: Compares n bytes from the given UTF-8 strings
*
* SMS-SPECIFIC ROUTINES CALLED:
*
**********************************************************************/
int   SMutf8nicmp     (const  utf8_t *s1, const  utf8_t  *s2, unsigned int n)
{
	return strnicmp( s1, s2, n);
}


