#
# This is a Perl script to test the pool operations
#
#use File::Path;
use strict;

#
# Set the datastream for a file handle to the passed in value.
#
sub SetDataStream(*$)
{
	my $fh = $_[0];
	my $dataStream = $_[1];

	my $result;
	my $command;
	my $error;

	$command = "<virtualIO><datastream name=\"$dataStream\"/></virtualIO>";
	seek $fh, 0, 0;
	if (!syswrite($fh, $command, length($command)))	
	{
		$result .= "Unable to send datastream command to NDS management.  ";
		seek $fh, 0, 0;
		if (sysread($fh, $error, 1000)) 
		{
			$result .= $error;
		}
		$result .= "\n";
	}
	return $result;
}

#
# Write a command to a file and get the result
#
sub WriteCommand(*$)
{
	my $fh = $_[0];
	my $command = $_[1];

	my $result;
	my $reply;
	my $error;

	print("command=$command\n");
	seek $fh, 0, 0;
	if (!syswrite($fh, $command, length($command)))
	{
		$result .= "Unable to send command to virtual file.  ";
		seek $fh, 0, 0;
		if (sysread($fh, $error, 1000)) 
		{
			$result .= $error;
		}
		$result .= "\n";
	}
	else
	{
		seek $fh, 0, 0;
		my $readLen = 999;
		while ($readLen)
		{
			$readLen = sysread($fh, $reply, 1000);
			$result .= $reply;
		}
	}
	print("result=$result\n");
	return $result;
}


#
# Parse the result of the list partition operation
#
sub ParseListPartitions($)
{
	my $xml = $_[0];
	my $result;
	my @xml;
	my $ret;
	my $deviceName;
	my $deviceID;
	my $partID;
	my $label;
	my $numSectors;

#	print "ndsResult xml=$xml\n";
	@xml = $xml =~ /<partitionInfo>(.*?)<\/partitionInfo>/gs;
	foreach $result (@xml)
	{
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 != 0)
			{
				return "Error $1 getting partition info\n";
			}
		}

		$result =~ /<partitionType>(.*?)<\/partitionType>/s;
		if ($1 != 105) # NSS partition?
		{
			next;
		}
		$deviceName = "";
		$deviceID = "";
		$partID = "";
		$label = "";
		$numSectors = "";
		if ($result =~ /<deviceName>(.*?)<\/deviceName>/s)
		{
			$deviceName = $1;
		}
		if ($result =~ /<deviceID>(.*?)<\/deviceID>/s)
		{
			$deviceID = $1;
		}
		if ($result =~ /<logicalPartitionID>(.*?)<\/logicalPartitionID>/s)
		{
			$partID = $1;
		}
		if ($result =~ /<label>(.*?)<\/label>/s)
		{
			$label = $1;
		}
		if ($result =~ /<numSectors>(.*?)<\/numSectors>/s)
		{
			$numSectors = $1;
		}
		$ret .= "Device ID: $deviceID     Name: $deviceName\n".
				"Partition ID: $partID    Label: $label\n".
				"Number of sectors: $numSectors\n".
				"********************************************\n";
#print "partitionInfo=$result\n";

	}
	return $ret;
}

#
# List partitions to display for creating a pool
#
sub ListPartitions()
{
	my $result;
	my $ret;

	$result = "List of NSS partitions\n";
	$result .= "********************************************\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><partition><listPartitions/>".
					"</partition></nssRequest>");
		$result .= ParseListPartitions($nssResult);
	}
	return $result;
}

#
# Parse the result of the list partition operation
#
sub ParseListPools($)
{
	my $xml = $_[0];
	my $result;
	my @xml;
	my $ret;
	my $poolName;
	my $poolState;
	my $totalBytes;
	my $freeBytes;
	my $enabledAttributes;

	print "ndsResult xml=$xml\n";
	@xml = $xml =~ /<poolInfo>(.*?)<\/poolInfo>/gs;
	$ret = "List of NSS pools\n";
	$ret .= "********************************************\n";
	foreach $result (@xml)
	{
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 != 0)
			{
				return "Error $1 getting partition info\n";
			}
		}

		$poolName = "";
		$poolState = "";
		if ($result =~ /<poolName>(.*?)<\/poolName>/s)
		{
			$poolName = $1;
		}
		if ($result =~ /<poolState>(.*?)<\/poolState>/s)
		{
			$poolState = $1;
		}
		if ($result =~ /<totalBytes>(.*?)<\/totalBytes>/s)
		{
			$totalBytes = $1;
		}
		if ($result =~ /<freeBytes>(.*?)<\/freeBytes>/s)
		{
			$freeBytes = $1;
		}
		if ($result =~ /<enabledAttributeBits>(.*?)<\/enabledAttributeBits>/s)
		{
			$enabledAttributes = $1;
		}

		$ret .= "Pool Name: $poolName\n".
				"State: $poolState    Attributes: $enabledAttributes\n".
				"Size: $totalBytes    Free: $freeBytes\n".
				"********************************************\n";
#print "partitionInfo=$result\n";

	}
	return $ret;
}

#
# Parse the result of adding or removing a pool
#
sub ParseResult($$)
{
	my $xml = $_[0];
	my $pool = $_[1];
	my $result = "";

	print "ndsResult xml=$xml\n";
	if ($xml =~ /<addPool>(.*)<\/addPool>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "Pool $pool successfully created\n";
			}
			else
			{
				return "Error $1 creating pool\n";
			}
		}
	}
	elsif ($xml =~ /<removePool>(.*)<\/removePool>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "Pool $pool successfully removed\n";
			}
			else
			{
				return "Error $1 removing pool $pool\n";
			}
		}
	}
	elsif ($xml =~ /<renamePool>(.*)<\/renamePool>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "Pool $pool successfully renamed\n";
			}
			else
			{
				return "Error $1 renaming pool $pool\n";
			}
		}
	}
	elsif ($xml =~ /<getNDSName>(.*)<\/getNDSName>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				if ($result =~ /<ndsName>(.*)<\/ndsName>/s)
				{
					my $ndsName = $1;
					if ($result =~ /<context>(.*)<\/context>/s)
					{
						return "The NDS name for $pool is $ndsName.\n".
								"The context of the object is $1.\n";
					}
					else
					{
						return "Unable to find element containing the NDS context.\n";
					}
				}
				else
				{
					return "Unable to find element containing the NDS name.\n";
				}
			}
			else
			{
				return "Error $1 getting the pool's NDS name\n";
			}
		}
	}
	else
	{
		return "Error parsing results of NSS operation\n";
	}
}

#
# List Pools
#
sub ListPools($)
{
	my $result;
	my $ret;
	my $server = $_[0];

	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><pool><listPools/>".
					"</pool></nssRequest>");
		$result .= ParseListPools($nssResult);
	}
	return $result;
}

#
# Add Pool
#
sub AddPool($$$$@)
{
	my $result;
	my $ret;
	my $server = $_[0];
	my $pool = $_[1];
	my $ndsName = $_[2];
	my $context = $_[3];
	my @partitions = $_[4];
	my $partitionID;

	$result = "Creating pool $pool($ndsName) on $server\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $request = "<nssRequest><pool><addPool>".
					"<lssType>ZLSS</lssType>".
					"<poolName>$pool</poolName>".
					"<ndsName>$ndsName</ndsName>".
					"<context>$context</context>".
#					"<ignoreShareState/>".
					"<partitions>";
		foreach $partitionID (@partitions)
		{
			$request .= "<partitionID>$partitionID</partitionID>";
		}
		$request .= "</partitions></addPool></pool></nssRequest>";
		my $nssResult = WriteCommand(NSSFILE, $request);
		$result .= ParseResult($nssResult, $pool);
	}
	return $result;
}

#
# Rename a pool
#
sub RenamePool($$)
{
	my $result;
	my $ret;
	my $pool = $_[0];
	my $newPool = $_[1];

	$result = "Renaming $pool to $newPool\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><pool><renamePool>".
					"<poolName>$pool</poolName>".
					"<newPoolName>$newPool</newPoolName>".
					"<newNDSName/>".
					"</renamePool></pool></nssRequest>");
		$result .= ParseResult($nssResult, $pool);
	}
	return $result;
}

#
# Remove Pool
#
sub RemovePool($$)
{
	my $result;
	my $ret;
	my $pool = $_[0];
	my $removeNDS = $_[1];
	my $remove = "";

	$removeNDS =~ tr/[A-Z]/[a-z]/;
	if ($removeNDS eq "n" || 
		$removeNDS eq "no")
	{
		$remove = "<dontRemoveNDSObject/>";
	}

	$result = "Removing $pool\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><pool><removePool>".
					"<poolName>$pool</poolName>".
					$remove.
#					"<ignoreShareState/>".
					"</removePool></pool></nssRequest>");
		$result .= ParseResult($nssResult, $pool);
	}
	return $result;
}

#
# Get NDS name for the pool
#
sub GetNDSName($)
{
	my $result;
	my $ret;
	my $pool = $_[0];

	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><pool><getNDSName>".
					"<poolName>$pool</poolName>".
					"</getNDSName></pool></nssRequest>");
		$result .= ParseResult($nssResult, $pool);
	}
	return $result;
}

#
# Main
#
{
	if ($#ARGV < 0 || $#ARGV > 0 || $ARGV[0] eq "/?" || $ARGV[0] eq "-?") 
	{
		print "USAGE: pool.pl server\n".
			"Example: pool server1 \n".
			"If running from a server use \".\" for the server name.";
		exit;
	}

	my $server = $ARGV[0];

	if ($server eq ".")
	{
		open(NSSFILE, "+<_admin:Manage_NSS\\manage.cmd") 
			or die "Error opening NSS management file ($!) on server";
	}
	else
	{
		open(NSSFILE, "+<\\\\$server\\_admin\\Manage_NSS\\manage.cmd") 
			or die "Error opening NSS management file ($!)";
	}

	MAIN: while(TRUE) 
	{
		my $action;

		do 
		{
			print	"1. List pools\n".
					"2. Add pool\n".
					"3. Remove pool\n".
					"4. Rename pool\n".
					"5. Get pool NDS name\n".
					"Select action: ";
			chomp($action = <STDIN>);
		}while ($action != 0  && ($action < 1 || $action > 5));

		if ($action == 1)
		{
			print ListPools($server);
		}
		elsif ($action == 2) 
		{
			print "Pool name: ";
			chomp(my $pool = <STDIN>);
			print "NDS name: ";
			chomp(my $ndsName = <STDIN>);
			print "Context: ";
			chomp(my $context = <STDIN>);
			my $numPartitions = 0;
			my @partitionIDs = ();
			print ListPartitions();
			while(TRUE)
			{
				print "Partition ID (return to quit): ";
				chomp(my $partitionID = <STDIN>);
				if ($partitionID eq "")
				{
					last;
				}
				$partitionIDs[$numPartitions] = $partitionID;
			}

			my @results = AddPool($server, $pool, $ndsName, $context, 
				@partitionIDs);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 3)
		{
			print "Pool name: ";
			chomp(my $pool = <STDIN>);
			print "Remove NDS name (default = yes)? ";
			chomp(my $remove = <STDIN>);
			my @results = RemovePool($pool, $remove);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 4)
		{
			my $poolName;
			my $newPoolName;

			print "pool name: ";
			chomp(my $poolName = <STDIN>);
			print "new pool name: ";
			chomp(my $newPoolName = <STDIN>);
			my @results = RenamePool($poolName, $newPoolName);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 5)
		{
			print "Pool name: ";
			chomp(my $pool = <STDIN>);
			print GetNDSName($pool);
			print "\n";
		}
		else
		{
			last MAIN;
		}
	}
	close(NSSFILE);
}