#
# This is a Perl script to test the volume operations
#
#use File::Path;

#
# Set the datastream for a file handle to the passed in value.
#
sub SetDataStream(*$)
{
	my $fh = $_[0];
	my $dataStream = $_[1];

	my $result;
	my $command;

	$command = "<virtualIO><datastream name=\"$dataStream\"/></virtualIO>";
	seek $fh, 0, 0;
	if (!syswrite($fh, $command, length($command)))	
	{
		$result .= "Unable to send datastream command to NDS management.  ";
		seek $fh, 0, 0;
		if (sysread($fh, $error, 1000)) 
		{
			$result .= $error;
		}
		$result .= "\n";
	}
	return $result;
}

#
# Write a command to a file and get the result
#
sub WriteCommand(*$)
{
	my $fh = $_[0];
	my $command = $_[1];

	my $result;

	print("command=$command\n");
	seek $fh, 0, 0;
	if (!syswrite($fh, $command, length($command)))
	{
		$result .= "Unable to send command to virtual file.  ";
		seek $fh, 0, 0;
		if (sysread($fh, $error, 1000)) 
		{
			$result .= $error;
		}
		$result .= "\n";
	}
	else
	{
		seek $fh, 0, 0;
		sysread($fh, $reply, 1000);
		$result .= $reply;
	}
#	print("result=$result\n");
	return $result;
}


#
# Parse the result of adding or removing a volume
#
sub ParseResult($$)
{
	my $xml = $_[0];
	my $name = $_[1];
	my $result = "";

	print "ndsResult xml=$xml\n";
	if ($xml =~ /<addVolume>(.*)<\/addVolume>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "$name successfully created\n";
			}
			else
			{
				if ($1 == -606)
				{
					return "$name already exists as an object in NDS\n";
				}
				else
				{
					return "Error $1 creating $name\n";
				}
			}
		}
	}
	elsif ($xml =~ /<removeVolume>(.*)<\/removeVolume>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "volume $name successfully removed\n";
			}
			else
			{
				if ($1 == -601)
				{
					return "Volume $name not found in NDS\n";
				}
				else
				{
					return "Error $1 removing volume $name\n";
				}
			}
		}
	}
	elsif ($xml =~ /<renameVolume>(.*)<\/renameVolume>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "volume $name successfully renamed\n";
			}
			else
			{
				if ($1 == -601)
				{
					return "Volume $name not found in NDS\n";
				}
				else
				{
					return "Error $1 renaming volume $name\n";
				}
			}
		}
	}
	elsif ($xml =~ /<modifyVolumeAttributes>(.*)<\/modifyVolumeAttributes>/s)
	{
		$result = $1;
		if ($result =~ /<result value=\"(-?\d+)\">/s)
		{
			if ($1 == 0)
			{
				return "volume $name successfully modified\n";
			}
			else
			{
				if ($1 == -601)
				{
					return "Volume $name not found in NDS\n";
				}
				else
				{
					return "Error $1 modifying volume $name\n";
				}
			}
		}
	}
	else
	{
		return "Error parsing results of NSS operation\n";
	}
}

#
# Create a volume
#
sub CreateVolume($$$$)
{
	my $result;
	my $ret;
	my $context = $_[0];
	my $volumeName = $_[1];
	my $poolName = $_[2];
	my $server = $_[3];

	$result = "Creating Volume \"$_[1]\"\n";
	#
	# Create a volume
	#
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><volume><addVolume state=\"mounted\">".
					"<volumeName>$volumeName</volumeName>".
					"<poolName>$poolName</poolName>".
					"<context/>".
					"<ndsPoolName/>".
					"<ndsName/>".
#					"<context>$context</context>".
#					"<ndsPoolName>kona_pool2_pool</ndsPoolName>".
#					"<ndsName>vol_$volumeName</ndsName>".
					"</addVolume></volume></nssRequest>");
		$result .= ParseResult($nssResult, $name);
	}
	return $result;
}

#
# Remove a volume
#
sub RemoveVolume($$)
{
	my $result;
	my $ret;
	my $vol = $_[0];
	my $removeNDS = $_[1];
	my $remove = "";

	$removeNDS =~ tr/[A-Z]/[a-z]/;
	if ($removeNDS eq "n" || 
		$removeNDS eq "no")
	{
		$remove = "<dontRemoveNDSObject/>";
	}

	$result = "Removing $vol\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><volume><removeVolume>".
					"<volumeName>$vol</volumeName>".
					$remove.
					"</removeVolume></volume></nssRequest>");
		$result .= ParseResult($nssResult, $vol);
	}
	return $result;
}

#
# Rename a volume
#
sub RenameVolume($$)
{
	my $result;
	my $ret;
	my $vol = $_[0];
	my $newVol = $_[1];

	$result = "Renaming $vol to $newVol\n";
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><volume><renameVolume>".
					"<volumeName>$vol</volumeName>".
					"<newVolumeName>$newVol</newVolumeName>".
					"<newNDSName/>".
					"</renameVolume></volume></nssRequest>");
		$result .= ParseResult($nssResult, $vol);
	}
	return $result;
}

#
# Modify a volume's attributes
#
sub ModifyVolume($%)
{
	(my $volumeName, my %attributes) = @_;

	$result = "Modifying Volume \"$volumeName\"\n";
	#
	# Modify the volume
	#
	if (!($ret = SetDataStream(NSSFILE, "command")))
	{
		my $value;
		my $atrElements;
		my $attrib;

		my @keys = keys %attributes;
		for $attrib (@keys)
		{
			if ($attrib eq 'volumeQuota')
			{
				$atrElements .= "<$attrib quota=\"".$attributes{$attrib}."\"/>";
			}
			elsif ($attrib eq 'shredding')
			{
				$atrElements .= "<$attrib count=\"".$attributes{$attrib}."\"/>";
			}
			else
			{
				$atrElements .= "<$attrib enabled=\"".$attributes{$attrib}."\"/>";
			}
		}
		$result .= $ret;
		my $nssResult = WriteCommand(NSSFILE, 
					"<nssRequest><volume><modifyVolumeAttributes>".
					"<volumeName>$volumeName</volumeName>".
					$atrElements.
					"</modifyVolumeAttributes></volume></nssRequest>");
		$result .= ParseResult($nssResult, $volumeName);
	}
	return $result;
}

#
# Main
#
{
	my $volumeName;
	my $poolName;
	my $ndsContext = "";

	if ($#ARGV < 0 || $#ARGV > 1 || $ARGV[0] eq "/?" || $ARGV[0] eq "-?") 
	{
		print "USAGE: volume.pl server [ndsContext]\n".
			"Example: volume server1 server1_tree\\novell\n".
			"If running from a server use \".\" for the server name.";
		exit;
	}

	my $server = $ARGV[0];
	if ($#ARGV > 0)
	{
		$ndsContext = $ARGV[1];
	}

	if ($server eq ".")
	{
		open(NSSFILE, "+<_admin:Manage_NSS\\manage.cmd") 
			or die "Error opening NSS management file ($!) on server";
	}
	else
	{
		open(NSSFILE, "+<\\\\$server\\_admin\\Manage_NSS\\manage.cmd") 
			or die "Error opening NSS management file ($!)";
	}

	MAIN: while(TRUE) 
	{
		do 
		{
			print	"1. Add a volume\n".
					"2. Remove a volume\n".
					"3. Update a volume\'s attributes\n".
					"4. Rename a volume\n".
					"Select action: ";
			chomp($action = <STDIN>);
		}while ($action != 0  && ($action < 1 || $action > 4));

		if ($action == 1) 
		{
			print "volume name: ";
			chomp(my $volumeName = <STDIN>);
			print "pool name: ";
			chomp(my $poolName = <STDIN>);

			my @results = CreateVolume($ndsContext, $volumeName, $poolName,
				$server);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 2)
		{
			print "volume name: ";
			chomp(my $volumeName = <STDIN>);
			print "Remove NDS name (default = yes)? ";
			chomp(my $remove = <STDIN>);
			my @results = RemoveVolume($volumeName, $remove);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 3)
		{
			my %attributes;

			print("Turn attributes on? ");
			chomp($action = lc(<STDIN>));
			if (($action eq 'y') || ($action eq 'yes'))
			{
				$attributes{volumeQuota} = "4096000000";
				$attributes{salvage} = "yes";
				$attributes{compression} = "yes";
				$attributes{directoryQuota} = "yes";
				$attributes{userQuota} = "yes";
				$attributes{flushFiles} = "yes";
				$attributes{shredding} = "5";
				$attributes{mfl} = "yes";
				$attributes{snapshot} = "yes";
				$attributes{backup} = "yes";
			}
			else
			{
				$attributes{volumeQuota} = "none";
				$attributes{salvage} = "no";
#				$attributes{compression} = "no";
				$attributes{directoryQuota} = "no";
				$attributes{userQuota} = "no";
				$attributes{flushFiles} = "no";
				$attributes{shredding} = "0";
				$attributes{mfl} = "no";
				$attributes{snapshot} = "no";
				$attributes{backup} = "no";
			}
			print "volume name: ";
			chomp(my $volumeName = <STDIN>);

			my @results = ModifyVolume($volumeName, %attributes);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		elsif ($action == 4)
		{
			my $newVolumeName;

			print "volume name: ";
			chomp(my $volumeName = <STDIN>);
			print "new volume name: ";
			chomp(my $newVolumeName = <STDIN>);
			my @results = RenameVolume($volumeName, $newVolumeName);
			print "---------------------------\n\n";
			print @results;
			print "\n---------------------------\n\n";
		}
		else
		{
			last MAIN;
		}
	}
	close(NSSFILE);
}