/****************************************************************************
 |	(C) Copyright 2001 Novell, Inc.
 |	All Rights Reserved.
 |
 |	This program is free software; you can redistribute it and/or
 |	modify it under the terms of version 2 of the GNU General Public
 |	License as published by the Free Software Foundation.
 |
 |	This program is distributed in the hope that it will be useful,
 |	but WITHOUT ANY WARRANTY; without even the implied warranty of
 |	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 |	GNU General Public License for more details.
 |
 |	You should have received a copy of the GNU General Public License
 |	along with this program; if not, contact Novell, Inc.
 |
 |	To contact Novell about this file by physical or electronic mail,
 |	you may find current contact information at www.novell.com
 |
 |***************************************************************************
 |
 |	 Novell Distributed File Services (DFS)
 |
 |---------------------------------------------------------------------------
 |
 | $Author: stoner $
 | $Date: 2005/04/13 16:38:08 $
 |
 | $RCSfile: jsmsg.h,v $
 | $Revision: 1.3 $
 |
 |---------------------------------------------------------------------------
 |	This module is used to:
 |		DFS communications declarations/interfaces for applications
 +-------------------------------------------------------------------------*/

#ifndef _JSMSG_H_
#define _JSMSG_H_

/* Dependencies */
#ifndef _OMNI_H_
#include <omni.h>		/* Defines types required by Jetstream, including GUID_t */
#endif
#if zNETWARE
#ifndef _OSMPKHDRS_H_
#include <osmpkhdrs.h>	/* Includes OS MPK definitions/declarations */
#endif
#endif
#ifndef _QUE_H_
#include <que.h>
#endif

#ifdef __cplusplus
extern "c"
{
#endif


#ifdef __linux__
#define THREAD pthread_t
typedef int SEMAPHORE;
#define MAX_NAME_LENGTH		100
#endif

/*
	Manifest constants
*/
#define		JS_LARGE_BUFFER_BYTES			(0x10000)				/* Jetstream's default large buffer size (in bytes) */
#define		JS_MEDIUM_BUFFER_BYTES			(0x1000)				/* Jetstream's default medium buffer size (in bytes) */
#define		JS_SMALL_BUFFER_BYTES			(0x100)					/* Jetstream's default small buffer size (in bytes) */
#define		JS_WAIT_FOREVER					(0xFFFFFFFF)			/* Used in functions that take a delay value (e.g. JS_AllocateMessage) to indicate the function should block */
#define		JS_ARRAY_STAT_BIT				(0x40000000)			/* Bit that indicates an array statistic in statistic IDs */
#define		JS_PEAK_STAT_BIT				(0x20000000)			/* Bit that indicates a peak statistic in statistic IDs */
#define		JS_ARRAY_STAT_MASK				(~JS_ARRAY_STAT_BIT)	/* AND mask to remove the ARRAY_STAT_BIT */
#define		JS_PEAK_STAT_MASK				(~JS_PEAK_STAT_BIT)		/* AND mask to remove the PEAK_STAT_BIT */
#define		JS_NAMED_CONNECT_REQUEST		(0x10)					/* ControlCode for a named connect initial request */
#define		JS_NAMED_CONNECT_RESPONSE		(0x11)					/* ControlCode for a named connect initial response */
#define		JS_PHYSICAL_CONNECT_REQUEST		(0x12)					/* ControlCode for a physical connect request */
#define		JS_PHYSICAL_CONNECT_RESPONSE	(0x13)					/* ControlCode for a physical connect response */
#define		JS_AUTHENTICATE_ME_AS_1			(0x14)					/* Control code for a connector to request the remote identity while supplying its own identity */
#define		JS_AUTHENTICATE_ME_AS_2			(0x15)					/* Control code for a response to a JS_AUTHENTICATE_ME_AS_1 message */
#define		JS_CONNECT_COMPLETE				(0x16)					/* ControlCode for a message containing a connection complete message */
#define		JS_DISCONNECT_REQUEST			(0x20)					/* ControlCode for a disconnect session intial request */
#define		JS_DISCONNECT_RESPONSE			(0x21)					/* ControlCode for a disconnect session response */
#define		JS_DISCONNECT_COMPLETE			(0x22)					/* ControlCode to indicate to a consumer that an asynchronous disconnect is complete */
#define		JS_DESTROY_USER_BUFFER			(0x30)					/* ControlCode to indicate to a consumer or transport that the message has a user buffer and it must be destroyed */
#define		JS_GET_STATISTICS_IDS			(0x40)					/* ControlCode to indicate a request for the list of supported statistics */
#define		JS_STATISTICS_IDS				(0x41)					/* ControlCode to indicate a response containing the list of supported statistics */
#define		JS_GET_STATISTICS_VALUES		(0x42)					/* ControlCode to retrieve a particular set of statistics values */
#define		JS_STATISTICS_VALUES			(0x43)					/* ControlCode to indicate a response containing the list of requested statistics */

/* Error codes */
#define		JS_GENERAL_ERROR					(0xFFFFFFFF)		/* Non-specific error code for message control code field or function return values */
#define		JS_ERR_INVALID_SESSION				(0xFFFFFFFE)		/* Supplied session is not found in the Jetstream session list or is disconnected */
#define		JS_ERR_MESSAGE_ALLOCATION_FAILED	(0xFFFFFFFD)		/* An attempt to allocate a Jetstream message failed */
#define		JS_ERR_TRANSPORT_LIST_FAILURE		(0xFFFFFFFC)		/* The access controls for the registered transport list have failed */
#define		JS_ERR_TRANSPORT_NOT_FOUND			(0xFFFFFFFB)		/* The transport module that created the supplied session is no longer registered */
#define		JS_ERR_INTERNAL_FAILURE				(0xFFFFFFFA)		/* Internal failure that may prevent further Jetstream operation */
#define		JS_ERR_OPERATION_WILL_NOT_COMPLETE	(0xFFFFFFF9)		/* Requested operation will not complete, probably due to Jetstream shutdown in progress */
#define		JS_ERR_CONSUMER_NOT_FOUND			(0xFFFFFFF8)		/* The consumer identified in the request is not not registered */


/*
	The following macros define constant of type GUID_t used in Jetstream.
	Due to limitations of the language it is not possible to use these
	constants in assignments or in arguments.  They may only be used as
	initializers in declarations, e.g. the following example use is allowed:

	GUID_t	pres_id = JS_PRESENTATION_LAYER_ID_INITIALIZER;

	SomeFunc(pres_id);


	but the following example uses are not and will fail to compile:

	GUID_t pres_id;

	pres_id = JS_PRESENTATION_LAYER_ID_INITIALIZER;
	SomeFunc(JS_PRESENTATION_LAYER_ID_INITIALIZER);

*/
#define		JS_PRESENTATION_LAYER_ID_INITIALIZER	{0x36662a8a, 0x9170, 0x11d3, 0xab, 0x89, {0x0, 0x8, 0xc7, 0xd3, 0xaf, 0x2d}}	/* ID for the Jetstream internal consumer */
#define		JS_NO_MODULE_INITIALIZER				{0, 0, 0, 0, 0, {0, 0, 0, 0, 0, 0}}												/* ID indicating no specific ID */


/* New types */
typedef		struct	JS_CommMsg_s						/* A Jetstream message */
{
	DQlink_t  		QueueLink;								/* Doubly linked list node pointers */
	unicode_t	*	Session;								/* Jetstream logical session the message is to be (was) transported with */
	GUID_t			SourceModule;							/* Source consumer's module ID */
	GUID_t			TargetModule;							/* Target consumer's module ID */
	SLONG			ControlCode;							/* Message control (positive) or error (negative) code */
	LONG			BufferLength;							/* The number of bytes available in MessageData */
	LONG			MessageLength;							/* The number of bytes in use in MessageData */
	void		*	MessageData;							/* The message payload */
	union
	{
		void	*	Ptr1;
		LONG		Long1;
	} OwnerData1;											/* For consumer use */
	union
	{
		void	*	Ptr2;
		LONG		Long2;
	} OwnerData2;											/* For consumer use */
} JS_CommMsg_s;

typedef struct JS_NamedConnectReq_s
{
	GUID_t		TransactionID;			/* Transaction ID of this request */
	unicode_t	Name[1];				/* First character of name, remainder of name extends beyond C structure */
} JS_NAMED_CONN_REQ_t;

typedef struct JS_TransportNamedConnectResp_s
{
	GUID_t		 TransactionID;			/* Transaction ID of the original request */
	GUID_t		 TransportID;			/* ID of the transport that resolved the name */
	BOOL		 Resolved;				/* TRUE if this response indicates the name was resolved */
	LONG		 ConnectDataLen;		/* Length of the following data */
	char		 ConnectData[1];		/* First byte of the physical connect request data, remainder of data extends beyond C structure */
} JS_TRANSPORT_NAMED_CONN_RESP_t;

typedef struct JS_PhysicalConnectReq_s
{
	GUID_t		TransactionID;			/* Transaction ID of the original request */
	LONG		ConnectDataLen;			/* Length of the following data */
	char		ConnectData[1];			/* First byte of the physical connect request data, remainder of data extends beyond C structure */
} JS_PHYSICAL_CONN_REQ_t;

typedef struct JS_PhysicalConnectResp_s
{
	GUID_t		TransactionID;			/* Transaction ID of original connect request */
	BOOL		Connected;				/* TRUE if connection was made */
} JS_PHYSICAL_CONN_RESP_t;

typedef struct JS_ConsumerNamedConnectResp_s
{
	GUID_t		TransactionID;			/* Transaction ID of the original request */
	BOOL		Connected;				/* TRUE if connection was made */
} JS_CONSUMER_NAMED_CONN_RESP_t;

typedef	struct	JS_DisconnectRequest_s
{
	GUID_t		 TransactionID;			/* Transaction ID of original request */
	void		*TransportSession;		/* Transport suppled session identity */
} JS_DISCONNECT_REQ_t;

typedef struct JS_DisconnectResponse_s
{
	GUID_t		 TransactionID;			/* Transaction ID of original request */
	STATUS		 status;				/* zOK if disconnected */
} JS_DISCONNECT_RESP_t;

typedef struct JS_ConnNameIsMsg_s
{
	GUID_t		TransactionID;			/* Transaction ID of original request */
	LONG		NameLen;				/* Length of connection name string in bytes (including any null terminator) */
	unicode_t	ConnName[1];			/* Name of connection.  NB: data extends beyond end of structure as defined */
} JS_CONN_NAME_IS_t;


/* Function prototypes */

/* Module Registration */
STATUS 				 JS_RegisterConsumer(GUID_t ConsumerID, SEMAPHORE RxSignal);							/* Register a Jetstream consumer */
STATUS 				 JS_DeregisterConsumer(GUID_t ModuleID, JS_CommMsg_s *ShutdownMessage);				/* De-register a Jetstream consumer */

/* Connection management */
unicode_t			*JS_ConnectToHost(GUID_t OriginatorID, unicode_t *TargetHostName, GUID_t TransactionID, BOOL Asynchronous, BOOL MutualAuth, unicode_t *UserID, char *Password, LONG Timeout);
																										/* Connect to a named host */
STATUS				 JS_DisconnectSession(GUID_t ConsumerID, unicode_t *Session, GUID_t TransactionID, BOOL Asynchronous);
																										/* Disconnect a session */
/* Message Communication */
JS_CommMsg_s		*JS_DequeueMessage(GUID_t ModuleID);													/* Used by a consumer to retrieve messages sent to it */
STATUS				 JS_QueueMessage(GUID_t SourceModule, JS_CommMsg_s * Message, GUID_t TargetModule);	/* Used by a consumer to transmit messages */
STATUS				 JS_RequestReceiveEvent(GUID_t ModuleID);											/* Used by a consumer to indicate it requires a semaphore signal when received data is available */

/* Dispatched session message communication */
JS_CommMsg_s		*JS_DequeueDispatchedMessage(GUID_t ModuleID, unicode_t *Session);					/* Used by a consumer to receive messages for a particular session independently of messages for other sessions */
STATUS				 JS_DispatchSession(GUID_t ModuleID, unicode_t *Session, SEMAPHORE RxSignal);		/* Used by a consumer to indicate that messages received on the supplied session should be queued for the consumer independently of those for other sessions */
STATUS				 JS_RequestDispatchedMessageReceiveEvent(GUID_t ModuleID, unicode_t *Session);		/* Used by a consumer to indicate that it requires a semaphore to be signaled when received data is available on a particular session */
STATUS				 JS_UndispatchSession(GUID_t ModuleID, unicode_t *Session, BOOL Discard);			/* Used by Jetstream to stop receiveing messages for a particular session independently of those for other sessions */

/* Message Management */
JS_CommMsg_s		*JS_AllocateMessage(GUID_t	ModuleID, size_t MessageSize, LONG MaxTimeToWait);		/* Supply a free message on demand */
STATUS				 JS_DuplicateMessage(JS_CommMsg_s *DestMsg, JS_CommMsg_s * SrcMsg);					/* Copy one message to another */
STATUS				 JS_GetDefaultMessageSizes(LONG *Sizes, LONG *Count);								/* Get the list of message size allocations that Jetstream supports */
STATUS				 JS_RetireMessage(GUID_t ModuleID, JS_CommMsg_s * Message);							/* Free a reviously allocated message */
STATUS				 JS_SetMessageSession(JS_CommMsg_s * Message, unicode_t * Session);					/* Associate a message with a logical session */


#ifdef __cplusplus
}
#endif

#endif
