#!/bin/bash
#
# Author: Frieder Schmidt <frieder.schmidt@microfocus.com>
#
# copyright (c) Novell Deutschland GmbH, 2001-2023. All rights reserved.
# GNU Public License
#
# zman_bundle_assign.sh					 8 Apr 2022
# last modified (extension for OES2023)                 25 May 2023


ARGV=$@
SCRIPT_NAME=$0

function debug()
{
        if [ "$1" != "0" ];then
                set -x
        fi
}


function usage()
{
        cat <<HERE

usage $0

        --parameter     | -p  <parameter file>  # file containing the parameters that describe
                                                # the ZCM zone structure
                                                # default: ../zman_params.cfg

	--output        | -o  <output file>     # file that will contain the zman commands for
                                                # the bundle assignments
                                                # default: zman_bundle_assign.in 

	--debug         | -d                    # optional


        example: $0 --prameter ../zman_params_OES2013.cfg --output zman_bundle_assign.in.OES2023

HERE
        exit 1
}


# retrieve cmd line parameters
function getopt()
{
        while [ $# -gt 0 ];do
                case $1 in --parameter|-p ) PARAM_FILE=$2; shift;;
                           --output|-o    ) OUT_FILE=$2; shift;;
                           --debug|-d     ) debug=1; debug $debug;;
                           --help|-h      ) clear; usage;;

                           *              ) clear; echo -e "\n\n\a\tUnknown parameter  > $1 <  provided!\n";
                                            usage;;
                esac
                shift
        done
}


function set_vars ()
{
        my_date=$(date +"%d.%m.%Y")
        my_time=$(date +"%H:%M")

	OUT_FILE_DEFAULT="zman_bundle_assign.in"
	PARAM_FILE_DEFAULT="../zman_params.cfg"

	OUT_FILE=${OUT_FILE:-${OUT_FILE_DEFAULT}}
	PARAM_FILE=${PARAM_FILE:-${PARAM_FILE_DEFAULT}}
}


function create_env ()
{
	# source parameter file
	source "${PARAM_FILE}"
	
	if [ -f "${OUT_FILE}" ]; then
		rm "${OUT_FILE}"
	fi

        # create zman input file
        cat <<HERE > ${OUT_FILE}
## ******  ${OUT_FILE} created on $my_date at $my_time   *******
## -------------------------------------------------------------------------

HERE
}


function assing_bundles ()
{
	folder="eDir-BASECONFIG"
	echo "##   Bundle Folder: ${folder}" >> ${OUT_FILE}
	for bundle in EDIR_MFC-disable-FIPS EDIR_NOV-NMAS-CONF; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in ${SERVER_CLASS}; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	folder="iMan-BASECONFIG"
	echo "##   Bundle Folder: ${folder}" >> ${OUT_FILE}
	for bundle in IMAN_NOV-change-config-xml IMAN_NOV-users; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in eDir-iMan eDir-iMan-DA; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	folder="OES-SERVICES"
	echo "##   Bundle Folder: ${folder}" >> ${OUT_FILE}
	# bundles for all OES systems
	for bundle in OESALL_NOV-NAM-CONF OESALL_NOV-NCPSERV-CONF OES2018SP3_MFC-DISABLE-OES-AGENTS; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in ${SERVER_CLASS}; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	# bundles for OES systems with NSS
	for bundle in OES11ALL_NOV-CHECK-MOVE OES11ALL_NOV-NLVM-CMD-COMPLETIONS OES11ALL_NOV-NLVM-CONF OESALL_NOV-BASHRC \
		      OESALL_NOV-CIFS-CONFIG OESALL_NOV-NSSSTART-CFG; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in Branch Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	# bundles for OES cluster nodes with PureFTP
	for bundle in OESALL_NOV-PureFTP-local-bind; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	for bundle in OESALL_NOV-PureFTP-SYSLOG_PRD-Cluster1; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in PRD; do 
		for class in Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	folder="SLES-SERVICES"
	echo "##   Bundle Folder: ${folder}" >> ${OUT_FILE}
	for bundle in SLESALL_Customer-SSH-Keys SLESALL_MFC-DISABLE-ZISLNX-SERVICE SLESALL_NOV-BOOTSPLASH-CONFIG \
		      SLESALL_NOV-no-SuSE-Register SLESALL_NOV-NTP-SYSCONFIG SLESALL_NOV-ZYPP-CONFIG; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in ${SERVER_CLASS}; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    for class in _Infrastructure; do
		echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${class} \
			       ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
	    done
	    echo "" >> ${OUT_FILE}
	done

	for bundle in SLESALL_NOV-SLPDA-CONFIG; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in eDir-iMan-DA eDir-UMC-DA; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
	
	for bundle in SLES12ALL_MFC-LFTP-CONFIG; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in Branch Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done

	for bundle in SLESALL_MFC-DHCP-BOND0; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done

	for bundle in SLES15ALL_MFC-enable-chrony-server; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in eDir eDir-DA eDir-iMan eDir-iMan-DA eDir-UMC eDir-UMC-DA; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done

	folder="SLES-STORAGE"
	echo "##   Bundle Folder: ${folder}" >> ${OUT_FILE}
	for bundle in SLESALL_NOV-MULTIPATH-CONF; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in ${DEVICE_ENVIRONMENTS}; do 
		for class in Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done

	for bundle in SLESALL_NOV-BINDINGS_PRD-Cluster1; do
	    echo "#    Assignments for Bundle: ${bundle}"	>> ${OUT_FILE}
	    for dev in PRD; do 
		for class in Cluster1; do
		    echo "zman bga ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${dev}_${class} \
				   ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${folder}/${bundle}" >> ${OUT_FILE}
		done
	    done
	    echo "" >> ${OUT_FILE}
	done
}


function main ()
{
        getopt  ${ARGV}
        set_vars
	create_env
	assing_bundles
}

main	
