#!/bin/bash
#
# Author: Frieder Schmidt <frieder.schmidt@microfocus.com>
#
# copyright (c) Novell Deutschland GmbH, 2001-2022. All rights reserved.
# GNU Public License
#
# zman_structure_create.sh				 8 Apr 2022
# last modified (extension for OES2023)			15 Nov 2022
# last modified (improved device folder handling
# 		 and update bundle assignment)		25 Apr 2023


ARGV=$@
POOL="No"
SCRIPT_NAME=$0

function debug()
{
	if [ "$1" != "0" ];then
		set -x
	fi
}


function usage()
{
        cat <<HERE

usage $0

	--parameter	| -p  <parameter file>	# file containing the parameters that describe
						# the desired ZCM zone structure
						# default: ../zman_params.cfg

	--output	| -o  <output file>	# file that will contain the zman commands that
						# will be used to create the desired ZCM zone
						# structure
						# default: zman_structure_create.in

	--assign	| -a			# Assign Update Bundle groups to Update Device groups
						# This is only required when using *zac* to deploy
						# updates which does not work with rpm 4.13 and later.
						# Not recommended when using btrfs. 
						# possible values: [y|Y|yes|Yes|YES]
						# default: Update Device groups will not be created
						# 	   Update Bundle groups will not be assigned
						#          to Update Device groups
						#          Registration Keys for Update Device groups
						#          will not be created

	--pool		| -P			# Creation of structures to include pool channels
						# possible values: [y|Y|yes|Yes|YES]
						# default: pool channels will NOT be included  

	--debug     	| -d 			# optional: enable debug


        example1: $0 --parameter ../zman_params_OES2013.cfg --output zman_structure_create.in.OES2023
        example2: $0 -p ../zman_params_OES2013.cfg -o zman_structure_create.in.OES2023 -P Yes -d

HERE
        exit 1
}


# retrieve cmd line parameters
function getopt()
{
	while [ $# -gt 0 ];do
		case $1 in --parameter|-p ) PARAM_FILE=$2; shift;;
			   --output|-o	  ) OUT_FILE=$2; shift;;
			   --assign|-a	  ) ASSIGN=$2; if grep -qi "Y"<<<${ASSIGN}; then ASSIGN="Y"; fi;
				  	    if [ ! "${ASSIGN}" = "Y" ]; then unknown_value $2; fi; shift;;
			   --pool|-P	  ) POOL=$2; if grep -qi "Y"<<<${POOL}; then POOL="Y"; fi; shift;;
			   --debug|-d	  ) debug=1; debug $debug;;
			   --help|-h	  ) clear; usage;;

			   *		  ) clear; echo -e "\n\n\t$0 $ARGV! \n\n\a\tUnknown parameter  > $1 <  provided!\n"; 
					    usage;;
		esac
		shift
	done
}


function unknown_value()
{
	clear
	echo -e "\n\n\t$0 $ARGV!"
	echo -e "\n\a\tUnknown value  > $1 <  provided!\n" 
	usage
}


function set_vars ()
{
	my_date=$(date +"%d.%m.%Y")
	my_time=$(date +"%H:%M")

	OUT_FILE_DEFAULT="zman_structure_create.in"
	PARAM_FILE_DEFAULT="../zman_params.cfg"

	OUT_FILE=${OUT_FILE:-${OUT_FILE_DEFAULT}}
        PARAM_FILE=${PARAM_FILE:-${PARAM_FILE_DEFAULT}}
}


function create_env ()
{
	# source parameter file
	source "${PARAM_FILE}"

	# compute sub-variables
	if [ -n  "${DEVICE_GROUP_FOLDER_CATEGORY}" ]; then
		  DEVICE_GROUP_CON_FOLDER="$(cut -d " " -f 1 <<<${DEVICE_GROUP_FOLDER_CATEGORY})"
		  DEVICE_GROUP_UPD_FOLDER="$(cut -d " " -f 2 <<<${DEVICE_GROUP_FOLDER_CATEGORY})"
	fi

	if [ -n  "${REG_KEY_FOLDERS}" ]; then
		  REG_KEY_CON_FOLDER="$(cut -d " " -f 1 <<<${REG_KEY_FOLDERS})"
		  REG_KEY_LOC_FOLDER="$(cut -d " " -f 2 <<<${REG_KEY_FOLDERS})"
		  REG_KEY_UPD_FOLDER="$(cut -d " " -f 3 <<<${REG_KEY_FOLDERS})"
	fi

	if [ -n  "${RELEASES}" ]; then
		  REG_KEY_RELEASES="$(sed 's/-//g' <<<${RELEASES})"
	fi	 

	if [ -n  "${DEVICE_FOLDERS}" ]; then
		  DEVICE_GROUP_FOLDER=$(cut -d " " -f 1 <<<$DEVICE_FOLDERS)
		  DEVICE_NOKEY_FOLDER=$(cut -d " " -f 2 <<<$DEVICE_FOLDERS)
		  DEVICE_SERVER_FOLDER=$(cut -d " " -f 3 <<<$DEVICE_FOLDERS)
	fi	 

	# create zman input file
	cat <<HERE > ${OUT_FILE}
## ******  ${OUT_FILE} created on $my_date at $my_time   *******
## -------------------------------------------------------------------------
HERE
}


function create_bundle_folders ()
{
	###  BUNDLES  ###

	# creating base folder for bundles
	if [ -n "${BASE_FOLDER}" ]; then
	     echo -e "\n## Creating Bundle Base Folder"	>> ${OUT_FILE}
	     echo "zman bfc ${BASE_FOLDER}"		>> ${OUT_FILE}
	fi

	# creating bundle folders
	if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" -o -n "${RELEASES}" ]; then
		echo -e "\n## Creating Bundle Folders" 	>> ${OUT_FILE}

		# configuration bundle folders
		if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
		     echo "zman bfc ${CONFIG_BUNDLE_BASE_FOLDER} ${BASE_FOLDER}" >> ${OUT_FILE}

		     echo -e "\n#    Configuration Bundle Folders" 	>> ${OUT_FILE}
		     for folder in ${CONFIG_BUNDLE_FOLDERS}; do
			  echo "zman bfc ${folder} ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}" >> ${OUT_FILE}
		     done
		fi

		# update bundle folders for SLES and OES Products
		first="Y"  ## formatting aid
		if [ -n "${RELEASES}" ]; then
			old_dist=""

			echo -e "\n#    Update Bundle Folders"  >> ${OUT_FILE}

			for rel in ${RELEASES}; do
			    dist=$(cut -d - -f 1 <<<$rel)
			    sp=$(cut -d - -f 2 <<<$rel)
			    if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then 
				  if [ -n "${first}" ]; then
					  echo "#    ${dist}" >> ${OUT_FILE}
			     		  first=""
				  else
					  echo -e "\n#    ${dist}" >> ${OUT_FILE}
				  fi
			    else
				  if [ -n "${first}" ]; then
					  echo "#    ${rel}" >> ${OUT_FILE}
			     		  first=""
				  else
					  echo -e "\n#    ${rel}" >> ${OUT_FILE}
				  fi
			    fi

			    if [ "${dist}" != "${old_dist}" ]; then
				  echo "zman bfc ${dist} ${BASE_FOLDER}" >> ${OUT_FILE}
			    fi
			    echo "zman bfc ${sp} ${BASE_FOLDER}/${dist}" >> ${OUT_FILE}

			    if [ "${dist}" == "SLES15" ]; then FOLDER_PREFIX="SLE-Product-"; else FOLDER_PREFIX=""; fi

			    if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then 
				echo "zman bfc ${FOLDER_PREFIX}${dist}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
				if [ "${POOL}" = "Y" ]; then
				     echo "zman bfc ${FOLDER_PREFIX}${dist}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
			   	fi
			    else
				echo "zman bfc ${FOLDER_PREFIX}${dist}-${sp}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
				if [ "${POOL}" = "Y" ]; then
				     echo "zman bfc ${FOLDER_PREFIX}${dist}-${sp}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
			    	fi
		    	    fi

			    # update bundle folders for SLE12 Modules
	                    if [ "${dist}" == "SLES12" -a -n "${SLE12_MODULES}" ]; then
				 echo -e "\n#    ${dist} Modules" 	>> ${OUT_FILE}
        	                 FOLDER_PREFIX="SLE-Module"
				 ver=${dist: -2:2}
				 for module in ${SLE12_MODULES};do
				     echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-Updates ${BASE_FOLDER}/${dist}" >> ${OUT_FILE}
				     if [ "${POOL}" = "Y" ]; then
					  echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-Pool ${BASE_FOLDER}/${dist}" >> ${OUT_FILE}
			     	     fi
			 	 done

				 if [ $((${sp:2:1}+0)) -ge 2 ]; then 
				      echo -e "\n#    ${dist} ${sp} Installer Updates" >> ${OUT_FILE}
				      FOLDER_PREFIX="SLES12-${sp}"
				      module="Installer"
				      echo "zman bfc ${FOLDER_PREFIX}-${module}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
				 fi
			    fi

			    # update bundle folders for SLE15 Modules
			    if [ "${dist}" == "SLES15" -a -n "${SLE15_MODULES}" ]; then
				 echo -e "\n#    ${dist} ${sp} Modules" 	>> ${OUT_FILE}
			         FOLDER_PREFIX="SLE-Module"
				 ver=${dist: -2:2}
				 for module in ${SLE15_MODULES};do
				     echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
				     if [ "${POOL}" = "Y" ]; then
					   echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-${sp}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
			     	     fi
			 	 done

				 echo -e "\n#    ${dist} ${sp} Installer Updates" >> ${OUT_FILE}
				 FOLDER_PREFIX="SLE15-${sp}"
				 module="Installer"
				 echo "zman bfc ${FOLDER_PREFIX}-${module}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
			    fi

			    # update bundle folders for SLES12 for OES2018
			    if [ -n "${OES2018_SLES_RELEASES}" ]; then
				  for oes_sles_rel in ${OES2018_SLES_RELEASES}; do
				      oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
			    	      sles_rel=$(cut -d - -f 3,4 <<<$oes_sles_rel)

				      if [ "$oes_rel" == "${dist}-${sp}" ]; then
					   echo -e "\n#    ${sles_rel} for ${oes_rel}" >> ${OUT_FILE}
					   echo "zman bfc ${oes_sles_rel}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
					   if [ "${POOL}" = "Y" ]; then
						 echo "zman bfc ${oes_sles_rel}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
					   fi

					   # update bundle folders for SLE12 Modules for OES2018
					   if [ -n "${OES2018_SLE12_MODULES}" ]; then
						echo -e "\n#    ${sles_rel} Modules for ${oes_rel}" >> ${OUT_FILE}
						sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)
						ver=${sles_ver: -2:2}
						FOLDER_PREFIX="${oes_rel}-SLE-Module"
						for module in ${OES2018_SLE12_MODULES};do
						    echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
						    if [ "${POOL}" = "Y" ]; then
				    			  echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
						    fi
						done
					   fi
		     		      fi		   
				  done
			    fi

			    # update bundle folders for SLES15 for OES2023
			    if [ -n "${OES2023_SLES_RELEASES}" ]; then
				 for oes_sles_rel in ${OES2023_SLES_RELEASES}; do
				     oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
				     sles_rel=$(cut -d - -f 3,4 <<<$oes_sles_rel)

				     if [ "$oes_rel" == "${dist}-${sp}" ]; then
					   if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then
						 FOLDER_PREFIX="${dist}-SLE-Product"
						 oes_rel=$(cut -d - -f 1 <<<$oes_sles_rel)
					   else
						 FOLDER_PREFIX="${dist}-${sp}-SLE-Product"
						 oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
					   fi

					   echo -e "\n#    ${sles_rel} for ${oes_rel}" >> ${OUT_FILE}
				 	   echo "zman bfc ${FOLDER_PREFIX}-${sles_rel}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
					   if [ "${POOL}" = "Y" ]; then
				 		 echo "zman bfc ${FOLDER_PREFIX}-${sles_rel}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
					   fi

					   # update bundle folders for SLE15 Modules for OES2023
					   if [ -n "${OES2023_SLE15_MODULES}" ]; then
				 		echo -e "\n#    ${sles_rel} Modules for ${oes_rel}" >> ${OUT_FILE}
						sles_sp=$(cut -d - -f 4 <<<$oes_sles_rel)
						sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)
						ver=${sles_ver: -2:2}

						for module in ${OES2023_SLE15_MODULES};do
						    if [ "${sp}" = "FCS" ]; then
							 FOLDER_PREFIX="${dist}-SLE-Module"
						    else
							 FOLDER_PREFIX="${dist}-${sp}-SLE-Module"
						    fi

						    echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Updates ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
						    if [ "${POOL}" = "Y" ]; then
							 echo "zman bfc ${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Pool ${BASE_FOLDER}/${dist}/${sp}" >> ${OUT_FILE}
						    fi
				   		done
			    		   fi
				     fi		   
				 done  ## oes_sles_rel in ${OES2023_SLES_RELEASES}	
			    fi
			    old_dist=${dist}

		        done ##  rel in ${RELEASES} 
		fi ##  [ -n "${RELEASES}" ]
	fi  ## [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" -o -n "${RELEASES}" ]
}


function create_bundle_groups ()
{
	# creating bundle groups
	if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}"  -o -n "${INFRA_SERVER_FOLDER}"  -o -n  "${DEVICE_ENVIRONMENTS}" ]; then
	echo -e "\n## Creating Bundle Groups" >> ${OUT_FILE}

	# configuration bundle groups
	if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
	     for env in ${DEVICE_ENVIRONMENTS}; do
		echo -e "\n#    Bundle Groups for Configuration Bundles in ${env}" >> ${OUT_FILE}
		for class in ${SERVER_CLASS}; do
		    echo "zman bgc ${env}_${class} ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}" >> ${OUT_FILE}
		done
	     done
	fi

	# installer update bundle groups
	for rel in ${RELEASES}; do
	    old_dist=""
            dist=$(cut -d - -f 1 <<<$rel)
            sp=$(cut -d - -f 2 <<<$rel)

            if [ "${dist}" != "${old_dist}" ]; then

		  # SLE12 Modules
		  if [ ${dist} == "SLES12" -a -n "${SLE12_MODULES}" ]; then
		       if [ $((${sp:2:1}+0)) -ge 2 ]; then 
			    FOLDER_PREFIX="SLES12-${sp}"
			    module="Installer"
			    echo -e "\n#    Bundle Groups for ${dist} Installer Updates" >> ${OUT_FILE}
			    echo "zman bgc ${FOLDER_PREFIX}-${module}-Updates ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}-Updates" >> ${OUT_FILE}
	 	       fi
		  fi

		  # SLE15 Modules
		  if [ ${dist} == "SLES15" -a -n "${SLE15_MODULES}" ]; then
		       FOLDER_PREFIX="SLE15-${sp}"
		       module="Installer"
		       echo -e "\n#    Bundle Groups for ${dist} Installer Updates" >> ${OUT_FILE}
		       echo "zman bgc ${FOLDER_PREFIX}-${module}-Updates ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}-Updates" >> ${OUT_FILE}
		  fi
	    fi
	done

	# update bundle groups
	for env in ${DEVICE_ENVIRONMENTS}; do
            for rel in ${RELEASES}; do
		echo -e "\n#    Bundle Groups for ${rel} Update Bundles in ${env}" >> ${OUT_FILE}
		old_dist=""
        	dist=$(cut -d - -f 1 <<<$rel)
        	sp=$(cut -d - -f 2 <<<$rel)

        	if [ "${dist}" != "${old_dist}" ]; then
	    	    if [ "${dist}" == "SLES15" ]; then FOLDER_PREFIX="SLE-Product-"; else FOLDER_PREFIX=""; fi

		    if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then
			 echo "zman bgc ${FOLDER_PREFIX}${dist}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-Updates" >> ${OUT_FILE}
		    else	    
			 echo "zman bgc ${FOLDER_PREFIX}${dist}-${sp}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-${sp}-Updates" >> ${OUT_FILE}
		    fi

    		    # SLE12 Modules
		    if [ ${dist} == "SLES12" -a -n "${SLE12_MODULES}" ]; then
			FOLDER_PREFIX="SLE-Module"
			ver=${dist: -2:2}
			for module in ${SLE12_MODULES};do
			    echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-Updates-${env} ${BASE_FOLDER}/${dist}/${FOLDER_PREFIX}-${module}${ver}-Updates" >> ${OUT_FILE}
	 		done
		    fi

    		    # SLE15 Modules
		    if [ ${dist} == "SLES15" -a -n "${SLE15_MODULES}" ]; then
			FOLDER_PREFIX="SLE-Module"
			ver=${dist: -2:2}
			for module in ${SLE15_MODULES};do
			    echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates" >> ${OUT_FILE}
	 		done
		    fi

		    # SLES12 for OES2018
	    	    if [ "${dist}" == "OES2018" ]; then
			 for oes_sles_rel in ${OES2018_SLES_RELEASES}; do
			     oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
			     if [ "$oes_rel" == "${dist}-${sp}" ]; then
		    	           echo "zman bgc ${oes_sles_rel}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${oes_sles_rel}-Updates" >> ${OUT_FILE}
			     fi
		         done	
			     
			 # SLE12 Modules for OES2018
			 if [ -n "${OES2018_SLE12_MODULES}" ]; then
				 FOLDER_PREFIX="${dist}-${sp}-SLE-Module"
				 sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)
				 ver=${sles_ver: -2:2}

				 for module in ${OES2018_SLE12_MODULES}; do
				     echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-Updates" >> ${OUT_FILE}
	 	     		 done
			  fi
		    fi  ## [ "${dist}" == "OES2018" ]


		    # SLES15 for OES2023
	    	    if [ "${dist}" == "OES2023" ]; then
			 for oes_sles_rel in ${OES2023_SLES_RELEASES}; do
			     oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
			     sles_rel=$(cut -d - -f 3,4 <<<$oes_sles_rel)
			     sles_sp=$(cut -d - -f 4 <<<$oes_sles_rel)
			     sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)

			     if [ "$oes_rel" == "${dist}-${sp}" ]; then
			 	   if grep -q "FCS" <<<${oes_sles_rel}; then oes_sles_rel=$(sed 's/FCS-//g' <<<${oes_sles_rel}); fi
				   if [ "${sp}" = "FCS" ]; then  
					 FOLDER_PREFIX="${dist}-SLE-Product"
			           else
				   	 FOLDER_PREFIX="${dist}-${sp}-SLE-Product"
				   fi

				   echo "zman bgc ${FOLDER_PREFIX}-${sles_rel}-Updates-${env} ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${sles_ver}-${sles_sp}-Updates" >> ${OUT_FILE}

				   # SLE15 Modules for OES2023
				   if [ -n "${OES2023_SLE15_MODULES}" ]; then
					   if [ "${sp}" = "FCS" ]; then
						FOLDER_PREFIX="${dist}-SLE-Module"
					   else
						FOLDER_PREFIX="${dist}-${sp}-SLE-Module"
					   fi

					   ver=${sles_ver: -2:2}

					   for module in ${OES2023_SLE15_MODULES}; do
					       echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Updates-${env} \
						       	      ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Updates" >> ${OUT_FILE}
					   done
				   fi
			     fi
		         done  ## oes_sles_rel in ${OES2023_SLES_RELEASES}	
		    fi  ## [ "${dist}" == "OES2023" ]

		fi  ## [ "${dist}" != "${old_dist}" ]
	    done  ## rel in ${RELEASES}
	done  ## env in ${DEVICE_ENVIRONMENTS}


	# common infrastructure servers group (optional)
	if [ -n "${INFRA_SERVER_FOLDER}" ]; then
	     	
	    # configuration bundle group
	    if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
		 echo -e "\n#    Bundle Groups for Configuration Bundles for Infrastructure Servers" >> ${OUT_FILE}
		 echo "zman bgc ${INFRA_SERVER_FOLDER} ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}" >> ${OUT_FILE}
	    fi

	    # update bundle groups
	    for rel in ${INFRA_SERVER_OS}; do
        	dist=$(cut -d - -f 1 <<<$rel)
        	sp=$(cut -d - -f 2 <<<$rel)
	    	if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then 
		      echo -e "\n#    Bundle Groups for ${dist} Update Bundles for Infrastructure Servers" >> ${OUT_FILE}
		else
		      echo -e "\n#    Bundle Groups for ${rel} Update Bundles for Infrastructure Servers" >> ${OUT_FILE}
		fi

		if [ "${dist}" == "SLES15" ]; then FOLDER_PREFIX="SLE-Product-"; else FOLDER_PREFIX=""; fi
		echo "zman bgc ${FOLDER_PREFIX}${dist}-${sp}-Updates-INFRA ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-${sp}-Updates" >> ${OUT_FILE}

   		# SLE12 Modules
		if [ ${dist} == "SLES12" -a -n "${INFRA_SERVER_SLE12_MODULES}" ]; then
		     FOLDER_PREFIX="SLE-Module"
		     ver=${dist: -2:2}
		     for module in ${INFRA_SERVER_SLE12_MODULES};do
			 echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-Updates-INFRA ${BASE_FOLDER}/${dist}/${FOLDER_PREFIX}-${module}${ver}-Updates" >> ${OUT_FILE}
		     done
		fi

		# SLE15 Modules
		if [ ${dist} == "SLES15" -a -n "${INFRA_SERVER_SLE15_MODULES}" ]; then
		     FOLDER_PREFIX="SLE-Module"
		     ver=${dist: -2:2}
		     for module in ${INFRA_SERVER_SLE15_MODULES};do
			 echo "zman bgc ${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates-INFRA ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates" >> ${OUT_FILE}
		     done
		fi
	    done
	fi
	fi
}	


function create_server_groups ()
{	
	###  DEVICES  ###
	if [ -n "${DEVICE_ENVIRONMENTS}" ]; then

		# creating base folder for devices
		if [ -n "${BASE_FOLDER}" ]; then
	     	     echo -e "\n## Creating Device Base Folders"	>> ${OUT_FILE}
		     echo "zman sfc ${BASE_FOLDER}"			>> ${OUT_FILE}
		fi

		# creating device folders
	     	echo ""	>> ${OUT_FILE}
		if [ -n "${DEVICE_FOLDERS}" ]; then
		      for folder in ${DEVICE_FOLDERS}; do echo "zman sfc ${folder} ${BASE_FOLDER}" >> ${OUT_FILE}; done
		fi      

		# adjust environments for common infrastructure servers group folder (optional)
		if [ -n "${INFRA_SERVER_FOLDER}" ]; then
		     OLD_DEVICE_ENVIRONMENTS="$DEVICE_ENVIRONMENTS"
		     DEVICE_ENVIRONMENTS="${INFRA_SERVER_FOLDER} ${DEVICE_ENVIRONMENTS}"
		fi

		for folder in $(sed "s/${DEVICE_NOKEY_FOLDER}\ //g" <<<${DEVICE_FOLDERS}); do
	     	    echo ""	>> ${OUT_FILE}
		    for env in ${DEVICE_ENVIRONMENTS}; do
			echo "zman sfc ${env} ${BASE_FOLDER}/${folder}" >> ${OUT_FILE}
		    done
		done

		# group folders
		echo -e "\n#    Creating Folders for Device Groups" >> ${OUT_FILE}
		for env in ${DEVICE_ENVIRONMENTS}; do
		    for type in ${DEVICE_GROUP_FOLDER_CATEGORY}; do
			echo "zman sfc ${type} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}" >> ${OUT_FILE}
		    done
		done

		# restore environments (optional)
		if [ -n "${OLD_DEVICE_ENVIRONMENTS}" ]; then
		     DEVICE_ENVIRONMENTS="$OLD_DEVICE_ENVIRONMENTS"
		     OLD_DEVICE_ENVIRONMENTS=""
		fi

		# server folders (optional)
		echo -e "\n#    Creating Folders for Devices" >> ${OUT_FILE}
		if [ -n "${SERVER_FOLDER}" ]; then
		    for env in ${DEVICE_ENVIRONMENTS}; do
			for class in ${SERVER_FOLDER}; do
			    echo "zman sfc ${class} ${BASE_FOLDER}/${DEVICE_SERVER_FOLDER}/${env}" >> ${OUT_FILE}
			done
		    done
		fi


		# creating device groups
		echo -e "\n## Creating Device Groups" >> ${OUT_FILE}

		# config groups
		if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
			first="Y"  ## formatting aid
			for env in ${DEVICE_ENVIRONMENTS}; do
			    if [ -n "${first}" ]; then
				 echo "#    Device Groups to assign Configuration Bundles in ${env}" >> ${OUT_FILE}
				 first=""
			    else
				 echo -e "\n#    Device Groups to assign Configuration Bundles in ${env}" >> ${OUT_FILE}
			    fi

			    for class in ${SERVER_CLASS}; do
				echo "zman sgc ${env}_${class} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_CON_FOLDER}" >> ${OUT_FILE}
			    done
			done
		fi

		# update groups
		if [ "${ASSIGN}" = "Y" ]; then
		     for env in ${DEVICE_ENVIRONMENTS}; do
			 echo -e "\n#    Device Groups to assign Update Bundles in ${env}" >> ${OUT_FILE}
			 for rel in ${RELEASES}; do
			     dist=$(cut -d - -f 1 <<<$rel)
			     sp=$(cut -d - -f 2 <<<$rel)

			     if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then
				  echo "zman sgc ${env}_${dist} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}" >> ${OUT_FILE}
			     else
				  echo "zman sgc ${env}_${dist}${sp} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}" >> ${OUT_FILE}
			     fi
    			 done
 		     done
		fi

		# common infrastructure servers group (optional)
		if [ -n "${INFRA_SERVER_FOLDER}" ]; then
		
		    # config groups
		    if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
			 echo -e "\n#    Device Group to assign Configuration Bundles to Infrastructure Servers" >> ${OUT_FILE}
		         echo "zman sgc ${INFRA_SERVER_FOLDER} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_CON_FOLDER}" >> ${OUT_FILE}
		    fi

		    # update groups
		    if [ "${ASSIGN}" = "Y" ]; then
			 echo -e "\n#    Device Groups to assign Update Bundles to Infrastructure Servers" >> ${OUT_FILE}
			 for rel in $(sed 's/-//g' <<<${INFRA_SERVER_OS}); do
			     echo "zman sgc _INFRA_${rel} ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_UPD_FOLDER}" >> ${OUT_FILE}
			 done
		    fi
		fi
	fi
}


function assign_bundle_groups ()
{
	###  BUNDLE GROUP ASSIGNMENTS  ###
	if [ -n "${DEVICE_ENVIRONMENTS}" -o -n "${RELEASES}" -o -n "${INFRA_SERVER_FOLDER}" ]; then
		echo -e "\n## Assigning Bundle Groups to Device Groups" >> ${OUT_FILE}
	fi

	if [ -n "${DEVICE_ENVIRONMENTS}" ]; then
	     # configuration bundle groups
	     if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
		     for env in ${DEVICE_ENVIRONMENTS}; do
			 echo -e "\n#    Assign Configuration Bundle Groups in ${env}" >> ${OUT_FILE}
			 for class in ${SERVER_CLASS}; do
			     echo "zman ba device ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${env}_${class} \
						    Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_CON_FOLDER}/${env}_${class} -r -i" >> ${OUT_FILE}
			 done
		     done
     	     fi

	     # update bundle groups
	     if [ "${ASSIGN}" = "Y" ]; then  
		  for env in ${DEVICE_ENVIRONMENTS}; do
		      old_rel=""
		      for rel in ${RELEASES}; do
			  echo -e "\n#    Assign Update Bundle Groups for ${rel} in ${env}" >> ${OUT_FILE}
			  dist=$(cut -d - -f 1 <<<$rel)
			  sp=$(cut -d - -f 2 <<<$rel)

			  if [ "${dist}" == "SLES15" ]; then FOLDER_PREFIX="SLE-Product-"; else FOLDER_PREFIX=""; fi

			  if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then
				target="${dist}"
				echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-Updates/${FOLDER_PREFIX}${dist}-Updates-${env} \
	     					       Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
			  else		     
			  	target="${dist}${sp}"
			   	echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-${sp}-Updates/${FOLDER_PREFIX}${dist}-${sp}-Updates-${env} \
	     					       Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
			  fi
		     

			  # SLE12 Modules
			  if [ ${dist} == "SLES12" -a -n "${SLE12_MODULES}" ]; then
			       FOLDER_PREFIX="SLE-Module"
			       ver=${dist: -2:2}
			       for module in ${SLE12_MODULES}; do
				   echo "zman ba device ${BASE_FOLDER}/${dist}/${FOLDER_PREFIX}-${module}${ver}-Updates/${FOLDER_PREFIX}-${module}${ver}-Updates-${env} \
	     						  Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
		    	       done
			  fi

			  # SLE15 Modules
			  if [ ${dist} == "SLES15" -a -n "${SLE15_MODULES}" ]; then
			       FOLDER_PREFIX="SLE-Module"
			       ver=${dist: -2:2}
			       for module in ${SLE15_MODULES};do
				   echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates-${env} \
	     						  Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
			       done
			  fi

			  # SLES12 for OES2018
			  if [ "${dist}" == "OES2018" ]; then
				for oes_sles_rel in ${OES2018_SLES_RELEASES}; do
				    oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
				    if [ "$oes_rel" == "${dist}-${sp}" ]; then
					  echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${oes_sles_rel}-Updates/${oes_sles_rel}-Updates-${env} \
	     	 						 Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
			  	    fi
			   	done

				# SLE12 Modules for OES2018
			   	if [ -n "${OES2018_SLE12_MODULES}" ]; then
				     FOLDER_PREFIX="${dist}-${sp}-SLE-Module"
				     sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)
				     ver=${sles_ver: -2:2}
				     for module in ${OES2018_SLE12_MODULES};do
	     	     			 echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-Updates/${FOLDER_PREFIX}-${module}${ver}-Updates-${env} \
	     	 		 				Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
				     done
				fi
			  fi

			  # SLES15 for OES2023
			  if [ "${dist}" == "OES2023" ]; then
			       if [ "${sp}" = "FCS" ]; then
				    FOLDER_PREFIX="${dist}-SLE-Product"
			       else
				    FOLDER_PREFIX="${dist}-${sp}-SLE-Product"
			       fi

			       for oes_sles_rel in ${OES2023_SLES_RELEASES}; do
				   oes_rel=$(cut -d - -f 1,2 <<<$oes_sles_rel)
				   sles_rel=$(cut -d - -f 3,4 <<<$oes_sles_rel)
				   sles_sp=$(cut -d - -f 4 <<<$oes_sles_rel)
				   sles_ver=$(cut -d - -f 3 <<<$oes_sles_rel)
				   ver=${sles_ver: -2:2}

				   if [ "$oes_rel" == "${dist}-${sp}" ]; then
					echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${sles_rel}-Updates/${FOLDER_PREFIX}-${sles_rel}-Updates-${env} \
							       Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${dist} -r" >> ${OUT_FILE}

					# SLE15 Modules for OES2023
					if [ -n "${OES2023_SLE15_MODULES}" ]; then
						if [ "${sp}" = "FCS" ]; then
						     FOLDER_PREFIX="${dist}-SLE-Module"
						else
						     FOLDER_PREFIX="${dist}-${sp}-SLE-Module"
						fi

						for module in ${OES2023_SLE15_MODULES};do
						    echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Updates/${FOLDER_PREFIX}-${module}${ver}-${sles_sp}-Updates-${env} \
	     	 							   Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${target} -r" >> ${OUT_FILE}
						done
					fi
				   fi  ## [ "$oes_rel" == "${dist}-${sp}

			       done  ## for oes_sles_rel in ${OES2023_SLES_RELEASES}
			  fi

		      done  ## for rel in ${RELEASES}
		  done  ## for env in ${DEVICE_ENVIRONMENTS}
	     fi  ## [ "${ASSIGN}" = "Y" ]
	fi  ## [ -n "${DEVICE_ENVIRONMENTS}" ]

	# common infrastructure servers group (optional)
	if [ -n "${INFRA_SERVER_FOLDER}" ]; then
		
	    # configuration bundle groups
	    if [ -n "${CONFIG_BUNDLE_BASE_FOLDER}" ]; then
		 echo -e "\n#    Assign Configuration Bundle Groups for Infrastructure Servers" >> ${OUT_FILE}
		 echo "zman ba device ${BASE_FOLDER}/${CONFIG_BUNDLE_BASE_FOLDER}/${CONFIG_BUNDLE_GROUPS_FOLDER}/${INFRA_SERVER_FOLDER} \
					Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_CON_FOLDER}/${INFRA_SERVER_FOLDER} -r -i" >> ${OUT_FILE}
	    fi

	    # update bundle groups
	    if [ "${ASSIGN}" = "Y" ]; then
		  for rel in ${INFRA_SERVER_OS}; do
		      echo -e "\n#    Assign Update Bundle Groups for ${rel} Infrastructure Servers" >> ${OUT_FILE}
		      dist=$(cut -d - -f 1 <<<$rel)
		      sp=$(cut -d - -f 2 <<<$rel)

		      if [ "${dist}" == "SLES15" ]; then FOLDER_PREFIX="SLE-Product-"; else FOLDER_PREFIX=""; fi

		      if [ "${sp}" = "FCS" -o "${sp}" = "GA" ]; then
			    target="${dist}"
		      else
	 		    target="${dist}${sp}"
		      fi

		      echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}${dist}-${sp}-Updates/${FOLDER_PREFIX}${dist}-${sp}-Updates-INFRA \
					     Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_UPD_FOLDER}/_INFRA_${target} -r" >> ${OUT_FILE}

		      # SLE12 Modules
		      if [ ${dist} == "SLES12" -a -n "${INFRA_SERVER_SLE12_MODULES}" ]; then
			   FOLDER_PREFIX="SLE-Module"
			   ver=${dist: -2:2}
			   for module in ${INFRA_SERVER_SLE12_MODULES};do
			       echo "zman ba device ${BASE_FOLDER}/${dist}/${FOLDER_PREFIX}-${module}${ver}-Updates/${FOLDER_PREFIX}-${module}${ver}-Updates-INFRA \
						      Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_UPD_FOLDER}/_INFRA_${target} -r" >> ${OUT_FILE}
			   done
		      fi

		      # SLE15 Modules
		      if [ ${dist} == "SLES15" -a -n "${INFRA_SERVER_SLE15_MODULES}" ]; then
			   FOLDER_PREFIX="SLE-Module"
			   ver=${dist: -2:2}
			   for module in ${INFRA_SERVER_SLE15_MODULES};do
			       echo "zman ba device ${BASE_FOLDER}/${dist}/${sp}/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates/${FOLDER_PREFIX}-${module}${ver}-${sp}-Updates-INFRA \
						      Servers/${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_UPD_FOLDER}/_INFRA_${target} -r" >> ${OUT_FILE}
			       done
		      fi
		  done  ## for rel in ${INFRA_SERVER_OS}
	    fi  ## [ "${ASSIGN}" = "Y" ]
	fi  ##  [ -n "${DEVICE_ENVIRONMENTS}" ]
}


function create_registration_keys ()
{
	###  REGISTRATION KEYS  ###
	if [ -n "${REG_KEY_FOLDERS}" ]; then
		# creating registration keys

		echo -e "\n## Creating Registration Key Folders" >> ${OUT_FILE}

		for folder in ${REG_KEY_FOLDERS}; do echo "zman rfc ${folder}" >> ${OUT_FILE}; done

		# Location keys
		echo -e "\n## Creating Registration Keys" >> ${OUT_FILE}
		first="Y"  ## formatting aid
		for env in ${DEVICE_ENVIRONMENTS}; do
		    if [ -n "${first}" ]; then
			 echo "#    Registration Keys for Device Folders in ${env}" >> ${OUT_FILE}
			 first=""
		    else
			 echo -e "\n#    Registration Keys for Device Folders in ${env}" >> ${OUT_FILE}
		    fi

		    for class in ${SERVER_FOLDER}; do
			echo "zman rck server ${env}_${class} ${REG_KEY_LOC_FOLDER} -f ${BASE_FOLDER}/${DEVICE_SERVER_FOLDER}/${env}/${class}" >> ${OUT_FILE}
		    done
		done

		# Config keys
		for env in ${DEVICE_ENVIRONMENTS}; do
		    echo -e "\n#    Registration Keys for Configuration Device Groups in ${env}" >> ${OUT_FILE}
		    for class in ${SERVER_CLASS}; do
			echo "zman rck server ${env}_${class}_GRP ${REG_KEY_CON_FOLDER} -f ${BASE_FOLDER}/${DEVICE_NOKEY_FOLDER}\
			       		      -g ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_CON_FOLDER}/${env}_${class}" >> ${OUT_FILE}
		    done
		done

		# Update keys
		if [ "${ASSIGN}" = "Y" ]; then
		      for env in ${DEVICE_ENVIRONMENTS}; do
			  echo -e "\n#    Registration Keys for Update Device Groups in ${env}" >> ${OUT_FILE}
			  for type in $(sed 's/-//g' <<<${RELEASES}); do
			      if egrep -q "FCS|GA" <<<${type}; then type=$(sed 's/FCS\|GA//g' <<<${type}); fi
			      echo "zman rck server ${env}_${type} ${REG_KEY_UPD_FOLDER} -f ${BASE_FOLDER}/${DEVICE_NOKEY_FOLDER}\
			      			    -g ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${env}/${DEVICE_GROUP_UPD_FOLDER}/${env}_${type}" >> ${OUT_FILE}
			  done
		      done
		fi

		# common infrastructure servers group (optional)
		if [ -n "${INFRA_SERVER_FOLDER}" ]; then
		     echo -e "\n#    Registration Key for Infrastructure Server Device Folder" >> ${OUT_FILE}
		     echo "zman rck server ${INFRA_SERVER_FOLDER} ${REG_KEY_LOC_FOLDER} -f ${BASE_FOLDER}/${DEVICE_SERVER_FOLDER}/${INFRA_SERVER_FOLDER}" >> ${OUT_FILE}
		     echo -e "\n#    Registration Key for Infrastructure Server Configuration Group" >> ${OUT_FILE}
		     echo "zman rck server ${INFRA_SERVER_FOLDER}_GRP ${REG_KEY_CON_FOLDER}\
			    		   -f ${BASE_FOLDER}/${DEVICE_NOKEY_FOLDER}\
					   -g ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_CON_FOLDER}/${INFRA_SERVER_FOLDER}" >> ${OUT_FILE}

		     # Update keys
		     if [ "${ASSIGN}" = "Y" ]; then
			  echo -e "\n#    Registration Keys for Infrastructure Server Update Groups" >> ${OUT_FILE}
			  for type in $(sed 's/-//g' <<<${INFRA_SERVER_OS}); do
			      echo "zman rck server _INFRA_${type} ${REG_KEY_UPD_FOLDER} -f ${BASE_FOLDER}/${DEVICE_NOKEY_FOLDER}\
						    -g ${BASE_FOLDER}/${DEVICE_GROUP_FOLDER}/${INFRA_SERVER_FOLDER}/${DEVICE_GROUP_UPD_FOLDER}/_INFRA_${type}" >> ${OUT_FILE}
			      done
		     fi
		fi
	fi
}


function create_subscription_folders ()
{	
	##  SUBSCRIPTION FOLDERS  ###
	if [ -n "${CUSTOMER_CENTERS}" ]; then
		echo -e "\n## Creating Subscription Folders" >> ${OUT_FILE}

		# subscription folders
		for cc in ${CUSTOMER_CENTERS}; do
            	    grep -qi "MicroFocus"<<<${cc}

            	if [ $? -eq 0 ]; then
                     folder="Micro Focus"
            	else
                     folder=${cc}
	    	fi

            	     echo "zman srfc \"$folder\"" >> ${OUT_FILE}
		     if [ "${POOL}" = "Y" ]; then
           		   echo "zman srfc RunOnceSubscriptions \"$folder\" --desc \"Subscriptions to mirror pool channels\"" >> ${OUT_FILE}
		     fi
		done
	fi
}


function main ()
{
	getopt  ${ARGV}
	set_vars
	create_env
	create_bundle_folders
	create_bundle_groups
	create_server_groups
	assign_bundle_groups
	create_registration_keys
	create_subscription_folders
}

main
exit $?
